use crate::core::models::{ConfigTelegram, MediaId, Message, Project, Version};
use crate::core::traits::{APISecret, Adapter, Builder, Publisher, SocialAPI, SocialBox};
use crate::db::functions::{insert::insert_message, read::get_message};
use crate::utils::gen_hash;
use anyhow::Result;
use async_trait::async_trait;
use chrono::prelude::*;
use regex::Regex;
use sea_orm::DatabaseConnection;
use serde::Deserialize;

/***************************/
/*-------------------------*/
/*--                     --*/
/*--       Models        --*/
/*--                     --*/
/*-------------------------*/
/***************************/
#[derive(Debug, Deserialize)]
pub struct TelegramMessage {
    pub ok: bool,
    pub result: TelegramMessageResult,
}

#[derive(Debug, Deserialize)]
pub struct TelegramMessageResult {
    pub message_id: u64,
    pub author_signature: String,
    pub date: i64,
    pub text: String,
}

/***************************/
/*-------------------------*/
/*--                     --*/
/*--        Types        --*/
/*--                     --*/
/*-------------------------*/
/***************************/
pub type TelegramBox<'a> = SocialBox<'a, ConfigTelegram>;
pub type TelegramAdapter<'a> = Adapter<'a, TelegramBox<'a>>;
impl APISecret for ConfigTelegram {}

/***************************/
/*-------------------------*/
/*--                     --*/
/*--     Social Box      --*/
/*--                     --*/
/*-------------------------*/
/***************************/
impl<'a> SocialAPI<'a> for TelegramBox<'a> {
    #[tokio::main]
    async fn send(&self, mut msg: String) -> Result<String> {
        msg.truncate(4096);
        let request_url = format!(
            "{}/bot{}/sendMessage?chat_id={}&text={}&parse_mode=HTML",
            self.get_api().base_url,
            self.get_api().secret,
            self.get_api().chat_id,
            msg
        );
        let response = self.get_client().post(&request_url).send().await?;
        let state_data: String = response.text().await?;
        Ok(state_data)
    }

    #[tokio::main]
    async fn recv(&self) -> Result<String> {
        let request_url = format!(
            "{}/bot{}/getUpdates?offset={}",
            self.get_api().base_url,
            self.get_api().secret,
            -1
        );
        let response = self.get_client().get(&request_url).send().await?;
        let message: String = response.text().await?;
        Ok(message)
    }

    #[tokio::main]
    async fn delete(&self, msgid: &str) -> Result<String> {
        let request_url = format!(
            "{}/bot{}/deleteMessage?chat_id={}&message_id={}",
            self.get_api().base_url,
            self.get_api().secret,
            self.get_api().chat_id,
            msgid
        );
        let response = self.get_client().get(&request_url).send().await?;
        let state_data: String = response.text().await?;
        Ok(state_data)
    }
}

/***************************/
/*-------------------------*/
/*--                     --*/
/*--      Builders       --*/
/*--                     --*/
/*-------------------------*/
/***************************/
impl<'a> Builder for TelegramBox<'a> {
    fn build(proj: &Project, version: &Version) -> String {
        version
            .data
            .lines()
            .map(|line| telegram_post_data(proj, version, line))
            .collect::<String>()
    }
}

fn telegram_post_data(proj: &Project, version: &Version, line: &str) -> String {
    let h1_re = Regex::new(r"^##\s").unwrap();
    let h2_re = Regex::new(r"^###\s").unwrap();
    let summary_re = Regex::new(r"\]:\s.+").unwrap();
    let commit_re = Regex::new(r"-\s\*(.+):\*\s([^\[\(]+)").unwrap();

    if h1_re.is_match(line) {
        let date = Utc.timestamp(version.timestamp, 0);
        format!(
            "<b>{} <u>{} v{}</u>%0A🎉 release {}-{}-{}%0A</b>",
            proj.emoji,
            proj.name,
            version.name,
            date.year(),
            date.month(),
            date.day(),
        )
    } else if h2_re.is_match(line) {
        format!("<b><u>{}</u></b>", line.trim_start_matches("### "))
    } else if summary_re.is_match(line) {
        let site_url = format!(
            "<a href=\"https://n3n.org/news/{}-{}-{}-{}\">here</a>",
            proj.name, version.major, version.minor, version.patch
        );
        format!(
            "🤩 star repo at {}%0A📜 full changelog {}%0A",
            proj.url, site_url
        )
    } else if commit_re.is_match(line) {
        let commit_cap = commit_re.captures(&line).unwrap();
        format!("- <b>{}</b> {}%0A", &commit_cap[1], &commit_cap[2])
    } else {
        format!("{}%0A", line)
    }
}

/***************************/
/*-------------------------*/
/*--                     --*/
/*--      Adapters       --*/
/*--                     --*/
/*-------------------------*/
/***************************/
#[async_trait]
impl<'a> Publisher<'a> for TelegramAdapter<'a> {
    async fn publish(&self) -> Result<()> {
        let stored_message =
            get_message(&self.get_conn(), MediaId::Telegram, &self.get_version()).await;
        if stored_message.is_err() {
            telegram_publish(
                &self.get_conn(),
                &self.get_client(),
                &self.get_project(),
                &self.get_version(),
            )
            .await?;
        } else {
            let stored_message = stored_message?;
            // let message_resp = self.get_client().recv()?;
            // if message_resp.find("Unknown Message").is_some() {
            // delete_message(&self.get_conn(), MediaId::Telegram, &stored_message)?;
            // telegram_publish(
            // &self.get_conn(),
            // &self.get_client(),
            // &self.get_project(),
            // &self.get_version(),
            // )?;
            // } else {
            // let got_message: TelegramMessage = serde_json::from_str(&message_resp)?;
            let got_message_hash = &self.get_version().hash;
            if &stored_message.version_hash != got_message_hash {
                telegram_publish(
                    &self.get_conn(),
                    &self.get_client(),
                    &self.get_project(),
                    &self.get_version(),
                )
                .await?;
            }
            // println!("{:#?}", stored_message);
        }
        Ok(())
    }
}

async fn telegram_publish<'e>(
    conn: &DatabaseConnection,
    client: &TelegramBox<'e>,
    project: &Project,
    version: &Version,
) -> Result<()> {
    let telegram_response = client.send(TelegramBox::build(project, version))?;
    let message: TelegramMessage = serde_json::from_str(&telegram_response)?;
    let sent_message = Message {
        media_id: MediaId::Telegram,
        message_hash: gen_hash(&message.result.text),
        message_id: message.result.message_id.to_string(),
        project_name: project.name.clone(),
        timestamp: message.result.date,
        version_hash: version.hash.clone(),
    };
    insert_message(conn, &sent_message).await?;
    Ok(())
}
