use crate::core::models::{MediaId, Message, Project, Version};
use anyhow::{bail, Result};
use rusqlite::{params, Connection};

pub fn get_id<T: rusqlite::types::ToSql, S: rusqlite::types::FromSql>(
    conn: &Connection,
    query: &str,
    object: &T,
) -> Result<S> {
    let mut stmt = conn.prepare(query)?;
    let mut id = stmt.query_and_then([&object], |row| row.get::<_, S>(0))?;
    if let Some(id) = id.next() {
        Ok(id?)
    } else {
        bail!("no unique id found inside database for query `{}`", query)
    }
}

pub fn get_branch(conn: &Connection, proj_id: usize) -> Result<String> {
    let mut stmt = conn.prepare("SELECT name FROM branch where project_id = ?")?;
    let mut id = stmt.query_and_then([proj_id], |row| row.get::<_, String>(0))?;
    if let Some(id) = id.next() {
        Ok(id?)
    } else {
        bail!("no matching branch found with project_id: {}", proj_id)
    }
}

pub fn get_all_tags(conn: &Connection, proj_id: usize) -> Result<Vec<String>> {
    let mut stmt = conn.prepare("SELECT DISTINCT name FROM tag WHERE project_id = ?")?;
    let tags = stmt.query_and_then([proj_id], |row| row.get::<_, String>(0))?;
    Ok(tags
        .filter_map(|tag| match tag {
            Ok(t) => Some(t),
            Err(_) => None,
        })
        .collect())
}

pub fn get_all_project(conn: &Connection) -> Result<Vec<Project>> {
    let mut stmt = conn.prepare(
        "SELECT id, name, description, url, emoji, is_featured, is_template FROM project",
    )?;
    let all_projects = stmt.query_map([], |row| {
        Ok(Project {
            name: row.get(1)?,
            desc: row.get(2)?,
            url: row.get(3)?,
            emoji: row.get(4)?,
            featured: row.get(5)?,
            template: row.get(6)?,
            branch: get_branch(&conn, row.get(0)?).unwrap(),
            tags: get_all_tags(&conn, row.get(0)?).unwrap(),
        })
    })?;

    Ok(all_projects
        .filter_map(|proj| match proj {
            Ok(p) => Some(p),
            Err(_) => None,
        })
        .collect())
}

pub fn get_changelog_hash(conn: &Connection, project: &Project) -> Result<String> {
    let hash_id: String = get_id(
        &conn,
        "SELECT hash FROM changelog WHERE project_id = (SELECT id FROM project WHERE name = ?)",
        &project.name,
    )?;
    Ok(hash_id)
}

#[allow(dead_code)]
pub fn get_version_hashlist(conn: &Connection, project: &Project) -> Result<Vec<String>> {
    let mut stmt = conn.prepare(
        "SELECT hash FROM version WHERE project_id = (SELECT id FROM project WHERE name = ?)",
    )?;
    let all_version_hash = stmt.query_and_then([&project.name], |row| row.get::<_, String>(0))?;
    Ok(all_version_hash
        .filter_map(|data| match data {
            Ok(d) => Some(d),
            Err(_) => None,
        })
        .collect::<Vec<String>>())
}

#[allow(dead_code)]
pub fn get_changelog_prevhash(conn: &Connection, project: &Project) -> Result<String> {
    let hash_id: String = get_id(
        &conn,
        "SELECT prev_hash FROM changelog WHERE project_id = (SELECT id FROM project WHERE name = ?)",
        &project.name,
    )?;
    Ok(hash_id)
}

pub fn get_hash_data(conn: &Connection, hash_id: String) -> Result<String> {
    let mut stmt = conn.prepare("SELECT data FROM hash WHERE id = ?")?;
    let hash_data = stmt.query_and_then([hash_id], |row| row.get::<_, String>(0))?;
    Ok(hash_data
        .filter_map(|data| match data {
            Ok(d) => Some(d),
            Err(_) => None,
        })
        .collect())
}

#[allow(dead_code)]
pub fn get_version(conn: &Connection, project: &Project) -> Result<Vec<Version>> {
    let mut stmt = conn.prepare(
        "SELECT hash, timestamp, name, major, minor, patch, codename, start_commit, end_commit 
        FROM version 
        WHERE project_id 
        IN (SELECT id 
            FROM project 
            WHERE name = ?)",
    )?;
    let all_versions = stmt.query_map([&project.name], |row| {
        Ok(Version {
            project_name: project.name.to_string(),
            hash: row.get(0)?,
            data: get_hash_data(&conn, row.get(0)?).unwrap(),
            timestamp: row.get(1)?,
            name: row.get(2)?,
            major: row.get(3)?,
            minor: row.get(4)?,
            patch: row.get(5)?,
            codename: row.get(6)?,
            start_commit: row.get(7)?,
            end_commit: row.get(8)?,
        })
    })?;

    Ok(all_versions
        .filter_map(|proj| match proj {
            Ok(p) => Some(p),
            Err(_) => None,
        })
        .collect())
}

pub fn get_message(conn: &Connection, media: MediaId, version: &Version) -> Result<Message> {
    let platform_name: &str = media.into();

    let mut stmt = conn.prepare("SELECT timestamp, message_id, message_hash FROM message WHERE social_media_id = (SELECT id FROM social_media WHERE platform_name = ?1) AND version_hash = ?2")?;

    let mut message = stmt.query_map(params![platform_name, version.hash], |row| {
        Ok(Message {
            media_id: media.clone(),
            project_name: version.project_name.clone(),
            version_hash: version.hash.clone(),
            timestamp: row.get(0)?,
            message_id: row.get(1)?,
            message_hash: row.get(2)?,
        })
    })?;

    Ok(if let Some(msg) = message.next() {
        msg?
    } else {
        bail!(
            "no message id found for project `{}` `{}` published on `{}`",
            version.project_name,
            version.name,
            platform_name
        )
    })
}
