use crate::core::models::{Changelog, Message, Project, Version};
use anyhow::Result;
use rusqlite::{params, Connection};

pub fn create_project(conn: &Connection, proj: &Project) -> Result<()> {
    // Add new project
    conn.execute("INSERT OR IGNORE INTO project (name, description, url, emoji, is_featured, is_template) VALUES (?1, ?2, ?3, ?4, ?5, ?6)", params![proj.name, proj.desc, proj.url, proj.emoji, proj.featured as u8, proj.template as u8])?;

    // Add new tag
    for t in proj.tags.iter() {
        conn.execute(
            "INSERT OR IGNORE INTO tag (name, project_id) VALUES (?1, (SELECT id FROM project WHERE name = ?2))",
            params![t, proj.name],
        )?;
    }

    // Add new branch
    conn.execute(
        "INSERT OR IGNORE INTO branch (name, project_id) VALUES (?1, (SELECT id FROM project WHERE name = ?2))",
        params![proj.branch, proj.name],
    )?;

    Ok(())
}

pub fn create_changelog(conn: &Connection, changelog: &Changelog) -> Result<()> {
    // Add hash data
    conn.execute(
        "INSERT OR IGNORE INTO hash (id, data) VALUES (?1, ?2)",
        params![changelog.hash, changelog.data],
    )?;

    // Add new changelog
    conn.execute(
            "INSERT OR IGNORE INTO changelog (project_id, hash, prev_hash) VALUES ((SELECT id FROM project WHERE name = ?1), ?2, ?3)",
            params![changelog.project_name, changelog.hash, String::new()],
    )?;

    Ok(())
}

pub fn create_version(conn: &Connection, version: &Version) -> Result<()> {
    // Add hash data
    conn.execute(
        "INSERT OR IGNORE INTO hash (id, data) VALUES (?1, ?2)",
        params![version.hash, version.data],
    )?;

    conn.execute(
        "INSERT OR IGNORE INTO version (project_id, hash, timestamp, name, major, minor, patch, codename, start_commit, end_commit) VALUES ((SELECT id FROM project WHERE name = ?1), ?2, ?3, ?4, ?5, ?6, ?7, ?8, ?9, ?10)",
        params![version.project_name, version.hash, version.timestamp, version.name, version.major, version.minor, version.patch, version.codename, version.start_commit, version.end_commit]
    )?;
    Ok(())
}

pub fn create_message(conn: &Connection, message: &Message) -> Result<()> {
    let platform_name: &str = message.media_id.into();

    conn.execute("INSERT OR IGNORE INTO message (social_media_id, project_id, version_hash, timestamp, message_id, message_hash) VALUES ((SELECT id FROM social_media WHERE platform_name = ?1), (SELECT id FROM project WHERE name = ?2), ?3, ?4, ?5, ?6)",params![platform_name, message.project_name, message.version_hash, message.timestamp, message.message_id, message.message_hash])?;
    Ok(())
}
