use crate::consts::DEFAULT_DB_PATH;
use crate::core::models::{Config, Project, Version};
use anyhow::Result;
use reqwest::Client;
use rusqlite::Connection;

pub trait APISecret {}

pub struct Adapter<'a, T: SocialAPI<'a>> {
    conn: Connection,
    client: &'a T,
    project: &'a Project,
    version: &'a Version,
}

impl<'a, T: SocialAPI<'a>> Adapter<'a, T> {
    pub fn get_conn(&self) -> &Connection {
        &self.conn
    }
    pub fn get_client(&self) -> &'a T {
        &self.client
    }
    pub fn get_project(&self) -> &'a Project {
        &self.project
    }
    pub fn get_version(&self) -> &'a Version {
        &self.version
    }
}

pub trait Publisher<'a> {
    fn new<T: SocialAPI<'a>>(
        config: &Config,
        client: &'a T,
        project: &'a Project,
        version: &'a Version,
    ) -> Result<Adapter<'a, T>> {
        let db_path = config.db_path.as_ref().unwrap_or(&DEFAULT_DB_PATH);
        Ok(Adapter {
            conn: Connection::open(&db_path)?,
            client,
            project,
            version,
        })
    }
    fn publish(&self) -> Result<()>;
}

pub trait Builder {
    fn build(proj: &Project, version: &Version) -> String;
}

pub struct SocialBox<'a, T: APISecret> {
    client: Client,
    api: &'a T,
}

impl<'a, T: APISecret> SocialBox<'a, T> {
    pub fn get_client(&self) -> &Client {
        &self.client
    }
    pub fn get_api(&self) -> &'a T {
        &self.api
    }
}

pub trait SocialAPI<'a> {
    fn new<T: 'a + APISecret>(client: Client, api: &T) -> SocialBox<T> {
        SocialBox { client, api }
    }

    #[tokio::main]
    async fn send(&self, msg: String) -> Result<String> {
        drop(msg);
        unimplemented!()
    }

    #[tokio::main]
    async fn recv(&self) -> Result<String> {
        unimplemented!()
    }
    #[tokio::main]
    async fn get(&self, msgid: &str) -> Result<String> {
        drop(msgid);
        unimplemented!()
    }

    #[tokio::main]
    async fn delete(&self, msgid: &str) -> Result<String> {
        drop(msgid);
        unimplemented!()
    }
}
