use super::log::N3NBox;
use super::models::{Config, Project};
use super::traits::{Publisher, SocialAPI};
use crate::api::discord::{DiscordAdapter, DiscordBox};
use crate::api::telegram::{TelegramAdapter, TelegramBox};
use crate::api::twitter::{TwitterAdapter, TwitterBox};
use crate::consts::{DEFAULT_DB_PATH, DEFAULT_SCAN_PERIOD};
use crate::db::functions::create::{create_changelog, create_project, create_version};
use crate::db::functions::delete::delete_project;
use crate::db::functions::select::{get_all_project, get_changelog_hash, get_version_hashlist};
use crate::db::functions::update::{update_changelog, update_project};
use crate::db::schema::create_database;
use anyhow::Result;
use rusqlite::Connection;
use std::thread;
use std::time::Duration;

pub fn create(config: &Config) -> Result<Connection> {
    let db_path = &config.db_path.as_ref().unwrap_or(&DEFAULT_DB_PATH);

    if std::fs::metadata(db_path).is_err() {
        let conn = Connection::open(&db_path)?;
        create_database(&conn)?;
        let all_n3n_boxes = N3NBox::fetch()?;
        for n3n in all_n3n_boxes.iter() {
            let project = n3n.get_project();
            create_project(&conn, project)?;

            if let Some(changelog) = n3n.get_changelog() {
                create_changelog(&conn, changelog)?;
            }
            if let Some(versions) = n3n.get_versions() {
                for version in versions.iter() {
                    create_version(&conn, version)?;
                }
            }
        }
        Ok(conn)
    } else {
        Ok(Connection::open(&db_path)?)
    }
}

pub fn delete(conn: &Connection, fetch_boxes: &Vec<N3NBox>) -> Result<()> {
    let upd_projects = get_all_project(&conn)?;
    let all_n3n_projects = fetch_boxes
        .iter()
        .map(|n3n_box| n3n_box.get_project())
        .collect::<Vec<&Project>>();
    for project in upd_projects.iter() {
        if !all_n3n_projects.contains(&project) {
            delete_project(&conn, project)?;
        }
    }
    Ok(())
}

pub fn start_sync(config: &Config) -> Result<()> {
    let conn = create(config)?;
    loop {
        let all_n3n_boxes = N3NBox::fetch()?;
        let old_projects = get_all_project(&conn)?;
        let old_proj_len = old_projects.len();
        for n3n_box in all_n3n_boxes.iter() {
            let now_proj = n3n_box.get_project();
            // Create project if not present already
            for (idx, old_proj) in old_projects.iter().enumerate() {
                match old_proj.name == now_proj.name {
                    true if old_proj != now_proj => update_project(&conn, now_proj, old_proj)?,
                    true => break,
                    false if idx == old_proj_len - 1 => create_project(&conn, now_proj)?,
                    false => continue,
                }
            }
            // store changelog if not already
            if let Some(changelog) = n3n_box.get_changelog() {
                let now_hash = &changelog.hash;
                let old_hash_wrapped = &get_changelog_hash(&conn, now_proj);
                if old_hash_wrapped.is_err() {
                    create_changelog(&conn, &changelog)?;
                } else {
                    let old_hash = old_hash_wrapped.as_ref().unwrap();
                    if now_hash != old_hash {
                        update_changelog(&conn, &changelog, old_hash)?;
                    }
                }
            }
            // parse versions if not already
            if let Some(versions) = n3n_box.get_versions() {
                for version in versions.iter() {
                    let now_hash = &version.hash;
                    let old_hashes = get_version_hashlist(&conn, now_proj)?;
                    if !old_hashes.contains(now_hash) {
                        create_version(&conn, version)?;
                    }
                }
            }
        }
        delete(&conn, &all_n3n_boxes)?;

        post_on_social_media(&config, &all_n3n_boxes)?;

        let scan_period = 60 * &config.scan_period.unwrap_or(DEFAULT_SCAN_PERIOD);
        thread::sleep(Duration::new(scan_period, 0));
    }
}

pub fn post_on_social_media(config: &Config, fetch_boxes: &Vec<N3NBox>) -> Result<()> {
    let twitter_client = TwitterBox::new(reqwest::Client::new(), &config.twitter);
    let telegram_client = TelegramBox::new(reqwest::Client::new(), &config.telegram);
    let discord_client = DiscordBox::new(reqwest::Client::new(), &config.discord);

    for n3n_box in fetch_boxes.iter() {
        if let Some(versions) = n3n_box.get_versions() {
            for version in versions.iter() {
                let twitter_adapter =
                    TwitterAdapter::new(&config, &twitter_client, n3n_box.get_project(), version)?;
                let discord_adapter =
                    DiscordAdapter::new(&config, &discord_client, n3n_box.get_project(), version)?;
                let telegram_adapter = TelegramAdapter::new(
                    &config,
                    &telegram_client,
                    n3n_box.get_project(),
                    version,
                )?;
                twitter_adapter.publish()?;
                discord_adapter.publish()?;
                telegram_adapter.publish()?;
            }
        }
    }
    Ok(())
}
