use serde::Deserialize;
use std::convert::From;

/***************************/
/*-------------------------*/
/*--                     --*/
/*--    Config Models    --*/
/*--                     --*/
/*-------------------------*/
/***************************/

#[derive(Deserialize, Debug)]
pub struct Config {
    pub db_path: Option<String>,
    pub scan_period: Option<u64>,

    // N3NX Network APIs Config
    pub n3nx: ConfigN3NX,
    pub telegram: ConfigTelegram,
    pub twitter: ConfigTwitter,
    pub discord: ConfigDiscord,
    pub reddit: ConfigReddit,
}

#[derive(Deserialize, Debug)]
pub struct ConfigN3NX {
    pub fetch_url: String,
}

#[derive(Deserialize, Debug)]
pub struct ConfigTelegram {
    pub base_url: String,
    pub secret: String,
    pub chat_id: String,
}

#[derive(Deserialize, Debug)]
pub struct ConfigTwitter {
    pub base_url: String,
    pub api_key: String,
    pub api_secret: String,
    pub bearer_token: String,
    pub access_token: String,
    pub token_secret: String,
}

#[derive(Deserialize, Debug)]
pub struct ConfigDiscord {
    pub base_url: String,
    pub webhook_id: String,
    pub webhook_token: String,
}

#[derive(Deserialize, Debug)]
pub struct ConfigReddit {
    pub api_key: String,
    pub api_secret: String,
}

/***************************/
/*-------------------------*/
/*--                     --*/
/*--     N3N Models      --*/
/*--                     --*/
/*-------------------------*/
/***************************/

#[derive(Deserialize, Debug, Default)]
pub struct AllProjects {
    pub projects: Vec<Project>,
}

#[derive(Deserialize, Debug, Default, PartialEq)]
pub struct Project {
    pub name: String,
    pub desc: String,
    pub url: String,
    pub tags: Vec<String>,
    pub featured: bool,
    pub emoji: String,
    pub template: bool,
    pub branch: String,
}

#[derive(Debug)]
pub struct Changelog {
    pub project_name: String,
    pub hash: String,
    pub data: String,
}

#[derive(Debug)]
pub struct Version {
    pub project_name: String,
    pub hash: String,
    pub data: String,
    pub timestamp: i64,
    pub name: String,
    pub major: u8,
    pub minor: u8,
    pub patch: u8,
    pub codename: String,
    pub start_commit: String,
    pub end_commit: String,
}

/***************************/
/*-------------------------*/
/*--                     --*/
/*--    Social Models    --*/
/*--                     --*/
/*-------------------------*/
/***************************/

#[derive(Debug)]
pub struct Message {
    pub media_id: MediaId,
    pub message_id: String,
    pub message_hash: String,
    pub timestamp: i64,
    pub project_name: String,
    pub version_hash: String,
}

#[derive(Clone, Copy, Debug)]
pub enum MediaId {
    Telegram,
    Twitter,
    Reddit,
    Discord,
    LinkedIn,
    Slack,
}

impl From<MediaId> for &str {
    fn from(item: MediaId) -> Self {
        match item {
            MediaId::Discord => "discord",
            MediaId::LinkedIn => "linkedin",
            MediaId::Reddit => "reddit",
            MediaId::Slack => "slack",
            MediaId::Telegram => "telegram",
            MediaId::Twitter => "twitter",
        }
    }
}
/***************************/
/*-------------------------*/
/*--                     --*/
/*--    Reddit Models    --*/
/*--                     --*/
/*-------------------------*/
/***************************/

// #[derive(Debug, Serialize)]
// pub struct RedditSend {
//     pub flair_id: String,
//     pub flair_text: String,
//     pub kind: String,
//     pub nsfw: bool,
//     pub resubmit: bool,
//     pub sr: String,
//     pub title: String,
//     pub text: String,
//     pub url: String,
// }
