use crate::core::models::{ConfigDiscord, MediaId, Message, Project, Version};
use crate::core::traits::{APISecret, Adapter, Builder, Publisher, SocialAPI, SocialBox};
use crate::db::functions::{create::create_message, delete::delete_message, select::get_message};
use crate::utils::gen_hash;
use anyhow::Result;
use chrono::prelude::*;
use regex::Regex;
use reqwest::header::{HeaderMap, CONTENT_TYPE};
use rusqlite::Connection;
use serde::{Deserialize, Serialize};

/***************************/
/*-------------------------*/
/*--                     --*/
/*--       Models        --*/
/*--                     --*/
/*-------------------------*/
/***************************/
#[derive(Debug, Serialize)]
pub struct DiscordSend {
    pub content: String,
}

#[derive(Deserialize, Debug)]
pub struct DiscordMessage {
    pub id: String,
    pub timestamp: String,
    pub content: String,
}

// #[derive(Deserialize, Debug)]
// pub struct DiscordError {
//     pub message: String,
//     pub code: i32,
// }

/***************************/
/*-------------------------*/
/*--                     --*/
/*--        Types        --*/
/*--                     --*/
/*-------------------------*/
/***************************/
pub type DiscordBox<'a> = SocialBox<'a, ConfigDiscord>;
pub type DiscordAdapter<'a> = Adapter<'a, DiscordBox<'a>>;
impl APISecret for ConfigDiscord {}

/***************************/
/*-------------------------*/
/*--                     --*/
/*--     Social Box      --*/
/*--                     --*/
/*-------------------------*/
/***************************/
impl<'a> SocialAPI<'a> for DiscordBox<'a> {
    #[tokio::main]
    async fn send(&self, mut msg: String) -> Result<String> {
        msg.truncate(2000);
        let url = format!(
            "{}/{}/{}?wait=true",
            self.get_api().base_url,
            self.get_api().webhook_id,
            self.get_api().webhook_token
        );
        let request = DiscordSend { content: msg };
        let mut heads = HeaderMap::new();
        heads.insert(CONTENT_TYPE, "application/json".parse()?);
        let response = self
            .get_client()
            .post(&url)
            .json(&request)
            .headers(heads)
            .send()
            .await?;
        let state_data: String = response.text().await?;
        Ok(state_data)
    }

    #[tokio::main]
    async fn get(&self, msgid: &str) -> Result<String> {
        let url = format!(
            "{}/{}/{}/messages/{}",
            self.get_api().base_url,
            self.get_api().webhook_id,
            self.get_api().webhook_token,
            msgid
        );
        let response = self.get_client().get(&url).send().await?;
        let state_data: String = response.text().await?;
        Ok(state_data)
    }

    #[tokio::main]
    async fn delete(&self, msgid: &str) -> Result<String> {
        let url = format!(
            "{}/{}/{}/messages/{}",
            self.get_api().base_url,
            self.get_api().webhook_id,
            self.get_api().webhook_token,
            msgid
        );
        let response = self.get_client().delete(&url).send().await?;
        let state_data: String = response.text().await?;
        Ok(state_data)
    }
}
/***************************/
/*-------------------------*/
/*--                     --*/
/*--      Builders       --*/
/*--                     --*/
/*-------------------------*/
/***************************/
impl<'a> Builder for DiscordBox<'a> {
    fn build(proj: &Project, version: &Version) -> String {
        version
            .data
            .lines()
            .map(|line| discord_post_data(proj, version, line))
            .collect::<String>()
    }
}

fn discord_post_data(proj: &Project, version: &Version, line: &str) -> String {
    let h1_re = Regex::new(r"^##\s").unwrap();
    let h2_re = Regex::new(r"^###\s").unwrap();
    let summary_re = Regex::new(r"\]:\s.+").unwrap();
    let commit_re = Regex::new(r"-\s\*(.+):\*\s([^\[\(]+)").unwrap();

    if h1_re.is_match(line) {
        let date = Utc.timestamp(version.timestamp, 0);
        format!(
            "**{} __{} v{}__**\n**🎉 release {}-{}-{}**\n",
            proj.emoji,
            proj.name,
            version.name,
            date.year(),
            date.month(),
            date.day(),
        )
    } else if h2_re.is_match(line) {
        format!("__**{}**__", line.trim_start_matches("### "))
    } else if summary_re.is_match(line) {
        let site_url = format!(
            "https://n3n.org/news/{}-{}-{}-{}",
            proj.name, version.major, version.minor, version.patch
        );
        format!(
            "🤩 star repo at {}\n📜 full changelog {}\n",
            proj.url, site_url
        )
    } else if commit_re.is_match(line) {
        let commit_cap = commit_re.captures(&line).unwrap();
        format!("- **{}** {}\n", &commit_cap[1], &commit_cap[2])
    } else {
        format!("{}\n", line)
    }
}

/***************************/
/*-------------------------*/
/*--                     --*/
/*--      Adapters       --*/
/*--                     --*/
/*-------------------------*/
/***************************/
impl<'a> Publisher<'a> for DiscordAdapter<'a> {
    fn publish(&self) -> Result<()> {
        let stored_message = get_message(&self.get_conn(), MediaId::Discord, &self.get_version());
        if stored_message.is_err() {
            discord_publish(
                &self.get_conn(),
                &self.get_client(),
                &self.get_project(),
                &self.get_version(),
            )?;
        } else {
            let stored_message = stored_message?;
            let message_resp = self.get_client().get(&stored_message.message_id)?;
            if message_resp.find("Unknown Message").is_some() {
                delete_message(&self.get_conn(), MediaId::Discord, &stored_message)?;
                discord_publish(
                    &self.get_conn(),
                    &self.get_client(),
                    &self.get_project(),
                    &self.get_version(),
                )?;
            } else {
                let got_message: DiscordMessage = serde_json::from_str(&message_resp)?;
                let got_message_hash = gen_hash(&got_message.content);
                if stored_message.message_hash != got_message_hash {
                    discord_publish(
                        &self.get_conn(),
                        &self.get_client(),
                        &self.get_project(),
                        &self.get_version(),
                    )?;
                }
                // println!("{:#?}", stored_message);
            }
        }
        Ok(())
    }
}

fn discord_publish(
    conn: &Connection,
    client: &DiscordBox,
    project: &Project,
    version: &Version,
) -> Result<()> {
    let discord_response = client.send(DiscordBox::build(project, version))?;
    let message: DiscordMessage = serde_json::from_str(&discord_response)?;
    let sent_message = Message {
        media_id: MediaId::Discord,
        message_hash: gen_hash(&message.content),
        message_id: message.id,
        project_name: project.name.clone(),
        timestamp: DateTime::parse_from_rfc3339(&message.timestamp)?.timestamp(),
        version_hash: version.hash.clone(),
    };
    create_message(conn, &sent_message)?;
    Ok(())
}
