use crate::core::models::{Changelog, Project, Version};
use anyhow::Result;
use rusqlite::{params, Connection};

#[allow(dead_code)]
pub fn update_project(path: &str, project: &Project, oldproject: &Project) -> Result<()> {
    let conn = Connection::open(&path)?;

    conn.execute("UPDATE project SET name = ?1, description = ?2, url = ?3, is_featured = ?4, emoji = ?5, is_template = ?6 WHERE name = ?7", params![project.name, project.desc, project.url, project.featured, project.emoji, project.template, oldproject.name])?;

    // delete old project tags
    for old_tag in oldproject.tags.iter() {
        conn.execute("DELETE FROM tag WHERE name = ?1 AND project_id = (SELECT id FROM project WHERE name = ?2)", params![old_tag, project.name])?;
    }

    // add new project tags
    for tag in project.tags.iter() {
        conn.execute("INSERT INTO tag (name, project_id) VALUES (?1, (SELECT id FROM project WHERE name = ?2))", params![tag, project.name])?;
    }

    // update default project branch
    conn.execute(
        "UPDATE branch SET name = ?1 WHERE project_id = (SELECT id FROM project WHERE name = ?2)",
        params![&project.branch, project.name],
    )?;

    Ok(())
}

#[allow(dead_code)]
pub fn update_project_name(path: &str, project: &Project, oldproject: &Project) -> Result<()> {
    let conn = Connection::open(&path)?;

    conn.execute(
        "UPDATE project SET name = ?1 where name = ?2",
        params![project.name, oldproject.name],
    )?;
    Ok(())
}

#[allow(dead_code)]
pub fn update_changelog(path: &str, changelog: &Changelog, oldchangelog_hash: &str) -> Result<()> {
    let conn = Connection::open(&path)?;

    conn.execute(
        "DELETE FROM hash WHERE id = (SELECT prev_hash FROM changelog WHERE project_id = (SELECT id FROM project WHERE name = ?))",
        params![changelog.project_name],
    )?;
    conn.execute(
        "INSERT OR IGNORE INTO hash (id, data) VALUES (?1, ?2)",
        params![changelog.hash, changelog.data],
    )?;

    conn.execute("UPDATE changelog SET project_id = (SELECT id FROM project WHERE name = ?1), hash = ?2, prev_hash = ?3 WHERE hash = ?3", params![changelog.project_name, changelog.hash, oldchangelog_hash])?;
    Ok(())
}

#[allow(dead_code)]
pub fn update_version(path: &str, version: &Version, oldversion: &Version) -> Result<()> {
    let conn = Connection::open(&path)?;

    conn.execute(
        "INSERT OR IGNORE INTO hash (id, data) VALUES (?1, ?2)",
        params![version.hash, version.data],
    )?;

    conn.execute("UPDATE version SET project_id = (SELECT id FROM project WHERE name = ?1), hash = ?2, release_date = ?3, name = ?4, major = ?5, minor = ?6, patch = ?7, codename = ?8, start_commit = ?9, end_commit = ?10 WHERE hash = ?11", params![version.project_name, version.hash, version.release_date, version.name, version.major, version.minor, version.patch, version.codename, version.start_commit, version.end_commit, oldversion.hash])?;
    Ok(())
}
