use anyhow::{bail, Result};
use chrono::prelude::*;
use hex::encode;
use regex::Regex;
use sha2::{Digest, Sha256};

pub fn clean_changelog(data: &str) -> String {
    let changelog_head = Regex::new(r"#\s[\s\S]+\)\.\n+").unwrap();
    changelog_head.split(data).collect()
}

pub fn build_version(data: &str) -> Result<(String, Vec<u8>)> {
    let version_mask = Regex::new(r"\[(\d\.|\d)+\]").unwrap();
    let version = version_mask
        .find(data)
        .unwrap()
        .as_str()
        .trim_matches(|c| c == '[' || c == ']')
        .to_string();
    let version_num: Vec<u8> = version
        .split('.')
        .map(|x| x.parse::<u8>().unwrap())
        .collect();
    Ok((version, version_num))
}

pub fn build_commit_range(data: &str) -> Result<(String, String)> {
    let commit_mask = Regex::new(r"\w{40}\.+\w{40}").unwrap();
    let commit_data = if commit_mask.is_match(data) {
        let raw_commit_id: Vec<String> = commit_mask
            .find(data)
            .unwrap()
            .as_str()
            .split("...")
            .map(|x| x.to_string())
            .collect();
        if raw_commit_id.len() < 2 {
            bail!("error: commit id index out of range (len < 2)")
        }
        (raw_commit_id[0].to_owned(), raw_commit_id[1].to_owned())
    } else {
        (String::new(), String::new())
    };
    Ok(commit_data)
}

pub fn build_date(data: &str) -> Result<chrono::DateTime<Utc>> {
    let date_mask = Regex::new(r"\d{4}(-\d{2})+").unwrap();
    let raw_date: Vec<u32> = date_mask
        .find(data)
        .unwrap()
        .as_str()
        .split('-')
        .map(|x| x.parse::<u32>().unwrap())
        .collect();
    let date = Utc
        .ymd(raw_date[0] as i32, raw_date[1], raw_date[2])
        .and_hms(0, 0, 0);
    Ok(date)
}

pub fn gen_hash(data: &str) -> String {
    let mut hasher = Sha256::new();
    hasher.update(data.as_bytes());
    encode(hasher.finalize())
}
