use super::log::N3NBox;
use super::models::{Config, Project};
use crate::api::{SocialAPI, TelegramBox};
use crate::builder::Builder;
use crate::db::functions::create::{create_changelog, create_project, create_version};
use crate::db::functions::delete::delete_project;
use crate::db::functions::select::{get_all_project, get_changelog_hash, get_version_hashlist};
use crate::db::functions::update::{update_changelog, update_project};
use crate::db::schema::create_database;
use anyhow::Result;
use std::thread;
use std::time::Duration;

const PERIOD: u64 = 30;

pub fn create(config: &Config) -> Result<()> {
    let db_path = &config.db_path;
    if std::fs::metadata(db_path).is_err() {
        create_database(db_path)?;
        let all_n3n_boxes = N3NBox::fetch()?;
        for n3n in all_n3n_boxes.iter() {
            let project = n3n.get_project();
            create_project(db_path, project)?;

            if let Some(changelog) = n3n.get_changelog() {
                create_changelog(db_path, changelog)?;
            }
            if let Some(versions) = n3n.get_versions() {
                for version in versions.iter() {
                    create_version(&config.db_path, version)?;
                }
            }
        }
    }
    Ok(())
}

pub fn delete(config: &Config, fetch_boxes: &Vec<N3NBox>) -> Result<()> {
    let db_path = &config.db_path;
    let upd_projects = get_all_project(db_path)?;
    let all_n3n_projects = fetch_boxes
        .iter()
        .map(|n3n_box| n3n_box.get_project())
        .collect::<Vec<&Project>>();
    for project in upd_projects.iter() {
        if !all_n3n_projects.contains(&project) {
            delete_project(db_path, project)?;
        }
    }
    Ok(())
}

pub fn start_sync(config: &Config) -> Result<()> {
    create(config)?;
    let db_path = &config.db_path;
    loop {
        let all_n3n_boxes = N3NBox::fetch()?;
        let old_projects = get_all_project(db_path)?;
        let old_proj_len = old_projects.len();
        for n3n_box in all_n3n_boxes.iter() {
            let now_proj = n3n_box.get_project();
            // Create project if not present already
            for (idx, old_proj) in old_projects.iter().enumerate() {
                match old_proj.name == now_proj.name {
                    true if old_proj != now_proj => {
                        println!("updated {:?}", now_proj);
                        update_project(db_path, now_proj, old_proj)?
                    }
                    true => break,
                    false if idx == old_proj_len - 1 => {
                        println!("created {:?}", now_proj);
                        create_project(db_path, now_proj)?
                    }
                    false => continue,
                }
            }
            // store changelog if not already
            if let Some(changelog) = n3n_box.get_changelog() {
                let now_hash = &changelog.hash;
                let old_hash_wrapped = &get_changelog_hash(db_path, now_proj);
                if old_hash_wrapped.is_err() {
                    create_changelog(db_path, &changelog)?;
                } else {
                    let old_hash = old_hash_wrapped.as_ref().unwrap();
                    if now_hash != old_hash {
                        update_changelog(db_path, &changelog, old_hash)?;
                    }
                }
            }
            // parse versions if not already
            if let Some(versions) = n3n_box.get_versions() {
                for version in versions.iter() {
                    let now_hash = &version.hash;
                    let old_hashes = get_version_hashlist(db_path, now_proj)?;
                    if !old_hashes.contains(now_hash) {
                        create_version(db_path, version)?;
                    }
                }
            }
        }
        delete(config, &all_n3n_boxes)?;

        post_on_social_media(&config, &all_n3n_boxes)?;

        thread::sleep(Duration::new(60 * PERIOD, 0));
    }
}

pub fn post_on_social_media(config: &Config, fetch_boxes: &Vec<N3NBox>) -> Result<()> {
    let telegram_client = TelegramBox::new(reqwest::Client::new(), &config.telegram);

    for n3n_box in fetch_boxes.iter() {
        if let Some(versions) = n3n_box.get_versions() {
            for version in versions.iter() {
                telegram_client.send(TelegramBox::build(n3n_box.get_project(), version))?;
            }
        }
    }
    Ok(())
}
