use super::models::{AllProjects, Changelog, Project, Version};
use super::utils::*;
use anyhow::Result;
use regex::Regex;
use reqwest;

pub struct N3NBox {
    project: Project,
    changelog: Option<Changelog>,
    versions: Option<Vec<Version>>,
}

impl N3NBox {
    pub fn fetch() -> Result<Vec<N3NBox>> {
        let mut n3n_boxes = Vec::new();
        let projects = fetch_project()?;
        for proj in projects.into_iter() {
            let changelog = fetch_changelog(&proj)?;
            let versions = fetch_version(&proj, &changelog)?;
            n3n_boxes.push(N3NBox {
                project: proj,
                changelog,
                versions,
            })
        }
        Ok(n3n_boxes)
    }

    pub fn get_project(&self) -> &Project {
        &self.project
    }

    pub fn get_changelog(&self) -> &Option<Changelog> {
        &self.changelog
    }

    pub fn get_versions(&self) -> &Option<Vec<Version>> {
        &self.versions
    }
}

#[tokio::main]
async fn fetch_project() -> Result<Vec<Project>> {
    let client = reqwest::Client::new();
    let data = "https://n3n.org/data/engine.json";
    let resp = client.get(data).send().await?;
    let n3n_data: AllProjects = resp.json().await?;
    Ok(n3n_data.projects)
}

#[tokio::main]
async fn fetch_changelog(project: &Project) -> Result<Option<Changelog>> {
    let client = reqwest::Client::new();
    if !project.template {
        let request_url = format!(
            "https://raw.githubusercontent.com/n3nx/{}/{}/CHANGELOG.md",
            project.name, project.branch
        );
        let response = client.get(&request_url).send().await?;
        let changelog_data_raw: String = response.text().await?;
        if changelog_data_raw.find("404:").is_some() {
            return Ok(None);
            // bail!("changelog not found for project `{}`", project.name);
        };
        let changelog_data = clean_changelog(&changelog_data_raw);
        let changelog_hash = gen_hash(&changelog_data);

        Ok(Some(Changelog {
            project_name: project.name.to_string(),
            data: changelog_data,
            hash: changelog_hash,
        }))
    } else {
        Ok(None)
    }
}

fn fetch_version(project: &Project, changelog: &Option<Changelog>) -> Result<Option<Vec<Version>>> {
    let mut all_versions: Vec<Version> = Vec::new();
    if let Some(valid_changelog) = changelog {
        let changelog_data = &valid_changelog.data[..];
        let split_marker = Regex::new(r"<!-.+\n").unwrap();
        for slice in split_marker
            .split(changelog_data)
            .filter_map(|x| match x.is_empty() {
                true => None,
                false => Some(x.trim()),
            })
        {
            let (version, version_num) = build_version(slice).unwrap();
            let (start_commit, end_commit) = build_commit_range(slice).unwrap();
            let date = build_date(slice).unwrap();
            let hash_id = gen_hash(slice);
            let version = Version {
                project_name: project.name.to_string(),
                hash: hash_id,
                data: slice.to_string(),
                release_date: date.to_rfc3339(),
                name: version,
                major: version_num[0],
                minor: version_num[1],
                patch: version_num[2],
                codename: String::new(),
                start_commit: start_commit,
                end_commit: end_commit,
            };
            all_versions.push(version)
        }
        Ok(Some(all_versions))
    } else {
        Ok(None)
    }
}
