use crate::api::{TelegramBox, TwitterBox};
use crate::core::models::{Project, Version};
use chrono::prelude::*;
use regex::Regex;

pub trait Builder {
    fn build(proj: &Project, version: &Version) -> String;
}

impl<'a> Builder for TelegramBox<'a> {
    fn build(proj: &Project, version: &Version) -> String {
        version
            .data
            .lines()
            .map(|line| telegram_post_data(proj, version, line))
            .collect::<String>()
    }
}

impl<'a> Builder for TwitterBox<'a> {
    fn build(proj: &Project, version: &Version) -> String {
        version
            .data
            .lines()
            .map(|line| twitter_post_data(proj, version, line))
            .collect::<String>()
    }
}

fn twitter_post_data(proj: &Project, version: &Version, line: &str) -> String {
    if line.find("## ") == Some(0) {
        let date = version.release_date.parse::<DateTime<Utc>>().unwrap();
        let tweet = format!(
            "{} {} v{}\n🎉 release {}-{}-{}\n🤩 repo at {}\n📜 changelog below",
            proj.emoji,
            proj.name,
            version.name,
            date.year(),
            date.month(),
            date.day(),
            proj.url
        );
        let tweet = build_tags(proj, tweet);
        format!(
            "{}\nhttps://n3n.org/news/{}-{}-{}-{}",
            tweet, proj.name, version.major, version.minor, version.patch
        )
    } else {
        String::new()
    }
}

fn build_tags(proj: &Project, line: String) -> String {
    let additional_tags = [
        "TrustIsUs",
        "cybersecurity",
        "infosec",
        "foss",
        "release",
        "opensource",
    ];
    let mut string = format!("{}\n", line);
    for tag in additional_tags.iter() {
        string = format!("{}#{} ", string, tag);
    }
    for tag in proj.tags.iter() {
        string = format!("{}#{} ", string, tag);
    }
    string
}

fn telegram_post_data(proj: &Project, version: &Version, line: &str) -> String {
    let h1_re = Regex::new(r"^##\s").unwrap();
    let h2_re = Regex::new(r"^###\s").unwrap();
    let summary_re = Regex::new(r"\]:\s.+").unwrap();
    let commit_re = Regex::new(r"-\s\*(.+):\*\s([^\[\(]+)").unwrap();

    if h1_re.is_match(line) {
        let date = version.release_date.parse::<DateTime<Utc>>().unwrap();
        format!(
            "<b>{} <u>{} v{}</u>%0A🎉 release {}-{}-{}%0A</b>",
            proj.emoji,
            proj.name,
            version.name,
            date.year(),
            date.month(),
            date.day(),
        )
    } else if h2_re.is_match(line) {
        format!("<b><u>{}</u></b>", line.trim_start_matches("### "))
    } else if summary_re.is_match(line) {
        format!(
            "🤩 star repo at {}%0A📜 full changelog {}%0A",
            proj.url,
            format!(
                "<a href=\"https://n3n.org/news/{}-{}-{}-{}\">here</a>",
                proj.name, version.major, version.minor, version.patch
            )
        )
    } else if commit_re.is_match(line) {
        let commit_cap = commit_re.captures(&line).unwrap();
        format!("- <b>{}</b> {}%0A", &commit_cap[1], &commit_cap[2])
    } else {
        format!("{}%0A", line)
    }
}
