use crate::core::models::APIs;
// use crate::models::TelegramMessageResult;
use crate::core::models::{Telegram, Twitter, TwitterPostReq};
use reqwest::header::{HeaderMap, AUTHORIZATION, CONTENT_TYPE};

pub struct SocialBox<'a, T: APIs> {
    client: reqwest::Client,
    api: &'a T,
}

pub trait SocialAPI<'a> {
    fn new<T: 'a + APIs>(client: reqwest::Client, api: &T) -> SocialBox<T> {
        SocialBox { client, api }
    }

    #[tokio::main]
    async fn send(&self, msg: String) -> Result<String, reqwest::Error> {
        drop(msg);
        Ok(String::default())
    }

    #[tokio::main]
    async fn recv(&self) -> Result<String, reqwest::Error> {
        Ok(String::default())
    }

    #[tokio::main]
    async fn delete(&self, msgid: u64) -> Result<String, reqwest::Error> {
        drop(msgid);
        Ok(String::default())
    }
}

/***************************/
/*-------------------------*/
/*--                     --*/
/*--    Telegram Box     --*/
/*--                     --*/
/*-------------------------*/
/***************************/

pub type TelegramBox<'a> = SocialBox<'a, Telegram>;
pub type TwitterBox<'a> = SocialBox<'a, Twitter>;

impl<'a> SocialAPI<'a> for TelegramBox<'a> {
    #[tokio::main]
    // async fn send(&self, msg: TelegramSend) -> Result<TelegramMessageResult, reqwest::Error> {
    async fn send(&self, msg: String) -> Result<String, reqwest::Error> {
        let request_url = format!(
            "{}/bot{}/sendMessage?chat_id={}&text={}&parse_mode=HTML",
            self.api.base_url, self.api.secret, self.api.chat_id, msg
        );
        let response = self
            .client
            .get(&request_url)
            // .headers(self.headers.clone())
            .send()
            .await?;
        // let state_data: TelegramMessageResult = response.json().await?;
        let state_data: String = response.text().await?;
        Ok(state_data)
    }

    #[tokio::main]
    async fn recv(&self) -> Result<String, reqwest::Error> {
        let request_url = format!("{}/bot{}/getUpdates", self.api.base_url, self.api.secret);
        let response = self.client.get(&request_url).send().await?;
        let message: String = response.json().await?;
        Ok(message)
    }

    #[tokio::main]
    async fn delete(&self, msgid: u64) -> Result<String, reqwest::Error> {
        let request_url = format!(
            "{}/bot{}/deleteMessage?chat_id={}&message_id={}",
            self.api.base_url, self.api.secret, self.api.chat_id, msgid
        );
        let response = self.client.get(&request_url).send().await?;
        let state_data: String = response.text().await?;
        Ok(state_data)
    }
}

/***************************/
/*-------------------------*/
/*--                     --*/
/*--     Twitter Box     --*/
/*--                     --*/
/*-------------------------*/
/***************************/

impl<'a> SocialAPI<'a> for TwitterBox<'a> {
    #[tokio::main]
    async fn send(&self, msg: String) -> Result<String, reqwest::Error> {
        let url = format!("{}/2/tweets", self.api.base_url);
        let request = TwitterPostReq { text: msg };
        let auth_header = self.gen_headers(&url);
        let mut heads = HeaderMap::new();
        heads.insert(CONTENT_TYPE, "application/json".parse().unwrap());
        heads.insert(AUTHORIZATION, auth_header.parse().unwrap());
        let response = self
            .client
            .post(&url)
            .json(&request)
            .headers(heads)
            .send()
            .await?;
        // let state_data: TelegramMessageResult = response.json().await?;
        let state_data: String = response.text().await?;
        Ok(state_data)
    }

    #[tokio::main]
    async fn recv(&self) -> Result<String, reqwest::Error> {
        // let url = format!("{}/2/tweets", self.api.base_url);
        // let request = TwitterPostReq { text: msg };
        // let auth_header = self.gen_headers(&url);
        // let mut heads = HeaderMap::new();
        // heads.insert(CONTENT_TYPE, "application/json".parse().unwrap());
        // heads.insert(AUTHORIZATION, auth_header.parse().unwrap());
        // let response = self
        //     .client
        //     .post(&url)
        //     .json(&request)
        //     .headers(heads)
        //     .send()
        //     .await?;
        // // let state_data: TelegramMessageResult = response.json().await?;
        // let state_data: String = response.text().await?;
        let message = String::default();
        Ok(message)
    }

    #[tokio::main]
    async fn delete(&self, msgid: u64) -> Result<String, reqwest::Error> {
        let url = format!("{}/2/tweets/{}", self.api.base_url, msgid);
        let mut heads = HeaderMap::new();
        let auth_header = self.gen_headers(&url);
        heads.insert(CONTENT_TYPE, "application/json".parse().unwrap());
        heads.insert(AUTHORIZATION, auth_header.parse().unwrap());
        let response = self.client.delete(&url).headers(heads).send().await?;
        let state_data: String = response.text().await?;
        Ok(state_data)
    }
}

use oauth1_header::{http::Method, Credentials};
use std::collections::HashMap;

impl<'a> TwitterBox<'a> {
    fn gen_headers(&self, url: &str) -> String {
        let client = Credentials::new(
            &self.api.api_key,
            &self.api.api_secret,
            &self.api.access_token,
            &self.api.token_secret,
        );
        let params = HashMap::new();
        client.auth(&Method::POST, &url, &params)
    }
}
