use crate::models::{AllProjects, Changelog, Project};
use hex::encode;
use reqwest;
use sha2::{Digest, Sha256};

pub struct N3NBox {
    pub client: reqwest::Client,
    // path: String,
    pub project: Project,
}

impl N3NBox {
    #[tokio::main]
    pub async fn fetch_all() -> Result<Vec<N3NBox>, reqwest::Error> {
        let mut n3n_boxes = Vec::new();
        let client = reqwest::Client::new();
        let data = "https://n3n.org/data/engine.json";
        let resp = client.get(data).send().await?;
        let n3n_data: AllProjects = resp.json().await?;
        for proj in n3n_data.projects.into_iter() {
            n3n_boxes.push(N3NBox {
                client: reqwest::Client::new(),
                project: proj,
            })
        }
        Ok(n3n_boxes)
    }

    #[tokio::main]
    pub async fn get(&self) -> Result<Changelog, reqwest::Error> {
        if !self.project.template {
            let request_url = format!(
                "https://raw.githubusercontent.com/n3nx/{}/{}/CHANGELOG.md",
                self.project.name, self.project.branch
            );
            let response = self
                .client
                .get(&request_url)
                // .headers(self.headers.clone())
                .send()
                .await?;
            let changelog_data_raw: String = response.text().await?;
            let changelog_hash: String = gen_hash(&changelog_data_raw);
            let changelog_data = if let Some(_) = changelog_data_raw.find("404:") {
                None
            } else {
                Some(changelog_data_raw)
            };
            Ok(Changelog {
                project: &self.project,
                data: changelog_data,
                hash: changelog_hash.clone(),
                prev_hash: changelog_hash,
            })
        } else {
            Ok(Changelog {
                project: &self.project,
                data: None,
                hash: String::new(),
                prev_hash: String::new(),
            })
        }
    }
}

fn gen_hash(data: &str) -> String {
    let mut hasher = Sha256::new();
    hasher.update(data.as_bytes());
    encode(hasher.finalize())
}
