use crate::api::{TelegramBox, TwitterBox};
use crate::models::Project;
use regex::Regex;

pub trait Builder {
    fn build(proj: &Project, data: &str) -> String;
}

impl<'a> Builder for TelegramBox<'a> {
    fn build(proj: &Project, data: &str) -> String {
        let mut version: Vec<u16> = Vec::new();
        data.lines()
            .map(|line| {
                let newline = telegram_fmt_head(proj, line, &mut version);
                telegram_fmt_lines(newline)
            })
            // .filter(|line| !line.is_empty())
            .collect::<String>()
    }
}

impl<'a> Builder for TwitterBox<'a> {
    fn build(proj: &Project, data: &str) -> String {
        data.lines()
            .map(|line| twitter_post_data(proj, line))
            // .filter(|line| !line.is_empty())
            .collect::<String>()
    }
}

fn twitter_post_data(proj: &Project, line: &str) -> String {
    if line.find("## ") == Some(0) {
        let version = build_version(line);
        let tweet = format!(
            "{} {} v{}.{}.{}\n🎉 release {}-{}-{}\n🤩 repo at {}\n📜 changelog below",
            proj.emoji,
            proj.name,
            version[0],
            version[1],
            version[2],
            version[3],
            version[4],
            version[5],
            proj.url
        );
        let tweet = build_tags(proj, tweet);
        twitter_build_url(proj, tweet, &version)
    } else {
        String::new()
    }
}

fn build_version(line: &str) -> Vec<u16> {
    let markdown_regex = Regex::new(r"[\W\D]+").unwrap();
    markdown_regex
        .split(line)
        .filter_map(|x| x.parse::<u16>().ok())
        .collect()
}

fn build_tags(proj: &Project, line: String) -> String {
    let additional_tags = [
        "TrustIsUs",
        "cybersecurity",
        "infosec",
        "foss",
        "release",
        "opensource",
    ];
    let mut string = format!("{}\n", line);
    for tag in additional_tags.iter() {
        string = format!("{}#{} ", string, tag);
    }
    for tag in proj.tags.iter() {
        string = format!("{}#{} ", string, tag);
    }
    string
}

fn twitter_build_url(proj: &Project, prev_line: String, version: &Vec<u16>) -> String {
    format!(
        "{}\nhttps://n3n.org/news/{}-{}-{}-{}",
        prev_line, proj.name, version[0], version[1], version[2]
    )
}

fn telegram_fmt_head(proj: &Project, line: &str, version: &mut Vec<u16>) -> String {
    let h1_re = Regex::new(r"^##\s").unwrap();
    let h2_re = Regex::new(r"^###\s").unwrap();
    let summary_re = Regex::new(r":\s(\.\./)+[\s\S]+").unwrap();

    if h1_re.is_match(line) {
        let mut ver = build_version(line);
        version.append(&mut ver);
        // if version.len() > 0 {
        format!(
            "<b>{} <u>{} v{}.{}.{}</u>%0A🎉 release {}-{}-{}%0A</b>",
            proj.emoji,
            proj.name,
            version[0],
            version[1],
            version[2],
            version[3],
            version[4],
            version[5],
        )
        // } else {
        // String::from("h1 placeholder")
        // }
    } else if h2_re.is_match(line) {
        format!("<b><u>{}</u></b>", line.trim_start_matches("### "))
    } else if summary_re.is_match(line) {
        // println!("{:?}", &version);
        format!(
            "🤩 star repo at {}%0A📜 full changelog {}%0A",
            proj.url,
            telegram_build_url(proj, "here", &version)
        )
    } else {
        format!("{}%0A", line)
    }
}

fn telegram_fmt_lines(mut line: String) -> String {
    if line.len() > 60 {
        let commit_re = Regex::new(r"\s\(\[[\s\S]+\)").unwrap();
        line = commit_re.replace(&line, "").parse().unwrap();

        let i = Regex::new(r"-\s\*").unwrap();
        line = i.replace(&line, "- <b>").parse().unwrap();
        let i_e = Regex::new(r":\*\s").unwrap();
        line = i_e.replace(&line, ":</b> ").parse().unwrap();

        let brkt = Regex::new(r"\[[\s\S]+\]").unwrap();
        line = brkt.replace(&line, "").parse().unwrap();
    }
    line
}

/// replaces the commit compare link with nice summary of project
// fn telegram_change_link(proj: &Project, line: String, version: &Vec<u16>) -> String {
//     let summary_re = Regex::new(r":\s(\.\./)+[\s\S]+").unwrap();
//     if summary_re.is_match(&line) {
//         format!(
//             "🤩 star repo at {}%0A📜 full changelog {}",
//             proj.url,
//             telegram_build_url(proj, "here", version)
//         )
//     } else {
//         line
//     }
// }

fn telegram_build_url(proj: &Project, link_name: &str, version: &Vec<u16>) -> String {
    if version.len() > 0 {
        format!(
            "<a href=\"https://n3n.org/news/{}-{}-{}-{}\">{}</a>",
            proj.name, version[0], version[1], version[2], link_name
        )
    } else {
        String::from("url placeholder")
    }
}
