// Runs tests for encoding / decoding for each type
use wkb_raster::{Raster, RasterBand, RasterDataSource, InMemoryRasterData, InMemoryRasterData::*, Endian};

fn run_encode_test_inner(endian: Endian, input: InMemoryRasterData, width: u16, height: u16) {
    let setup = Raster {
        endian,
        version: 0,
        scale_x: 500.0,
        scale_y: 1.0,
        ip_x: 0.0,
        ip_y: 0.0,
        skew_x: 0.0,
        skew_y: 0.0,
        srid: 4326,
        width,
        height,
        bands: vec![RasterBand {
            is_nodata_value: false,
            data: RasterDataSource::InMemory(input),
        }],
    };

    let encoded = setup.clone().to_wkb_string();
    let decoded = Raster::from_wkb_string(&encoded.as_bytes()).unwrap();
    if decoded != setup {
        use std::process::exit;
        println!("expected: {:#?}\n\ngot:{:#?}", setup, decoded);
        exit(1);
    }
}

fn run_encode_test(input: InMemoryRasterData, width: u16, height: u16) {
    run_encode_test_inner(Endian::Big, input.clone(), width, height);
    run_encode_test_inner(Endian::Little, input, width, height);
}

fn run_decode_test(decode: &[u8]) {
    let _ = Raster::from_wkb_string(decode);
}

fn main() {

    run_encode_test(Bool1Bit { data: vec![
        vec![true, false, true],
        vec![false, false, false],
        vec![false, true, false],
    ], nodata: None }, 3, 3);

    run_encode_test(UInt2 { data: vec![
        vec![50, 38, 58],
        vec![20, 10, 90],
        vec![78, 5, 0],
    ], nodata: None }, 3, 3);

    run_encode_test(UInt4 { data: vec![
        vec![9, 89, 23],
        vec![49, 8, 17],
        vec![90, 83, 48],
    ], nodata: None }, 3, 3);

    run_encode_test(Int8 { data: vec![
        vec![-9, -25, 23],
        vec![49, -50, 19],
        vec![4, 18, 4],
    ], nodata: None }, 3, 3);

    run_encode_test(UInt8 { data: vec![
        vec![45, 255, 0],
        vec![49, 39, 77],
        vec![3, 15, 10],
    ], nodata: None }, 3, 3);

    run_encode_test(Int16 { data: vec![
        vec![3939, 255, 0],
        vec![49, 3939, 15],
        vec![405, 15, -2927],
    ], nodata: None }, 3, 3);

    run_encode_test(UInt16 { data: vec![
        vec![3939, 255, 0],
        vec![49, 3939, 15],
        vec![405, 15, 2927],
    ], nodata: None }, 3, 3);

    run_encode_test(UInt32 { data: vec![
        vec![3939, 25500, 482],
        vec![49893, 393900, 15],
        vec![405120, 150, 2927580],
    ], nodata: None }, 3, 3);

    run_encode_test(Int32 { data: vec![
        vec![3939, 25500, 482],
        vec![-49893, -393900, 15],
        vec![405120, 150, -29580],
    ], nodata: None }, 3, 3);

    run_encode_test(Float32 { data: vec![
        vec![3939.38, 255.482, 0.49],
        vec![49.382, 3939.3729, 15.28],
        vec![405.298, 15.47, -2927.284],
    ], nodata: None }, 3, 3);

    run_encode_test(Float64 { data: vec![
        vec![3939.38, 255.482, 0.49],
        vec![49.382, 3939.3729, 15.28],
        vec![405.298, 15.47, -2927.284],
    ], nodata: None }, 3, 3);

    run_decode_test(b"010000010037afeaa3af484b3f37afeaa3af484bbfed79045be58c2b40814248bd0887494000000000000000000000000000000000e61000003400180045008000800080008000800080008000800080008000800080008000800080008000800080008000800080008000800080008000800080008000800080008000800080008000800080008000800080008000800080008000800080008000800080008000800080008000806f006c006e007000710071007200740074007400740073007200720071007200730073007300760076007500740075007600740075007500740074007400740073007300760073007200730074006f006b006a006a006b0071007200740074007700790081008e00700070007000710070007000710072007400720071007100710071007300740073007400750075007700770075007400750074007300740075007500740073007300720072007300740075007600740072006d006a006a006c006f00700071007400750074007b007300740073007300730073007200710073007200710072007300730073007200730074007500760077007600750074007600750073007400750076007600760074007400720073007600780078007500720073006f006b006b006b006e007300790079007300730071007300740072007300730073007300720072007200720072007100720072007300730074007500750075007600740073007400750075007400740075007500750075007500740074007500770077007600770076006d006b006b006b0071007700770074007500720072007100720073007300730074007300730073007300730071007200730071007200750075007500750075007600750074007300720072007300750074007200710073007400730073007500750073007400740070006b006b006b006d007200770076007400740074007200720073007200720073007400740073007300720072007200720073007400750074007500740074007500760076007400740074007600760075007300730074007400730073007300740074007400740071006e006b006b006b006d0074007900780074007500740072007200730072007100720072007300720071007300750073007200750075007400730073007300730073007500760077007700760076007400730074007500740075007600740073007500760075007400730070006b006b006b006e0075007a00740074007500760074007400750074007300720072007200730074007400730073007400750075007400740074007400740075007500760078007700760075007400740074007400740074007500760076007600750075007600760071006b006b006b006f007600780076007500740075007500730074007400720072007300730073007100710072007300750074007500750075007400750076007500750076007600750075007600750075007400730074007500750075007500760078007800750074006e006b006b006d006e0077007900770075007600760074007100710070007100720073007400740073007300750074007200730075007400750075007400730076007500740075007600750077007700750075007600760074007400760077007700750074007500760070006b006b006c0076007700760075007600770076007400720072007200720072007200730073007400740073007300740075007500740073007300740075007400740076007500740075007500750076007600750075007400750076007600750077007700770073006c006b006d0077007600750075007500750074007300720074007600760074007400750075007400720071007200730074007400750074007500740074007500750074007500740076007600760076007600740074007400750074007400750078007700760076006d006b006c0076007500730074007400740073007300710073007600770077007700780076007400730070006f007100730074007400750076007400740075007600750075007500750076007400750076007500760075007500740076007800770077007700770072006b006b00750074007400750074007200720072007200730075007500750075007600750076007500710070007200740073007300730073007200730076007600750074007500750076007400750075007400740075007500760077007800780077007700760075006e006b00760074007200720073007400740072007300730072007200720073007400740073007100700070007000710072007400740073007400730074007500760076007500750075007400740074007300730073007500780078007600760078007c007e007a0075006c0077007600740073007300730074007400740072007200720073007500740075007600720071007100710071007300740075007400730072007200720074007400740074007400740075007500740073007300750076007700760076007a007d007c007800770075007500740074007400740073007200740075007300720073007400750077007700760075007400730074007400740073007400740073007400730072007500740073007300740073007300730073007300750076007500750076007700770078007a007900780076007500730074007400750074007300740075007300710073007500750076007600770076007400730074007500760074007300730073007400750075007400740076007600750075007400740073007400750075007500740074007500760077007800780079007700770077007500750075007300730073007300740072007300740076007500740075007500730074007400740073007300730074007400750076007700740073007500760077007600760075007300720074007500760076007600760075007400770079007a007a0077007b0079007500740074007500740073007400740074007300740073007300740074007300730073007400740072007100720074007500760076007500750077007800770075007500750075007500740074007500750075007500760077007700780079007b007600790078007400740074007500740074007200730073007400750074007500760075007300730075007500730072007200700071007300750075007300730074007500740074007300740075007500740074007400720073007200730074007600770077007a0074007600760075007500750074007500740074007400740073007400750075007600740072007400740074007300730073007200730073007300730073007300730074007500740074007500750075007400750074007500750075007500740074007500750077007500750075007400740074007400740075007500740074007400740074007400740073007300730073007400740073007100720074007500730073007600750073007300730074007400750076007600750075007500760075007500740074007500750076007600");
}