use crate::{types, Vertex};
use wgpu::util::DeviceExt;
use winit::{event::WindowEvent, window::Window};

use crate::util::*;

/// A class used for rendering using any number of render pipelines.
pub struct Renderer {
    clear_color: wgpu::Color,
    surface: wgpu::Surface,
    device: wgpu::Device,
    queue: wgpu::Queue,
    config: wgpu::SurfaceConfiguration,
    size: winit::dpi::PhysicalSize<u32>,

    camera: types::Camera,
    camera_uniform: types::CameraUniform,
    camera_buffer: wgpu::Buffer,
    camera_bind_group: wgpu::BindGroup,

    shader: wgpu::ShaderModule,
    render_pipelines: Vec<Box<dyn types::RenderPipeline>>,
}
impl Renderer {
    /// Creates a new renderer given a window.
    /// # Requirements:
    /// - any of the supported backends
    /// - all native features must be supported by the rendering device
    pub async fn new(window: &Window) -> anyhow::Result<Self> {
        let size = window.inner_size();

        let instance = wgpu::Instance::new(wgpu::Backends::all());
        let surface = unsafe { instance.create_surface(&window) };

        let adapter = instance
            .request_adapter(&wgpu::RequestAdapterOptions {
                power_preference: wgpu::PowerPreference::HighPerformance,
                compatible_surface: Some(&surface),
                force_fallback_adapter: false,
            })
            .await
            .unwrap();

        let (device, queue) = adapter
            .request_device(
                &wgpu::DeviceDescriptor {
                    label: Some("rendering device"),
                    features: wgpu::Features::all_native_mask(),
                    limits: wgpu::Limits::default(),
                },
                None,
            )
            .await?;

        let config = wgpu::SurfaceConfiguration {
            usage: wgpu::TextureUsages::RENDER_ATTACHMENT,
            format: surface.get_preferred_format(&adapter).unwrap(),
            width: size.width,
            height: size.height,
            present_mode: wgpu::PresentMode::Fifo,
        };

        // the default value, camera can be changed with the `with_camera` method.
        let mut camera = types::Camera {
            eye: (0.0, 0.0, 2.0).into(),
            target: (0.0, 0.0, 0.0).into(),
            up: cgmath::Vector3::unit_y(),
            aspect: config.width as f32 / config.height as f32,
            fovy: 45.0,
            znear: 0.1,
            zfar: 100.0,
        };

        let mut camera_uniform = types::CameraUniform::new();
        camera_uniform.update_view_proj(&camera);

        let camera_buffer = device.create_buffer_init(&wgpu::util::BufferInitDescriptor {
            label: Some("camera buffer"),
            contents: bytemuck::cast_slice(&[camera_uniform]),
            usage: wgpu::BufferUsages::UNIFORM | wgpu::BufferUsages::COPY_DST,
        });

        let camera_bind_group_layout =
            device.create_bind_group_layout(&wgpu::BindGroupLayoutDescriptor {
                label: Some("camera bind group layout"),
                entries: &[wgpu::BindGroupLayoutEntry {
                    binding: 0,
                    visibility: wgpu::ShaderStages::VERTEX,
                    ty: wgpu::BindingType::Buffer {
                        ty: wgpu::BufferBindingType::Uniform,
                        has_dynamic_offset: false,
                        min_binding_size: None,
                    },
                    count: None,
                }],
            });

        let camera_bind_group = device.create_bind_group(&wgpu::BindGroupDescriptor {
            label: Some("camera bind group"),
            layout: &camera_bind_group_layout,
            entries: &[wgpu::BindGroupEntry {
                binding: 0,
                resource: camera_buffer.as_entire_binding(),
            }],
        });

        let shader = device.create_shader_module(&wgpu::ShaderModuleDescriptor {
            label: Some("shader"),
            source: wgpu::ShaderSource::Wgsl(read_file_to_str("shader.wgsl").unwrap().into()),
        });

        Ok(Self {
            clear_color: wgpu::Color {
                r: 0.0,
                g: 0.0,
                b: 0.0,
                a: 1.0,
            },
            size,
            surface,
            device,
            queue,
            config,

            camera,
            camera_uniform,
            camera_buffer,
            camera_bind_group,

            shader,
            render_pipelines: Vec::new(),
        })
    }

    /// Sets the camera of the renderer.
    pub fn with_camera(mut self, camera: types::Camera) -> Self {
        self.camera = camera;
        self.update();
        self
    }

    pub fn add_pipeline(&mut self, pipeline: Box<dyn types::RenderPipeline>) {
        self.render_pipelines.push(pipeline);
    }

    /// Loads a texture from a file.
    pub fn get_texture(
        &self,
        label: &str,
        texture_path: &std::path::Path,
    ) -> anyhow::Result<types::Texture> {
        use std::fs::File;
        use std::io::BufReader;

        let file = File::open(texture_path)?;
        let reader = BufReader::new(file);
        let bytes = reader.buffer();

        types::Texture::from_bytes(label, &self.device, &self.queue, bytes)
    }

    /// Passively updates the state of the Renderer.
    pub fn update(&mut self) {
        self.camera_uniform.update_view_proj(&self.camera);
        self.queue.write_buffer(
            &self.camera_buffer,
            0,
            bytemuck::cast_slice(&[self.camera_uniform]),
        );
    }

    /// Runs on every iteration of the window's event_loop.
    /// Renders without altering the state of the Renderer.
    pub async fn render(&mut self) -> Result<(), wgpu::SurfaceError> {
        // get frame to render to
        let output = self.surface.get_current_texture()?;
        let view = output
            .texture
            .create_view(&wgpu::TextureViewDescriptor::default());

        // used to create actual GPU commands
        let mut encoder = self
            .device
            .create_command_encoder(&wgpu::CommandEncoderDescriptor {
                label: Some("renderer command encoder"),
            });

        {
            let mut render_pass = encoder.begin_render_pass(&wgpu::RenderPassDescriptor {
                label: Some("renderer render pass"),
                color_attachments: &[wgpu::RenderPassColorAttachment {
                    view: &view,
                    resolve_target: None,
                    ops: wgpu::Operations {
                        load: wgpu::LoadOp::Clear(self.clear_color.clone()),
                        store: true,
                    },
                }],
                depth_stencil_attachment: None, // TODO: implement depth buffer
            });

            render_pass.set_bind_group(1, &self.camera_bind_group, &[]);

            for render_pipeline in &self.render_pipelines {
                // render texture?
                match render_pipeline.get_texture() {
                    Some(texture) => {
                        render_pass.set_pipeline(render_pipeline.get_pipeline());
                        render_pass
                            .set_vertex_buffer(1, render_pipeline.get_instance_buffer().slice(..));

                        render_pass.set_bind_group(0, render_pipeline.get_bind_group(), &[]);
                        render_pass
                            .set_vertex_buffer(0, render_pipeline.get_vertex_buffer().slice(..));
                        render_pass.set_index_buffer(
                            render_pipeline.get_index_buffer().slice(..),
                            wgpu::IndexFormat::Uint32,
                        );

                        render_pass.draw_indexed(
                            0..render_pipeline.get_num_indices(),
                            0,
                            render_pipeline.get_instances(),
                        );
                    }
                    None => {}
                }
            }
        }

        self.queue.submit(std::iter::once(encoder.finish()));
        output.present();

        Ok(())
    }
}
