use std::ops::Range;
use wgpu::util::DeviceExt;

use crate::{types, Vertex};

/// Abstraction for the different render pipelines `Renderer` can use.
pub trait RenderPipeline {
    fn get_pipeline(&self) -> &wgpu::RenderPipeline;
    /// Returns a range of instances to be drawn.
    fn get_instances(&self) -> Range<u32>;
    /// Returns the instance buffer.
    fn get_instance_buffer(&self) -> &wgpu::Buffer;
    /// Returns the bind group to be used when rendering.
    fn get_bind_group(&self) -> &wgpu::BindGroup;
    /// Used by `TexturePipeline`.
    fn get_texture(&self) -> Option<&types::Texture> {
        None
    }
    /// Returns the vertex buffer.
    /// # Panics
    /// This function will panic by default, since working with raw vertex buffers isn't a good practice
    /// and therefore requires being overridden by the user.
    fn get_vertex_buffer(&self) -> &wgpu::Buffer {
        unimplemented!()
    }

    /// Returns the index buffer.
    /// # Panics
    /// This function will panic by default, since working with raw index buffers isn't a good practice
    /// and therefore requires being overridden by the user.
    fn get_index_buffer(&self) -> &wgpu::Buffer {
        unimplemented!()
    }

    /// Returns the number of indices.
    /// # Panics
    /// This function will panic by default, since working with raw indices isn't a good practice
    /// and therefore requires being overridden by the user.
    fn get_num_indices(&self) -> u32 {
        unimplemented!()
    }
}

/// A render pipeline used to render a single texture.
/// Can render as many different instances as needed.
pub struct TexturePipeline<'device> {
    device: &'device wgpu::Device,
    render_pipeline: wgpu::RenderPipeline,
    texture: types::Texture,
    texture_buffer: wgpu::Buffer,
    texture_bind_group: wgpu::BindGroup,
    instances: Vec<types::Instance>,
    instances_raw: Vec<types::InstanceRaw>,
    instance_buffer: wgpu::Buffer,

    vertex_buffer: wgpu::Buffer,
    index_buffer: wgpu::Buffer,
    num_indices: usize,
}
impl<'device> TexturePipeline<'device> {
    pub fn new(
        label: &str,
        texture_bytes: &[u8],
        device: &'device wgpu::Device,
        queue: &wgpu::Queue,
        config: &wgpu::SurfaceConfiguration,
        camera_bind_group_layout: &wgpu::BindGroupLayout,
        shader: &wgpu::ShaderModule,
    ) -> anyhow::Result<Self> {
        let texture = types::Texture::from_bytes(label, device, queue, texture_bytes).unwrap();

        let texture_buffer = device.create_buffer_init(&wgpu::util::BufferInitDescriptor {
            label: Some(&(label.to_owned() + " buffer")),
            contents: texture_bytes,
            usage: wgpu::BufferUsages::VERTEX,
        });

        let texture_bind_group_layout =
            device.create_bind_group_layout(&wgpu::BindGroupLayoutDescriptor {
                label: Some(&(label.to_owned() + " bind group layout")),
                entries: &[
                    wgpu::BindGroupLayoutEntry {
                        binding: 0,
                        visibility: wgpu::ShaderStages::FRAGMENT,
                        ty: wgpu::BindingType::Texture {
                            multisampled: false,
                            view_dimension: wgpu::TextureViewDimension::D2,
                            sample_type: wgpu::TextureSampleType::Float { filterable: true },
                        },
                        count: None,
                    },
                    wgpu::BindGroupLayoutEntry {
                        binding: 1,
                        visibility: wgpu::ShaderStages::FRAGMENT,
                        ty: wgpu::BindingType::Sampler(wgpu::SamplerBindingType::Filtering),
                        count: None,
                    },
                ],
            });

        let texture_bind_group = device.create_bind_group(&wgpu::BindGroupDescriptor {
            label: Some(&(label.to_owned() + " bind group")),
            layout: &texture_bind_group_layout,
            entries: &[wgpu::BindGroupEntry {
                binding: 0,
                resource: texture_buffer.as_entire_binding(),
            }],
        });

        let render_pipeline_layout =
            device.create_pipeline_layout(&wgpu::PipelineLayoutDescriptor {
                label: Some("render pipeline layout"),
                bind_group_layouts: &[&texture_bind_group_layout, &camera_bind_group_layout],
                push_constant_ranges: &[],
            });

        let render_pipeline = device.create_render_pipeline(&wgpu::RenderPipelineDescriptor {
            label: Some(&(label.to_owned() + " render pipeline")),
            layout: Some(&render_pipeline_layout),
            vertex: wgpu::VertexState {
                module: &shader,
                entry_point: "vs_main",
                buffers: &[types::ModelVertex::desc(), types::InstanceRaw::desc()],
            },
            fragment: Some(wgpu::FragmentState {
                module: &shader,
                entry_point: "fs_main",
                targets: &[wgpu::ColorTargetState {
                    format: config.format,
                    blend: Some(wgpu::BlendState::REPLACE),
                    write_mask: wgpu::ColorWrites::ALL,
                }],
            }),
            primitive: wgpu::PrimitiveState {
                // every 3 vertices will correspond to 1 triangle
                topology: wgpu::PrimitiveTopology::TriangleList,
                strip_index_format: None,
                // tells WGPU how to tell whether a vertex is facing forwards or not
                // counter-clockwise == facing forward
                front_face: wgpu::FrontFace::Ccw,
                // tells WGPU how to handle vertices that aren't facing forward
                cull_mode: Some(wgpu::Face::Back),
                polygon_mode: wgpu::PolygonMode::Fill, // required
                unclipped_depth: false,
                conservative: false,
            },
            depth_stencil: Some(wgpu::DepthStencilState {
                format: types::Texture::DEPTH_FORMAT,
                depth_write_enabled: true,
                // tells WGPU when to discard a new pixel
                depth_compare: wgpu::CompareFunction::Less,
                stencil: wgpu::StencilState::default(),
                bias: wgpu::DepthBiasState::default(),
            }),
            multisample: wgpu::MultisampleState {
                count: 1,
                mask: !0,                         // use all samples
                alpha_to_coverage_enabled: false, // has to do with anti-aliasing
            },
            multiview: None,
        });

        let instance_buffer = device.create_buffer_init(&wgpu::util::BufferInitDescriptor {
            label: None,
            contents: &[],
            usage: wgpu::BufferUsages::VERTEX,
        });

        let vertex_buffer = device.create_buffer_init(&wgpu::util::BufferInitDescriptor {
            label: None,
            contents: &[],
            usage: wgpu::BufferUsages::VERTEX | wgpu::BufferUsages::COPY_DST,
        });

        let index_buffer = device.create_buffer_init(&wgpu::util::BufferInitDescriptor {
            label: None,
            contents: &[],
            usage: wgpu::BufferUsages::VERTEX | wgpu::BufferUsages::COPY_DST,
        });

        Ok(Self {
            device,
            render_pipeline,
            texture,
            texture_buffer,
            texture_bind_group,
            instances: Vec::new(),
            instances_raw: Vec::new(),
            instance_buffer,
            vertex_buffer,
            index_buffer,
            num_indices: 0usize,
        })
    }

    pub fn add_instance(&mut self, instance: types::Instance) -> anyhow::Result<()> {
        self.instances.push(instance);
        self.instances_raw.push(instance.to_raw());

        self.instance_buffer = self
            .device
            .create_buffer_init(&wgpu::util::BufferInitDescriptor {
                label: None,
                contents: bytemuck::cast_slice(&self.instances_raw),
                usage: wgpu::BufferUsages::VERTEX,
            });

        Ok(())
    }

    pub fn set_vertex_buffer(&mut self, buf: &[u8]) -> anyhow::Result<()> {
        self.vertex_buffer = self
            .device
            .create_buffer_init(&wgpu::util::BufferInitDescriptor {
                label: None,
                contents: buf,
                usage: wgpu::BufferUsages::VERTEX | wgpu::BufferUsages::COPY_DST,
            });

        Ok(())
    }

    pub fn set_index_buffer(&mut self, buf: &[u8]) -> anyhow::Result<()> {
        self.index_buffer = self
            .device
            .create_buffer_init(&wgpu::util::BufferInitDescriptor {
                label: None,
                contents: buf,
                usage: wgpu::BufferUsages::VERTEX | wgpu::BufferUsages::COPY_DST,
            });
        self.num_indices = buf.len();

        Ok(())
    }
}
impl<'a> RenderPipeline for TexturePipeline<'a> {
    fn get_pipeline(&self) -> &wgpu::RenderPipeline {
        &self.render_pipeline
    }

    fn get_instances(&self) -> Range<u32> {
        0..self.instances.len() as u32
    }

    fn get_instance_buffer(&self) -> &wgpu::Buffer {
        &self.instance_buffer
    }

    fn get_bind_group(&self) -> &wgpu::BindGroup {
        &self.texture_bind_group
    }

    fn get_texture(&self) -> Option<&types::Texture> {
        Some(&self.texture)
    }

    fn get_vertex_buffer(&self) -> &wgpu::Buffer {
        &self.vertex_buffer
    }

    fn get_index_buffer(&self) -> &wgpu::Buffer {
        &self.index_buffer
    }

    fn get_num_indices(&self) -> u32 {
        self.num_indices as u32
    }
}
