use cgmath::Matrix4;

/// A matrix used to convert an OpenGL matrix to a WGPU matrix by multiplication.
#[rustfmt::skip]
pub const OPENGL_TO_WGPU_MATRIX: cgmath::Matrix4<f32> = cgmath::Matrix4::new(
    1.0, 0.0, 0.0, 0.0,
    0.0, 1.0, 0.0, 0.0,
    0.0, 0.0, 0.5, 0.0,
    0.0, 0.0, 0.5, 1.0,
);

/// A basic thread-safe camera type.
pub struct Camera {
    pub eye: cgmath::Point3<f32>,
    pub target: cgmath::Point3<f32>,
    pub up: cgmath::Vector3<f32>,
    pub aspect: f32,
    pub fovy: f32,
    pub zfar: f32,
    pub znear: f32,
}
impl Camera {
    pub fn build_view_proj_matrix(&self) -> Matrix4<f32> {
        // moves the world to be at the position and rotation the camera is looking at.
        let view = cgmath::Matrix4::look_at_rh(self.eye, self.target, self.up);
        // wrap the scene to give the effect of depth
        let proj = cgmath::perspective(cgmath::Deg(self.fovy), self.aspect, self.znear, self.zfar);

        OPENGL_TO_WGPU_MATRIX * proj * view // multiplication order is very important when dealing with matrices
    }
}

#[repr(C)]
#[derive(Clone, Copy, Debug, bytemuck::Pod, bytemuck::Zeroable)]
pub struct CameraUniform {
    // cgmath can't be used with bytemuck, so usage of "raw" matrices is required
    view_proj: [[f32; 4]; 4],
}
impl CameraUniform {
    pub fn new() -> Self {
        use cgmath::SquareMatrix;
        Self {
            view_proj: cgmath::Matrix4::identity().into(),
        }
    }

    pub fn update_view_proj(&mut self, camera: &Camera) {
        self.view_proj = camera.build_view_proj_matrix().into();
    }
}

#[derive(Clone, Copy)]
pub enum CameraControllerMovement {
    /// Movement along the X axis.
    X(f32),
    /// Movement along the Y axis.
    Y(f32),
    /// Movement along the Z axis.
    Z(f32),
    /// Rotation around the X axis.
    RotX(f32),
    /// Rotation around the Y axis.
    RotY(f32),
    /// Rotation around the Z axis.
    RotZ(f32),
}

/// Used to move and/or rotate the camera.
pub struct CameraController<'a> {
    camera: &'a mut Camera,
}
impl<'a> CameraController<'a> {
    pub fn new(camera: &'a mut Camera) -> Self {
        Self { camera: camera }
    }

    /// Moves the camera in a single direction at a time.
    pub fn update_camera(&mut self, movement: CameraControllerMovement) {
        use cgmath::InnerSpace;
        // defining directions, used for movement calculations
        let forward = self.camera.target - self.camera.eye;
        let right = forward.normalize().cross(self.camera.up);

        use crate::CameraControllerMovement::*;
        match movement {
            X(x) => self.camera.eye += right.normalize() * x,
            Y(y) => self.camera.eye += self.camera.up * y,
            Z(z) => self.camera.eye += forward.normalize() * z,
            RotX(rot) => {
                self.camera.eye = self.camera.target
                    - (forward + self.camera.up * rot).normalize() * forward.magnitude()
            }
            RotY(rot) => {
                self.camera.eye =
                    self.camera.target - (forward + forward * rot).normalize() * forward.magnitude()
            }
            RotZ(rot) => {
                self.camera.eye = self.camera.target
                    - (forward + self.camera.up * rot).normalize() * forward.magnitude()
            }
        }
    }

    /// Runs `update_camera` on every movement provided.
    pub fn update_camera_batch(&mut self, movements: &[CameraControllerMovement]) {
        for movement in movements {
            self.update_camera(*movement);
        }
    }
}
