//! EventHandler trait and event parameter structures.

#![allow(unused_variables)]

use windows::Win32::Foundation::*;

#[cfg(feature = "raw_input")]
use crate::raw_input;
use crate::{device::*, geometry::*, ime::*, window::Window};
use std::path::Path;

/// A `EventHandler::draw` parameter.
pub struct Draw<'a> {
    pub window: &'a Window,
}

/// A `EventHandler::activated` parameter.
pub struct Activated<'a> {
    pub window: &'a Window,
}

/// A `EventHandler::inactivated` parameter.
pub struct Inactivated<'a> {
    pub window: &'a Window,
}

/// A `EventHandler::closed` parameter.
pub struct Closed<'a> {
    pub window: &'a Window,
}

/// `EventHandler::moved` parameters.
pub struct Moved<'a> {
    pub window: &'a Window,
    pub position: ScreenPosition,
}

/// `EventHandler::resizing` parameters.
pub struct Resizing<'a> {
    pub window: &'a Window,
    pub size: PhysicalSize<u32>,
}

/// `EventHandler::resized` parameters.
pub struct Resized<'a> {
    pub window: &'a Window,
    pub size: PhysicalSize<u32>,
}

/// `EventHandler::dpi_changed` parameters.
pub struct DpiChanged<'a> {
    pub window: &'a Window,
    pub new_dpi: u32,
}

/// `EventHandler::mouse_input` parameters.
pub struct MouseInput<'a> {
    pub window: &'a Window,
    pub button: MouseButton,
    pub button_state: KeyState,
    pub mouse_state: MouseState<'a>,
}

/// `EventHandler::cursor_moved` parameters.
pub struct CursorMoved<'a> {
    pub window: &'a Window,
    pub mouse_state: MouseState<'a>,
}

/// `EventHandler::cursor_entered` parameters.
pub struct CursorEntered<'a> {
    pub window: &'a Window,
    pub mouse_state: MouseState<'a>,
}

/// `EventHandler::cursor_leaved` parameters.
pub struct CursorLeaved<'a> {
    pub window: &'a Window,
    pub mouse_state: MouseState<'a>,
}

/// `EventHandler::mouse_wheel` parameters.
pub struct MouseWheel<'a> {
    pub window: &'a Window,
    pub axis: MouseWheelAxis,
    pub distance: i32,
    pub mouse_state: MouseState<'a>,
}

/// `EventHandler::key_input` parameters.
pub struct KeyInput<'a> {
    pub window: &'a Window,
    pub key_code: KeyCode,
    pub state: KeyState,
    pub prev_pressed: bool,
}

/// `EventHandler::char_input` parameters.
pub struct CharInput<'a> {
    pub window: &'a Window,
    pub c: char,
}

/// A `EventHandler::ime_start_composition` parameter.
pub struct ImeStartComposition<'a> {
    pub window: &'a Window,
}

/// `EventHandler::ime_composition` parameters.
pub struct ImeComposition<'a> {
    pub window: &'a Window,
    pub composition: &'a Composition,
    pub candidate_list: Option<&'a CandidateList>,
}

/// `EventHandler::ime_end_composition` parameters.
pub struct ImeEndComposition<'a> {
    pub window: &'a Window,
    pub result: Option<&'a str>,
}

/// `EventHandler::drop_files` parameters.
pub struct DropFiles<'a> {
    pub window: &'a Window,
    pub paths: &'a [&'a Path],
    pub position: PhysicalPosition<i32>,
}

/// `EventHandler::raw_input` parameters.
#[cfg(feature = "raw_input")]
pub struct RawInput<'a> {
    pub window: &'a Window,
    pub data: &'a raw_input::InputData,
}

/// `EventHandler::raw_input_device_change` parameters.
#[cfg(feature = "raw_input")]
pub struct RawInputDeviceChange<'a> {
    pub window: &'a Window,
    pub device: &'a raw_input::Device,
    pub state: raw_input::DeviceChangeState,
}

/// `EventHandler::other` parameters.
#[derive(Debug)]
pub struct Other {
    pub hwnd: HWND,
    pub message: u32,
    pub wparam: WPARAM,
    pub lparam: LPARAM,
}

/// Trait that must implements for handling events.
pub trait EventHandler {
    /// This is called when there are no events.
    ///
    /// only passed `RunType::Idle` to `Context::run`.
    fn idle(&mut self) {}

    /// This is called before a event.
    ///
    /// only passed `RunType::Idle` to `Context::run`.
    fn pre_processing(&mut self) {}

    /// This is called after a event.
    ///
    /// only passed `RunType::Idle` to `Context::run`.
    fn post_processing(&mut self) {}

    /// This is called when the window needs redrawing.
    fn draw(&mut self, ev: Draw) {}

    /// This is called when the window has been activated.
    fn activated(&mut self, ev: Activated) {}

    /// This is called when the window has been inactivated.
    fn inactivated(&mut self, ev: Inactivated) {}

    /// This is called when the window has been closed.
    fn closed(&mut self, ev: Closed) {}

    /// This is called when the window has been moved.
    fn moved(&mut self, ev: Moved) {}

    /// This is called when the window is resizing.
    fn resizing(&mut self, ev: Resizing) {}

    /// This is called when the window has been resized.
    fn resized(&mut self, ev: Resized) {}

    /// This is called when the window's DPI has been changed.
    fn dpi_changed(&mut self, ev: DpiChanged) {}

    /// This is called when the mouse button has been pressed and released on the window.
    fn mouse_input(&mut self, ev: MouseInput) {}

    /// This is called when the cursor has been moved on the window.
    fn cursor_moved(&mut self, ev: CursorMoved) {}

    /// This is called when the cursor has been entered the window.
    fn cursor_entered(&mut self, ev: CursorEntered) {}

    /// This is called when the cursor has been leaved the window.
    fn cursor_leaved(&mut self, ev: CursorLeaved) {}

    /// This is called when the mouse wheel has been rotated or tilted.
    fn mouse_wheel(&mut self, ev: MouseWheel) {}

    /// This is called when the keyboard key has been pressed and released.
    fn key_input(&mut self, ev: KeyInput) {}

    /// This is called when the keyboard key has been inputed the character.
    fn char_input(&mut self, ev: CharInput) {}

    /// This is called when the IME starts composition.
    fn ime_start_composition(&mut self, ev: ImeStartComposition) {}

    /// This is called when the IME composition status has been changed.
    fn ime_composition(&mut self, ev: ImeComposition) {}

    /// This is called when the IME ends composition.
    fn ime_end_composition(&mut self, ev: ImeEndComposition) {}

    /// This is called when files have been dropped on the window.
    fn drop_files(&mut self, ev: DropFiles) {}

    /// This is called when raw data has been inputed.
    #[cfg(feature = "raw_input")]
    fn raw_input(&mut self, ev: RawInput) {}

    /// This is called when a device state has been changead.
    #[cfg(feature = "raw_input")]
    fn raw_input_device_change(&mut self, ev: RawInputDeviceChange) {}

    /// This is called when none of defined events.
    fn other(&mut self, ev: Other) -> Option<isize> {
        None
    }
}
