use crate::geometry::*;
use std::sync::Once;
use windows::Win32::{Foundation::*, Graphics::Gdi::*, UI::HiDpi::*, UI::WindowsAndMessaging::*};

pub fn get_dpi_from_point(pt: ScreenPosition) -> u32 {
    unsafe {
        let mut dpi_x = 0;
        let mut _dpi_y = 0;
        GetDpiForMonitor(
            MonitorFromPoint(POINT { x: pt.x, y: pt.y }, MONITOR_DEFAULTTOPRIMARY),
            MDT_DEFAULT,
            &mut dpi_x,
            &mut _dpi_y,
        )
        .ok();
        dpi_x
    }
}

pub fn adjust_window_rect(
    size: PhysicalSize<u32>,
    style: WINDOW_STYLE,
    ex_style: WINDOW_EX_STYLE,
    dpi: u32,
) -> RECT {
    unsafe {
        let mut rc = RECT {
            left: 0,
            top: 0,
            right: size.width as i32,
            bottom: size.height as i32,
        };
        AdjustWindowRectExForDpi(&mut rc, style, false, ex_style, dpi);
        rc
    }
}

pub fn enable_dpi_awareness() {
    static ENABLE_DPI_AWARENESS: Once = Once::new();
    unsafe {
        ENABLE_DPI_AWARENESS.call_once(|| {
            if !SetProcessDpiAwarenessContext(DPI_AWARENESS_CONTEXT_PER_MONITOR_AWARE_V2).as_bool()
                && !SetProcessDpiAwarenessContext(DPI_AWARENESS_CONTEXT_PER_MONITOR_AWARE).as_bool()
            {
                SetProcessDpiAwareness(PROCESS_PER_MONITOR_DPI_AWARE).ok();
            }
        });
    }
}

pub fn enable_gui_thread() {
    unsafe {
        IsGUIThread(true);
    }
}

pub fn child_windows(hwnd: HWND) -> Vec<HWND> {
    unsafe extern "system" fn callback(hwnd: HWND, lparam: LPARAM) -> BOOL {
        let v = (lparam.0 as *mut Vec<HWND>).as_mut().unwrap();
        v.push(hwnd);
        true.into()
    }

    unsafe {
        let mut v = Vec::new();
        EnumChildWindows(hwnd, Some(callback), LPARAM(&mut v as *mut Vec<HWND> as _));
        v
    }
}
