use crate::co;
use crate::co::prelude::FormattedError;

const_no_debug_display! { HRESULT: u32;
	/// A native COM
	/// [error code](https://docs.microsoft.com/en-us/openspecs/windows_protocols/ms-erref/0642cb2f-2075-4469-918c-4441e69c548a)
	/// (`u32`).
	///
	/// Implements the standard
	/// [`Error`](https://doc.rust-lang.org/beta/std/error/trait.Error.html)
	/// trait.
	///
	/// Implements the
	/// [`Debug`](https://doc.rust-lang.org/std/fmt/trait.Debug.html) and
	/// [`Display`](https://doc.rust-lang.org/std/fmt/trait.Display.html) traits
	/// to show the error code along with the error description, taken from
	/// [`FormattedError`](crate::prelude::FormattedError) trait. For example,
	/// the code below:
	///
	/// ```rust,ignore
	/// use winsafe::co::HRESULT;
	///
	/// println!("{}", HRESULT::E_INVALIDARG);
	/// ```
	///
	/// Will print:
	///
	/// ```text
	/// [0x80070057 2147942487] The parameter is incorrect.
	/// ```
	=>
	=>
	/// Operation successful.
	S_OK 0
	S_FALSE 1

	E_UNEXPECTED 0x8000_ffff
	E_NOTIMPL 0x8000_4001
	E_OUTOFMEMORY 0x8007_000e
	E_INVALIDARG 0x8007_0057
	E_NOINTERFACE 0x8000_4002
	E_POINTER 0x8000_4003
	E_HANDLE 0x8007_0006
	E_ABORT 0x8000_4004
	E_FAIL 0x8000_4005
	E_ACCESSDENIED 0x8007_0005
	E_PENDING 0x8000_000a
	E_BOUNDS 0x8000_000b
	E_CHANGED_STATE 0x8000_000c
	E_ILLEGAL_STATE_CHANGE 0x8000_000d
	E_ILLEGAL_METHOD_CALL 0x8000_000e

	RO_E_METADATA_NAME_NOT_FOUND 0x8000_000f
	RO_E_METADATA_NAME_IS_NAMESPACE 0x8000_0010
	RO_E_METADATA_INVALID_TYPE_FORMAT 0x8000_0011
	RO_E_INVALID_METADATA_FILE 0x8000_0012
	RO_E_CLOSED 0x8000_0013
	RO_E_EXCLUSIVE_WRITE 0x8000_0014
	RO_E_CHANGE_NOTIFICATION_IN_PROGRESS 0x8000_0015
	RO_E_ERROR_STRING_NOT_FOUND 0x8000_0016

	E_STRING_NOT_NULL_TERMINATED 0x8000_0017
	E_ILLEGAL_DELEGATE_ASSIGNMENT 0x8000_0018
	E_ASYNC_OPERATION_NOT_STARTED 0x8000_0019
	E_APPLICATION_EXITING 0x8000_001a
	E_APPLICATION_VIEW_EXITING 0x8000_001b

	RO_E_MUST_BE_AGILE 0x8000_001c
	RO_E_UNSUPPORTED_FROM_MTA 0x8000_001d
	RO_E_COMMITTED 0x8000_001e
	RO_E_BLOCKED_CROSS_ASTA_CALL 0x8000_001f
	RO_E_CANNOT_ACTIVATE_FULL_TRUST_SERVER 0x8000_0020
	RO_E_CANNOT_ACTIVATE_UNIVERSAL_APPLICATION_SERVER 0x8000_0021

	RPC_E_CALL_REJECTED 0x8001_0001
	RPC_E_CALL_CANCELED 0x8001_0002
	RPC_E_CANTPOST_INSENDCALL 0x8001_0003
	RPC_E_CANTCALLOUT_INASYNCCALL 0x8001_0004
	RPC_E_CANTCALLOUT_INEXTERNALCALL 0x8001_0005
	RPC_E_CONNECTION_TERMINATED 0x8001_0006
	RPC_E_SERVER_DIED 0x8001_0007
	RPC_E_CLIENT_DIED 0x8001_0008
	RPC_E_INVALID_DATAPACKET 0x8001_0009
	RPC_E_CANTTRANSMIT_CALL 0x8001_000a
	RPC_E_CLIENT_CANTMARSHAL_DATA 0x8001_000b
	RPC_E_CLIENT_CANTUNMARSHAL_DATA 0x8001_000c
	RPC_E_SERVER_CANTMARSHAL_DATA 0x8001_000d
	RPC_E_SERVER_CANTUNMARSHAL_DATA 0x8001_000e
	RPC_E_INVALID_DATA 0x8001_000f
	RPC_E_INVALID_PARAMETER 0x8001_0010
	RPC_E_CANTCALLOUT_AGAIN 0x8001_0011
	RPC_E_SERVER_DIED_DNE 0x8001_0012
	RPC_E_SYS_CALL_FAILED 0x8001_0100
	RPC_E_OUT_OF_RESOURCES 0x8001_0101
	RPC_E_ATTEMPTED_MULTITHREAD 0x8001_0102
	RPC_E_NOT_REGISTERED 0x8001_0103
	RPC_E_FAULT 0x8001_0104
	RPC_E_SERVERFAULT 0x8001_0105
	RPC_E_CHANGED_MODE 0x8001_0106
	RPC_E_INVALIDMETHOD 0x8001_0107
	RPC_E_DISCONNECTED 0x8001_0108
	RPC_E_RETRY 0x8001_0109
	RPC_E_SERVERCALL_RETRYLATER 0x8001_010a
	RPC_E_SERVERCALL_REJECTED 0x8001_010b
	RPC_E_INVALID_CALLDATA 0x8001_010c
	RPC_E_CANTCALLOUT_ININPUTSYNCCALL 0x8001_010d
	RPC_E_WRONG_THREAD 0x8001_010e
	RPC_E_THREAD_NOT_INIT 0x8001_010f
	RPC_E_VERSION_MISMATCH 0x8001_0110
	RPC_E_INVALID_HEADER 0x8001_0111
	RPC_E_INVALID_EXTENSION 0x8001_0112
	RPC_E_INVALID_IPID 0x8001_0113
	RPC_E_INVALID_OBJECT 0x8001_0114
	RPC_E_CALL_COMPLETE 0x8001_0117
	RPC_E_UNSECURE_CALL 0x8001_0118
	RPC_E_TOO_LATE 0x8001_0119
	RPC_E_NO_GOOD_SECURITY_PACKAGES 0x8001_011a
	RPC_E_ACCESS_DENIED 0x8001_011b
	RPC_E_REMOTE_DISABLED 0x8001_011c
	RPC_E_INVALID_OBJREF 0x8001_011d
	RPC_E_NO_CONTEXT 0x8001_011e
	RPC_E_TIMEOUT 0x8001_011f
	RPC_E_NO_SYNC 0x8001_0120
	RPC_E_FULLSIC_REQUIRED 0x8001_0121
	RPC_E_INVALID_STD_NAME 0x8001_0122

	CO_E_FAILEDTOIMPERSONATE 0x8001_0123
	CO_E_FAILEDTOGETSECCTX 0x8001_0124
	CO_E_FAILEDTOOPENTHREADTOKEN 0x8001_0125
	CO_E_FAILEDTOGETTOKENINFO 0x8001_0126
	CO_E_TRUSTEEDOESNTMATCHCLIENT 0x8001_0127
	CO_E_FAILEDTOQUERYCLIENTBLANKET 0x8001_0128
	CO_E_FAILEDTOSETDACL 0x8001_0129
	CO_E_ACCESSCHECKFAILED 0x8001_012a
	CO_E_NETACCESSAPIFAILED 0x8001_012b
	CO_E_WRONGTRUSTEENAMESYNTAX 0x8001_012c
	CO_E_INVALIDSID 0x8001_012d
	CO_E_CONVERSIONFAILED 0x8001_012e
	CO_E_NOMATCHINGSIDFOUND 0x8001_012f
	CO_E_LOOKUPACCSIDFAILED 0x8001_0130
	CO_E_NOMATCHINGNAMEFOUND 0x8001_0131
	CO_E_LOOKUPACCNAMEFAILED 0x8001_0132
	CO_E_SETSERLHNDLFAILED 0x8001_0133
	CO_E_FAILEDTOGETWINDIR 0x8001_0134
	CO_E_PATHTOOLONG 0x8001_0135
	CO_E_FAILEDTOGENUUID 0x8001_0136
	CO_E_FAILEDTOCREATEFILE 0x8001_0137
	CO_E_FAILEDTOCLOSEHANDLE 0x8001_0138
	CO_E_EXCEEDSYSACLLIMIT 0x8001_0139
	CO_E_ACESINWRONGORDER 0x8001_013a
	CO_E_INCOMPATIBLESTREAMVERSION 0x8001_013b
	CO_E_FAILEDTOOPENPROCESSTOKEN 0x8001_013c
	CO_E_DECODEFAILED 0x8001_013d
	CO_E_ACNOTINITIALIZED 0x8001_013f
	CO_E_CANCEL_DISABLED 0x8001_0140

	RPC_E_UNEXPECTED 0x8001_ffff

	ERROR_AUDITING_DISABLED 0xc009_0001
	ERROR_ALL_SIDS_FILTERED 0xc009_0002
	ERROR_BIZRULES_NOT_ENABLED 0xc009_0003

	CO_E_NOTINITIALIZED 0x8004_01f0
	CO_E_ALREADYINITIALIZED 0x8004_01f1
	CO_E_CANTDETERMINECLASS 0x8004_01f2
	CO_E_CLASSSTRING 0x8004_01f3
	CO_E_IIDSTRING 0x8004_01f4
	CO_E_APPNOTFOUND 0x8004_01f5
	CO_E_APPSINGLEUSE 0x8004_01f6
	CO_E_ERRORINAPP 0x8004_01f7
	CO_E_DLLNOTFOUND 0x8004_01f8
	CO_E_ERRORINDLL 0x8004_01f9
	CO_E_WRONGOSFORAPP 0x8004_01fa
	CO_E_OBJNOTREG 0x8004_01fb
	CO_E_OBJISREG 0x8004_01fc
	CO_E_OBJNOTCONNECTED 0x8004_01fd
	CO_E_APPDIDNTREG 0x8004_01fe
	CO_E_RELEASED 0x8004_01ff

	MF_E_PLATFORM_NOT_INITIALIZED 0xc00d_36b0
	MF_E_BUFFERTOOSMALL 0xc00d_36b1
	MF_E_INVALIDREQUEST 0xc00d_36b2
	MF_E_INVALIDSTREAMNUMBER 0xc00d_36b3
	MF_E_INVALIDMEDIATYPE 0xc00d_36b4
	MF_E_NOTACCEPTING 0xc00d_36b5
	MF_E_NOT_INITIALIZED 0xc00d_36b6
	MF_E_UNSUPPORTED_REPRESENTATION 0xc00d_36b7
	MF_E_NO_MORE_TYPES 0xc00d_36b9
	MF_E_UNSUPPORTED_SERVICE 0xc00d_36ba
	MF_E_UNEXPECTED 0xc00d_36bb
	MF_E_INVALIDNAME 0xc00d_36bc
	MF_E_INVALIDTYPE 0xc00d_36bd
	MF_E_INVALID_FILE_FORMAT 0xc00d_36be
	MF_E_INVALIDINDEX 0xc00d_36bf
	MF_E_INVALID_TIMESTAMP 0xc00d_36c0
	MF_E_UNSUPPORTED_SCHEME 0xc00d_36c3
	MF_E_UNSUPPORTED_BYTESTREAM_TYPE 0xc00d_36c4
	MF_E_UNSUPPORTED_TIME_FORMAT 0xc00d_36c5
	MF_E_NO_SAMPLE_TIMESTAMP 0xc00d_36c8
	MF_E_NO_SAMPLE_DURATION 0xc00d_36c9
	MF_E_INVALID_STREAM_DATA 0xc00d_36cb
	MF_E_RT_UNAVAILABLE 0xc00d_36cf
	MF_E_UNSUPPORTED_RATE 0xc00d_36d0
	MF_E_THINNING_UNSUPPORTED 0xc00d_36d1
	MF_E_REVERSE_UNSUPPORTED 0xc00d_36d2
	MF_E_UNSUPPORTED_RATE_TRANSITION 0xc00d_36d3
	MF_E_RATE_CHANGE_PREEMPTED 0xc00d_36d4
	MF_E_NOT_FOUND 0xc00d_36d5
	MF_E_NOT_AVAILABLE 0xc00d_36d6
	MF_E_NO_CLOCK 0xc00d_36d7
	MF_S_MULTIPLE_BEGIN 0x000d_36d8
	MF_E_MULTIPLE_BEGIN 0xc00d_36d9
	MF_E_MULTIPLE_SUBSCRIBERS 0xc00d_36da
	MF_E_TIMER_ORPHANED 0xc00d_36db
	MF_E_STATE_TRANSITION_PENDING 0xc00d_36dc
	MF_E_UNSUPPORTED_STATE_TRANSITION 0xc00d_36dd
	MF_E_UNRECOVERABLE_ERROR_OCCURRED 0xc00d_36de
	MF_E_SAMPLE_HAS_TOO_MANY_BUFFERS 0xc00d_36df
	MF_E_SAMPLE_NOT_WRITABLE 0xc00d_36e0
	MF_E_INVALID_KEY 0xc00d_36e2
	MF_E_BAD_STARTUP_VERSION 0xc00d_36e3
	MF_E_UNSUPPORTED_CAPTION 0xc00d_36e4
	MF_E_INVALID_POSITION 0xc00d_36e5
	MF_E_ATTRIBUTENOTFOUND 0xc00d_36e6
	MF_E_PROPERTY_TYPE_NOT_ALLOWED 0xc00d_36e7
	MF_E_TOPO_INVALID_OPTIONAL_NODE 0xc00d_520e
	MF_E_TOPO_CANNOT_FIND_DECRYPTOR 0xc00d_5211
	MF_E_TOPO_CODEC_NOT_FOUND 0xc00d_5212
	MF_E_TOPO_CANNOT_CONNECT 0xc00d_5213
	MF_E_TOPO_UNSUPPORTED 0xc00d_5214
	MF_E_TOPO_INVALID_TIME_ATTRIBUTES 0xc00d_5215
	MF_E_TOPO_LOOPS_IN_TOPOLOGY 0xC00d_5216
	MF_E_TOPO_MISSING_PRESENTATION_DESCRIPTOR 0xc00d_5217
	MF_E_TOPO_MISSING_STREAM_DESCRIPTOR 0xc00d_5218
	MF_E_TOPO_STREAM_DESCRIPTOR_NOT_SELECTED 0xc00d_5219
	MF_E_TOPO_MISSING_SOURCE 0xc00d_521a
	MF_E_TOPO_SINK_ACTIVATES_UNSUPPORTED 0xc00d_521b
	MF_E_TRANSFORM_TYPE_NOT_SET 0xc00d_6d60
}

impl std::error::Error for HRESULT {
	fn source(&self) -> Option<&(dyn std::error::Error + 'static)> {
		None
	}
}

impl std::fmt::Debug for HRESULT {
	fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
		if self.0 > 0xffff {
			write!(f, "[{:#010x} {}] {}",
				self.0, self.0, self.FormatMessage())
		} else {
			write!(f, "[{:#06x} {}] {}",
				self.0, self.0, self.FormatMessage())
		}
	}
}

impl std::fmt::Display for HRESULT {
	fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
		<Self as std::fmt::Debug>::fmt(self, f) // delegate to Debug trait
	}
}

impl FormattedError for HRESULT {}

impl HRESULT {
	/// [`HRESULT_CODE`](https://docs.microsoft.com/en-us/windows/win32/api/winerror/nf-winerror-hresult_code)
	/// method. Originally a macro.
	pub fn code(self) -> u16 {
		(self.0 & 0xffff) as u16
	}

	/// [`HRESULT_FACILITY`](https://docs.microsoft.com/en-us/windows/win32/api/winerror/nf-winerror-hresult_facility)
	/// method. Originally a macro.
	pub fn facility(self) -> co::FACILITY {
		co::FACILITY((self.0 >> 16) & 0x1fff)
	}

	/// [`HRESULT_SEVERITY`](https://docs.microsoft.com/en-us/windows/win32/api/winerror/nf-winerror-hresult_severity)
	/// method. Originally a macro.
	pub fn severity(self) -> u8 {
		((self.0 >> 31) & 0x1) as u8
	}
}
