#![allow(non_snake_case, non_upper_case_globals)]

use crate::co;
use crate::ffi::kernel32;
use crate::funcs::GetLastError;
use crate::handles::HLOCAL;
use crate::structs::LANGID;
use crate::various::WString;

pub_struct_const_no_debug_display! { ERROR, u32,
	/// A Windows
	/// [system error code](https://docs.microsoft.com/en-us/windows/win32/debug/system-error-codes)
	/// (`u32`) retrieved by
	/// [`GetLastError`](crate::GetLastError) function, or an
	/// [`HRESULT`](https://docs.microsoft.com/en-us/openspecs/windows_protocols/ms-erref/0642cb2f-2075-4469-918c-4441e69c548a).
	///
	/// Implements the
	/// [`Error`](https://doc.rust-lang.org/beta/std/error/trait.Error.html)
	/// trait.
	///
	/// Implements the
	/// [`Debug`](https://doc.rust-lang.org/std/fmt/trait.Debug.html) and
	/// [`Display`](https://doc.rust-lang.org/std/fmt/trait.Display.html) traits
	/// to show the error code along with the error description, taken from
	/// [`ERROR::FormatMessage`](crate::co::ERROR::FormatMessage). For example,
	/// the code below:
	///
	/// ```rust,ignore
	/// use winsafe::co::ERROR;
	///
	/// println!("{}", ERROR::OUT_OF_PAPER);
	/// ```
	///
	/// Will print:
	///
	/// ```
	/// [0x001c 28] The printer is out of paper.
	/// ```
	=>
	/// The operation completed successfully.
	SUCCESS, 0
	INVALID_FUNCTION, 1
	FILE_NOT_FOUND, 2
	PATH_NOT_FOUND, 3
	TOO_MANY_OPEN_FILES, 4
	ACCESS_DENIED, 5
	INVALID_HANDLE, 6
	ARENA_TRASHED, 7
	NOT_ENOUGH_MEMORY, 8
	INVALID_BLOCK, 9
	BAD_ENVIRONMENT, 10
	BAD_FORMAT, 11
	INVALID_ACCESS, 12
	INVALID_DATA, 13
	OUTOFMEMORY, 14
	INVALID_DRIVE, 15
	CURRENT_DIRECTORY, 16
	NOT_SAME_DEVICE, 17
	NO_MORE_FILES, 18
	WRITE_PROTECT, 19
	BAD_UNIT, 20
	NOT_READY, 21
	BAD_COMMAND, 22
	CRC, 23
	BAD_LENGTH, 24
	SEEK, 25
	NOT_DOS_DISK, 26
	SECTOR_NOT_FOUND, 27
	OUT_OF_PAPER, 28
	WRITE_FAULT, 29
	READ_FAULT, 30
	GEN_FAILURE, 31
	SHARING_VIOLATION, 32
	LOCK_VIOLATION, 33
	WRONG_DISK, 34
	SHARING_BUFFER_EXCEEDED, 36
	HANDLE_EOF, 38
	HANDLE_DISK_FULL, 39
	NOT_SUPPORTED, 50
	REM_NOT_LIST, 51
	DUP_NAME, 52
	BAD_NETPATH, 53
	NETWORK_BUSY, 54
	DEV_NOT_EXIST, 55
	TOO_MANY_CMDS, 56
	ADAP_HDW_ERR, 57
	BAD_NET_RESP, 58
	UNEXP_NET_ERR, 59
	BAD_REM_ADAP, 60
	PRINTQ_FULL, 61
	NO_SPOOL_SPACE, 62
	PRINT_CANCELLED, 63
	NETNAME_DELETED, 64
	NETWORK_ACCESS_DENIED, 65
	BAD_DEV_TYPE, 66
	BAD_NET_NAME, 67
	TOO_MANY_NAMES, 68
	TOO_MANY_SESS, 69
	SHARING_PAUSED, 70
	REQ_NOT_ACCEP, 71
	REDIR_PAUSED, 72
	FILE_EXISTS, 80
	CANNOT_MAKE, 82
	FAIL_I24, 83
	OUT_OF_STRUCTURES, 84
	ALREADY_ASSIGNED, 85
	INVALID_PASSWORD, 86
	INVALID_PARAMETER, 87
	NET_WRITE_FAULT, 88
	NO_PROC_SLOTS, 89
	TOO_MANY_SEMAPHORES, 100
	EXCL_SEM_ALREADY_OWNED, 101
	SEM_IS_SET, 102
	TOO_MANY_SEM_REQUESTS, 103
	INVALID_AT_INTERRUPT_TIME, 104
	SEM_OWNER_DIED, 105
	SEM_USER_LIMIT, 106
	DISK_CHANGE, 107
	DRIVE_LOCKED, 108
	BROKEN_PIPE, 109
	OPEN_FAILED, 110
	BUFFER_OVERFLOW, 111
	DISK_FULL, 112
	NO_MORE_SEARCH_HANDLES, 113
	INVALID_TARGET_HANDLE, 114
	INVALID_CATEGORY, 117
	INVALID_VERIFY_SWITCH, 118
	BAD_DRIVER_LEVEL, 119
	CALL_NOT_IMPLEMENTED, 120
	SEM_TIMEOUT, 121
	INSUFFICIENT_BUFFER, 122
	INVALID_NAME, 123
	INVALID_LEVEL, 124
	NO_VOLUME_LABEL, 125
	MOD_NOT_FOUND, 126
	PROC_NOT_FOUND, 127
	WAIT_NO_CHILDREN, 128
	CHILD_NOT_COMPLETE, 129
	DIRECT_ACCESS_HANDLE, 130
	NEGATIVE_SEEK, 131
	SEEK_ON_DEVICE, 132
	IS_JOIN_TARGET, 133
	IS_JOINED, 134
	IS_SUBSTED, 135
	NOT_JOINED, 136
	NOT_SUBSTED, 137
	JOIN_TO_JOIN, 138
	SUBST_TO_SUBST, 139
	JOIN_TO_SUBST, 140
	SUBST_TO_JOIN, 141
	BUSY_DRIVE, 142
	SAME_DRIVE, 143
	DIR_NOT_ROOT, 144
	DIR_NOT_EMPTY, 145
	IS_SUBST_PATH, 146
	IS_JOIN_PATH, 147
	PATH_BUSY, 148
	IS_SUBST_TARGET, 149
	SYSTEM_TRACE, 150
	INVALID_EVENT_COUNT, 151
	TOO_MANY_MUXWAITERS, 152
	INVALID_LIST_FORMAT, 153
	LABEL_TOO_LONG, 154
	TOO_MANY_TCBS, 155
	SIGNAL_REFUSED, 156
	DISCARDED, 157
	NOT_LOCKED, 158
	BAD_THREADID_ADDR, 159
	BAD_ARGUMENTS, 160
	BAD_PATHNAME, 161
	SIGNAL_PENDING, 162
	MAX_THRDS_REACHED, 164
	LOCK_FAILED, 167
	BUSY, 170
	DEVICE_SUPPORT_IN_PROGRESS, 171
	CANCEL_VIOLATION, 173
	ATOMIC_LOCKS_NOT_SUPPORTED, 174
	INVALID_SEGMENT_NUMBER, 180
	INVALID_ORDINAL, 182
	ALREADY_EXISTS, 183
	INVALID_FLAG_NUMBER, 186
	SEM_NOT_FOUND, 187
	INVALID_STARTING_CODESEG, 188
	INVALID_STACKSEG, 189
	INVALID_MODULETYPE, 190
	INVALID_EXE_SIGNATURE, 191
	EXE_MARKED_INVALID, 192
	BAD_EXE_FORMAT, 193
	ITERATED_DATA_EXCEEDS_64k, 194
	INVALID_MINALLOCSIZE, 195
	DYNLINK_FROM_INVALID_RING, 196
	IOPL_NOT_ENABLED, 197
	INVALID_SEGDPL, 198
	AUTODATASEG_EXCEEDS_64k, 199
	RING2SEG_MUST_BE_MOVABLE, 200
	RELOC_CHAIN_XEEDS_SEGLIM, 201
	INFLOOP_IN_RELOC_CHAIN, 202
	ENVVAR_NOT_FOUND, 203
	NO_SIGNAL_SENT, 205
	FILENAME_EXCED_RANGE, 206
	RING2_STACK_IN_USE, 207
	META_EXPANSION_TOO_LONG, 208
	INVALID_SIGNAL_NUMBER, 209
	THREAD_1_INACTIVE, 210
	LOCKED, 212
	TOO_MANY_MODULES, 214
	NESTING_NOT_ALLOWED, 215
	EXE_MACHINE_TYPE_MISMATCH, 216
	EXE_CANNOT_MODIFY_SIGNED_BINARY, 217
	EXE_CANNOT_MODIFY_STRONG_SIGNED_BINARY, 218
	FILE_CHECKED_OUT, 220
	CHECKOUT_REQUIRED, 221
	BAD_FILE_TYPE, 222
	FILE_TOO_LARGE, 223
	FORMS_AUTH_REQUIRED, 224
	VIRUS_INFECTED, 225
	VIRUS_DELETED, 226
	PIPE_LOCAL, 229
	BAD_PIPE, 230
	PIPE_BUSY, 231
	NO_DATA, 232
	PIPE_NOT_CONNECTED, 233
	MORE_DATA, 234
	NO_WORK_DONE, 235
	VC_DISCONNECTED, 240
	INVALID_EA_NAME, 254
	EA_LIST_INCONSISTENT, 255
	NO_MORE_ITEMS, 259
	CANNOT_COPY, 266
	DIRECTORY, 267
	EAS_DIDNT_FIT, 275
	EA_FILE_CORRUPT, 276
	EA_TABLE_FULL, 277
	INVALID_EA_HANDLE, 278
	EAS_NOT_SUPPORTED, 282
	NOT_OWNER, 288
	TOO_MANY_POSTS, 298
	PARTIAL_COPY, 299
	OPLOCK_NOT_GRANTED, 300
	INVALID_OPLOCK_PROTOCOL, 301
	DISK_TOO_FRAGMENTED, 302
	DELETE_PENDING, 303
	INCOMPATIBLE_WITH_GLOBAL_SHORT_NAME_REGISTRY_SETTING, 304
	SHORT_NAMES_NOT_ENABLED_ON_VOLUME, 305
	SECURITY_STREAM_IS_INCONSISTENT, 306
	INVALID_LOCK_RANGE, 307
	IMAGE_SUBSYSTEM_NOT_PRESENT, 308
	NOTIFICATION_GUID_ALREADY_DEFINED, 309
	INVALID_EXCEPTION_HANDLER, 310
	DUPLICATE_PRIVILEGES, 311
	NO_RANGES_PROCESSED, 312
	NOT_ALLOWED_ON_SYSTEM_FILE, 313
	DISK_RESOURCES_EXHAUSTED, 314
	INVALID_TOKEN, 315
	DEVICE_FEATURE_NOT_SUPPORTED, 316
	MR_MID_NOT_FOUND, 317
	SCOPE_NOT_FOUND, 318
	UNDEFINED_SCOPE, 319
	INVALID_CAP, 320
	DEVICE_UNREACHABLE, 321
	DEVICE_NO_RESOURCES, 322
	DATA_CHECKSUM_ERROR, 323
	INTERMIXED_KERNEL_EA_OPERATION, 324
	FILE_LEVEL_TRIM_NOT_SUPPORTED, 326
	OFFSET_ALIGNMENT_VIOLATION, 327
	INVALID_FIELD_IN_PARAMETER_LIST, 328
	OPERATION_IN_PROGRESS, 329
	BAD_DEVICE_PATH, 330
	TOO_MANY_DESCRIPTORS, 331
	SCRUB_DATA_DISABLED, 332
	NOT_REDUNDANT_STORAGE, 333
	RESIDENT_FILE_NOT_SUPPORTED, 334
	COMPRESSED_FILE_NOT_SUPPORTED, 335
	DIRECTORY_NOT_SUPPORTED, 336
	NOT_READ_FROM_COPY, 337
	FT_WRITE_FAILURE, 338
	FT_DI_SCAN_REQUIRED, 339
	INVALID_KERNEL_INFO_VERSION, 340
	INVALID_PEP_INFO_VERSION, 341
	OBJECT_NOT_EXTERNALLY_BACKED, 342
	EXTERNAL_BACKING_PROVIDER_UNKNOWN, 343
	COMPRESSION_NOT_BENEFICIAL, 344
	STORAGE_TOPOLOGY_ID_MISMATCH, 345
	BLOCKED_BY_PARENTAL_CONTROLS, 346
	BLOCK_TOO_MANY_REFERENCES, 347
	MARKED_TO_DISALLOW_WRITES, 348
	ENCLAVE_FAILURE, 349
	FAIL_NOACTION_REBOOT, 350
	FAIL_SHUTDOWN, 351
	FAIL_RESTART, 352
	MAX_SESSIONS_REACHED, 353
	NETWORK_ACCESS_DENIED_EDP, 354
	DEVICE_HINT_NAME_BUFFER_TOO_SMALL, 355
	EDP_POLICY_DENIES_OPERATION, 356
	EDP_DPL_POLICY_CANT_BE_SATISFIED, 357
	CLOUD_FILE_SYNC_ROOT_METADATA_CORRUPT, 358
	DEVICE_IN_MAINTENANCE, 359
	NOT_SUPPORTED_ON_DAX, 360
	DAX_MAPPING_EXISTS, 361
	CLOUD_FILE_PROVIDER_NOT_RUNNING, 362
	CLOUD_FILE_METADATA_CORRUPT, 363
	CLOUD_FILE_METADATA_TOO_LARGE, 364
	CLOUD_FILE_PROPERTY_BLOB_TOO_LARGE, 365
	CLOUD_FILE_PROPERTY_BLOB_CHECKSUM_MISMATCH, 366
	CHILD_PROCESS_BLOCKED, 367
	STORAGE_LOST_DATA_PERSISTENCE, 368
	FILE_SYSTEM_VIRTUALIZATION_UNAVAILABLE, 369
	FILE_SYSTEM_VIRTUALIZATION_METADATA_CORRUPT, 370
	FILE_SYSTEM_VIRTUALIZATION_BUSY, 371
	FILE_SYSTEM_VIRTUALIZATION_PROVIDER_UNKNOWN, 372
	GDI_HANDLE_LEAK, 373
	CLOUD_FILE_TOO_MANY_PROPERTY_BLOBS, 374
	CLOUD_FILE_PROPERTY_VERSION_NOT_SUPPORTED, 375
	NOT_A_CLOUD_FILE, 376
	CLOUD_FILE_NOT_IN_SYNC, 377
	CLOUD_FILE_ALREADY_CONNECTED, 378
	CLOUD_FILE_NOT_SUPPORTED, 379
	CLOUD_FILE_INVALID_REQUEST, 380
	CLOUD_FILE_READ_ONLY_VOLUME, 381
	CLOUD_FILE_CONNECTED_PROVIDER_ONLY, 382
	CLOUD_FILE_VALIDATION_FAILED, 383
	SMB1_NOT_AVAILABLE, 384
	FILE_SYSTEM_VIRTUALIZATION_INVALID_OPERATION, 385
	CLOUD_FILE_AUTHENTICATION_FAILED, 386
	CLOUD_FILE_INSUFFICIENT_RESOURCES, 387
	CLOUD_FILE_NETWORK_UNAVAILABLE, 388
	CLOUD_FILE_UNSUCCESSFUL, 389
	CLOUD_FILE_NOT_UNDER_SYNC_ROOT, 390
	CLOUD_FILE_IN_USE, 391
	CLOUD_FILE_PINNED, 392
	CLOUD_FILE_REQUEST_ABORTED, 393
	CLOUD_FILE_PROPERTY_CORRUPT, 394
	CLOUD_FILE_ACCESS_DENIED, 395
	CLOUD_FILE_INCOMPATIBLE_HARDLINKS, 396
	CLOUD_FILE_PROPERTY_LOCK_CONFLICT, 397
	CLOUD_FILE_REQUEST_CANCELED, 398
	EXTERNAL_SYSKEY_NOT_SUPPORTED, 399
	THREAD_MODE_ALREADY_BACKGROUND, 400
	THREAD_MODE_NOT_BACKGROUND, 401
	PROCESS_MODE_ALREADY_BACKGROUND, 402
	PROCESS_MODE_NOT_BACKGROUND, 403
	CLOUD_FILE_PROVIDER_TERMINATED, 404
	NOT_A_CLOUD_SYNC_ROOT, 405
	FILE_PROTECTED_UNDER_DPL, 406
	VOLUME_NOT_CLUSTER_ALIGNED, 407
	NO_PHYSICALLY_ALIGNED_FREE_SPACE_FOUND, 408
	APPX_FILE_NOT_ENCRYPTED, 409
	RWRAW_ENCRYPTED_FILE_NOT_ENCRYPTED, 410
	RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILEOFFSET, 411
	RWRAW_ENCRYPTED_INVALID_EDATAINFO_FILERANGE, 412
	RWRAW_ENCRYPTED_INVALID_EDATAINFO_PARAMETER, 413
	LINUX_SUBSYSTEM_NOT_PRESENT, 414
	FT_READ_FAILURE, 415
	STORAGE_RESERVE_ID_INVALID, 416
	STORAGE_RESERVE_DOES_NOT_EXIST, 417
	STORAGE_RESERVE_ALREADY_EXISTS, 418
	STORAGE_RESERVE_NOT_EMPTY, 419
	NOT_A_DAX_VOLUME, 420
	NOT_DAX_MAPPABLE, 421
	TIME_SENSITIVE_THREAD, 422
	DPL_NOT_SUPPORTED_FOR_USER, 423
	CASE_DIFFERING_NAMES_IN_DIR, 424
	FILE_NOT_SUPPORTED, 425
	CLOUD_FILE_REQUEST_TIMEOUT, 426
	NO_TASK_QUEUE, 427
	SRC_SRV_DLL_LOAD_FAILED, 428
	NOT_SUPPORTED_WITH_BTT, 429
	ENCRYPTION_DISABLED, 430
	ENCRYPTING_METADATA_DISALLOWED, 431
	CANT_CLEAR_ENCRYPTION_FLAG, 432
	NO_SUCH_DEVICE, 433
	CAPAUTHZ_NOT_DEVUNLOCKED, 450
	CAPAUTHZ_CHANGE_TYPE, 451
	CAPAUTHZ_NOT_PROVISIONED, 452
	CAPAUTHZ_NOT_AUTHORIZED, 453
	CAPAUTHZ_NO_POLICY, 454
	CAPAUTHZ_DB_CORRUPTED, 455
	CAPAUTHZ_SCCD_INVALID_CATALOG, 456
	CAPAUTHZ_SCCD_NO_AUTH_ENTITY, 457
	CAPAUTHZ_SCCD_PARSE_ERROR, 458
	CAPAUTHZ_SCCD_DEV_MODE_REQUIRED, 459
	CAPAUTHZ_SCCD_NO_CAPABILITY_MATCH, 460
	PNP_QUERY_REMOVE_DEVICE_TIMEOUT, 480
	PNP_QUERY_REMOVE_RELATED_DEVICE_TIMEOUT, 481
	PNP_QUERY_REMOVE_UNRELATED_DEVICE_TIMEOUT, 482
	DEVICE_HARDWARE_ERROR, 483
	INVALID_ADDRESS, 487
	VRF_CFG_ENABLED, 1183
	PARTITION_TERMINATING, 1184
	USER_PROFILE_LOAD, 500
	ARITHMETIC_OVERFLOW, 534
	PIPE_CONNECTED, 535
	PIPE_LISTENING, 536
	VERIFIER_STOP, 537
	ABIOS_ERROR, 538
	WX86_WARNING, 539
	WX86_ERROR, 540
	TIMER_NOT_CANCELED, 541
	UNWIND, 542
	BAD_STACK, 543
	INVALID_UNWIND_TARGET, 544
	INVALID_PORT_ATTRIBUTES, 545
	PORT_MESSAGE_TOO_LONG, 546
	INVALID_QUOTA_LOWER, 547
	DEVICE_ALREADY_ATTACHED, 548
	INSTRUCTION_MISALIGNMENT, 549
	PROFILING_NOT_STARTED, 550
	PROFILING_NOT_STOPPED, 551
	COULD_NOT_INTERPRET, 552
	PROFILING_AT_LIMIT, 553
	CANT_WAIT, 554
	CANT_TERMINATE_SELF, 555
	UNEXPECTED_MM_CREATE_ERR, 556
	UNEXPECTED_MM_MAP_ERROR, 557
	UNEXPECTED_MM_EXTEND_ERR, 558
	BAD_FUNCTION_TABLE, 559
	NO_GUID_TRANSLATION, 560
	INVALID_LDT_SIZE, 561
	INVALID_LDT_OFFSET, 563
	INVALID_LDT_DESCRIPTOR, 564
	TOO_MANY_THREADS, 565
	THREAD_NOT_IN_PROCESS, 566
	PAGEFILE_QUOTA_EXCEEDED, 567
	LOGON_SERVER_CONFLICT, 568
	SYNCHRONIZATION_REQUIRED, 569
	NET_OPEN_FAILED, 570
	IO_PRIVILEGE_FAILED, 571
	CONTROL_C_EXIT, 572
	MISSING_SYSTEMFILE, 573
	UNHANDLED_EXCEPTION, 574
	APP_INIT_FAILURE, 575
	PAGEFILE_CREATE_FAILED, 576
	INVALID_IMAGE_HASH, 577
	NO_PAGEFILE, 578
	ILLEGAL_FLOAT_CONTEXT, 579
	NO_EVENT_PAIR, 580
	DOMAIN_CTRLR_CONFIG_ERROR, 581
	ILLEGAL_CHARACTER, 582
	UNDEFINED_CHARACTER, 583
	FLOPPY_VOLUME, 584
	BIOS_FAILED_TO_CONNECT_INTERRUPT, 585
	BACKUP_CONTROLLER, 586
	MUTANT_LIMIT_EXCEEDED, 587
	FS_DRIVER_REQUIRED, 588
	CANNOT_LOAD_REGISTRY_FILE, 589
	DEBUG_ATTACH_FAILED, 590
	SYSTEM_PROCESS_TERMINATED, 591
	DATA_NOT_ACCEPTED, 592
	VDM_HARD_ERROR, 593
	DRIVER_CANCEL_TIMEOUT, 594
	REPLY_MESSAGE_MISMATCH, 595
	LOST_WRITEBEHIND_DATA, 596
	CLIENT_SERVER_PARAMETERS_INVALID, 597
	NOT_TINY_STREAM, 598
	STACK_OVERFLOW_READ, 599
	CONVERT_TO_LARGE, 600
	FOUND_OUT_OF_SCOPE, 601
	ALLOCATE_BUCKET, 602
	MARSHALL_OVERFLOW, 603
	INVALID_VARIANT, 604
	BAD_COMPRESSION_BUFFER, 605
	AUDIT_FAILED, 606
	TIMER_RESOLUTION_NOT_SET, 607
	INSUFFICIENT_LOGON_INFO, 608
	BAD_DLL_ENTRYPOINT, 609
	BAD_SERVICE_ENTRYPOINT, 610
	IP_ADDRESS_CONFLICT1, 611
	IP_ADDRESS_CONFLICT2, 612
	REGISTRY_QUOTA_LIMIT, 613
	NO_CALLBACK_ACTIVE, 614
	PWD_TOO_SHORT, 615
	PWD_TOO_RECENT, 616
	PWD_HISTORY_CONFLICT, 617
	UNSUPPORTED_COMPRESSION, 618
	INVALID_HW_PROFILE, 619
	INVALID_PLUGPLAY_DEVICE_PATH, 620
	QUOTA_LIST_INCONSISTENT, 621
	EVALUATION_EXPIRATION, 622
	ILLEGAL_DLL_RELOCATION, 623
	DLL_INIT_FAILED_LOGOFF, 624
	VALIDATE_CONTINUE, 625
	NO_MORE_MATCHES, 626
	RANGE_LIST_CONFLICT, 627
	SERVER_SID_MISMATCH, 628
	CANT_ENABLE_DENY_ONLY, 629
	FLOAT_MULTIPLE_FAULTS, 630
	FLOAT_MULTIPLE_TRAPS, 631
	NOINTERFACE, 632
	DRIVER_FAILED_SLEEP, 633
	CORRUPT_SYSTEM_FILE, 634
	COMMITMENT_MINIMUM, 635
	PNP_RESTART_ENUMERATION, 636
	SYSTEM_IMAGE_BAD_SIGNATURE, 637
	PNP_REBOOT_REQUIRED, 638
	INSUFFICIENT_POWER, 639
	MULTIPLE_FAULT_VIOLATION, 640
	SYSTEM_SHUTDOWN, 641
	PORT_NOT_SET, 642
	DS_VERSION_CHECK_FAILURE, 643
	RANGE_NOT_FOUND, 644
	NOT_SAFE_MODE_DRIVER, 646
	FAILED_DRIVER_ENTRY, 647
	DEVICE_ENUMERATION_ERROR, 648
	MOUNT_POINT_NOT_RESOLVED, 649
	INVALID_DEVICE_OBJECT_PARAMETER, 650
	MCA_OCCURED, 651
	DRIVER_DATABASE_ERROR, 652
	SYSTEM_HIVE_TOO_LARGE, 653
	DRIVER_FAILED_PRIOR_UNLOAD, 654
	VOLSNAP_PREPARE_HIBERNATE, 655
	HIBERNATION_FAILURE, 656
	PWD_TOO_LONG, 657
	FILE_SYSTEM_LIMITATION, 665
	ASSERTION_FAILURE, 668
	ACPI_ERROR, 669
	WOW_ASSERTION, 670
	PNP_BAD_MPS_TABLE, 671
	PNP_TRANSLATION_FAILED, 672
	PNP_IRQ_TRANSLATION_FAILED, 673
	PNP_INVALID_ID, 674
	WAKE_SYSTEM_DEBUGGER, 675
	HANDLES_CLOSED, 676
	EXTRANEOUS_INFORMATION, 677
	RXACT_COMMIT_NECESSARY, 678
	MEDIA_CHECK, 679
	GUID_SUBSTITUTION_MADE, 680
	STOPPED_ON_SYMLINK, 681
	LONGJUMP, 682
	PLUGPLAY_QUERY_VETOED, 683
	UNWIND_CONSOLIDATE, 684
	REGISTRY_HIVE_RECOVERED, 685
	DLL_MIGHT_BE_INSECURE, 686
	DLL_MIGHT_BE_INCOMPATIBLE, 687
	DBG_EXCEPTION_NOT_HANDLED, 688
	DBG_REPLY_LATER, 689
	DBG_UNABLE_TO_PROVIDE_HANDLE, 690
	DBG_TERMINATE_THREAD, 691
	DBG_TERMINATE_PROCESS, 692
	DBG_CONTROL_C, 693
	DBG_PRINTEXCEPTION_C, 694
	DBG_RIPEXCEPTION, 695
	DBG_CONTROL_BREAK, 696
	DBG_COMMAND_EXCEPTION, 697
	OBJECT_NAME_EXISTS, 698
	THREAD_WAS_SUSPENDED, 699
	IMAGE_NOT_AT_BASE, 700
	RXACT_STATE_CREATED, 701
	SEGMENT_NOTIFICATION, 702
	BAD_CURRENT_DIRECTORY, 703
	FT_READ_RECOVERY_FROM_BACKUP, 704
	FT_WRITE_RECOVERY, 705
	IMAGE_MACHINE_TYPE_MISMATCH, 706
	RECEIVE_PARTIAL, 707
	RECEIVE_EXPEDITED, 708
	RECEIVE_PARTIAL_EXPEDITED, 709
	EVENT_DONE, 710
	EVENT_PENDING, 711
	CHECKING_FILE_SYSTEM, 712
	FATAL_APP_EXIT, 713
	PREDEFINED_HANDLE, 714
	WAS_UNLOCKED, 715
	SERVICE_NOTIFICATION, 716
	WAS_LOCKED, 717
	LOG_HARD_ERROR, 718
	ALREADY_WIN32, 719
	IMAGE_MACHINE_TYPE_MISMATCH_EXE, 720
	NO_YIELD_PERFORMED, 721
	TIMER_RESUME_IGNORED, 722
	ARBITRATION_UNHANDLED, 723
	CARDBUS_NOT_SUPPORTED, 724
	MP_PROCESSOR_MISMATCH, 725
	HIBERNATED, 726
	RESUME_HIBERNATION, 727
	FIRMWARE_UPDATED, 728
	DRIVERS_LEAKING_LOCKED_PAGES, 729
	WAKE_SYSTEM, 730
	WAIT_1, 731
	WAIT_2, 732
	WAIT_3, 733
	WAIT_63, 734
	ABANDONED_WAIT_0, 735
	ABANDONED_WAIT_63, 736
	USER_APC, 737
	KERNEL_APC, 738
	ALERTED, 739
	ELEVATION_REQUIRED, 740
	REPARSE, 741
	OPLOCK_BREAK_IN_PROGRESS, 742
	VOLUME_MOUNTED, 743
	RXACT_COMMITTED, 744
	NOTIFY_CLEANUP, 745
	PRIMARY_TRANSPORT_CONNECT_FAILED, 746
	PAGE_FAULT_TRANSITION, 747
	PAGE_FAULT_DEMAND_ZERO, 748
	PAGE_FAULT_COPY_ON_WRITE, 749
	PAGE_FAULT_GUARD_PAGE, 750
	PAGE_FAULT_PAGING_FILE, 751
	CACHE_PAGE_LOCKED, 752
	CRASH_DUMP, 753
	BUFFER_ALL_ZEROS, 754
	REPARSE_OBJECT, 755
	RESOURCE_REQUIREMENTS_CHANGED, 756
	TRANSLATION_COMPLETE, 757
	NOTHING_TO_TERMINATE, 758
	PROCESS_NOT_IN_JOB, 759
	PROCESS_IN_JOB, 760
	VOLSNAP_HIBERNATE_READY, 761
	FSFILTER_OP_COMPLETED_SUCCESSFULLY, 762
	INTERRUPT_VECTOR_ALREADY_CONNECTED, 763
	INTERRUPT_STILL_CONNECTED, 764
	WAIT_FOR_OPLOCK, 765
	DBG_EXCEPTION_HANDLED, 766
	DBG_CONTINUE, 767
	CALLBACK_POP_STACK, 768
	COMPRESSION_DISABLED, 769
	CANTFETCHBACKWARDS, 770
	CANTSCROLLBACKWARDS, 771
	ROWSNOTRELEASED, 772
	BAD_ACCESSOR_FLAGS, 773
	ERRORS_ENCOUNTERED, 774
	NOT_CAPABLE, 775
	REQUEST_OUT_OF_SEQUENCE, 776
	VERSION_PARSE_ERROR, 777
	BADSTARTPOSITION, 778
	MEMORY_HARDWARE, 779
	DISK_REPAIR_DISABLED, 780
	INSUFFICIENT_RESOURCE_FOR_SPECIFIED_SHARED_SECTION_SIZE, 781
	SYSTEM_POWERSTATE_TRANSITION, 782
	SYSTEM_POWERSTATE_COMPLEX_TRANSITION, 783
	MCA_EXCEPTION, 784
	ACCESS_AUDIT_BY_POLICY, 785
	ACCESS_DISABLED_NO_SAFER_UI_BY_POLICY, 786
	ABANDON_HIBERFILE, 787
	LOST_WRITEBEHIND_DATA_NETWORK_DISCONNECTED, 788
	LOST_WRITEBEHIND_DATA_NETWORK_SERVER_ERROR, 789
	LOST_WRITEBEHIND_DATA_LOCAL_DISK_ERROR, 790
	BAD_MCFG_TABLE, 791
	DISK_REPAIR_REDIRECTED, 792
	DISK_REPAIR_UNSUCCESSFUL, 793
	CORRUPT_LOG_OVERFULL, 794
	CORRUPT_LOG_CORRUPTED, 795
	CORRUPT_LOG_UNAVAILABLE, 796
	CORRUPT_LOG_DELETED_FULL, 797
	CORRUPT_LOG_CLEARED, 798
	ORPHAN_NAME_EXHAUSTED, 799
	OPLOCK_SWITCHED_TO_NEW_HANDLE, 800
	CANNOT_GRANT_REQUESTED_OPLOCK, 801
	CANNOT_BREAK_OPLOCK, 802
	OPLOCK_HANDLE_CLOSED, 803
	NO_ACE_CONDITION, 804
	INVALID_ACE_CONDITION, 805
	FILE_HANDLE_REVOKED, 806
	IMAGE_AT_DIFFERENT_BASE, 807
	ENCRYPTED_IO_NOT_POSSIBLE, 808
	FILE_METADATA_OPTIMIZATION_IN_PROGRESS, 809
	QUOTA_ACTIVITY, 810
	HANDLE_REVOKED, 811
	CALLBACK_INVOKE_INLINE, 812
	CPU_SET_INVALID, 813
	ENCLAVE_NOT_TERMINATED, 814
	ENCLAVE_VIOLATION, 815
	EA_ACCESS_DENIED, 994
	OPERATION_ABORTED, 995
	IO_INCOMPLETE, 996
	IO_PENDING, 997
	NOACCESS, 998
	SWAPERROR, 999
	STACK_OVERFLOW, 1001
	INVALID_MESSAGE, 1002
	CAN_NOT_COMPLETE, 1003
	INVALID_FLAGS, 1004
	UNRECOGNIZED_VOLUME, 1005
	FILE_INVALID, 1006
	FULLSCREEN_MODE, 1007
	NO_TOKEN, 1008
	BADDB, 1009
	BADKEY, 1010
	CANTOPEN, 1011
	CANTREAD, 1012
	CANTWRITE, 1013
	REGISTRY_RECOVERED, 1014
	REGISTRY_CORRUPT, 1015
	REGISTRY_IO_FAILED, 1016
	NOT_REGISTRY_FILE, 1017
	KEY_DELETED, 1018
	NO_LOG_SPACE, 1019
	KEY_HAS_CHILDREN, 1020
	CHILD_MUST_BE_VOLATILE, 1021
	NOTIFY_ENUM_DIR, 1022
	DEPENDENT_SERVICES_RUNNING, 1051
	INVALID_SERVICE_CONTROL, 1052
	SERVICE_REQUEST_TIMEOUT, 1053
	SERVICE_NO_THREAD, 1054
	SERVICE_DATABASE_LOCKED, 1055
	SERVICE_ALREADY_RUNNING, 1056
	INVALID_SERVICE_ACCOUNT, 1057
	SERVICE_DISABLED, 1058
	CIRCULAR_DEPENDENCY, 1059
	SERVICE_DOES_NOT_EXIST, 1060
	SERVICE_CANNOT_ACCEPT_CTRL, 1061
	SERVICE_NOT_ACTIVE, 1062
	FAILED_SERVICE_CONTROLLER_CONNECT, 1063
	EXCEPTION_IN_SERVICE, 1064
	DATABASE_DOES_NOT_EXIST, 1065
	SERVICE_SPECIFIC_ERROR, 1066
	PROCESS_ABORTED, 1067
	SERVICE_DEPENDENCY_FAIL, 1068
	SERVICE_LOGON_FAILED, 1069
	SERVICE_START_HANG, 1070
	INVALID_SERVICE_LOCK, 1071
	SERVICE_MARKED_FOR_DELETE, 1072
	SERVICE_EXISTS, 1073
	ALREADY_RUNNING_LKG, 1074
	SERVICE_DEPENDENCY_DELETED, 1075
	BOOT_ALREADY_ACCEPTED, 1076
	SERVICE_NEVER_STARTED, 1077
	DUPLICATE_SERVICE_NAME, 1078
	DIFFERENT_SERVICE_ACCOUNT, 1079
	CANNOT_DETECT_DRIVER_FAILURE, 1080
	CANNOT_DETECT_PROCESS_ABORT, 1081
	NO_RECOVERY_PROGRAM, 1082
	SERVICE_NOT_IN_EXE, 1083
	NOT_SAFEBOOT_SERVICE, 1084
	END_OF_MEDIA, 1100
	FILEMARK_DETECTED, 1101
	BEGINNING_OF_MEDIA, 1102
	SETMARK_DETECTED, 1103
	NO_DATA_DETECTED, 1104
	PARTITION_FAILURE, 1105
	INVALID_BLOCK_LENGTH, 1106
	DEVICE_NOT_PARTITIONED, 1107
	UNABLE_TO_LOCK_MEDIA, 1108
	UNABLE_TO_UNLOAD_MEDIA, 1109
	MEDIA_CHANGED, 1110
	BUS_RESET, 1111
	NO_MEDIA_IN_DRIVE, 1112
	NO_UNICODE_TRANSLATION, 1113
	DLL_INIT_FAILED, 1114
	SHUTDOWN_IN_PROGRESS, 1115
	NO_SHUTDOWN_IN_PROGRESS, 1116
	IO_DEVICE, 1117
	SERIAL_NO_DEVICE, 1118
	IRQ_BUSY, 1119
	MORE_WRITES, 1120
	COUNTER_TIMEOUT, 1121
	FLOPPY_ID_MARK_NOT_FOUND, 1122
	FLOPPY_WRONG_CYLINDER, 1123
	FLOPPY_UNKNOWN_ERROR, 1124
	FLOPPY_BAD_REGISTERS, 1125
	DISK_RECALIBRATE_FAILED, 1126
	DISK_OPERATION_FAILED, 1127
	DISK_RESET_FAILED, 1128
	EOM_OVERFLOW, 1129
	NOT_ENOUGH_SERVER_MEMORY, 1130
	POSSIBLE_DEADLOCK, 1131
	MAPPED_ALIGNMENT, 1132
	SET_POWER_STATE_VETOED, 1140
	SET_POWER_STATE_FAILED, 1141
	TOO_MANY_LINKS, 1142
	OLD_WIN_VERSION, 1150
	APP_WRONG_OS, 1151
	SINGLE_INSTANCE_APP, 1152
	RMODE_APP, 1153
	INVALID_DLL, 1154
	NO_ASSOCIATION, 1155
	DDE_FAIL, 1156
	DLL_NOT_FOUND, 1157
	NO_MORE_USER_HANDLES, 1158
	MESSAGE_SYNC_ONLY, 1159
	SOURCE_ELEMENT_EMPTY, 1160
	DESTINATION_ELEMENT_FULL, 1161
	ILLEGAL_ELEMENT_ADDRESS, 1162
	MAGAZINE_NOT_PRESENT, 1163
	DEVICE_REINITIALIZATION_NEEDED, 1164
	DEVICE_REQUIRES_CLEANING, 1165
	DEVICE_DOOR_OPEN, 1166
	DEVICE_NOT_CONNECTED, 1167
	NOT_FOUND, 1168
	NO_MATCH, 1169
	SET_NOT_FOUND, 1170
	POINT_NOT_FOUND, 1171
	NO_TRACKING_SERVICE, 1172
	NO_VOLUME_ID, 1173
	UNABLE_TO_REMOVE_REPLACED, 1175
	UNABLE_TO_MOVE_REPLACEMENT, 1176
	UNABLE_TO_MOVE_REPLACEMENT_2, 1177
	JOURNAL_DELETE_IN_PROGRESS, 1178
	JOURNAL_NOT_ACTIVE, 1179
	POTENTIAL_FILE_FOUND, 1180
	JOURNAL_ENTRY_DELETED, 1181
	SHUTDOWN_IS_SCHEDULED, 1190
	SHUTDOWN_USERS_LOGGED_ON, 1191
	BAD_DEVICE, 1200
	CONNECTION_UNAVAIL, 1201
	DEVICE_ALREADY_REMEMBERED, 1202
	NO_NET_OR_BAD_PATH, 1203
	BAD_PROVIDER, 1204
	CANNOT_OPEN_PROFILE, 1205
	BAD_PROFILE, 1206
	NOT_CONTAINER, 1207
	EXTENDED_ERROR, 1208
	INVALID_GROUPNAME, 1209
	INVALID_COMPUTERNAME, 1210
	INVALID_EVENTNAME, 1211
	INVALID_DOMAINNAME, 1212
	INVALID_SERVICENAME, 1213
	INVALID_NETNAME, 1214
	INVALID_SHARENAME, 1215
	INVALID_PASSWORDNAME, 1216
	INVALID_MESSAGENAME, 1217
	INVALID_MESSAGEDEST, 1218
	SESSION_CREDENTIAL_CONFLICT, 1219
	REMOTE_SESSION_LIMIT_EXCEEDED, 1220
	DUP_DOMAINNAME, 1221
	NO_NETWORK, 1222
	CANCELLED, 1223
	USER_MAPPED_FILE, 1224
	CONNECTION_REFUSED, 1225
	GRACEFUL_DISCONNECT, 1226
	ADDRESS_ALREADY_ASSOCIATED, 1227
	ADDRESS_NOT_ASSOCIATED, 1228
	CONNECTION_INVALID, 1229
	CONNECTION_ACTIVE, 1230
	NETWORK_UNREACHABLE, 1231
	HOST_UNREACHABLE, 1232
	PROTOCOL_UNREACHABLE, 1233
	PORT_UNREACHABLE, 1234
	REQUEST_ABORTED, 1235
	CONNECTION_ABORTED, 1236
	RETRY, 1237
	CONNECTION_COUNT_LIMIT, 1238
	LOGIN_TIME_RESTRICTION, 1239
	LOGIN_WKSTA_RESTRICTION, 1240
	INCORRECT_ADDRESS, 1241
	ALREADY_REGISTERED, 1242
	SERVICE_NOT_FOUND, 1243
	NOT_AUTHENTICATED, 1244
	NOT_LOGGED_ON, 1245
	CONTINUE, 1246
	ALREADY_INITIALIZED, 1247
	NO_MORE_DEVICES, 1248
	NO_SUCH_SITE, 1249
	DOMAIN_CONTROLLER_EXISTS, 1250
	ONLY_IF_CONNECTED, 1251
	OVERRIDE_NOCHANGES, 1252
	BAD_USER_PROFILE, 1253
	NOT_SUPPORTED_ON_SBS, 1254
	SERVER_SHUTDOWN_IN_PROGRESS, 1255
	HOST_DOWN, 1256
	NON_ACCOUNT_SID, 1257
	NON_DOMAIN_SID, 1258
	APPHELP_BLOCK, 1259
	ACCESS_DISABLED_BY_POLICY, 1260
	REG_NAT_CONSUMPTION, 1261
	CSCSHARE_OFFLINE, 1262
	PKINIT_FAILURE, 1263
	SMARTCARD_SUBSYSTEM_FAILURE, 1264
	DOWNGRADE_DETECTED, 1265
	MACHINE_LOCKED, 1271
	SMB_GUEST_LOGON_BLOCKED, 1272
	CALLBACK_SUPPLIED_INVALID_DATA, 1273
	SYNC_FOREGROUND_REFRESH_REQUIRED, 1274
	DRIVER_BLOCKED, 1275
	INVALID_IMPORT_OF_NON_DLL, 1276
	ACCESS_DISABLED_WEBBLADE, 1277
	ACCESS_DISABLED_WEBBLADE_TAMPER, 1278
	RECOVERY_FAILURE, 1279
	ALREADY_FIBER, 1280
	ALREADY_THREAD, 1281
	STACK_BUFFER_OVERRUN, 1282
	PARAMETER_QUOTA_EXCEEDED, 1283
	DEBUGGER_INACTIVE, 1284
	DELAY_LOAD_FAILED, 1285
	VDM_DISALLOWED, 1286
	UNIDENTIFIED_ERROR, 1287
	INVALID_CRUNTIME_PARAMETER, 1288
	BEYOND_VDL, 1289
	INCOMPATIBLE_SERVICE_SID_TYPE, 1290
	DRIVER_PROCESS_TERMINATED, 1291
	IMPLEMENTATION_LIMIT, 1292
	PROCESS_IS_PROTECTED, 1293
	SERVICE_NOTIFY_CLIENT_LAGGING, 1294
	DISK_QUOTA_EXCEEDED, 1295
	CONTENT_BLOCKED, 1296
	INCOMPATIBLE_SERVICE_PRIVILEGE, 1297
	APP_HANG, 1298
	INVALID_LABEL, 1299
	NOT_ALL_ASSIGNED, 1300
	SOME_NOT_MAPPED, 1301
	NO_QUOTAS_FOR_ACCOUNT, 1302
	LOCAL_USER_SESSION_KEY, 1303
	NULL_LM_PASSWORD, 1304
	UNKNOWN_REVISION, 1305
	REVISION_MISMATCH, 1306
	INVALID_OWNER, 1307
	INVALID_PRIMARY_GROUP, 1308
	NO_IMPERSONATION_TOKEN, 1309
	CANT_DISABLE_MANDATORY, 1310
	NO_LOGON_SERVERS, 1311
	NO_SUCH_LOGON_SESSION, 1312
	NO_SUCH_PRIVILEGE, 1313
	PRIVILEGE_NOT_HELD, 1314
	INVALID_ACCOUNT_NAME, 1315
	USER_EXISTS, 1316
	NO_SUCH_USER, 1317
	GROUP_EXISTS, 1318
	NO_SUCH_GROUP, 1319
	MEMBER_IN_GROUP, 1320
	MEMBER_NOT_IN_GROUP, 1321
	LAST_ADMIN, 1322
	WRONG_PASSWORD, 1323
	ILL_FORMED_PASSWORD, 1324
	PASSWORD_RESTRICTION, 1325
	LOGON_FAILURE, 1326
	ACCOUNT_RESTRICTION, 1327
	INVALID_LOGON_HOURS, 1328
	INVALID_WORKSTATION, 1329
	PASSWORD_EXPIRED, 1330
	ACCOUNT_DISABLED, 1331
	NONE_MAPPED, 1332
	TOO_MANY_LUIDS_REQUESTED, 1333
	LUIDS_EXHAUSTED, 1334
	INVALID_SUB_AUTHORITY, 1335
	INVALID_ACL, 1336
	INVALID_SID, 1337
	INVALID_SECURITY_DESCR, 1338
	BAD_INHERITANCE_ACL, 1340
	SERVER_DISABLED, 1341
	SERVER_NOT_DISABLED, 1342
	INVALID_ID_AUTHORITY, 1343
	ALLOTTED_SPACE_EXCEEDED, 1344
	INVALID_GROUP_ATTRIBUTES, 1345
	BAD_IMPERSONATION_LEVEL, 1346
	CANT_OPEN_ANONYMOUS, 1347
	BAD_VALIDATION_CLASS, 1348
	BAD_TOKEN_TYPE, 1349
	NO_SECURITY_ON_OBJECT, 1350
	CANT_ACCESS_DOMAIN_INFO, 1351
	INVALID_SERVER_STATE, 1352
	INVALID_DOMAIN_STATE, 1353
	INVALID_DOMAIN_ROLE, 1354
	NO_SUCH_DOMAIN, 1355
	DOMAIN_EXISTS, 1356
	DOMAIN_LIMIT_EXCEEDED, 1357
	INTERNAL_DB_CORRUPTION, 1358
	INTERNAL_ERROR, 1359
	GENERIC_NOT_MAPPED, 1360
	BAD_DESCRIPTOR_FORMAT, 1361
	NOT_LOGON_PROCESS, 1362
	LOGON_SESSION_EXISTS, 1363
	NO_SUCH_PACKAGE, 1364
	BAD_LOGON_SESSION_STATE, 1365
	LOGON_SESSION_COLLISION, 1366
	INVALID_LOGON_TYPE, 1367
	CANNOT_IMPERSONATE, 1368
	RXACT_INVALID_STATE, 1369
	RXACT_COMMIT_FAILURE, 1370
	SPECIAL_ACCOUNT, 1371
	SPECIAL_GROUP, 1372
	SPECIAL_USER, 1373
	MEMBERS_PRIMARY_GROUP, 1374
	TOKEN_ALREADY_IN_USE, 1375
	NO_SUCH_ALIAS, 1376
	MEMBER_NOT_IN_ALIAS, 1377
	MEMBER_IN_ALIAS, 1378
	ALIAS_EXISTS, 1379
	LOGON_NOT_GRANTED, 1380
	TOO_MANY_SECRETS, 1381
	SECRET_TOO_LONG, 1382
	INTERNAL_DB_ERROR, 1383
	TOO_MANY_CONTEXT_IDS, 1384
	LOGON_TYPE_NOT_GRANTED, 1385
	NT_CROSS_ENCRYPTION_REQUIRED, 1386
	NO_SUCH_MEMBER, 1387
	INVALID_MEMBER, 1388
	TOO_MANY_SIDS, 1389
	LM_CROSS_ENCRYPTION_REQUIRED, 1390
	NO_INHERITANCE, 1391
	FILE_CORRUPT, 1392
	DISK_CORRUPT, 1393
	NO_USER_SESSION_KEY, 1394
	LICENSE_QUOTA_EXCEEDED, 1395
	WRONG_TARGET_NAME, 1396
	MUTUAL_AUTH_FAILED, 1397
	TIME_SKEW, 1398
	CURRENT_DOMAIN_NOT_ALLOWED, 1399
	INVALID_WINDOW_HANDLE, 1400
	INVALID_MENU_HANDLE, 1401
	INVALID_CURSOR_HANDLE, 1402
	INVALID_ACCEL_HANDLE, 1403
	INVALID_HOOK_HANDLE, 1404
	INVALID_DWP_HANDLE, 1405
	TLW_WITH_WSCHILD, 1406
	CANNOT_FIND_WND_CLASS, 1407
	WINDOW_OF_OTHER_THREAD, 1408
	HOTKEY_ALREADY_REGISTERED, 1409
	CLASS_ALREADY_EXISTS, 1410
	CLASS_DOES_NOT_EXIST, 1411
	CLASS_HAS_WINDOWS, 1412
	INVALID_INDEX, 1413
	INVALID_ICON_HANDLE, 1414
	PRIVATE_DIALOG_INDEX, 1415
	LISTBOX_ID_NOT_FOUND, 1416
	NO_WILDCARD_CHARACTERS, 1417
	CLIPBOARD_NOT_OPEN, 1418
	HOTKEY_NOT_REGISTERED, 1419
	WINDOW_NOT_DIALOG, 1420
	CONTROL_ID_NOT_FOUND, 1421
	INVALID_COMBOBOX_MESSAGE, 1422
	WINDOW_NOT_COMBOBOX, 1423
	INVALID_EDIT_HEIGHT, 1424
	DC_NOT_FOUND, 1425
	INVALID_HOOK_FILTER, 1426
	INVALID_FILTER_PROC, 1427
	HOOK_NEEDS_HMOD, 1428
	GLOBAL_ONLY_HOOK, 1429
	JOURNAL_HOOK_SET, 1430
	HOOK_NOT_INSTALLED, 1431
	INVALID_LB_MESSAGE, 1432
	SETCOUNT_ON_BAD_LB, 1433
	LB_WITHOUT_TABSTOPS, 1434
	DESTROY_OBJECT_OF_OTHER_THREAD, 1435
	CHILD_WINDOW_MENU, 1436
	NO_SYSTEM_MENU, 1437
	INVALID_MSGBOX_STYLE, 1438
	INVALID_SPI_VALUE, 1439
	SCREEN_ALREADY_LOCKED, 1440
	HWNDS_HAVE_DIFF_PARENT, 1441
	NOT_CHILD_WINDOW, 1442
	INVALID_GW_COMMAND, 1443
	INVALID_THREAD_ID, 1444
	NON_MDICHILD_WINDOW, 1445
	POPUP_ALREADY_ACTIVE, 1446
	NO_SCROLLBARS, 1447
	INVALID_SCROLLBAR_RANGE, 1448
	INVALID_SHOWWIN_COMMAND, 1449
	NO_SYSTEM_RESOURCES, 1450
	NONPAGED_SYSTEM_RESOURCES, 1451
	PAGED_SYSTEM_RESOURCES, 1452
	WORKING_SET_QUOTA, 1453
	PAGEFILE_QUOTA, 1454
	COMMITMENT_LIMIT, 1455
	MENU_ITEM_NOT_FOUND, 1456
	INVALID_KEYBOARD_HANDLE, 1457
	HOOK_TYPE_NOT_ALLOWED, 1458
	REQUIRES_INTERACTIVE_WINDOWSTATION, 1459
	TIMEOUT, 1460
	INVALID_MONITOR_HANDLE, 1461
	INCORRECT_SIZE, 1462
	SYMLINK_CLASS_DISABLED, 1463
	SYMLINK_NOT_SUPPORTED, 1464
	XML_PARSE_ERROR, 1465
	XMLDSIG_ERROR, 1466
	RESTART_APPLICATION, 1467
	WRONG_COMPARTMENT, 1468
	AUTHIP_FAILURE, 1469
	NO_NVRAM_RESOURCES, 1470
	NOT_GUI_PROCESS, 1471
	EVENTLOG_FILE_CORRUPT, 1500
	EVENTLOG_CANT_START, 1501
	LOG_FILE_FULL, 1502
	EVENTLOG_FILE_CHANGED, 1503
	CONTAINER_ASSIGNED, 1504
	JOB_NO_CONTAINER, 1505
	INVALID_TASK_NAME, 1550
	INVALID_TASK_INDEX, 1551
	THREAD_ALREADY_IN_TASK, 1552
	INSTALL_SERVICE_FAILURE, 1601
	INSTALL_USEREXIT, 1602
	INSTALL_FAILURE, 1603
	INSTALL_SUSPEND, 1604
	UNKNOWN_PRODUCT, 1605
	UNKNOWN_FEATURE, 1606
	UNKNOWN_COMPONENT, 1607
	UNKNOWN_PROPERTY, 1608
	INVALID_HANDLE_STATE, 1609
	BAD_CONFIGURATION, 1610
	INDEX_ABSENT, 1611
	INSTALL_SOURCE_ABSENT, 1612
	INSTALL_PACKAGE_VERSION, 1613
	PRODUCT_UNINSTALLED, 1614
	BAD_QUERY_SYNTAX, 1615
	INVALID_FIELD, 1616
	DEVICE_REMOVED, 1617
	INSTALL_ALREADY_RUNNING, 1618
	INSTALL_PACKAGE_OPEN_FAILED, 1619
	INSTALL_PACKAGE_INVALID, 1620
	INSTALL_UI_FAILURE, 1621
	INSTALL_LOG_FAILURE, 1622
	INSTALL_LANGUAGE_UNSUPPORTED, 1623
	INSTALL_TRANSFORM_FAILURE, 1624
	INSTALL_PACKAGE_REJECTED, 1625
	FUNCTION_NOT_CALLED, 1626
	FUNCTION_FAILED, 1627
	INVALID_TABLE, 1628
	DATATYPE_MISMATCH, 1629
	UNSUPPORTED_TYPE, 1630
	CREATE_FAILED, 1631
	INSTALL_TEMP_UNWRITABLE, 1632
	INSTALL_PLATFORM_UNSUPPORTED, 1633
	INSTALL_NOTUSED, 1634
	PATCH_PACKAGE_OPEN_FAILED, 1635
	PATCH_PACKAGE_INVALID, 1636
	PATCH_PACKAGE_UNSUPPORTED, 1637
	PRODUCT_VERSION, 1638
	INVALID_COMMAND_LINE, 1639
	INSTALL_REMOTE_DISALLOWED, 1640
	SUCCESS_REBOOT_INITIATED, 1641
	PATCH_TARGET_NOT_FOUND, 1642
	PATCH_PACKAGE_REJECTED, 1643
	INSTALL_TRANSFORM_REJECTED, 1644
	INSTALL_REMOTE_PROHIBITED, 1645
	PATCH_REMOVAL_UNSUPPORTED, 1646
	UNKNOWN_PATCH, 1647
	PATCH_NO_SEQUENCE, 1648
	PATCH_REMOVAL_DISALLOWED, 1649
	INVALID_PATCH_XML, 1650
	PATCH_MANAGED_ADVERTISED_PRODUCT, 1651
	INSTALL_SERVICE_SAFEBOOT, 1652
	FAIL_FAST_EXCEPTION, 1653
	INSTALL_REJECTED, 1654
	DYNAMIC_CODE_BLOCKED, 1655
	NOT_SAME_OBJECT, 1656
	STRICT_CFG_VIOLATION, 1657
	SET_CONTEXT_DENIED, 1660
	CROSS_PARTITION_VIOLATION, 1661
	INVALID_USER_BUFFER, 1784
	UNRECOGNIZED_MEDIA, 1785
	NO_TRUST_LSA_SECRET, 1786
	NO_TRUST_SAM_ACCOUNT, 1787
	TRUSTED_DOMAIN_FAILURE, 1788
	TRUSTED_RELATIONSHIP_FAILURE, 1789
	TRUST_FAILURE, 1790
	NETLOGON_NOT_STARTED, 1792
	ACCOUNT_EXPIRED, 1793
	REDIRECTOR_HAS_OPEN_HANDLES, 1794
	PRINTER_DRIVER_ALREADY_INSTALLED, 1795
	UNKNOWN_PORT, 1796
	UNKNOWN_PRINTER_DRIVER, 1797
	UNKNOWN_PRINTPROCESSOR, 1798
	INVALID_SEPARATOR_FILE, 1799
	INVALID_PRIORITY, 1800
	INVALID_PRINTER_NAME, 1801
	PRINTER_ALREADY_EXISTS, 1802
	INVALID_PRINTER_COMMAND, 1803
	INVALID_DATATYPE, 1804
	INVALID_ENVIRONMENT, 1805
	NOLOGON_INTERDOMAIN_TRUST_ACCOUNT, 1807
	NOLOGON_WORKSTATION_TRUST_ACCOUNT, 1808
	NOLOGON_SERVER_TRUST_ACCOUNT, 1809
	DOMAIN_TRUST_INCONSISTENT, 1810
	SERVER_HAS_OPEN_HANDLES, 1811
	RESOURCE_DATA_NOT_FOUND, 1812
	RESOURCE_TYPE_NOT_FOUND, 1813
	RESOURCE_NAME_NOT_FOUND, 1814
	RESOURCE_LANG_NOT_FOUND, 1815
	NOT_ENOUGH_QUOTA, 1816
	INVALID_TIME, 1901
	INVALID_FORM_NAME, 1902
	INVALID_FORM_SIZE, 1903
	ALREADY_WAITING, 1904
	PRINTER_DELETED, 1905
	INVALID_PRINTER_STATE, 1906
	PASSWORD_MUST_CHANGE, 1907
	DOMAIN_CONTROLLER_NOT_FOUND, 1908
	ACCOUNT_LOCKED_OUT, 1909
	NO_SITENAME, 1919
	CANT_ACCESS_FILE, 1920
	CANT_RESOLVE_FILENAME, 1921
	KM_DRIVER_BLOCKED, 1930
	CONTEXT_EXPIRED, 1931
	PER_USER_TRUST_QUOTA_EXCEEDED, 1932
	ALL_USER_TRUST_QUOTA_EXCEEDED, 1933
	USER_DELETE_TRUST_QUOTA_EXCEEDED, 1934
	AUTHENTICATION_FIREWALL_FAILED, 1935
	REMOTE_PRINT_CONNECTIONS_BLOCKED, 1936
	NTLM_BLOCKED, 1937
	PASSWORD_CHANGE_REQUIRED, 1938
	LOST_MODE_LOGON_RESTRICTION, 1939
	INVALID_PIXEL_FORMAT, 2000
	BAD_DRIVER, 2001
	INVALID_WINDOW_STYLE, 2002
	METAFILE_NOT_SUPPORTED, 2003
	TRANSFORM_NOT_SUPPORTED, 2004
	CLIPPING_NOT_SUPPORTED, 2005
	INVALID_CMM, 2010
	INVALID_PROFILE, 2011
	TAG_NOT_FOUND, 2012
	TAG_NOT_PRESENT, 2013
	DUPLICATE_TAG, 2014
	PROFILE_NOT_ASSOCIATED_WITH_DEVICE, 2015
	PROFILE_NOT_FOUND, 2016
	INVALID_COLORSPACE, 2017
	ICM_NOT_ENABLED, 2018
	DELETING_ICM_XFORM, 2019
	INVALID_TRANSFORM, 2020
	COLORSPACE_MISMATCH, 2021
	INVALID_COLORINDEX, 2022
	PROFILE_DOES_NOT_MATCH_DEVICE, 2023
	CONNECTED_OTHER_PASSWORD, 2108
	CONNECTED_OTHER_PASSWORD_DEFAULT, 2109
	BAD_USERNAME, 2202
	NOT_CONNECTED, 2250
	OPEN_FILES, 2401
	ACTIVE_CONNECTIONS, 2402
	DEVICE_IN_USE, 2404
	UNKNOWN_PRINT_MONITOR, 3000
	PRINTER_DRIVER_IN_USE, 3001
	SPOOL_FILE_NOT_FOUND, 3002
	SPL_NO_STARTDOC, 3003
	SPL_NO_ADDJOB, 3004
	PRINT_PROCESSOR_ALREADY_INSTALLED, 3005
	PRINT_MONITOR_ALREADY_INSTALLED, 3006
	INVALID_PRINT_MONITOR, 3007
	PRINT_MONITOR_IN_USE, 3008
	PRINTER_HAS_JOBS_QUEUED, 3009
	SUCCESS_REBOOT_REQUIRED, 3010
	SUCCESS_RESTART_REQUIRED, 3011
	PRINTER_NOT_FOUND, 3012
	PRINTER_DRIVER_WARNED, 3013
	PRINTER_DRIVER_BLOCKED, 3014
	PRINTER_DRIVER_PACKAGE_IN_USE, 3015
	CORE_DRIVER_PACKAGE_NOT_FOUND, 3016
	FAIL_REBOOT_REQUIRED, 3017
	FAIL_REBOOT_INITIATED, 3018
	PRINTER_DRIVER_DOWNLOAD_NEEDED, 3019
	PRINT_JOB_RESTART_REQUIRED, 3020
	INVALID_PRINTER_DRIVER_MANIFEST, 3021
	PRINTER_NOT_SHAREABLE, 3022
	REQUEST_PAUSED, 3050
	APPEXEC_CONDITION_NOT_SATISFIED, 3060
	APPEXEC_HANDLE_INVALIDATED, 3061
	APPEXEC_INVALID_HOST_GENERATION, 3062
	APPEXEC_UNEXPECTED_PROCESS_REGISTRATION, 3063
	APPEXEC_INVALID_HOST_STATE, 3064
	APPEXEC_NO_DONOR, 3065
	APPEXEC_HOST_ID_MISMATCH, 3066
	APPEXEC_UNKNOWN_USER, 3067
	IO_REISSUE_AS_CACHED, 3950
	WINS_INTERNAL, 4000
	CAN_NOT_DEL_LOCAL_WINS, 4001
	STATIC_INIT, 4002
	INC_BACKUP, 4003
	FULL_BACKUP, 4004
	REC_NON_EXISTENT, 4005
	RPL_NOT_ALLOWED, 4006
	DHCP_ADDRESS_CONFLICT, 4100
	WMI_GUID_NOT_FOUND, 4200
	WMI_INSTANCE_NOT_FOUND, 4201
	WMI_ITEMID_NOT_FOUND, 4202
	WMI_TRY_AGAIN, 4203
	WMI_DP_NOT_FOUND, 4204
	WMI_UNRESOLVED_INSTANCE_REF, 4205
	WMI_ALREADY_ENABLED, 4206
	WMI_GUID_DISCONNECTED, 4207
	WMI_SERVER_UNAVAILABLE, 4208
	WMI_DP_FAILED, 4209
	WMI_INVALID_MOF, 4210
	WMI_INVALID_REGINFO, 4211
	WMI_ALREADY_DISABLED, 4212
	WMI_READ_ONLY, 4213
	WMI_SET_FAILURE, 4214
	NOT_APPCONTAINER, 4250
	APPCONTAINER_REQUIRED, 4251
	NOT_SUPPORTED_IN_APPCONTAINER, 4252
	INVALID_PACKAGE_SID_LENGTH, 4253
	INVALID_MEDIA, 4300
	INVALID_LIBRARY, 4301
	INVALID_MEDIA_POOL, 4302
	DRIVE_MEDIA_MISMATCH, 4303
	MEDIA_OFFLINE, 4304
	LIBRARY_OFFLINE, 4305
	EMPTY, 4306
	NOT_EMPTY, 4307
	MEDIA_UNAVAILABLE, 4308
	RESOURCE_DISABLED, 4309
	INVALID_CLEANER, 4310
	UNABLE_TO_CLEAN, 4311
	OBJECT_NOT_FOUND, 4312
	DATABASE_FAILURE, 4313
	DATABASE_FULL, 4314
	MEDIA_INCOMPATIBLE, 4315
	RESOURCE_NOT_PRESENT, 4316
	INVALID_OPERATION, 4317
	MEDIA_NOT_AVAILABLE, 4318
	DEVICE_NOT_AVAILABLE, 4319
	REQUEST_REFUSED, 4320
	INVALID_DRIVE_OBJECT, 4321
	LIBRARY_FULL, 4322
	MEDIUM_NOT_ACCESSIBLE, 4323
	UNABLE_TO_LOAD_MEDIUM, 4324
	UNABLE_TO_INVENTORY_DRIVE, 4325
	UNABLE_TO_INVENTORY_SLOT, 4326
	UNABLE_TO_INVENTORY_TRANSPORT, 4327
	TRANSPORT_FULL, 4328
	CONTROLLING_IEPORT, 4329
	UNABLE_TO_EJECT_MOUNTED_MEDIA, 4330
	CLEANER_SLOT_SET, 4331
	CLEANER_SLOT_NOT_SET, 4332
	CLEANER_CARTRIDGE_SPENT, 4333
	UNEXPECTED_OMID, 4334
	CANT_DELETE_LAST_ITEM, 4335
	MESSAGE_EXCEEDS_MAX_SIZE, 4336
	VOLUME_CONTAINS_SYS_FILES, 4337
	INDIGENOUS_TYPE, 4338
	NO_SUPPORTING_DRIVES, 4339
	CLEANER_CARTRIDGE_INSTALLED, 4340
	IEPORT_FULL, 4341
	FILE_OFFLINE, 4350
	REMOTE_STORAGE_NOT_ACTIVE, 4351
	REMOTE_STORAGE_MEDIA_ERROR, 4352
	NOT_A_REPARSE_POINT, 4390
	REPARSE_ATTRIBUTE_CONFLICT, 4391
	INVALID_REPARSE_DATA, 4392
	REPARSE_TAG_INVALID, 4393
	REPARSE_TAG_MISMATCH, 4394
	REPARSE_POINT_ENCOUNTERED, 4395
	APP_DATA_NOT_FOUND, 4400
	APP_DATA_EXPIRED, 4401
	APP_DATA_CORRUPT, 4402
	APP_DATA_LIMIT_EXCEEDED, 4403
	APP_DATA_REBOOT_REQUIRED, 4404
	SECUREBOOT_ROLLBACK_DETECTED, 4420
	SECUREBOOT_POLICY_VIOLATION, 4421
	SECUREBOOT_INVALID_POLICY, 4422
	SECUREBOOT_POLICY_PUBLISHER_NOT_FOUND, 4423
	SECUREBOOT_POLICY_NOT_SIGNED, 4424
	SECUREBOOT_NOT_ENABLED, 4425
	SECUREBOOT_FILE_REPLACED, 4426
	SECUREBOOT_POLICY_NOT_AUTHORIZED, 4427
	SECUREBOOT_POLICY_UNKNOWN, 4428
	SECUREBOOT_POLICY_MISSING_ANTIROLLBACKVERSION, 4429
	SECUREBOOT_PLATFORM_ID_MISMATCH, 4430
	SECUREBOOT_POLICY_ROLLBACK_DETECTED, 4431
	SECUREBOOT_POLICY_UPGRADE_MISMATCH, 4432
	SECUREBOOT_REQUIRED_POLICY_FILE_MISSING, 4433
	SECUREBOOT_NOT_BASE_POLICY, 4434
	SECUREBOOT_NOT_SUPPLEMENTAL_POLICY, 4435
	OFFLOAD_READ_FLT_NOT_SUPPORTED, 4440
	OFFLOAD_WRITE_FLT_NOT_SUPPORTED, 4441
	OFFLOAD_READ_FILE_NOT_SUPPORTED, 4442
	OFFLOAD_WRITE_FILE_NOT_SUPPORTED, 4443
	ALREADY_HAS_STREAM_ID, 4444
	SMR_GARBAGE_COLLECTION_REQUIRED, 4445
	WOF_WIM_HEADER_CORRUPT, 4446
	WOF_WIM_RESOURCE_TABLE_CORRUPT, 4447
	WOF_FILE_RESOURCE_TABLE_CORRUPT, 4448
	VOLUME_NOT_SIS_ENABLED, 4500
	SYSTEM_INTEGRITY_ROLLBACK_DETECTED, 4550
	SYSTEM_INTEGRITY_POLICY_VIOLATION, 4551
	SYSTEM_INTEGRITY_INVALID_POLICY, 4552
	SYSTEM_INTEGRITY_POLICY_NOT_SIGNED, 4553
	SYSTEM_INTEGRITY_TOO_MANY_POLICIES, 4554
	SYSTEM_INTEGRITY_SUPPLEMENTAL_POLICY_NOT_AUTHORIZED, 4555
	VSM_NOT_INITIALIZED, 4560
	VSM_DMA_PROTECTION_NOT_IN_USE, 4561
	PLATFORM_MANIFEST_NOT_AUTHORIZED, 4570
	PLATFORM_MANIFEST_INVALID, 4571
	PLATFORM_MANIFEST_FILE_NOT_AUTHORIZED, 4572
	PLATFORM_MANIFEST_CATALOG_NOT_AUTHORIZED, 4573
	PLATFORM_MANIFEST_BINARY_ID_NOT_FOUND, 4574
	PLATFORM_MANIFEST_NOT_ACTIVE, 4575
	PLATFORM_MANIFEST_NOT_SIGNED, 4576
	DEPENDENT_RESOURCE_EXISTS, 5001
	DEPENDENCY_NOT_FOUND, 5002
	DEPENDENCY_ALREADY_EXISTS, 5003
	RESOURCE_NOT_ONLINE, 5004
	HOST_NODE_NOT_AVAILABLE, 5005
	RESOURCE_NOT_AVAILABLE, 5006
	RESOURCE_NOT_FOUND, 5007
	SHUTDOWN_CLUSTER, 5008
	CANT_EVICT_ACTIVE_NODE, 5009
	OBJECT_ALREADY_EXISTS, 5010
	OBJECT_IN_LIST, 5011
	GROUP_NOT_AVAILABLE, 5012
	GROUP_NOT_FOUND, 5013
	GROUP_NOT_ONLINE, 5014
	HOST_NODE_NOT_RESOURCE_OWNER, 5015
	HOST_NODE_NOT_GROUP_OWNER, 5016
	RESMON_CREATE_FAILED, 5017
	RESMON_ONLINE_FAILED, 5018
	RESOURCE_ONLINE, 5019
	QUORUM_RESOURCE, 5020
	NOT_QUORUM_CAPABLE, 5021
	CLUSTER_SHUTTING_DOWN, 5022
	INVALID_STATE, 5023
	RESOURCE_PROPERTIES_STORED, 5024
	NOT_QUORUM_CLASS, 5025
	CORE_RESOURCE, 5026
	QUORUM_RESOURCE_ONLINE_FAILED, 5027
	QUORUMLOG_OPEN_FAILED, 5028
	CLUSTERLOG_CORRUPT, 5029
	CLUSTERLOG_RECORD_EXCEEDS_MAXSIZE, 5030
	CLUSTERLOG_EXCEEDS_MAXSIZE, 5031
	CLUSTERLOG_CHKPOINT_NOT_FOUND, 5032
	CLUSTERLOG_NOT_ENOUGH_SPACE, 5033
	QUORUM_OWNER_ALIVE, 5034
	NETWORK_NOT_AVAILABLE, 5035
	NODE_NOT_AVAILABLE, 5036
	ALL_NODES_NOT_AVAILABLE, 5037
	RESOURCE_FAILED, 5038
	CLUSTER_INVALID_NODE, 5039
	CLUSTER_NODE_EXISTS, 5040
	CLUSTER_JOIN_IN_PROGRESS, 5041
	CLUSTER_NODE_NOT_FOUND, 5042
	CLUSTER_LOCAL_NODE_NOT_FOUND, 5043
	CLUSTER_NETWORK_EXISTS, 5044
	CLUSTER_NETWORK_NOT_FOUND, 5045
	CLUSTER_NETINTERFACE_EXISTS, 5046
	CLUSTER_NETINTERFACE_NOT_FOUND, 5047
	CLUSTER_INVALID_REQUEST, 5048
	CLUSTER_INVALID_NETWORK_PROVIDER, 5049
	CLUSTER_NODE_DOWN, 5050
	CLUSTER_NODE_UNREACHABLE, 5051
	CLUSTER_NODE_NOT_MEMBER, 5052
	CLUSTER_JOIN_NOT_IN_PROGRESS, 5053
	CLUSTER_INVALID_NETWORK, 5054
	CLUSTER_NODE_UP, 5056
	CLUSTER_IPADDR_IN_USE, 5057
	CLUSTER_NODE_NOT_PAUSED, 5058
	CLUSTER_NO_SECURITY_CONTEXT, 5059
	CLUSTER_NETWORK_NOT_INTERNAL, 5060
	CLUSTER_NODE_ALREADY_UP, 5061
	CLUSTER_NODE_ALREADY_DOWN, 5062
	CLUSTER_NETWORK_ALREADY_ONLINE, 5063
	CLUSTER_NETWORK_ALREADY_OFFLINE, 5064
	CLUSTER_NODE_ALREADY_MEMBER, 5065
	CLUSTER_LAST_INTERNAL_NETWORK, 5066
	CLUSTER_NETWORK_HAS_DEPENDENTS, 5067
	INVALID_OPERATION_ON_QUORUM, 5068
	DEPENDENCY_NOT_ALLOWED, 5069
	CLUSTER_NODE_PAUSED, 5070
	NODE_CANT_HOST_RESOURCE, 5071
	CLUSTER_NODE_NOT_READY, 5072
	CLUSTER_NODE_SHUTTING_DOWN, 5073
	CLUSTER_JOIN_ABORTED, 5074
	CLUSTER_INCOMPATIBLE_VERSIONS, 5075
	CLUSTER_MAXNUM_OF_RESOURCES_EXCEEDED, 5076
	CLUSTER_SYSTEM_CONFIG_CHANGED, 5077
	CLUSTER_RESOURCE_TYPE_NOT_FOUND, 5078
	CLUSTER_RESTYPE_NOT_SUPPORTED, 5079
	CLUSTER_RESNAME_NOT_FOUND, 5080
	CLUSTER_NO_RPC_PACKAGES_REGISTERED, 5081
	CLUSTER_OWNER_NOT_IN_PREFLIST, 5082
	CLUSTER_DATABASE_SEQMISMATCH, 5083
	RESMON_INVALID_STATE, 5084
	CLUSTER_GUM_NOT_LOCKER, 5085
	QUORUM_DISK_NOT_FOUND, 5086
	DATABASE_BACKUP_CORRUPT, 5087
	CLUSTER_NODE_ALREADY_HAS_DFS_ROOT, 5088
	RESOURCE_PROPERTY_UNCHANGEABLE, 5089
	NO_ADMIN_ACCESS_POINT, 5090
	CLUSTER_MEMBERSHIP_INVALID_STATE, 5890
	CLUSTER_QUORUMLOG_NOT_FOUND, 5891
	CLUSTER_MEMBERSHIP_HALT, 5892
	CLUSTER_INSTANCE_ID_MISMATCH, 5893
	CLUSTER_NETWORK_NOT_FOUND_FOR_IP, 5894
	CLUSTER_PROPERTY_DATA_TYPE_MISMATCH, 5895
	CLUSTER_EVICT_WITHOUT_CLEANUP, 5896
	CLUSTER_PARAMETER_MISMATCH, 5897
	NODE_CANNOT_BE_CLUSTERED, 5898
	CLUSTER_WRONG_OS_VERSION, 5899
	CLUSTER_CANT_CREATE_DUP_CLUSTER_NAME, 5900
	CLUSCFG_ALREADY_COMMITTED, 5901
	CLUSCFG_ROLLBACK_FAILED, 5902
	CLUSCFG_SYSTEM_DISK_DRIVE_LETTER_CONFLICT, 5903
	CLUSTER_OLD_VERSION, 5904
	CLUSTER_MISMATCHED_COMPUTER_ACCT_NAME, 5905
	CLUSTER_NO_NET_ADAPTERS, 5906
	CLUSTER_POISONED, 5907
	CLUSTER_GROUP_MOVING, 5908
	CLUSTER_RESOURCE_TYPE_BUSY, 5909
	RESOURCE_CALL_TIMED_OUT, 5910
	INVALID_CLUSTER_IPV6_ADDRESS, 5911
	CLUSTER_INTERNAL_INVALID_FUNCTION, 5912
	CLUSTER_PARAMETER_OUT_OF_BOUNDS, 5913
	CLUSTER_PARTIAL_SEND, 5914
	CLUSTER_REGISTRY_INVALID_FUNCTION, 5915
	CLUSTER_INVALID_STRING_TERMINATION, 5916
	CLUSTER_INVALID_STRING_FORMAT, 5917
	CLUSTER_DATABASE_TRANSACTION_IN_PROGRESS, 5918
	CLUSTER_DATABASE_TRANSACTION_NOT_IN_PROGRESS, 5919
	CLUSTER_NULL_DATA, 5920
	CLUSTER_PARTIAL_READ, 5921
	CLUSTER_PARTIAL_WRITE, 5922
	CLUSTER_CANT_DESERIALIZE_DATA, 5923
	DEPENDENT_RESOURCE_PROPERTY_CONFLICT, 5924
	CLUSTER_NO_QUORUM, 5925
	CLUSTER_INVALID_IPV6_NETWORK, 5926
	CLUSTER_INVALID_IPV6_TUNNEL_NETWORK, 5927
	QUORUM_NOT_ALLOWED_IN_THIS_GROUP, 5928
	DEPENDENCY_TREE_TOO_COMPLEX, 5929
	EXCEPTION_IN_RESOURCE_CALL, 5930
	CLUSTER_RHS_FAILED_INITIALIZATION, 5931
	CLUSTER_NOT_INSTALLED, 5932
	CLUSTER_RESOURCES_MUST_BE_ONLINE_ON_THE_SAME_NODE, 5933
	CLUSTER_MAX_NODES_IN_CLUSTER, 5934
	CLUSTER_TOO_MANY_NODES, 5935
	CLUSTER_OBJECT_ALREADY_USED, 5936
	NONCORE_GROUPS_FOUND, 5937
	FILE_SHARE_RESOURCE_CONFLICT, 5938
	CLUSTER_EVICT_INVALID_REQUEST, 5939
	CLUSTER_SINGLETON_RESOURCE, 5940
	CLUSTER_GROUP_SINGLETON_RESOURCE, 5941
	CLUSTER_RESOURCE_PROVIDER_FAILED, 5942
	CLUSTER_RESOURCE_CONFIGURATION_ERROR, 5943
	CLUSTER_GROUP_BUSY, 5944
	CLUSTER_NOT_SHARED_VOLUME, 5945
	CLUSTER_INVALID_SECURITY_DESCRIPTOR, 5946
	CLUSTER_SHARED_VOLUMES_IN_USE, 5947
	CLUSTER_USE_SHARED_VOLUMES_API, 5948
	CLUSTER_BACKUP_IN_PROGRESS, 5949
	NON_CSV_PATH, 5950
	CSV_VOLUME_NOT_LOCAL, 5951
	CLUSTER_WATCHDOG_TERMINATING, 5952
	CLUSTER_RESOURCE_VETOED_MOVE_INCOMPATIBLE_NODES, 5953
	CLUSTER_INVALID_NODE_WEIGHT, 5954
	CLUSTER_RESOURCE_VETOED_CALL, 5955
	RESMON_SYSTEM_RESOURCES_LACKING, 5956
	CLUSTER_RESOURCE_VETOED_MOVE_NOT_ENOUGH_RESOURCES_ON_DESTINATION, 5957
	CLUSTER_RESOURCE_VETOED_MOVE_NOT_ENOUGH_RESOURCES_ON_SOURCE, 5958
	CLUSTER_GROUP_QUEUED, 5959
	CLUSTER_RESOURCE_LOCKED_STATUS, 5960
	CLUSTER_SHARED_VOLUME_FAILOVER_NOT_ALLOWED, 5961
	CLUSTER_NODE_DRAIN_IN_PROGRESS, 5962
	CLUSTER_DISK_NOT_CONNECTED, 5963
	DISK_NOT_CSV_CAPABLE, 5964
	RESOURCE_NOT_IN_AVAILABLE_STORAGE, 5965
	CLUSTER_SHARED_VOLUME_REDIRECTED, 5966
	CLUSTER_SHARED_VOLUME_NOT_REDIRECTED, 5967
	CLUSTER_CANNOT_RETURN_PROPERTIES, 5968
	CLUSTER_RESOURCE_CONTAINS_UNSUPPORTED_DIFF_AREA_FOR_SHARED_VOLUMES, 5969
	CLUSTER_RESOURCE_IS_IN_MAINTENANCE_MODE, 5970
	CLUSTER_AFFINITY_CONFLICT, 5971
	CLUSTER_RESOURCE_IS_REPLICA_VIRTUAL_MACHINE, 5972
	CLUSTER_UPGRADE_INCOMPATIBLE_VERSIONS, 5973
	CLUSTER_UPGRADE_FIX_QUORUM_NOT_SUPPORTED, 5974
	CLUSTER_UPGRADE_RESTART_REQUIRED, 5975
	CLUSTER_UPGRADE_IN_PROGRESS, 5976
	CLUSTER_UPGRADE_INCOMPLETE, 5977
	CLUSTER_NODE_IN_GRACE_PERIOD, 5978
	CLUSTER_CSV_IO_PAUSE_TIMEOUT, 5979
	NODE_NOT_ACTIVE_CLUSTER_MEMBER, 5980
	CLUSTER_RESOURCE_NOT_MONITORED, 5981
	CLUSTER_RESOURCE_DOES_NOT_SUPPORT_UNMONITORED, 5982
	CLUSTER_RESOURCE_IS_REPLICATED, 5983
	CLUSTER_NODE_ISOLATED, 5984
	CLUSTER_NODE_QUARANTINED, 5985
	CLUSTER_DATABASE_UPDATE_CONDITION_FAILED, 5986
	CLUSTER_SPACE_DEGRADED, 5987
	CLUSTER_TOKEN_DELEGATION_NOT_SUPPORTED, 5988
	CLUSTER_CSV_INVALID_HANDLE, 5989
	CLUSTER_CSV_SUPPORTED_ONLY_ON_COORDINATOR, 5990
	GROUPSET_NOT_AVAILABLE, 5991
	GROUPSET_NOT_FOUND, 5992
	GROUPSET_CANT_PROVIDE, 5993
	CLUSTER_FAULT_DOMAIN_PARENT_NOT_FOUND, 5994
	CLUSTER_FAULT_DOMAIN_INVALID_HIERARCHY, 5995
	CLUSTER_FAULT_DOMAIN_FAILED_S2D_VALIDATION, 5996
	CLUSTER_FAULT_DOMAIN_S2D_CONNECTIVITY_LOSS, 5997
	CLUSTER_INVALID_INFRASTRUCTURE_FILESERVER_NAME, 5998
	CLUSTERSET_MANAGEMENT_CLUSTER_UNREACHABLE, 5999
	ENCRYPTION_FAILED, 6000
	DECRYPTION_FAILED, 6001
	FILE_ENCRYPTED, 6002
	NO_RECOVERY_POLICY, 6003
	NO_EFS, 6004
	WRONG_EFS, 6005
	NO_USER_KEYS, 6006
	FILE_NOT_ENCRYPTED, 6007
	NOT_EXPORT_FORMAT, 6008
	FILE_READ_ONLY, 6009
	DIR_EFS_DISALLOWED, 6010
	EFS_SERVER_NOT_TRUSTED, 6011
	BAD_RECOVERY_POLICY, 6012
	EFS_ALG_BLOB_TOO_BIG, 6013
	VOLUME_NOT_SUPPORT_EFS, 6014
	EFS_DISABLED, 6015
	EFS_VERSION_NOT_SUPPORT, 6016
	CS_ENCRYPTION_INVALID_SERVER_RESPONSE, 6017
	CS_ENCRYPTION_UNSUPPORTED_SERVER, 6018
	CS_ENCRYPTION_EXISTING_ENCRYPTED_FILE, 6019
	CS_ENCRYPTION_NEW_ENCRYPTED_FILE, 6020
	CS_ENCRYPTION_FILE_NOT_CSE, 6021
	ENCRYPTION_POLICY_DENIES_OPERATION, 6022
	WIP_ENCRYPTION_FAILED, 6023
	NO_BROWSER_SERVERS_FOUND, 6118
	LOG_SECTOR_INVALID, 6600
	LOG_SECTOR_PARITY_INVALID, 6601
	LOG_SECTOR_REMAPPED, 6602
	LOG_BLOCK_INCOMPLETE, 6603
	LOG_INVALID_RANGE, 6604
	LOG_BLOCKS_EXHAUSTED, 6605
	LOG_READ_CONTEXT_INVALID, 6606
	LOG_RESTART_INVALID, 6607
	LOG_BLOCK_VERSION, 6608
	LOG_BLOCK_INVALID, 6609
	LOG_READ_MODE_INVALID, 6610
	LOG_NO_RESTART, 6611
	LOG_METADATA_CORRUPT, 6612
	LOG_METADATA_INVALID, 6613
	LOG_METADATA_INCONSISTENT, 6614
	LOG_RESERVATION_INVALID, 6615
	LOG_CANT_DELETE, 6616
	LOG_CONTAINER_LIMIT_EXCEEDED, 6617
	LOG_START_OF_LOG, 6618
	LOG_POLICY_ALREADY_INSTALLED, 6619
	LOG_POLICY_NOT_INSTALLED, 6620
	LOG_POLICY_INVALID, 6621
	LOG_POLICY_CONFLICT, 6622
	LOG_PINNED_ARCHIVE_TAIL, 6623
	LOG_RECORD_NONEXISTENT, 6624
	LOG_RECORDS_RESERVED_INVALID, 6625
	LOG_SPACE_RESERVED_INVALID, 6626
	LOG_TAIL_INVALID, 6627
	LOG_FULL, 6628
	COULD_NOT_RESIZE_LOG, 6629
	LOG_MULTIPLEXED, 6630
	LOG_DEDICATED, 6631
	LOG_ARCHIVE_NOT_IN_PROGRESS, 6632
	LOG_ARCHIVE_IN_PROGRESS, 6633
	LOG_EPHEMERAL, 6634
	LOG_NOT_ENOUGH_CONTAINERS, 6635
	LOG_CLIENT_ALREADY_REGISTERED, 6636
	LOG_CLIENT_NOT_REGISTERED, 6637
	LOG_FULL_HANDLER_IN_PROGRESS, 6638
	LOG_CONTAINER_READ_FAILED, 6639
	LOG_CONTAINER_WRITE_FAILED, 6640
	LOG_CONTAINER_OPEN_FAILED, 6641
	LOG_CONTAINER_STATE_INVALID, 6642
	LOG_STATE_INVALID, 6643
	LOG_PINNED, 6644
	LOG_METADATA_FLUSH_FAILED, 6645
	LOG_INCONSISTENT_SECURITY, 6646
	LOG_APPENDED_FLUSH_FAILED, 6647
	LOG_PINNED_RESERVATION, 6648
	INVALID_TRANSACTION, 6700
	TRANSACTION_NOT_ACTIVE, 6701
	TRANSACTION_REQUEST_NOT_VALID, 6702
	TRANSACTION_NOT_REQUESTED, 6703
	TRANSACTION_ALREADY_ABORTED, 6704
	TRANSACTION_ALREADY_COMMITTED, 6705
	TM_INITIALIZATION_FAILED, 6706
	RESOURCEMANAGER_READ_ONLY, 6707
	TRANSACTION_NOT_JOINED, 6708
	TRANSACTION_SUPERIOR_EXISTS, 6709
	CRM_PROTOCOL_ALREADY_EXISTS, 6710
	TRANSACTION_PROPAGATION_FAILED, 6711
	CRM_PROTOCOL_NOT_FOUND, 6712
	TRANSACTION_INVALID_MARSHALL_BUFFER, 6713
	CURRENT_TRANSACTION_NOT_VALID, 6714
	TRANSACTION_NOT_FOUND, 6715
	RESOURCEMANAGER_NOT_FOUND, 6716
	ENLISTMENT_NOT_FOUND, 6717
	TRANSACTIONMANAGER_NOT_FOUND, 6718
	TRANSACTIONMANAGER_NOT_ONLINE, 6719
	TRANSACTIONMANAGER_RECOVERY_NAME_COLLISION, 6720
	TRANSACTION_NOT_ROOT, 6721
	TRANSACTION_OBJECT_EXPIRED, 6722
	TRANSACTION_RESPONSE_NOT_ENLISTED, 6723
	TRANSACTION_RECORD_TOO_LONG, 6724
	IMPLICIT_TRANSACTION_NOT_SUPPORTED, 6725
	TRANSACTION_INTEGRITY_VIOLATED, 6726
	TRANSACTIONMANAGER_IDENTITY_MISMATCH, 6727
	RM_CANNOT_BE_FROZEN_FOR_SNAPSHOT, 6728
	TRANSACTION_MUST_WRITETHROUGH, 6729
	TRANSACTION_NO_SUPERIOR, 6730
	HEURISTIC_DAMAGE_POSSIBLE, 6731
	TRANSACTIONAL_CONFLICT, 6800
	RM_NOT_ACTIVE, 6801
	RM_METADATA_CORRUPT, 6802
	DIRECTORY_NOT_RM, 6803
	TRANSACTIONS_UNSUPPORTED_REMOTE, 6805
	LOG_RESIZE_INVALID_SIZE, 6806
	OBJECT_NO_LONGER_EXISTS, 6807
	STREAM_MINIVERSION_NOT_FOUND, 6808
	STREAM_MINIVERSION_NOT_VALID, 6809
	MINIVERSION_INACCESSIBLE_FROM_SPECIFIED_TRANSACTION, 6810
	CANT_OPEN_MINIVERSION_WITH_MODIFY_INTENT, 6811
	CANT_CREATE_MORE_STREAM_MINIVERSIONS, 6812
	REMOTE_FILE_VERSION_MISMATCH, 6814
	HANDLE_NO_LONGER_VALID, 6815
	NO_TXF_METADATA, 6816
	LOG_CORRUPTION_DETECTED, 6817
	CANT_RECOVER_WITH_HANDLE_OPEN, 6818
	RM_DISCONNECTED, 6819
	ENLISTMENT_NOT_SUPERIOR, 6820
	RECOVERY_NOT_NEEDED, 6821
	RM_ALREADY_STARTED, 6822
	FILE_IDENTITY_NOT_PERSISTENT, 6823
	CANT_BREAK_TRANSACTIONAL_DEPENDENCY, 6824
	CANT_CROSS_RM_BOUNDARY, 6825
	TXF_DIR_NOT_EMPTY, 6826
	INDOUBT_TRANSACTIONS_EXIST, 6827
	TM_VOLATILE, 6828
	ROLLBACK_TIMER_EXPIRED, 6829
	TXF_ATTRIBUTE_CORRUPT, 6830
	EFS_NOT_ALLOWED_IN_TRANSACTION, 6831
	TRANSACTIONAL_OPEN_NOT_ALLOWED, 6832
	LOG_GROWTH_FAILED, 6833
	TRANSACTED_MAPPING_UNSUPPORTED_REMOTE, 6834
	TXF_METADATA_ALREADY_PRESENT, 6835
	TRANSACTION_SCOPE_CALLBACKS_NOT_SET, 6836
	TRANSACTION_REQUIRED_PROMOTION, 6837
	CANNOT_EXECUTE_FILE_IN_TRANSACTION, 6838
	TRANSACTIONS_NOT_FROZEN, 6839
	TRANSACTION_FREEZE_IN_PROGRESS, 6840
	NOT_SNAPSHOT_VOLUME, 6841
	NO_SAVEPOINT_WITH_OPEN_FILES, 6842
	DATA_LOST_REPAIR, 6843
	SPARSE_NOT_ALLOWED_IN_TRANSACTION, 6844
	TM_IDENTITY_MISMATCH, 6845
	FLOATED_SECTION, 6846
	CANNOT_ACCEPT_TRANSACTED_WORK, 6847
	CANNOT_ABORT_TRANSACTIONS, 6848
	BAD_CLUSTERS, 6849
	COMPRESSION_NOT_ALLOWED_IN_TRANSACTION, 6850
	VOLUME_DIRTY, 6851
	NO_LINK_TRACKING_IN_TRANSACTION, 6852
	OPERATION_NOT_SUPPORTED_IN_TRANSACTION, 6853
	EXPIRED_HANDLE, 6854
	TRANSACTION_NOT_ENLISTED, 6855
	CTX_WINSTATION_NAME_INVALID, 7001
	CTX_INVALID_PD, 7002
	CTX_PD_NOT_FOUND, 7003
	CTX_WD_NOT_FOUND, 7004
	CTX_CANNOT_MAKE_EVENTLOG_ENTRY, 7005
	CTX_SERVICE_NAME_COLLISION, 7006
	CTX_CLOSE_PENDING, 7007
	CTX_NO_OUTBUF, 7008
	CTX_MODEM_INF_NOT_FOUND, 7009
	CTX_INVALID_MODEMNAME, 7010
	CTX_MODEM_RESPONSE_ERROR, 7011
	CTX_MODEM_RESPONSE_TIMEOUT, 7012
	CTX_MODEM_RESPONSE_NO_CARRIER, 7013
	CTX_MODEM_RESPONSE_NO_DIALTONE, 7014
	CTX_MODEM_RESPONSE_BUSY, 7015
	CTX_MODEM_RESPONSE_VOICE, 7016
	CTX_TD_ERROR, 7017
	CTX_WINSTATION_NOT_FOUND, 7022
	CTX_WINSTATION_ALREADY_EXISTS, 7023
	CTX_WINSTATION_BUSY, 7024
	CTX_BAD_VIDEO_MODE, 7025
	CTX_GRAPHICS_INVALID, 7035
	CTX_LOGON_DISABLED, 7037
	CTX_NOT_CONSOLE, 7038
	CTX_CLIENT_QUERY_TIMEOUT, 7040
	CTX_CONSOLE_DISCONNECT, 7041
	CTX_CONSOLE_CONNECT, 7042
	CTX_SHADOW_DENIED, 7044
	CTX_WINSTATION_ACCESS_DENIED, 7045
	CTX_INVALID_WD, 7049
	CTX_SHADOW_INVALID, 7050
	CTX_SHADOW_DISABLED, 7051
	CTX_CLIENT_LICENSE_IN_USE, 7052
	CTX_CLIENT_LICENSE_NOT_SET, 7053
	CTX_LICENSE_NOT_AVAILABLE, 7054
	CTX_LICENSE_CLIENT_INVALID, 7055
	CTX_LICENSE_EXPIRED, 7056
	CTX_SHADOW_NOT_RUNNING, 7057
	CTX_SHADOW_ENDED_BY_MODE_CHANGE, 7058
	ACTIVATION_COUNT_EXCEEDED, 7059
	CTX_WINSTATIONS_DISABLED, 7060
	CTX_ENCRYPTION_LEVEL_REQUIRED, 7061
	CTX_SESSION_IN_USE, 7062
	CTX_NO_FORCE_LOGOFF, 7063
	CTX_ACCOUNT_RESTRICTION, 7064
	RDP_PROTOCOL_ERROR, 7065
	CTX_CDM_CONNECT, 7066
	CTX_CDM_DISCONNECT, 7067
	CTX_SECURITY_LAYER_ERROR, 7068
	TS_INCOMPATIBLE_SESSIONS, 7069
	TS_VIDEO_SUBSYSTEM_ERROR, 7070
	DS_NOT_INSTALLED, 8200
	DS_MEMBERSHIP_EVALUATED_LOCALLY, 8201
	DS_NO_ATTRIBUTE_OR_VALUE, 8202
	DS_INVALID_ATTRIBUTE_SYNTAX, 8203
	DS_ATTRIBUTE_TYPE_UNDEFINED, 8204
	DS_ATTRIBUTE_OR_VALUE_EXISTS, 8205
	DS_BUSY, 8206
	DS_UNAVAILABLE, 8207
	DS_NO_RIDS_ALLOCATED, 8208
	DS_NO_MORE_RIDS, 8209
	DS_INCORRECT_ROLE_OWNER, 8210
	DS_RIDMGR_INIT_ERROR, 8211
	DS_OBJ_CLASS_VIOLATION, 8212
	DS_CANT_ON_NON_LEAF, 8213
	DS_CANT_ON_RDN, 8214
	DS_CANT_MOD_OBJ_CLASS, 8215
	DS_CROSS_DOM_MOVE_ERROR, 8216
	DS_GC_NOT_AVAILABLE, 8217
	SHARED_POLICY, 8218
	POLICY_OBJECT_NOT_FOUND, 8219
	POLICY_ONLY_IN_DS, 8220
	PROMOTION_ACTIVE, 8221
	NO_PROMOTION_ACTIVE, 8222
	DS_OPERATIONS_ERROR, 8224
	DS_PROTOCOL_ERROR, 8225
	DS_TIMELIMIT_EXCEEDED, 8226
	DS_SIZELIMIT_EXCEEDED, 8227
	DS_ADMIN_LIMIT_EXCEEDED, 8228
	DS_COMPARE_FALSE, 8229
	DS_COMPARE_TRUE, 8230
	DS_AUTH_METHOD_NOT_SUPPORTED, 8231
	DS_STRONG_AUTH_REQUIRED, 8232
	DS_INAPPROPRIATE_AUTH, 8233
	DS_AUTH_UNKNOWN, 8234
	DS_REFERRAL, 8235
	DS_UNAVAILABLE_CRIT_EXTENSION, 8236
	DS_CONFIDENTIALITY_REQUIRED, 8237
	DS_INAPPROPRIATE_MATCHING, 8238
	DS_CONSTRAINT_VIOLATION, 8239
	DS_NO_SUCH_OBJECT, 8240
	DS_ALIAS_PROBLEM, 8241
	DS_INVALID_DN_SYNTAX, 8242
	DS_IS_LEAF, 8243
	DS_ALIAS_DEREF_PROBLEM, 8244
	DS_UNWILLING_TO_PERFORM, 8245
	DS_LOOP_DETECT, 8246
	DS_NAMING_VIOLATION, 8247
	DS_OBJECT_RESULTS_TOO_LARGE, 8248
	DS_AFFECTS_MULTIPLE_DSAS, 8249
	DS_SERVER_DOWN, 8250
	DS_LOCAL_ERROR, 8251
	DS_ENCODING_ERROR, 8252
	DS_DECODING_ERROR, 8253
	DS_FILTER_UNKNOWN, 8254
	DS_PARAM_ERROR, 8255
	DS_NOT_SUPPORTED, 8256
	DS_NO_RESULTS_RETURNED, 8257
	DS_CONTROL_NOT_FOUND, 8258
	DS_CLIENT_LOOP, 8259
	DS_REFERRAL_LIMIT_EXCEEDED, 8260
	DS_SORT_CONTROL_MISSING, 8261
	DS_OFFSET_RANGE_ERROR, 8262
	DS_RIDMGR_DISABLED, 8263
	DS_ROOT_MUST_BE_NC, 8301
	DS_ADD_REPLICA_INHIBITED, 8302
	DS_ATT_NOT_DEF_IN_SCHEMA, 8303
	DS_MAX_OBJ_SIZE_EXCEEDED, 8304
	DS_OBJ_STRING_NAME_EXISTS, 8305
	DS_NO_RDN_DEFINED_IN_SCHEMA, 8306
	DS_RDN_DOESNT_MATCH_SCHEMA, 8307
	DS_NO_REQUESTED_ATTS_FOUND, 8308
	DS_USER_BUFFER_TO_SMALL, 8309
	DS_ATT_IS_NOT_ON_OBJ, 8310
	DS_ILLEGAL_MOD_OPERATION, 8311
	DS_OBJ_TOO_LARGE, 8312
	DS_BAD_INSTANCE_TYPE, 8313
	DS_MASTERDSA_REQUIRED, 8314
	DS_OBJECT_CLASS_REQUIRED, 8315
	DS_MISSING_REQUIRED_ATT, 8316
	DS_ATT_NOT_DEF_FOR_CLASS, 8317
	DS_ATT_ALREADY_EXISTS, 8318
	DS_CANT_ADD_ATT_VALUES, 8320
	DS_SINGLE_VALUE_CONSTRAINT, 8321
	DS_RANGE_CONSTRAINT, 8322
	DS_ATT_VAL_ALREADY_EXISTS, 8323
	DS_CANT_REM_MISSING_ATT, 8324
	DS_CANT_REM_MISSING_ATT_VAL, 8325
	DS_ROOT_CANT_BE_SUBREF, 8326
	DS_NO_CHAINING, 8327
	DS_NO_CHAINED_EVAL, 8328
	DS_NO_PARENT_OBJECT, 8329
	DS_PARENT_IS_AN_ALIAS, 8330
	DS_CANT_MIX_MASTER_AND_REPS, 8331
	DS_CHILDREN_EXIST, 8332
	DS_OBJ_NOT_FOUND, 8333
	DS_ALIASED_OBJ_MISSING, 8334
	DS_BAD_NAME_SYNTAX, 8335
	DS_ALIAS_POINTS_TO_ALIAS, 8336
	DS_CANT_DEREF_ALIAS, 8337
	DS_OUT_OF_SCOPE, 8338
	DS_OBJECT_BEING_REMOVED, 8339
	DS_CANT_DELETE_DSA_OBJ, 8340
	DS_GENERIC_ERROR, 8341
	DS_DSA_MUST_BE_INT_MASTER, 8342
	DS_CLASS_NOT_DSA, 8343
	DS_INSUFF_ACCESS_RIGHTS, 8344
	DS_ILLEGAL_SUPERIOR, 8345
	DS_ATTRIBUTE_OWNED_BY_SAM, 8346
	DS_NAME_TOO_MANY_PARTS, 8347
	DS_NAME_TOO_LONG, 8348
	DS_NAME_VALUE_TOO_LONG, 8349
	DS_NAME_UNPARSEABLE, 8350
	DS_NAME_TYPE_UNKNOWN, 8351
	DS_NOT_AN_OBJECT, 8352
	DS_SEC_DESC_TOO_SHORT, 8353
	DS_SEC_DESC_INVALID, 8354
	DS_NO_DELETED_NAME, 8355
	DS_SUBREF_MUST_HAVE_PARENT, 8356
	DS_NCNAME_MUST_BE_NC, 8357
	DS_CANT_ADD_SYSTEM_ONLY, 8358
	DS_CLASS_MUST_BE_CONCRETE, 8359
	DS_INVALID_DMD, 8360
	DS_OBJ_GUID_EXISTS, 8361
	DS_NOT_ON_BACKLINK, 8362
	DS_NO_CROSSREF_FOR_NC, 8363
	DS_SHUTTING_DOWN, 8364
	DS_UNKNOWN_OPERATION, 8365
	DS_INVALID_ROLE_OWNER, 8366
	DS_COULDNT_CONTACT_FSMO, 8367
	DS_CROSS_NC_DN_RENAME, 8368
	DS_CANT_MOD_SYSTEM_ONLY, 8369
	DS_REPLICATOR_ONLY, 8370
	DS_OBJ_CLASS_NOT_DEFINED, 8371
	DS_OBJ_CLASS_NOT_SUBCLASS, 8372
	DS_NAME_REFERENCE_INVALID, 8373
	DS_CROSS_REF_EXISTS, 8374
	DS_CANT_DEL_MASTER_CROSSREF, 8375
	DS_SUBTREE_NOTIFY_NOT_NC_HEAD, 8376
	DS_NOTIFY_FILTER_TOO_COMPLEX, 8377
	DS_DUP_RDN, 8378
	DS_DUP_OID, 8379
	DS_DUP_MAPI_ID, 8380
	DS_DUP_SCHEMA_ID_GUID, 8381
	DS_DUP_LDAP_DISPLAY_NAME, 8382
	DS_SEMANTIC_ATT_TEST, 8383
	DS_SYNTAX_MISMATCH, 8384
	DS_EXISTS_IN_MUST_HAVE, 8385
	DS_EXISTS_IN_MAY_HAVE, 8386
	DS_NONEXISTENT_MAY_HAVE, 8387
	DS_NONEXISTENT_MUST_HAVE, 8388
	DS_AUX_CLS_TEST_FAIL, 8389
	DS_NONEXISTENT_POSS_SUP, 8390
	DS_SUB_CLS_TEST_FAIL, 8391
	DS_BAD_RDN_ATT_ID_SYNTAX, 8392
	DS_EXISTS_IN_AUX_CLS, 8393
	DS_EXISTS_IN_SUB_CLS, 8394
	DS_EXISTS_IN_POSS_SUP, 8395
	DS_RECALCSCHEMA_FAILED, 8396
	DS_TREE_DELETE_NOT_FINISHED, 8397
	DS_CANT_DELETE, 8398
	DS_ATT_SCHEMA_REQ_ID, 8399
	DS_BAD_ATT_SCHEMA_SYNTAX, 8400
	DS_CANT_CACHE_ATT, 8401
	DS_CANT_CACHE_CLASS, 8402
	DS_CANT_REMOVE_ATT_CACHE, 8403
	DS_CANT_REMOVE_CLASS_CACHE, 8404
	DS_CANT_RETRIEVE_DN, 8405
	DS_MISSING_SUPREF, 8406
	DS_CANT_RETRIEVE_INSTANCE, 8407
	DS_CODE_INCONSISTENCY, 8408
	DS_DATABASE_ERROR, 8409
	DS_GOVERNSID_MISSING, 8410
	DS_MISSING_EXPECTED_ATT, 8411
	DS_NCNAME_MISSING_CR_REF, 8412
	DS_SECURITY_CHECKING_ERROR, 8413
	DS_SCHEMA_NOT_LOADED, 8414
	DS_SCHEMA_ALLOC_FAILED, 8415
	DS_ATT_SCHEMA_REQ_SYNTAX, 8416
	DS_GCVERIFY_ERROR, 8417
	DS_DRA_SCHEMA_MISMATCH, 8418
	DS_CANT_FIND_DSA_OBJ, 8419
	DS_CANT_FIND_EXPECTED_NC, 8420
	DS_CANT_FIND_NC_IN_CACHE, 8421
	DS_CANT_RETRIEVE_CHILD, 8422
	DS_SECURITY_ILLEGAL_MODIFY, 8423
	DS_CANT_REPLACE_HIDDEN_REC, 8424
	DS_BAD_HIERARCHY_FILE, 8425
	DS_BUILD_HIERARCHY_TABLE_FAILED, 8426
	DS_CONFIG_PARAM_MISSING, 8427
	DS_COUNTING_AB_INDICES_FAILED, 8428
	DS_HIERARCHY_TABLE_MALLOC_FAILED, 8429
	DS_INTERNAL_FAILURE, 8430
	DS_UNKNOWN_ERROR, 8431
	DS_ROOT_REQUIRES_CLASS_TOP, 8432
	DS_REFUSING_FSMO_ROLES, 8433
	DS_MISSING_FSMO_SETTINGS, 8434
	DS_UNABLE_TO_SURRENDER_ROLES, 8435
	DS_DRA_GENERIC, 8436
	DS_DRA_INVALID_PARAMETER, 8437
	DS_DRA_BUSY, 8438
	DS_DRA_BAD_DN, 8439
	DS_DRA_BAD_NC, 8440
	DS_DRA_DN_EXISTS, 8441
	DS_DRA_INTERNAL_ERROR, 8442
	DS_DRA_INCONSISTENT_DIT, 8443
	DS_DRA_CONNECTION_FAILED, 8444
	DS_DRA_BAD_INSTANCE_TYPE, 8445
	DS_DRA_OUT_OF_MEM, 8446
	DS_DRA_MAIL_PROBLEM, 8447
	DS_DRA_REF_ALREADY_EXISTS, 8448
	DS_DRA_REF_NOT_FOUND, 8449
	DS_DRA_OBJ_IS_REP_SOURCE, 8450
	DS_DRA_DB_ERROR, 8451
	DS_DRA_NO_REPLICA, 8452
	DS_DRA_ACCESS_DENIED, 8453
	DS_DRA_NOT_SUPPORTED, 8454
	DS_DRA_RPC_CANCELLED, 8455
	DS_DRA_SOURCE_DISABLED, 8456
	DS_DRA_SINK_DISABLED, 8457
	DS_DRA_NAME_COLLISION, 8458
	DS_DRA_SOURCE_REINSTALLED, 8459
	DS_DRA_MISSING_PARENT, 8460
	DS_DRA_PREEMPTED, 8461
	DS_DRA_ABANDON_SYNC, 8462
	DS_DRA_SHUTDOWN, 8463
	DS_DRA_INCOMPATIBLE_PARTIAL_SET, 8464
	DS_DRA_SOURCE_IS_PARTIAL_REPLICA, 8465
	DS_DRA_EXTN_CONNECTION_FAILED, 8466
	DS_INSTALL_SCHEMA_MISMATCH, 8467
	DS_DUP_LINK_ID, 8468
	DS_NAME_ERROR_RESOLVING, 8469
	DS_NAME_ERROR_NOT_FOUND, 8470
	DS_NAME_ERROR_NOT_UNIQUE, 8471
	DS_NAME_ERROR_NO_MAPPING, 8472
	DS_NAME_ERROR_DOMAIN_ONLY, 8473
	DS_NAME_ERROR_NO_SYNTACTICAL_MAPPING, 8474
	DS_CONSTRUCTED_ATT_MOD, 8475
	DS_WRONG_OM_OBJ_CLASS, 8476
	DS_DRA_REPL_PENDING, 8477
	DS_DS_REQUIRED, 8478
	DS_INVALID_LDAP_DISPLAY_NAME, 8479
	DS_NON_BASE_SEARCH, 8480
	DS_CANT_RETRIEVE_ATTS, 8481
	DS_BACKLINK_WITHOUT_LINK, 8482
	DS_EPOCH_MISMATCH, 8483
	DS_SRC_NAME_MISMATCH, 8484
	DS_SRC_AND_DST_NC_IDENTICAL, 8485
	DS_DST_NC_MISMATCH, 8486
	DS_NOT_AUTHORITIVE_FOR_DST_NC, 8487
	DS_SRC_GUID_MISMATCH, 8488
	DS_CANT_MOVE_DELETED_OBJECT, 8489
	DS_PDC_OPERATION_IN_PROGRESS, 8490
	DS_CROSS_DOMAIN_CLEANUP_REQD, 8491
	DS_ILLEGAL_XDOM_MOVE_OPERATION, 8492
	DS_CANT_WITH_ACCT_GROUP_MEMBERSHPS, 8493
	DS_NC_MUST_HAVE_NC_PARENT, 8494
	DS_CR_IMPOSSIBLE_TO_VALIDATE, 8495
	DS_DST_DOMAIN_NOT_NATIVE, 8496
	DS_MISSING_INFRASTRUCTURE_CONTAINER, 8497
	DS_CANT_MOVE_ACCOUNT_GROUP, 8498
	DS_CANT_MOVE_RESOURCE_GROUP, 8499
	DS_INVALID_SEARCH_FLAG, 8500
	DS_NO_TREE_DELETE_ABOVE_NC, 8501
	DS_COULDNT_LOCK_TREE_FOR_DELETE, 8502
	DS_COULDNT_IDENTIFY_OBJECTS_FOR_TREE_DELETE, 8503
	DS_SAM_INIT_FAILURE, 8504
	DS_SENSITIVE_GROUP_VIOLATION, 8505
	DS_CANT_MOD_PRIMARYGROUPID, 8506
	DS_ILLEGAL_BASE_SCHEMA_MOD, 8507
	DS_NONSAFE_SCHEMA_CHANGE, 8508
	DS_SCHEMA_UPDATE_DISALLOWED, 8509
	DS_CANT_CREATE_UNDER_SCHEMA, 8510
	DS_INSTALL_NO_SRC_SCH_VERSION, 8511
	DS_INSTALL_NO_SCH_VERSION_IN_INIFILE, 8512
	DS_INVALID_GROUP_TYPE, 8513
	DS_NO_NEST_GLOBALGROUP_IN_MIXEDDOMAIN, 8514
	DS_NO_NEST_LOCALGROUP_IN_MIXEDDOMAIN, 8515
	DS_GLOBAL_CANT_HAVE_LOCAL_MEMBER, 8516
	DS_GLOBAL_CANT_HAVE_UNIVERSAL_MEMBER, 8517
	DS_UNIVERSAL_CANT_HAVE_LOCAL_MEMBER, 8518
	DS_GLOBAL_CANT_HAVE_CROSSDOMAIN_MEMBER, 8519
	DS_LOCAL_CANT_HAVE_CROSSDOMAIN_LOCAL_MEMBER, 8520
	DS_HAVE_PRIMARY_MEMBERS, 8521
	DS_STRING_SD_CONVERSION_FAILED, 8522
	DS_NAMING_MASTER_GC, 8523
	DS_DNS_LOOKUP_FAILURE, 8524
	DS_COULDNT_UPDATE_SPNS, 8525
	DS_CANT_RETRIEVE_SD, 8526
	DS_KEY_NOT_UNIQUE, 8527
	DS_WRONG_LINKED_ATT_SYNTAX, 8528
	DS_SAM_NEED_BOOTKEY_PASSWORD, 8529
	DS_SAM_NEED_BOOTKEY_FLOPPY, 8530
	DS_CANT_START, 8531
	DS_INIT_FAILURE, 8532
	DS_NO_PKT_PRIVACY_ON_CONNECTION, 8533
	DS_SOURCE_DOMAIN_IN_FOREST, 8534
	DS_DESTINATION_DOMAIN_NOT_IN_FOREST, 8535
	DS_DESTINATION_AUDITING_NOT_ENABLED, 8536
	DS_CANT_FIND_DC_FOR_SRC_DOMAIN, 8537
	DS_SRC_OBJ_NOT_GROUP_OR_USER, 8538
	DS_SRC_SID_EXISTS_IN_FOREST, 8539
	DS_SRC_AND_DST_OBJECT_CLASS_MISMATCH, 8540
	SAM_INIT_FAILURE, 8541
	DS_DRA_SCHEMA_INFO_SHIP, 8542
	DS_DRA_SCHEMA_CONFLICT, 8543
	DS_DRA_EARLIER_SCHEMA_CONFLICT, 8544
	DS_DRA_OBJ_NC_MISMATCH, 8545
	DS_NC_STILL_HAS_DSAS, 8546
	DS_GC_REQUIRED, 8547
	DS_LOCAL_MEMBER_OF_LOCAL_ONLY, 8548
	DS_NO_FPO_IN_UNIVERSAL_GROUPS, 8549
	DS_CANT_ADD_TO_GC, 8550
	DS_NO_CHECKPOINT_WITH_PDC, 8551
	DS_SOURCE_AUDITING_NOT_ENABLED, 8552
	DS_CANT_CREATE_IN_NONDOMAIN_NC, 8553
	DS_INVALID_NAME_FOR_SPN, 8554
	DS_FILTER_USES_CONTRUCTED_ATTRS, 8555
	DS_UNICODEPWD_NOT_IN_QUOTES, 8556
	DS_MACHINE_ACCOUNT_QUOTA_EXCEEDED, 8557
	DS_MUST_BE_RUN_ON_DST_DC, 8558
	DS_SRC_DC_MUST_BE_SP4_OR_GREATER, 8559
	DS_CANT_TREE_DELETE_CRITICAL_OBJ, 8560
	DS_INIT_FAILURE_CONSOLE, 8561
	DS_SAM_INIT_FAILURE_CONSOLE, 8562
	DS_FOREST_VERSION_TOO_HIGH, 8563
	DS_DOMAIN_VERSION_TOO_HIGH, 8564
	DS_FOREST_VERSION_TOO_LOW, 8565
	DS_DOMAIN_VERSION_TOO_LOW, 8566
	DS_INCOMPATIBLE_VERSION, 8567
	DS_LOW_DSA_VERSION, 8568
	DS_NO_BEHAVIOR_VERSION_IN_MIXEDDOMAIN, 8569
	DS_NOT_SUPPORTED_SORT_ORDER, 8570
	DS_NAME_NOT_UNIQUE, 8571
	DS_MACHINE_ACCOUNT_CREATED_PRENT4, 8572
	DS_OUT_OF_VERSION_STORE, 8573
	DS_INCOMPATIBLE_CONTROLS_USED, 8574
	DS_NO_REF_DOMAIN, 8575
	DS_RESERVED_LINK_ID, 8576
	DS_LINK_ID_NOT_AVAILABLE, 8577
	DS_AG_CANT_HAVE_UNIVERSAL_MEMBER, 8578
	DS_MODIFYDN_DISALLOWED_BY_INSTANCE_TYPE, 8579
	DS_NO_OBJECT_MOVE_IN_SCHEMA_NC, 8580
	DS_MODIFYDN_DISALLOWED_BY_FLAG, 8581
	DS_MODIFYDN_WRONG_GRANDPARENT, 8582
	DS_NAME_ERROR_TRUST_REFERRAL, 8583
	NOT_SUPPORTED_ON_STANDARD_SERVER, 8584
	DS_CANT_ACCESS_REMOTE_PART_OF_AD, 8585
	DS_CR_IMPOSSIBLE_TO_VALIDATE_V2, 8586
	DS_THREAD_LIMIT_EXCEEDED, 8587
	DS_NOT_CLOSEST, 8588
	DS_CANT_DERIVE_SPN_WITHOUT_SERVER_REF, 8589
	DS_SINGLE_USER_MODE_FAILED, 8590
	DS_NTDSCRIPT_SYNTAX_ERROR, 8591
	DS_NTDSCRIPT_PROCESS_ERROR, 8592
	DS_DIFFERENT_REPL_EPOCHS, 8593
	DS_DRS_EXTENSIONS_CHANGED, 8594
	DS_REPLICA_SET_CHANGE_NOT_ALLOWED_ON_DISABLED_CR, 8595
	DS_NO_MSDS_INTID, 8596
	DS_DUP_MSDS_INTID, 8597
	DS_EXISTS_IN_RDNATTID, 8598
	DS_AUTHORIZATION_FAILED, 8599
	DS_INVALID_SCRIPT, 8600
	DS_REMOTE_CROSSREF_OP_FAILED, 8601
	DS_CROSS_REF_BUSY, 8602
	DS_CANT_DERIVE_SPN_FOR_DELETED_DOMAIN, 8603
	DS_CANT_DEMOTE_WITH_WRITEABLE_NC, 8604
	DS_DUPLICATE_ID_FOUND, 8605
	DS_INSUFFICIENT_ATTR_TO_CREATE_OBJECT, 8606
	DS_GROUP_CONVERSION_ERROR, 8607
	DS_CANT_MOVE_APP_BASIC_GROUP, 8608
	DS_CANT_MOVE_APP_QUERY_GROUP, 8609
	DS_ROLE_NOT_VERIFIED, 8610
	DS_WKO_CONTAINER_CANNOT_BE_SPECIAL, 8611
	DS_DOMAIN_RENAME_IN_PROGRESS, 8612
	DS_EXISTING_AD_CHILD_NC, 8613
	DS_REPL_LIFETIME_EXCEEDED, 8614
	DS_DISALLOWED_IN_SYSTEM_CONTAINER, 8615
	DS_LDAP_SEND_QUEUE_FULL, 8616
	DS_DRA_OUT_SCHEDULE_WINDOW, 8617
	DS_POLICY_NOT_KNOWN, 8618
	NO_SITE_SETTINGS_OBJECT, 8619
	NO_SECRETS, 8620
	NO_WRITABLE_DC_FOUND, 8621
	DS_NO_SERVER_OBJECT, 8622
	DS_NO_NTDSA_OBJECT, 8623
	DS_NON_ASQ_SEARCH, 8624
	DS_AUDIT_FAILURE, 8625
	DS_INVALID_SEARCH_FLAG_SUBTREE, 8626
	DS_INVALID_SEARCH_FLAG_TUPLE, 8627
	DS_HIERARCHY_TABLE_TOO_DEEP, 8628
	DS_DRA_CORRUPT_UTD_VECTOR, 8629
	DS_DRA_SECRETS_DENIED, 8630
	DS_RESERVED_MAPI_ID, 8631
	DS_MAPI_ID_NOT_AVAILABLE, 8632
	DS_DRA_MISSING_KRBTGT_SECRET, 8633
	DS_DOMAIN_NAME_EXISTS_IN_FOREST, 8634
	DS_FLAT_NAME_EXISTS_IN_FOREST, 8635
	INVALID_USER_PRINCIPAL_NAME, 8636
	DS_OID_MAPPED_GROUP_CANT_HAVE_MEMBERS, 8637
	DS_OID_NOT_FOUND, 8638
	DS_DRA_RECYCLED_TARGET, 8639
	DS_DISALLOWED_NC_REDIRECT, 8640
	DS_HIGH_ADLDS_FFL, 8641
	DS_HIGH_DSA_VERSION, 8642
	DS_LOW_ADLDS_FFL, 8643
	DOMAIN_SID_SAME_AS_LOCAL_WORKSTATION, 8644
	DS_UNDELETE_SAM_VALIDATION_FAILED, 8645
	INCORRECT_ACCOUNT_TYPE, 8646
	DS_SPN_VALUE_NOT_UNIQUE_IN_FOREST, 8647
	DS_UPN_VALUE_NOT_UNIQUE_IN_FOREST, 8648
	DS_MISSING_FOREST_TRUST, 8649
	DS_VALUE_KEY_NOT_UNIQUE, 8650
	IPSEC_QM_POLICY_EXISTS, 13000
	IPSEC_QM_POLICY_NOT_FOUND, 13001
	IPSEC_QM_POLICY_IN_USE, 13002
	IPSEC_MM_POLICY_EXISTS, 13003
	IPSEC_MM_POLICY_NOT_FOUND, 13004
	IPSEC_MM_POLICY_IN_USE, 13005
	IPSEC_MM_FILTER_EXISTS, 13006
	IPSEC_MM_FILTER_NOT_FOUND, 13007
	IPSEC_TRANSPORT_FILTER_EXISTS, 13008
	IPSEC_TRANSPORT_FILTER_NOT_FOUND, 13009
	IPSEC_MM_AUTH_EXISTS, 13010
	IPSEC_MM_AUTH_NOT_FOUND, 13011
	IPSEC_MM_AUTH_IN_USE, 13012
	IPSEC_DEFAULT_MM_POLICY_NOT_FOUND, 13013
	IPSEC_DEFAULT_MM_AUTH_NOT_FOUND, 13014
	IPSEC_DEFAULT_QM_POLICY_NOT_FOUND, 13015
	IPSEC_TUNNEL_FILTER_EXISTS, 13016
	IPSEC_TUNNEL_FILTER_NOT_FOUND, 13017
	IPSEC_MM_FILTER_PENDING_DELETION, 13018
	IPSEC_TRANSPORT_FILTER_PENDING_DELETION, 13019
	IPSEC_TUNNEL_FILTER_PENDING_DELETION, 13020
	IPSEC_MM_POLICY_PENDING_DELETION, 13021
	IPSEC_MM_AUTH_PENDING_DELETION, 13022
	IPSEC_QM_POLICY_PENDING_DELETION, 13023
	IPSEC_IKE_NEG_STATUS_BEGIN, 13800
	IPSEC_IKE_AUTH_FAIL, 13801
	IPSEC_IKE_ATTRIB_FAIL, 13802
	IPSEC_IKE_NEGOTIATION_PENDING, 13803
	IPSEC_IKE_GENERAL_PROCESSING_ERROR, 13804
	IPSEC_IKE_TIMED_OUT, 13805
	IPSEC_IKE_NO_CERT, 13806
	IPSEC_IKE_SA_DELETED, 13807
	IPSEC_IKE_SA_REAPED, 13808
	IPSEC_IKE_MM_ACQUIRE_DROP, 13809
	IPSEC_IKE_QM_ACQUIRE_DROP, 13810
	IPSEC_IKE_QUEUE_DROP_MM, 13811
	IPSEC_IKE_QUEUE_DROP_NO_MM, 13812
	IPSEC_IKE_DROP_NO_RESPONSE, 13813
	IPSEC_IKE_MM_DELAY_DROP, 13814
	IPSEC_IKE_QM_DELAY_DROP, 13815
	IPSEC_IKE_ERROR, 13816
	IPSEC_IKE_CRL_FAILED, 13817
	IPSEC_IKE_INVALID_KEY_USAGE, 13818
	IPSEC_IKE_INVALID_CERT_TYPE, 13819
	IPSEC_IKE_NO_PRIVATE_KEY, 13820
	IPSEC_IKE_SIMULTANEOUS_REKEY, 13821
	IPSEC_IKE_DH_FAIL, 13822
	IPSEC_IKE_CRITICAL_PAYLOAD_NOT_RECOGNIZED, 13823
	IPSEC_IKE_INVALID_HEADER, 13824
	IPSEC_IKE_NO_POLICY, 13825
	IPSEC_IKE_INVALID_SIGNATURE, 13826
	IPSEC_IKE_KERBEROS_ERROR, 13827
	IPSEC_IKE_NO_PUBLIC_KEY, 13828
	IPSEC_IKE_PROCESS_ERR, 13829
	IPSEC_IKE_PROCESS_ERR_SA, 13830
	IPSEC_IKE_PROCESS_ERR_PROP, 13831
	IPSEC_IKE_PROCESS_ERR_TRANS, 13832
	IPSEC_IKE_PROCESS_ERR_KE, 13833
	IPSEC_IKE_PROCESS_ERR_ID, 13834
	IPSEC_IKE_PROCESS_ERR_CERT, 13835
	IPSEC_IKE_PROCESS_ERR_CERT_REQ, 13836
	IPSEC_IKE_PROCESS_ERR_HASH, 13837
	IPSEC_IKE_PROCESS_ERR_SIG, 13838
	IPSEC_IKE_PROCESS_ERR_NONCE, 13839
	IPSEC_IKE_PROCESS_ERR_NOTIFY, 13840
	IPSEC_IKE_PROCESS_ERR_DELETE, 13841
	IPSEC_IKE_PROCESS_ERR_VENDOR, 13842
	IPSEC_IKE_INVALID_PAYLOAD, 13843
	IPSEC_IKE_LOAD_SOFT_SA, 13844
	IPSEC_IKE_SOFT_SA_TORN_DOWN, 13845
	IPSEC_IKE_INVALID_COOKIE, 13846
	IPSEC_IKE_NO_PEER_CERT, 13847
	IPSEC_IKE_PEER_CRL_FAILED, 13848
	IPSEC_IKE_POLICY_CHANGE, 13849
	IPSEC_IKE_NO_MM_POLICY, 13850
	IPSEC_IKE_NOTCBPRIV, 13851
	IPSEC_IKE_SECLOADFAIL, 13852
	IPSEC_IKE_FAILSSPINIT, 13853
	IPSEC_IKE_FAILQUERYSSP, 13854
	IPSEC_IKE_SRVACQFAIL, 13855
	IPSEC_IKE_SRVQUERYCRED, 13856
	IPSEC_IKE_GETSPIFAIL, 13857
	IPSEC_IKE_INVALID_FILTER, 13858
	IPSEC_IKE_OUT_OF_MEMORY, 13859
	IPSEC_IKE_ADD_UPDATE_KEY_FAILED, 13860
	IPSEC_IKE_INVALID_POLICY, 13861
	IPSEC_IKE_UNKNOWN_DOI, 13862
	IPSEC_IKE_INVALID_SITUATION, 13863
	IPSEC_IKE_DH_FAILURE, 13864
	IPSEC_IKE_INVALID_GROUP, 13865
	IPSEC_IKE_ENCRYPT, 13866
	IPSEC_IKE_DECRYPT, 13867
	IPSEC_IKE_POLICY_MATCH, 13868
	IPSEC_IKE_UNSUPPORTED_ID, 13869
	IPSEC_IKE_INVALID_HASH, 13870
	IPSEC_IKE_INVALID_HASH_ALG, 13871
	IPSEC_IKE_INVALID_HASH_SIZE, 13872
	IPSEC_IKE_INVALID_ENCRYPT_ALG, 13873
	IPSEC_IKE_INVALID_AUTH_ALG, 13874
	IPSEC_IKE_INVALID_SIG, 13875
	IPSEC_IKE_LOAD_FAILED, 13876
	IPSEC_IKE_RPC_DELETE, 13877
	IPSEC_IKE_BENIGN_REINIT, 13878
	IPSEC_IKE_INVALID_RESPONDER_LIFETIME_NOTIFY, 13879
	IPSEC_IKE_INVALID_MAJOR_VERSION, 13880
	IPSEC_IKE_INVALID_CERT_KEYLEN, 13881
	IPSEC_IKE_MM_LIMIT, 13882
	IPSEC_IKE_NEGOTIATION_DISABLED, 13883
	IPSEC_IKE_QM_LIMIT, 13884
	IPSEC_IKE_MM_EXPIRED, 13885
	IPSEC_IKE_PEER_MM_ASSUMED_INVALID, 13886
	IPSEC_IKE_CERT_CHAIN_POLICY_MISMATCH, 13887
	IPSEC_IKE_UNEXPECTED_MESSAGE_ID, 13888
	IPSEC_IKE_INVALID_AUTH_PAYLOAD, 13889
	IPSEC_IKE_DOS_COOKIE_SENT, 13890
	IPSEC_IKE_SHUTTING_DOWN, 13891
	IPSEC_IKE_CGA_AUTH_FAILED, 13892
	IPSEC_IKE_PROCESS_ERR_NATOA, 13893
	IPSEC_IKE_INVALID_MM_FOR_QM, 13894
	IPSEC_IKE_QM_EXPIRED, 13895
	IPSEC_IKE_TOO_MANY_FILTERS, 13896
	IPSEC_IKE_NEG_STATUS_END, 13897
	IPSEC_IKE_KILL_DUMMY_NAP_TUNNEL, 13898
	IPSEC_IKE_INNER_IP_ASSIGNMENT_FAILURE, 13899
	IPSEC_IKE_REQUIRE_CP_PAYLOAD_MISSING, 13900
	IPSEC_KEY_MODULE_IMPERSONATION_NEGOTIATION_PENDING, 13901
	IPSEC_IKE_COEXISTENCE_SUPPRESS, 13902
	IPSEC_IKE_RATELIMIT_DROP, 13903
	IPSEC_IKE_PEER_DOESNT_SUPPORT_MOBIKE, 13904
	IPSEC_IKE_AUTHORIZATION_FAILURE, 13905
	IPSEC_IKE_STRONG_CRED_AUTHORIZATION_FAILURE, 13906
	IPSEC_IKE_AUTHORIZATION_FAILURE_WITH_OPTIONAL_RETRY, 13907
	IPSEC_IKE_STRONG_CRED_AUTHORIZATION_AND_CERTMAP_FAILURE, 13908
	IPSEC_IKE_NEG_STATUS_EXTENDED_END, 13909
	IPSEC_BAD_SPI, 13910
	IPSEC_SA_LIFETIME_EXPIRED, 13911
	IPSEC_WRONG_SA, 13912
	IPSEC_REPLAY_CHECK_FAILED, 13913
	IPSEC_INVALID_PACKET, 13914
	IPSEC_INTEGRITY_CHECK_FAILED, 13915
	IPSEC_CLEAR_TEXT_DROP, 13916
	IPSEC_AUTH_FIREWALL_DROP, 13917
	IPSEC_THROTTLE_DROP, 13918
	IPSEC_DOSP_BLOCK, 13925
	IPSEC_DOSP_RECEIVED_MULTICAST, 13926
	IPSEC_DOSP_INVALID_PACKET, 13927
	IPSEC_DOSP_STATE_LOOKUP_FAILED, 13928
	IPSEC_DOSP_MAX_ENTRIES, 13929
	IPSEC_DOSP_KEYMOD_NOT_ALLOWED, 13930
	IPSEC_DOSP_NOT_INSTALLED, 13931
	IPSEC_DOSP_MAX_PER_IP_RATELIMIT_QUEUES, 13932
	SXS_SECTION_NOT_FOUND, 14000
	SXS_CANT_GEN_ACTCTX, 14001
	SXS_INVALID_ACTCTXDATA_FORMAT, 14002
	SXS_ASSEMBLY_NOT_FOUND, 14003
	SXS_MANIFEST_FORMAT_ERROR, 14004
	SXS_MANIFEST_PARSE_ERROR, 14005
	SXS_ACTIVATION_CONTEXT_DISABLED, 14006
	SXS_KEY_NOT_FOUND, 14007
	SXS_VERSION_CONFLICT, 14008
	SXS_WRONG_SECTION_TYPE, 14009
	SXS_THREAD_QUERIES_DISABLED, 14010
	SXS_PROCESS_DEFAULT_ALREADY_SET, 14011
	SXS_UNKNOWN_ENCODING_GROUP, 14012
	SXS_UNKNOWN_ENCODING, 14013
	SXS_INVALID_XML_NAMESPACE_URI, 14014
	SXS_ROOT_MANIFEST_DEPENDENCY_NOT_INSTALLED, 14015
	SXS_LEAF_MANIFEST_DEPENDENCY_NOT_INSTALLED, 14016
	SXS_INVALID_ASSEMBLY_IDENTITY_ATTRIBUTE, 14017
	SXS_MANIFEST_MISSING_REQUIRED_DEFAULT_NAMESPACE, 14018
	SXS_MANIFEST_INVALID_REQUIRED_DEFAULT_NAMESPACE, 14019
	SXS_PRIVATE_MANIFEST_CROSS_PATH_WITH_REPARSE_POINT, 14020
	SXS_DUPLICATE_DLL_NAME, 14021
	SXS_DUPLICATE_WINDOWCLASS_NAME, 14022
	SXS_DUPLICATE_CLSID, 14023
	SXS_DUPLICATE_IID, 14024
	SXS_DUPLICATE_TLBID, 14025
	SXS_DUPLICATE_PROGID, 14026
	SXS_DUPLICATE_ASSEMBLY_NAME, 14027
	SXS_FILE_HASH_MISMATCH, 14028
	SXS_POLICY_PARSE_ERROR, 14029
	SXS_XML_E_MISSINGQUOTE, 14030
	SXS_XML_E_COMMENTSYNTAX, 14031
	SXS_XML_E_BADSTARTNAMECHAR, 14032
	SXS_XML_E_BADNAMECHAR, 14033
	SXS_XML_E_BADCHARINSTRING, 14034
	SXS_XML_E_XMLDECLSYNTAX, 14035
	SXS_XML_E_BADCHARDATA, 14036
	SXS_XML_E_MISSINGWHITESPACE, 14037
	SXS_XML_E_EXPECTINGTAGEND, 14038
	SXS_XML_E_MISSINGSEMICOLON, 14039
	SXS_XML_E_UNBALANCEDPAREN, 14040
	SXS_XML_E_INTERNALERROR, 14041
	SXS_XML_E_UNEXPECTED_WHITESPACE, 14042
	SXS_XML_E_INCOMPLETE_ENCODING, 14043
	SXS_XML_E_MISSING_PAREN, 14044
	SXS_XML_E_EXPECTINGCLOSEQUOTE, 14045
	SXS_XML_E_MULTIPLE_COLONS, 14046
	SXS_XML_E_INVALID_DECIMAL, 14047
	SXS_XML_E_INVALID_HEXIDECIMAL, 14048
	SXS_XML_E_INVALID_UNICODE, 14049
	SXS_XML_E_WHITESPACEORQUESTIONMARK, 14050
	SXS_XML_E_UNEXPECTEDENDTAG, 14051
	SXS_XML_E_UNCLOSEDTAG, 14052
	SXS_XML_E_DUPLICATEATTRIBUTE, 14053
	SXS_XML_E_MULTIPLEROOTS, 14054
	SXS_XML_E_INVALIDATROOTLEVEL, 14055
	SXS_XML_E_BADXMLDECL, 14056
	SXS_XML_E_MISSINGROOT, 14057
	SXS_XML_E_UNEXPECTEDEOF, 14058
	SXS_XML_E_BADPEREFINSUBSET, 14059
	SXS_XML_E_UNCLOSEDSTARTTAG, 14060
	SXS_XML_E_UNCLOSEDENDTAG, 14061
	SXS_XML_E_UNCLOSEDSTRING, 14062
	SXS_XML_E_UNCLOSEDCOMMENT, 14063
	SXS_XML_E_UNCLOSEDDECL, 14064
	SXS_XML_E_UNCLOSEDCDATA, 14065
	SXS_XML_E_RESERVEDNAMESPACE, 14066
	SXS_XML_E_INVALIDENCODING, 14067
	SXS_XML_E_INVALIDSWITCH, 14068
	SXS_XML_E_BADXMLCASE, 14069
	SXS_XML_E_INVALID_STANDALONE, 14070
	SXS_XML_E_UNEXPECTED_STANDALONE, 14071
	SXS_XML_E_INVALID_VERSION, 14072
	SXS_XML_E_MISSINGEQUALS, 14073
	SXS_PROTECTION_RECOVERY_FAILED, 14074
	SXS_PROTECTION_PUBLIC_KEY_TOO_SHORT, 14075
	SXS_PROTECTION_CATALOG_NOT_VALID, 14076
	SXS_UNTRANSLATABLE_HRESULT, 14077
	SXS_PROTECTION_CATALOG_FILE_MISSING, 14078
	SXS_MISSING_ASSEMBLY_IDENTITY_ATTRIBUTE, 14079
	SXS_INVALID_ASSEMBLY_IDENTITY_ATTRIBUTE_NAME, 14080
	SXS_ASSEMBLY_MISSING, 14081
	SXS_CORRUPT_ACTIVATION_STACK, 14082
	SXS_CORRUPTION, 14083
	SXS_EARLY_DEACTIVATION, 14084
	SXS_INVALID_DEACTIVATION, 14085
	SXS_MULTIPLE_DEACTIVATION, 14086
	SXS_PROCESS_TERMINATION_REQUESTED, 14087
	SXS_RELEASE_ACTIVATION_CONTEXT, 14088
	SXS_SYSTEM_DEFAULT_ACTIVATION_CONTEXT_EMPTY, 14089
	SXS_INVALID_IDENTITY_ATTRIBUTE_VALUE, 14090
	SXS_INVALID_IDENTITY_ATTRIBUTE_NAME, 14091
	SXS_IDENTITY_DUPLICATE_ATTRIBUTE, 14092
	SXS_IDENTITY_PARSE_ERROR, 14093
	MALFORMED_SUBSTITUTION_STRING, 14094
	SXS_INCORRECT_PUBLIC_KEY_TOKEN, 14095
	UNMAPPED_SUBSTITUTION_STRING, 14096
	SXS_ASSEMBLY_NOT_LOCKED, 14097
	SXS_COMPONENT_STORE_CORRUPT, 14098
	ADVANCED_INSTALLER_FAILED, 14099
	XML_ENCODING_MISMATCH, 14100
	SXS_MANIFEST_IDENTITY_SAME_BUT_CONTENTS_DIFFERENT, 14101
	SXS_IDENTITIES_DIFFERENT, 14102
	SXS_ASSEMBLY_IS_NOT_A_DEPLOYMENT, 14103
	SXS_FILE_NOT_PART_OF_ASSEMBLY, 14104
	SXS_MANIFEST_TOO_BIG, 14105
	SXS_SETTING_NOT_REGISTERED, 14106
	SXS_TRANSACTION_CLOSURE_INCOMPLETE, 14107
	SMI_PRIMITIVE_INSTALLER_FAILED, 14108
	GENERIC_COMMAND_FAILED, 14109
	SXS_FILE_HASH_MISSING, 14110
	SXS_DUPLICATE_ACTIVATABLE_CLASS, 14111
	EVT_INVALID_CHANNEL_PATH, 15000
	EVT_INVALID_QUERY, 15001
	EVT_PUBLISHER_METADATA_NOT_FOUND, 15002
	EVT_EVENT_TEMPLATE_NOT_FOUND, 15003
	EVT_INVALID_PUBLISHER_NAME, 15004
	EVT_INVALID_EVENT_DATA, 15005
	EVT_CHANNEL_NOT_FOUND, 15007
	EVT_MALFORMED_XML_TEXT, 15008
	EVT_SUBSCRIPTION_TO_DIRECT_CHANNEL, 15009
	EVT_CONFIGURATION_ERROR, 15010
	EVT_QUERY_RESULT_STALE, 15011
	EVT_QUERY_RESULT_INVALID_POSITION, 15012
	EVT_NON_VALIDATING_MSXML, 15013
	EVT_FILTER_ALREADYSCOPED, 15014
	EVT_FILTER_NOTELTSET, 15015
	EVT_FILTER_INVARG, 15016
	EVT_FILTER_INVTEST, 15017
	EVT_FILTER_INVTYPE, 15018
	EVT_FILTER_PARSEERR, 15019
	EVT_FILTER_UNSUPPORTEDOP, 15020
	EVT_FILTER_UNEXPECTEDTOKEN, 15021
	EVT_INVALID_OPERATION_OVER_ENABLED_DIRECT_CHANNEL, 15022
	EVT_INVALID_CHANNEL_PROPERTY_VALUE, 15023
	EVT_INVALID_PUBLISHER_PROPERTY_VALUE, 15024
	EVT_CHANNEL_CANNOT_ACTIVATE, 15025
	EVT_FILTER_TOO_COMPLEX, 15026
	EVT_MESSAGE_NOT_FOUND, 15027
	EVT_MESSAGE_ID_NOT_FOUND, 15028
	EVT_UNRESOLVED_VALUE_INSERT, 15029
	EVT_UNRESOLVED_PARAMETER_INSERT, 15030
	EVT_MAX_INSERTS_REACHED, 15031
	EVT_EVENT_DEFINITION_NOT_FOUND, 15032
	EVT_MESSAGE_LOCALE_NOT_FOUND, 15033
	EVT_VERSION_TOO_OLD, 15034
	EVT_VERSION_TOO_NEW, 15035
	EVT_CANNOT_OPEN_CHANNEL_OF_QUERY, 15036
	EVT_PUBLISHER_DISABLED, 15037
	EVT_FILTER_OUT_OF_RANGE, 15038
	EC_SUBSCRIPTION_CANNOT_ACTIVATE, 15080
	EC_LOG_DISABLED, 15081
	EC_CIRCULAR_FORWARDING, 15082
	EC_CREDSTORE_FULL, 15083
	EC_CRED_NOT_FOUND, 15084
	EC_NO_ACTIVE_CHANNEL, 15085
	MUI_FILE_NOT_FOUND, 15100
	MUI_INVALID_FILE, 15101
	MUI_INVALID_RC_CONFIG, 15102
	MUI_INVALID_LOCALE_NAME, 15103
	MUI_INVALID_ULTIMATEFALLBACK_NAME, 15104
	MUI_FILE_NOT_LOADED, 15105
	RESOURCE_ENUM_USER_STOP, 15106
	MUI_INTLSETTINGS_UILANG_NOT_INSTALLED, 15107
	MUI_INTLSETTINGS_INVALID_LOCALE_NAME, 15108
	MRM_RUNTIME_NO_DEFAULT_OR_NEUTRAL_RESOURCE, 15110
	MRM_INVALID_PRICONFIG, 15111
	MRM_INVALID_FILE_TYPE, 15112
	MRM_UNKNOWN_QUALIFIER, 15113
	MRM_INVALID_QUALIFIER_VALUE, 15114
	MRM_NO_CANDIDATE, 15115
	MRM_NO_MATCH_OR_DEFAULT_CANDIDATE, 15116
	MRM_RESOURCE_TYPE_MISMATCH, 15117
	MRM_DUPLICATE_MAP_NAME, 15118
	MRM_DUPLICATE_ENTRY, 15119
	MRM_INVALID_RESOURCE_IDENTIFIER, 15120
	MRM_FILEPATH_TOO_LONG, 15121
	MRM_UNSUPPORTED_DIRECTORY_TYPE, 15122
	MRM_INVALID_PRI_FILE, 15126
	MRM_NAMED_RESOURCE_NOT_FOUND, 15127
	MRM_MAP_NOT_FOUND, 15135
	MRM_UNSUPPORTED_PROFILE_TYPE, 15136
	MRM_INVALID_QUALIFIER_OPERATOR, 15137
	MRM_INDETERMINATE_QUALIFIER_VALUE, 15138
	MRM_AUTOMERGE_ENABLED, 15139
	MRM_TOO_MANY_RESOURCES, 15140
	MRM_UNSUPPORTED_FILE_TYPE_FOR_MERGE, 15141
	MRM_UNSUPPORTED_FILE_TYPE_FOR_LOAD_UNLOAD_PRI_FILE, 15142
	MRM_NO_CURRENT_VIEW_ON_THREAD, 15143
	DIFFERENT_PROFILE_RESOURCE_MANAGER_EXIST, 15144
	OPERATION_NOT_ALLOWED_FROM_SYSTEM_COMPONENT, 15145
	MRM_DIRECT_REF_TO_NON_DEFAULT_RESOURCE, 15146
	MRM_GENERATION_COUNT_MISMATCH, 15147
	PRI_MERGE_VERSION_MISMATCH, 15148
	PRI_MERGE_MISSING_SCHEMA, 15149
	PRI_MERGE_LOAD_FILE_FAILED, 15150
	PRI_MERGE_ADD_FILE_FAILED, 15151
	PRI_MERGE_WRITE_FILE_FAILED, 15152
	PRI_MERGE_MULTIPLE_PACKAGE_FAMILIES_NOT_ALLOWED, 15153
	PRI_MERGE_MULTIPLE_MAIN_PACKAGES_NOT_ALLOWED, 15154
	PRI_MERGE_BUNDLE_PACKAGES_NOT_ALLOWED, 15155
	PRI_MERGE_MAIN_PACKAGE_REQUIRED, 15156
	PRI_MERGE_RESOURCE_PACKAGE_REQUIRED, 15157
	PRI_MERGE_INVALID_FILE_NAME, 15158
	MRM_PACKAGE_NOT_FOUND, 15159
	MRM_MISSING_DEFAULT_LANGUAGE, 15160
	MCA_INVALID_CAPABILITIES_STRING, 15200
	MCA_INVALID_VCP_VERSION, 15201
	MCA_MONITOR_VIOLATES_MCCS_SPECIFICATION, 15202
	MCA_MCCS_VERSION_MISMATCH, 15203
	MCA_UNSUPPORTED_MCCS_VERSION, 15204
	MCA_INTERNAL_ERROR, 15205
	MCA_INVALID_TECHNOLOGY_TYPE_RETURNED, 15206
	MCA_UNSUPPORTED_COLOR_TEMPERATURE, 15207
	AMBIGUOUS_SYSTEM_DEVICE, 15250
	SYSTEM_DEVICE_NOT_FOUND, 15299
	HASH_NOT_SUPPORTED, 15300
	HASH_NOT_PRESENT, 15301
	SECONDARY_IC_PROVIDER_NOT_REGISTERED, 15321
	GPIO_CLIENT_INFORMATION_INVALID, 15322
	GPIO_VERSION_NOT_SUPPORTED, 15323
	GPIO_INVALID_REGISTRATION_PACKET, 15324
	GPIO_OPERATION_DENIED, 15325
	GPIO_INCOMPATIBLE_CONNECT_MODE, 15326
	GPIO_INTERRUPT_ALREADY_UNMASKED, 15327
	CANNOT_SWITCH_RUNLEVEL, 15400
	INVALID_RUNLEVEL_SETTING, 15401
	RUNLEVEL_SWITCH_TIMEOUT, 15402
	RUNLEVEL_SWITCH_AGENT_TIMEOUT, 15403
	RUNLEVEL_SWITCH_IN_PROGRESS, 15404
	SERVICES_FAILED_AUTOSTART, 15405
	COM_TASK_STOP_PENDING, 15501
	INSTALL_OPEN_PACKAGE_FAILED, 15600
	INSTALL_PACKAGE_NOT_FOUND, 15601
	INSTALL_INVALID_PACKAGE, 15602
	INSTALL_RESOLVE_DEPENDENCY_FAILED, 15603
	INSTALL_OUT_OF_DISK_SPACE, 15604
	INSTALL_NETWORK_FAILURE, 15605
	INSTALL_REGISTRATION_FAILURE, 15606
	INSTALL_DEREGISTRATION_FAILURE, 15607
	INSTALL_CANCEL, 15608
	INSTALL_FAILED, 15609
	REMOVE_FAILED, 15610
	PACKAGE_ALREADY_EXISTS, 15611
	NEEDS_REMEDIATION, 15612
	INSTALL_PREREQUISITE_FAILED, 15613
	PACKAGE_REPOSITORY_CORRUPTED, 15614
	INSTALL_POLICY_FAILURE, 15615
	PACKAGE_UPDATING, 15616
	DEPLOYMENT_BLOCKED_BY_POLICY, 15617
	PACKAGES_IN_USE, 15618
	RECOVERY_FILE_CORRUPT, 15619
	INVALID_STAGED_SIGNATURE, 15620
	DELETING_EXISTING_APPLICATIONDATA_STORE_FAILED, 15621
	INSTALL_PACKAGE_DOWNGRADE, 15622
	SYSTEM_NEEDS_REMEDIATION, 15623
	APPX_INTEGRITY_FAILURE_CLR_NGEN, 15624
	RESILIENCY_FILE_CORRUPT, 15625
	INSTALL_FIREWALL_SERVICE_NOT_RUNNING, 15626
	PACKAGE_MOVE_FAILED, 15627
	INSTALL_VOLUME_NOT_EMPTY, 15628
	INSTALL_VOLUME_OFFLINE, 15629
	INSTALL_VOLUME_CORRUPT, 15630
	NEEDS_REGISTRATION, 15631
	INSTALL_WRONG_PROCESSOR_ARCHITECTURE, 15632
	DEV_SIDELOAD_LIMIT_EXCEEDED, 15633
	INSTALL_OPTIONAL_PACKAGE_REQUIRES_MAIN_PACKAGE, 15634
	PACKAGE_NOT_SUPPORTED_ON_FILESYSTEM, 15635
	PACKAGE_MOVE_BLOCKED_BY_STREAMING, 15636
	INSTALL_OPTIONAL_PACKAGE_APPLICATIONID_NOT_UNIQUE, 15637
	PACKAGE_STAGING_ONHOLD, 15638
	INSTALL_INVALID_RELATED_SET_UPDATE, 15639
	INSTALL_OPTIONAL_PACKAGE_REQUIRES_MAIN_PACKAGE_FULLTRUST_CAPABILITY, 15640
	DEPLOYMENT_BLOCKED_BY_USER_LOG_OFF, 15641
	PROVISION_OPTIONAL_PACKAGE_REQUIRES_MAIN_PACKAGE_PROVISIONED, 15642
	PACKAGES_REPUTATION_CHECK_FAILED, 15643
	PACKAGES_REPUTATION_CHECK_TIMEDOUT, 15644
	DEPLOYMENT_OPTION_NOT_SUPPORTED, 15645
	APPINSTALLER_ACTIVATION_BLOCKED, 15646
	REGISTRATION_FROM_REMOTE_DRIVE_NOT_SUPPORTED, 15647
	APPX_RAW_DATA_WRITE_FAILED, 15648
	DEPLOYMENT_BLOCKED_BY_VOLUME_POLICY_PACKAGE, 15649
	DEPLOYMENT_BLOCKED_BY_VOLUME_POLICY_MACHINE, 15650
	DEPLOYMENT_BLOCKED_BY_PROFILE_POLICY, 15651
	DEPLOYMENT_FAILED_CONFLICTING_MUTABLE_PACKAGE_DIRECTORY, 15652
	SINGLETON_RESOURCE_INSTALLED_IN_ACTIVE_USER, 15653
	DIFFERENT_VERSION_OF_PACKAGED_SERVICE_INSTALLED, 15654
	SERVICE_EXISTS_AS_NON_PACKAGED_SERVICE, 15655
	PACKAGED_SERVICE_REQUIRES_ADMIN_PRIVILEGES, 15656
	STATE_LOAD_STORE_FAILED, 15800
	STATE_GET_VERSION_FAILED, 15801
	STATE_SET_VERSION_FAILED, 15802
	STATE_STRUCTURED_RESET_FAILED, 15803
	STATE_OPEN_CONTAINER_FAILED, 15804
	STATE_CREATE_CONTAINER_FAILED, 15805
	STATE_DELETE_CONTAINER_FAILED, 15806
	STATE_READ_SETTING_FAILED, 15807
	STATE_WRITE_SETTING_FAILED, 15808
	STATE_DELETE_SETTING_FAILED, 15809
	STATE_QUERY_SETTING_FAILED, 15810
	STATE_READ_COMPOSITE_SETTING_FAILED, 15811
	STATE_WRITE_COMPOSITE_SETTING_FAILED, 15812
	STATE_ENUMERATE_CONTAINER_FAILED, 15813
	STATE_ENUMERATE_SETTINGS_FAILED, 15814
	STATE_COMPOSITE_SETTING_VALUE_SIZE_LIMIT_EXCEEDED, 15815
	STATE_SETTING_VALUE_SIZE_LIMIT_EXCEEDED, 15816
	STATE_SETTING_NAME_SIZE_LIMIT_EXCEEDED, 15817
	STATE_CONTAINER_NAME_SIZE_LIMIT_EXCEEDED, 15818
	API_UNAVAILABLE, 15841

	/// Operation successful.
	S_OK, 0
	S_FALSE, 1

	E_UNEXPECTED, 0x8000_ffff
	E_NOTIMPL, 0x8000_4001
	E_OUTOFMEMORY, 0x8007_000e
	E_INVALIDARG, 0x8007_0057
	E_NOINTERFACE, 0x8000_4002
	E_POINTER, 0x8000_4003
	E_HANDLE, 0x8007_0006
	E_ABORT, 0x8000_4004
	E_FAIL, 0x8000_4005
	E_ACCESSDENIED, 0x8007_0005
	E_PENDING, 0x8000_000a
	E_BOUNDS, 0x8000_000b
	E_CHANGED_STATE, 0x8000_000c
	E_ILLEGAL_STATE_CHANGE, 0x8000_000d
	E_ILLEGAL_METHOD_CALL, 0x8000_000e

	RPC_E_CALL_REJECTED, 0x8001_0001
	RPC_E_CALL_CANCELED, 0x8001_0002
	RPC_E_CANTPOST_INSENDCALL, 0x8001_0003
	RPC_E_CANTCALLOUT_INASYNCCALL, 0x8001_0004
	RPC_E_CANTCALLOUT_INEXTERNALCALL, 0x8001_0005
	RPC_E_CONNECTION_TERMINATED, 0x8001_0006
	RPC_E_SERVER_DIED, 0x8001_0007
	RPC_E_CLIENT_DIED, 0x8001_0008
	RPC_E_INVALID_DATAPACKET, 0x8001_0009
	RPC_E_CANTTRANSMIT_CALL, 0x8001_000a
	RPC_E_CLIENT_CANTMARSHAL_DATA, 0x8001_000b
	RPC_E_CLIENT_CANTUNMARSHAL_DATA, 0x8001_000c
	RPC_E_SERVER_CANTMARSHAL_DATA, 0x8001_000d
	RPC_E_SERVER_CANTUNMARSHAL_DATA, 0x8001_000e
	RPC_E_INVALID_DATA, 0x8001_000f
	RPC_E_INVALID_PARAMETER, 0x8001_0010
	RPC_E_CANTCALLOUT_AGAIN, 0x8001_0011
	RPC_E_SERVER_DIED_DNE, 0x8001_0012
	RPC_E_SYS_CALL_FAILED, 0x8001_0100
	RPC_E_OUT_OF_RESOURCES, 0x8001_0101
	RPC_E_ATTEMPTED_MULTITHREAD, 0x8001_0102
	RPC_E_NOT_REGISTERED, 0x8001_0103
	RPC_E_FAULT, 0x8001_0104
	RPC_E_SERVERFAULT, 0x8001_0105
	RPC_E_CHANGED_MODE, 0x8001_0106
	RPC_E_INVALIDMETHOD, 0x8001_0107
	RPC_E_DISCONNECTED, 0x8001_0108
	RPC_E_RETRY, 0x8001_0109
	RPC_E_SERVERCALL_RETRYLATER, 0x8001_010a
	RPC_E_SERVERCALL_REJECTED, 0x8001_010b
	RPC_E_INVALID_CALLDATA, 0x8001_010c
	RPC_E_CANTCALLOUT_ININPUTSYNCCALL, 0x8001_010d
	RPC_E_WRONG_THREAD, 0x8001_010e
	RPC_E_THREAD_NOT_INIT, 0x8001_010f
	RPC_E_VERSION_MISMATCH, 0x8001_0110
	RPC_E_INVALID_HEADER, 0x8001_0111
	RPC_E_INVALID_EXTENSION, 0x8001_0112
	RPC_E_INVALID_IPID, 0x8001_0113
	RPC_E_INVALID_OBJECT, 0x8001_0114
	RPC_E_CALL_COMPLETE, 0x8001_0117
	RPC_E_UNSECURE_CALL, 0x8001_0118
	RPC_E_TOO_LATE, 0x8001_0119
	RPC_E_NO_GOOD_SECURITY_PACKAGES, 0x8001_011a
	RPC_E_ACCESS_DENIED, 0x8001_011b
	RPC_E_REMOTE_DISABLED, 0x8001_011c
	RPC_E_INVALID_OBJREF, 0x8001_011d
	RPC_E_NO_CONTEXT, 0x8001_011e
	RPC_E_TIMEOUT, 0x8001_011f
	RPC_E_NO_SYNC, 0x8001_0120
	RPC_E_FULLSIC_REQUIRED, 0x8001_0121
	RPC_E_INVALID_STD_NAME, 0x8001_0122
	RPC_E_UNEXPECTED, 0x8001_ffff

	CO_E_NOTINITIALIZED, 0x8004_01f0
	CO_E_ALREADYINITIALIZED, 0x8004_01f1
	CO_E_CANTDETERMINECLASS, 0x8004_01f2
	CO_E_CLASSSTRING, 0x8004_01f3
	CO_E_IIDSTRING, 0x8004_01f4
	CO_E_APPNOTFOUND, 0x8004_01f5
	CO_E_APPSINGLEUSE, 0x8004_01f6
	CO_E_ERRORINAPP, 0x8004_01f7
	CO_E_DLLNOTFOUND, 0x8004_01f8
	CO_E_ERRORINDLL, 0x8004_01f9
	CO_E_WRONGOSFORAPP, 0x8004_01fa
	CO_E_OBJNOTREG, 0x8004_01fb
	CO_E_OBJISREG, 0x8004_01fc
	CO_E_OBJNOTCONNECTED, 0x8004_01fd
	CO_E_APPDIDNTREG, 0x8004_01fe
	CO_E_RELEASED, 0x8004_01ff

	MF_E_PLATFORM_NOT_INITIALIZED, 0xc00d_36b0
	MF_E_BUFFERTOOSMALL, 0xc00d_36b1
	MF_E_INVALIDREQUEST, 0xc00d_36b2
	MF_E_INVALIDSTREAMNUMBER, 0xc00d_36b3
	MF_E_INVALIDMEDIATYPE, 0xc00d_36b4
	MF_E_NOTACCEPTING, 0xc00d_36b5
	MF_E_NOT_INITIALIZED, 0xc00d_36b6
	MF_E_UNSUPPORTED_REPRESENTATION, 0xc00d_36b7
	MF_E_NO_MORE_TYPES, 0xc00d_36b9
	MF_E_UNSUPPORTED_SERVICE, 0xc00d_36ba
	MF_E_UNEXPECTED, 0xc00d_36bb
	MF_E_INVALIDNAME, 0xc00d_36bc
	MF_E_INVALIDTYPE, 0xc00d_36bd
	MF_E_INVALID_FILE_FORMAT, 0xc00d_36be
	MF_E_INVALIDINDEX, 0xc00d_36bf
	MF_E_INVALID_TIMESTAMP, 0xc00d_36c0
	MF_E_UNSUPPORTED_SCHEME, 0xc00d_36c3
	MF_E_UNSUPPORTED_BYTESTREAM_TYPE, 0xc00d_36c4
	MF_E_UNSUPPORTED_TIME_FORMAT, 0xc00d_36c5
	MF_E_NO_SAMPLE_TIMESTAMP, 0xc00d_36c8
	MF_E_NO_SAMPLE_DURATION, 0xc00d_36c9
	MF_E_INVALID_STREAM_DATA, 0xc00d_36cb
	MF_E_RT_UNAVAILABLE, 0xc00d_36cf
	MF_E_UNSUPPORTED_RATE, 0xc00d_36d0
	MF_E_THINNING_UNSUPPORTED, 0xc00d_36d1
	MF_E_REVERSE_UNSUPPORTED, 0xc00d_36d2
	MF_E_UNSUPPORTED_RATE_TRANSITION, 0xc00d_36d3
	MF_E_RATE_CHANGE_PREEMPTED, 0xc00d_36d4
	MF_E_NOT_FOUND, 0xc00d_36d5
	MF_E_NOT_AVAILABLE, 0xc00d_36d6
	MF_E_NO_CLOCK, 0xc00d_36d7
	MF_S_MULTIPLE_BEGIN, 0x000d_36d8
	MF_E_MULTIPLE_BEGIN, 0xc00d_36d9
	MF_E_MULTIPLE_SUBSCRIBERS, 0xc00d_36da
	MF_E_TIMER_ORPHANED, 0xc00d_36db
	MF_E_STATE_TRANSITION_PENDING, 0xc00d_36dc
	MF_E_UNSUPPORTED_STATE_TRANSITION, 0xc00d_36dd
	MF_E_UNRECOVERABLE_ERROR_OCCURRED, 0xc00d_36de
	MF_E_SAMPLE_HAS_TOO_MANY_BUFFERS, 0xc00d_36df
	MF_E_SAMPLE_NOT_WRITABLE, 0xc00d_36e0
	MF_E_INVALID_KEY, 0xc00d_36e2
	MF_E_BAD_STARTUP_VERSION, 0xc00d_36e3
	MF_E_UNSUPPORTED_CAPTION, 0xc00d_36e4
	MF_E_INVALID_POSITION, 0xc00d_36e5
	MF_E_ATTRIBUTENOTFOUND, 0xc00d_36e6
	MF_E_PROPERTY_TYPE_NOT_ALLOWED, 0xc00d_36e7
	MF_E_TOPO_INVALID_OPTIONAL_NODE, 0xc00d_520e
	MF_E_TOPO_CANNOT_FIND_DECRYPTOR, 0xc00d_5211
	MF_E_TOPO_CODEC_NOT_FOUND, 0xc00d_5212
	MF_E_TOPO_CANNOT_CONNECT, 0xc00d_5213
	MF_E_TOPO_UNSUPPORTED, 0xc00d_5214
	MF_E_TOPO_INVALID_TIME_ATTRIBUTES, 0xc00d_5215
	MF_E_TOPO_LOOPS_IN_TOPOLOGY, 0xC00d_5216
	MF_E_TOPO_MISSING_PRESENTATION_DESCRIPTOR, 0xc00d_5217
	MF_E_TOPO_MISSING_STREAM_DESCRIPTOR, 0xc00d_5218
	MF_E_TOPO_STREAM_DESCRIPTOR_NOT_SELECTED, 0xc00d_5219
	MF_E_TOPO_MISSING_SOURCE, 0xc00d_521a
	MF_E_TOPO_SINK_ACTIVATES_UNSUPPORTED, 0xc00d_521b
	MF_E_TRANSFORM_TYPE_NOT_SET, 0xc00d_6d60
}

impl std::error::Error for ERROR {
	fn source(&self) -> Option<&(dyn std::error::Error + 'static)> {
		None
	}
}

impl std::fmt::Debug for ERROR {
	fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
		if self.0 > 0xffff {
			write!(f, "[{:#010x} {}] {}",
				self.0, self.0, self.FormatMessage())
		} else {
			write!(f, "[{:#06x} {}] {}",
				self.0, self.0, self.FormatMessage())
		}
	}
}

impl std::fmt::Display for ERROR {
	fn fmt(&self, f: &mut std::fmt::Formatter) -> std::fmt::Result {
		<Self as std::fmt::Debug>::fmt(self, f) // delegate to Debug trait
	}
}

impl ERROR {
	/// Returns the textual description of the system error, by calling
	/// [`FormatMessage`](https://docs.microsoft.com/en-us/windows/win32/api/winbase/nf-winbase-formatmessagew)
	/// function.
	pub fn FormatMessage(self) -> String {
		unsafe {
			let mut ptr_buf: *mut u16 = std::ptr::null_mut();
			match kernel32::FormatMessageW(
				co::FORMAT_MESSAGE::ALLOCATE_BUFFER.0
					| co::FORMAT_MESSAGE::FROM_SYSTEM.0
					| co::FORMAT_MESSAGE::IGNORE_INSERTS.0,
				std::ptr::null(),
				self.0,
				LANGID::new(co::LANG::NEUTRAL, co::SUBLANG::DEFAULT).0 as _,
				(&mut ptr_buf as *mut *mut u16) as _, // pass pointer to pointer
				0,
				std::ptr::null_mut(),
			) {
				0 => format!( // never fails, returns a message instead
					"FormatMessage failed to format error {:#06x}: error {:#06x}.",
					self.0, GetLastError(),
				),
				nchars => {
					let final_str = WString::from_wchars_count(ptr_buf, nchars as _);
					match (HLOCAL { ptr: ptr_buf as _ }).LocalFree() {
						Ok(()) => final_str.to_string(),
						Err(err) => format!(
							"LocalFree failed after formatting error {:#06x}: error {:#06x}.",
							self.0, err.0),
					}
				},
			}
		}
	}
}
