#[path = "errors.rs"]
mod errors;
use errors::{Error};
use winapi::um::tlhelp32::{TH32CS_SNAPPROCESS, TH32CS_SNAPMODULE, TH32CS_SNAPMODULE32, MODULEENTRY32W, PROCESSENTRY32W, CreateToolhelp32Snapshot, Process32FirstW, Process32NextW};
use winapi::um::handleapi::INVALID_HANDLE_VALUE;
use winapi::shared::minwindef::{HINSTANCE__, MAX_PATH};
use std::mem;
use std::os::windows::ffi::OsStringExt;
use winapi::um::tlhelp32::Module32NextW;

/// Writes a value to memory, internally.
pub fn write(addr:u64, values:&[u8]) -> Result<(), Error> {
    let result = unsafe{
        winapi::um::memoryapi::WriteProcessMemory(
            winapi::um::processthreadsapi::GetCurrentProcess(), 
            (winapi::um::libloaderapi::GetModuleHandleA(std::ptr::null()) as u64 + addr) as winapi::shared::minwindef::LPVOID,  
            values.as_ptr() as winapi::shared::minwindef::LPCVOID, 
            values.len() as winapi::shared::basetsd::SIZE_T, 
            std::ptr::null_mut()
        )
    };
    match result {
        1 => {
            Ok(())
        }
        0 => {
            Err(Error::MemoryWrite(addr, unsafe { winapi::um::errhandlingapi::GetLastError() }))
        }
        _ => {
            Ok(())
        }
    }
}

/// Reads a value from memory, internally.
pub fn read(addr:u64, size: usize) -> Result<Vec<u8>, Error> {
    let copy:&mut [u8] = &mut vec![0; size];
    let result = unsafe {
        winapi::um::memoryapi::ReadProcessMemory(
            winapi::um::processthreadsapi::GetCurrentProcess(), 
            (winapi::um::libloaderapi::GetModuleHandleA(std::ptr::null()) as u64 + addr) as winapi::shared::minwindef::LPVOID, 
            copy.as_mut_ptr() as winapi::shared::minwindef::LPVOID,
            size as winapi::shared::basetsd::SIZE_T,
            std::ptr::null_mut()
        )
    };
    match result {
        1 => {
            Ok(copy.to_vec())
        }
        0 => {
            Err(Error::MemoryRead(addr, unsafe { winapi::um::errhandlingapi::GetLastError() }))
        }
        _ => {
            Ok(copy.to_vec())
        }
    }
}

/// Writes to a procces memory address, externally.
pub fn write_process(pid:u32, addr:u64, values:&[u8]) -> Result<(), Error> {
    let handle = unsafe {
        winapi::um::processthreadsapi::OpenProcess(
            winapi::um::winnt::PROCESS_ALL_ACCESS, 
            0, 
            pid as winapi::shared::minwindef::DWORD
        )
    };
    let result = unsafe {
        winapi::um::memoryapi::WriteProcessMemory(
            handle, 
            addr as winapi::shared::minwindef::LPVOID,  
            values.as_ptr() as winapi::shared::minwindef::LPCVOID, 
            values.len() as winapi::shared::basetsd::SIZE_T, 
            std::ptr::null_mut()
        )
    };
    unsafe {winapi::um::handleapi::CloseHandle(handle);}
    match result {
        1 => {
            Ok(())
        }
        0 => {
            Err(Error::ProcessMemoryWrite(addr, unsafe{winapi::um::errhandlingapi::GetLastError()}))
        }
        _ => {
            Ok(())
        }
    }
}

/// Reads a process memory address, externally.
pub fn read_process(pid:u32, addr:u64, size: usize) -> Result<Vec<u8>, Error> {
    let copy:&mut [u8] = &mut vec![0; size];
    let handle = unsafe {
        winapi::um::processthreadsapi::OpenProcess(
            winapi::um::winnt::PROCESS_ALL_ACCESS, 
            0, 
            pid as winapi::shared::minwindef::DWORD
        )
    };
    let result = unsafe {
        winapi::um::memoryapi::ReadProcessMemory(
            handle,
            addr as winapi::shared::minwindef::LPVOID, 
            copy.as_mut_ptr() as winapi::shared::minwindef::LPVOID,
            size as winapi::shared::basetsd::SIZE_T,
            std::ptr::null_mut()
        )
    };
    unsafe {winapi::um::handleapi::CloseHandle(handle);}
    match result {
        1 => {
            Ok(copy.to_vec())
        }
        0 => {
            Err(Error::ProcessMemoryRead(addr, unsafe{winapi::um::errhandlingapi::GetLastError()}))
        }
        _ => {
            Ok(copy.to_vec())
        }
    }
}


/// Writes a value to memory, internally.
pub fn write32(addr:u32, values:&[u8]) -> Result<(), Error> {
    let result = unsafe{
        winapi::um::memoryapi::WriteProcessMemory(
            winapi::um::processthreadsapi::GetCurrentProcess(), 
            (winapi::um::libloaderapi::GetModuleHandleA(std::ptr::null()) as u32 + addr) as winapi::shared::minwindef::LPVOID,  
            values.as_ptr() as winapi::shared::minwindef::LPCVOID, 
            values.len() as winapi::shared::basetsd::SIZE_T, 
            std::ptr::null_mut()
        )
    };
    match result {
        1 => {
            Ok(())
        }
        0 => {
            Err(Error::MemoryWrite32(addr, unsafe { winapi::um::errhandlingapi::GetLastError() }))
        }
        _ => {
            Ok(())
        }
    }
}

/// Reads a value from memory, internally.
pub fn read32(addr:u32, size: usize) -> Result<Vec<u8>, Error> {
    let copy:&mut [u8] = &mut vec![0; size];
    let result = unsafe {
        winapi::um::memoryapi::ReadProcessMemory(
            winapi::um::processthreadsapi::GetCurrentProcess(), 
            (winapi::um::libloaderapi::GetModuleHandleA(std::ptr::null()) as u32 + addr) as winapi::shared::minwindef::LPVOID, 
            copy.as_mut_ptr() as winapi::shared::minwindef::LPVOID,
            size as winapi::shared::basetsd::SIZE_T,
            std::ptr::null_mut()
        )
    };
    match result {
        1 => {
            Ok(copy.to_vec())
        }
        0 => {
            Err(Error::MemoryRead32(addr, unsafe { winapi::um::errhandlingapi::GetLastError() }))
        }
        _ => {
            Ok(copy.to_vec())
        }
    }
}

/// Writes to a procces memory address, externally.
pub fn write_process32(pid:u32, addr:u32, values:&[u8]) -> Result<(), Error> {
    let handle = unsafe {
        winapi::um::processthreadsapi::OpenProcess(
            winapi::um::winnt::PROCESS_ALL_ACCESS, 
            0, 
            pid as winapi::shared::minwindef::DWORD
        )
    };
    let result = unsafe {
        winapi::um::memoryapi::WriteProcessMemory(
            handle, 
            addr as winapi::shared::minwindef::LPVOID,  
            values.as_ptr() as winapi::shared::minwindef::LPCVOID, 
            values.len() as winapi::shared::basetsd::SIZE_T, 
            std::ptr::null_mut()
        )
    };
    unsafe {winapi::um::handleapi::CloseHandle(handle);}
    match result {
        1 => {
            Ok(())
        }
        0 => {
            Err(Error::ProcessMemoryWrite32(addr, unsafe{winapi::um::errhandlingapi::GetLastError()}))
        }
        _ => {
            Ok(())
        }
    }
}

/// Reads a process memory address, externally.
pub fn read_process32(pid:u32, addr:u32, size: usize) -> Result<Vec<u8>, Error> {
    let copy:&mut [u8] = &mut vec![0; size];
    let handle = unsafe {
        winapi::um::processthreadsapi::OpenProcess(
            winapi::um::winnt::PROCESS_ALL_ACCESS, 
            0, 
            pid as winapi::shared::minwindef::DWORD
        )
    };
    let result = unsafe {
        winapi::um::memoryapi::ReadProcessMemory(
            handle,
            addr as winapi::shared::minwindef::LPVOID, 
            copy.as_mut_ptr() as winapi::shared::minwindef::LPVOID,
            size as winapi::shared::basetsd::SIZE_T,
            std::ptr::null_mut()
        )
    };
    unsafe {winapi::um::handleapi::CloseHandle(handle);}
    match result {
        1 => {
            Ok(copy.to_vec())
        }
        0 => {
            Err(Error::ProcessMemoryRead32(addr, unsafe{winapi::um::errhandlingapi::GetLastError()}))
        }
        _ => {
            Ok(copy.to_vec())
        }
    }
}

/// Get the base address of the module with the given name.
pub fn get_module_base(pid:u32, name:&str) -> Result<u64, Error> {
    let snapshot_handle = unsafe { CreateToolhelp32Snapshot(TH32CS_SNAPMODULE | TH32CS_SNAPMODULE32, pid) };
    if snapshot_handle == INVALID_HANDLE_VALUE {
        return Err(Error::GetModuleBase { module: name.to_owned(), error: unsafe { winapi::um::errhandlingapi::GetLastError() } });
    }
    else {
        let mut module_entry : MODULEENTRY32W = MODULEENTRY32W {
            dwSize: mem::size_of::<MODULEENTRY32W>() as u32,
            th32ModuleID: 0,
            th32ProcessID: 0,
            GlblcntUsage: 0,
            ProccntUsage: 0,
            modBaseAddr: 0 as *mut u8,
            modBaseSize: 0,
            hModule: 0 as *mut HINSTANCE__,
            szModule: [0; 256],
            szExePath: [0; MAX_PATH],
        };
        let mut success : i32 = 1;
        while success == 1 {
            let module_name = std::ffi::OsString::from_wide(&module_entry.szModule);
            match module_name.into_string() {
                Ok(s) => {
                    if s.contains(name) {
                        return Ok(module_entry.modBaseAddr as u64);
                    }
                },
                Err(_) => {
                    return Err(Error::GetModuleBase { module: name.to_owned(), error: unsafe { winapi::um::errhandlingapi::GetLastError() } });
                }
            }
            success = unsafe { Module32NextW(snapshot_handle, &mut module_entry) };
        }
    }
    Err(Error::GetModuleBase { module: name.to_owned(), error: unsafe { winapi::um::errhandlingapi::GetLastError() } })
}

/// Get the process ID of the process with the given name.
/// Returns either an error or a 0 if it fails.
/// https://users.rust-lang.org/t/comparing-a-string-to-an-array-of-i8/5120
pub fn find_pid_by_name(name: &str) -> Result<u32, Error> {
    // Create a snapshot of the current processes
    let processes_snap_handle = unsafe { CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, 0) };

    if processes_snap_handle == INVALID_HANDLE_VALUE {
        return Err(Error::GetPid { name: name.to_owned(), error: unsafe { winapi::um::errhandlingapi::GetLastError() } });
    }

    // Initialise a process entry. In order to use `Process32First`, you need to set `dwSize`.
    let mut process_entry : PROCESSENTRY32W = PROCESSENTRY32W {
        dwSize: mem::size_of::<PROCESSENTRY32W>() as u32,
        cntUsage: 0,
        th32ProcessID: 0,
        th32DefaultHeapID: 0,
        th32ModuleID: 0,
        cntThreads: 0,
        th32ParentProcessID: 0,
        pcPriClassBase: 0,
        dwFlags: 0,
        szExeFile: [0; MAX_PATH],
    };

    // Get the first process from the snapshot.
    match unsafe { Process32FirstW(processes_snap_handle, &mut process_entry) } {
        1 => {
            // First process worked, loop to find the process with the correct name.
            let mut process_success : i32 = 1;

            // Loop through all processes until we find one hwere `szExeFile` == `name`.
            while process_success == 1 {
                // TODO: Compare `szExeFile` to `name`.
                let process_name = std::ffi::OsString::from_wide(&process_entry.szExeFile);

                match process_name.into_string() {
                    Ok(s) => {
                        if s.contains(name) {
                            // We found the process, return the PID.
                            return Ok(process_entry.th32ProcessID as u32);
                        }
                    },
                    Err(_) => {
                        return Err(Error::GetPid { name: name.to_owned(), error: unsafe { winapi::um::errhandlingapi::GetLastError() } });
                    }
                }

                process_success = unsafe { Process32NextW(processes_snap_handle, &mut process_entry) };
            }

            unsafe { winapi::um::handleapi::CloseHandle(processes_snap_handle) };
            Ok(0)
        },
        0|_ => {
            unsafe { winapi::um::handleapi::CloseHandle(processes_snap_handle) };
            Ok(0)
        }
    }
}