/// A struct representing a single search result. It's generated by the SearchBuilder and should
/// not be created directly. Instead, use the one SearchBuilder creates for you.
/// It's important to know that only some values are in every search result and the rest of them are
/// optional.
#[derive(Clone)]
pub struct SearchResult {
    /// The title of the search result, usually the title of the article it represents
    title: String,
    /// The namespace of the article the search result represents
    namespace: usize,
    /// The page id of the article the search result represents
    page_id: i32,
    /// The size of the article
    size: Option<i32>,

    /// The word count of the article
    wordcount: Option<i32>,
    /// The date and time the article was last modified
    timestamp: Option<String>,

    /// A small preview of the article
    snippet: Option<String>,
    /// A title snippet of the article
    title_snippet: Option<String>,
    /// The snippet for the category
    category_snippet: Option<String>,

    /// The title of the redirect
    redirect_title: Option<String>,
    /// The snippet of the redirect
    redirect_snippet: Option<String>,

    /// The title of the section
    section_title: Option<String>,
    /// The snippet of the section
    section_snippet: Option<String>,

    /// If it's a file match
    file_match: Option<bool>,
}

/// A helper macro for generating getter functions in the SearchResult struct
macro_rules! build_getter {
    ($(#[$meta :meta])* $value: ident, $return_type: ty) => {
        $(#[$meta])*
        pub fn $value(&self) -> Option<$return_type> {
            match &self.$value {
                Some(ref value) => Some(value),
                None => None,
            }
        }
    };
}

impl SearchResult {
    /// Creates a new struct with the given metadata. This function should not be used directly
    #[allow(clippy::too_many_arguments)]
    pub fn new(
        title: String,
        namespace: usize,
        page_id: i32,
        size: Option<i32>,
        wordcount: Option<i32>,
        timestamp: Option<String>,
        snippet: Option<String>,
        title_snippet: Option<String>,
        category_snippet: Option<String>,
        redirect_title: Option<String>,
        redirect_snippet: Option<String>,
        section_title: Option<String>,
        section_snippet: Option<String>,
        is_file_match: Option<bool>,
    ) -> Self {
        Self {
            title,
            namespace,
            page_id,
            size,
            wordcount,
            timestamp,
            snippet,
            title_snippet,
            category_snippet,
            redirect_title,
            redirect_snippet,
            section_title,
            section_snippet,
            file_match: is_file_match,
        }
    }

    /// The title of the search result, usually the title of the article it represents
    pub fn title(&self) -> &str {
        &self.title
    }

    /// The namespace of the article the search result represents
    pub fn namespace(&self) -> &usize {
        &self.namespace
    }

    /// The page id of the article the search result represents
    pub fn page_id(&self) -> &i32 {
        &self.page_id
    }

    build_getter!(
        /// The size of the article
        size,
        &i32
    );
    build_getter!(
        /// The word count of the article
        wordcount,
        &i32
    );
    build_getter!(
        /// The date and time the article was last modified
        timestamp,
        &str
    );

    build_getter!(
        /// A small preview of the article
        snippet,
        &str
    );
    build_getter!(
        /// A title snippet for the article
        title_snippet,
        &str
    );
    build_getter!(
        /// The snippet for the category
        category_snippet,
        &str
    );

    build_getter!(
        /// The title of the redirect
        redirect_title,
        &str
    );
    build_getter!(
        /// The snippet of the redirect
        redirect_snippet,
        &str
    );

    build_getter!(
        /// The title of the section
        section_title,
        &str
    );
    build_getter!(
        /// The snippet of the section
        section_snippet,
        &str
    );

    /// If it's a file match
    pub fn is_file_match(&self) -> Option<&bool> {
        self.file_match.as_ref()
    }
}
