use std::env;
use std::path::PathBuf;
use cc;

const LIBC_TYPES: &[&str] = &[
    "in_addr",
    "in6_addr",
    "sockaddr",
    "sockaddr_in",
    "sockaddr_in6",
    "timespec",
];

const USELESS_TYPES: &[&str] = &["in6_addr__bindgen_ty_1"];

fn main() {
    let mut bindings = bindgen::Builder::default()
        .header("c/wireguard.h")
        .impl_debug(true)
        .allowlist_function("wg_.*")
        .bitfield_enum("wg_peer_flags")
        .bitfield_enum("wg_device_flags")
        .parse_callbacks(Box::new(bindgen::CargoCallbacks))
        .raw_line("extern crate libc;");

    for libc_type in LIBC_TYPES {
        bindings = bindings
            .blocklist_type(libc_type)
            .raw_line(format!("use libc::{};", libc_type));
    }

    for useless_type in USELESS_TYPES {
        bindings = bindings.blocklist_type(useless_type);
    }

    let bindings = bindings.generate().expect("Unable to generate bindings");

    let out_path = PathBuf::from(env::var("OUT_DIR").unwrap());

    bindings
        .write_to_file(out_path.join("bindings.rs"))
        .expect("Couldn't write bindings!");

    cc::Build::new()
        .file("c/wireguard.c")
        .warnings(true)
        .extra_warnings(true)
        .warnings_into_errors(true)
        .flag_if_supported("-Wno-unused-parameter")
        .compile("wireguard");
}
