//! Low Level Bindings for WebView2 SDK.
#![cfg(windows)]
#![allow(
    clippy::missing_safety_doc,
    non_snake_case,
    clippy::upper_case_acronyms
)]

// Generated by idl2rs.

use com::{
    com_interface,
    interfaces::{iunknown::IUnknownVTable, IUnknown},
};
use std::ffi::c_void;
use winapi::shared::minwindef::{ULONG, *};
use winapi::shared::ntdef::*;
use winapi::shared::windef::*;
use winapi::um::oaidl::VARIANT;
use winapi::um::objidlbase::STATSTG;

/// Represents a reference to a delegate that receives change notifications.
#[repr(C)]
#[derive(Debug, Eq, PartialEq, Copy, Clone)]
pub struct EventRegistrationToken {
    value: i64,
}

#[com_interface("0c733a30-2a1c-11ce-ade5-00aa0044773d")]
pub trait ISequentialStream: IUnknown {
    unsafe fn read(&self, pv: *mut c_void, cb: ULONG, pcbRead: *mut ULONG) -> HRESULT;
    unsafe fn write(&self, pv: *const c_void, cb: ULONG, pcbWritten: *mut ULONG) -> HRESULT;
}

#[com_interface("0000000c-0000-0000-C000-000000000046")]
pub trait IStream: ISequentialStream {
    unsafe fn seek(
        &self,
        dlibMove: LARGE_INTEGER,
        dwOrigin: DWORD,
        plibNewPosition: *mut ULARGE_INTEGER,
    ) -> HRESULT;
    unsafe fn set_size(&self, libNewSize: ULARGE_INTEGER) -> HRESULT;
    unsafe fn copy_to(
        &self,
        pstm: *mut *mut IStreamVTable,
        cb: ULARGE_INTEGER,
        pcbRead: *mut ULARGE_INTEGER,
        pcbWritten: *mut ULARGE_INTEGER,
    ) -> HRESULT;
    unsafe fn commit(&self, grfCommitFlags: DWORD) -> HRESULT;
    unsafe fn revert(&self) -> HRESULT;
    unsafe fn lock_region(
        &self,
        libOffset: ULARGE_INTEGER,
        cb: ULARGE_INTEGER,
        dwLockType: DWORD,
    ) -> HRESULT;
    unsafe fn unlock_region(
        &self,
        libOffset: ULARGE_INTEGER,
        cb: ULARGE_INTEGER,
        dwLockType: DWORD,
    ) -> HRESULT;
    unsafe fn stat(&self, pstatstg: *mut STATSTG, grfStatFlag: DWORD) -> HRESULT;
    unsafe fn clone(&self, ppstm: *mut *mut *mut IStreamVTable) -> HRESULT;
}

/// DLL export to create a WebView2 environment with a custom version of Edge,
/// user data directory and/or additional options.
///
/// browserExecutableFolder is the relative path to the folder that
/// contains the embedded Edge. The embedded Edge can be obtained by
/// copying the version named folder of an installed Edge, like
/// 73.0.52.0 sub folder of an installed 73.0.52.0 Edge. The folder
/// should have msedge.exe, msedge.dll, and so on.
/// Use null or empty string for browserExecutableFolder to create
/// WebView using Edge installed on the machine, in which case the
/// API will try to find a compatible version of Edge installed on the
/// machine according to the channel preference trying to find first
/// per user install and then per machine install.
///
/// The default channel search order is stable, beta, dev, and canary.
/// When there is an override WEBVIEW2_RELEASE_CHANNEL_PREFERENCE environment
/// variable or applicable releaseChannelPreference registry value
/// with the value of 1, the channel search order is reversed.
///
/// userDataFolder can be
/// specified to change the default user data folder location for
/// WebView2. The path can be an absolute file path or a relative file path
/// that is interpreted as relative to the current process's executable.
/// Otherwise, for UWP apps, the default user data folder will be
/// the app data folder for the package; for non-UWP apps,
/// the default user data folder `{Executable File Name}.WebView2`
/// will be created in the same directory next to the app executable.
/// WebView2 creation can fail if the executable is running in a directory
/// that the process doesn't have permission to create a new folder in.
/// The app is responsible to clean up its user data folder
/// when it is done.
///
/// Note that as a browser process might be shared among WebViews,
/// WebView creation will fail with HRESULT_FROM_WIN32(ERROR_INVALID_STATE) if
/// the specified options does not match the options of the WebViews that are
/// currently running in the shared browser process.
///
/// environment_created_handler is the handler result to the async operation
/// which will contain the WebView2Environment that got created.
///
/// The browserExecutableFolder, userDataFolder and additionalBrowserArguments
/// of the environmentOptions may be overridden by
/// values either specified in environment variables or in the registry.
///
/// When creating a WebView2Environment the following environment variables
/// are checked:
///
/// ```
/// WEBVIEW2_BROWSER_EXECUTABLE_FOLDER
/// WEBVIEW2_USER_DATA_FOLDER
/// WEBVIEW2_ADDITIONAL_BROWSER_ARGUMENTS
/// WEBVIEW2_RELEASE_CHANNEL_PREFERENCE
/// ```
///
/// If an override environment variable is found then we use the
/// browserExecutableFolder, userDataFolder and additionalBrowserArguments
/// values as replacements for the corresponding values in
/// CreateCoreWebView2EnvironmentWithOptions parameters.
///
/// While not strictly overrides, there exists additional environment variables
/// that can be set:
///
/// ```
/// WEBVIEW2_WAIT_FOR_SCRIPT_DEBUGGER
/// ```
///
/// When found with a non-empty value, this indicates that the WebView is being
/// launched under a script debugger. In this case, the WebView will issue a
/// `Page.waitForDebugger` CDP command that will cause script execution inside the
/// WebView to pause on launch, until a debugger issues a corresponding
/// `Runtime.runIfWaitingForDebugger` CDP command to resume execution.
/// Note: There is no registry key equivalent of this environment variable.
///
/// ```
/// WEBVIEW2_PIPE_FOR_SCRIPT_DEBUGGER
/// ```
///
/// When found with a non-empty value, this indicates that the WebView is being
/// launched under a script debugger that also supports host applications that
/// use multiple WebViews. The value is used as the identifier for a named pipe
/// that will be opened and written to when a new WebView is created by the host
/// application. The payload will match that of the remote-debugging-port JSON
/// target and can be used by the external debugger to attach to a specific
/// WebView instance.
/// The format of the pipe created by the debugger should be:
/// `\\.\pipe\WebView2\Debugger\{app_name}\{pipe_name}`
/// where:
///
/// - `{app_name}` is the host application exe filename, e.g. WebView2Example.exe
/// - `{pipe_name}` is the value set for WEBVIEW2_PIPE_FOR_SCRIPT_DEBUGGER.
///
/// To enable debugging of the targets identified by the JSON you will also need
/// to set the WEBVIEW2_ADDITIONAL_BROWSER_ARGUMENTS environment variable to
/// send `--remote-debugging-port={port_num}`
/// where:
///
/// - `{port_num}` is the port on which the CDP server will bind.
///
/// Be aware that setting both the WEBVIEW2_PIPE_FOR_SCRIPT_DEBUGGER and
/// WEBVIEW2_ADDITIONAL_BROWSER_ARGUMENTS environment variables will cause the
/// WebViews hosted in your application and their contents to be exposed to
/// 3rd party applications such as debuggers.
///
/// Note: There is no registry key equivalent of this environment variable.
///
/// If none of those environment variables exist, then the registry is examined next.
/// The following registry keys are checked:
///
/// ```
/// [{Root}\Software\Policies\Microsoft\EmbeddedBrowserWebView\LoaderOverride\{AppId}]
/// "releaseChannelPreference"=dword:00000000
/// "browserExecutableFolder"=""
/// "userDataFolder"=""
/// "additionalBrowserArguments"=""
/// ```
///
/// In the unlikely scenario where some instances of WebView are open during
/// a browser update we could end up blocking the deletion of old Edge browsers.
/// To avoid running out of disk space a new WebView creation will fail
/// with the next error if it detects that there are many old versions present.
///
/// ```
/// ERROR_DISK_FULL
/// ```
///
/// The default maximum number of Edge versions allowed is 20.
///
/// The maximum number of old Edge versions allowed can be overwritten with the value
/// of the following environment variable.
///
/// ```
/// WEBVIEW2_MAX_INSTANCES
/// ```
///
/// If the Webview depends on an installed Edge and it is uninstalled
/// any subsequent creation will fail with the next error
///
/// ```
/// ERROR_PRODUCT_UNINSTALLED
/// ```
///
/// First we check with Root as HKLM and then HKCU.
/// AppId is first set to the Application User Model ID of the caller's process,
/// then if there's no corresponding registry key the AppId is
/// set to the executable name of the caller's process, or if that
/// isn't a registry key then '*'. If an override registry key is found then we
/// use the browserExecutableFolder, userDataFolder and additionalBrowserArguments
/// registry values as replacements for the corresponding values in
/// CreateCoreWebView2EnvironmentWithOptions parameters.
extern "stdcall" {
    pub fn CreateCoreWebView2EnvironmentWithOptions(
        browserExecutableFolder: PCWSTR,
        userDataFolder: PCWSTR,
        environment_options: *mut *mut ICoreWebView2EnvironmentOptionsVTable,
        environment_created_handler: *mut *mut ICoreWebView2CreateCoreWebView2EnvironmentCompletedHandlerVTable,
    ) -> HRESULT;
}

/// Get the browser version info including channel name if it is not the stable channel
/// or the Embedded Edge.
/// Channel names are beta, dev, and canary.
/// If an override exists for the browserExecutableFolder or the channel preference,
/// the override will be used.
/// If there isn't an override, then the parameter passed to
/// GetAvailableCoreWebView2BrowserVersionString is used.
extern "stdcall" {
    pub fn GetAvailableCoreWebView2BrowserVersionString(
        browser_executable_folder: PCWSTR,
        version_info: *mut LPWSTR,
    ) -> HRESULT;
}

/// This method is for anyone want to compare version correctly to determine
/// which version is newer, older or same. It can be used to determine whether
/// to use webview2 or certain feature base on version.
/// Sets the value of result to -1, 0 or 1 if version1 is less than, equal or
/// greater than version2 respectively.
/// Returns E_INVALIDARG if it fails to parse any of the version strings or any
/// input parameter is null.
/// Input can directly use the versionInfo obtained from
/// GetAvailableCoreWebView2BrowserVersionString, channel info will be ignored.
extern "stdcall" {
    pub fn CompareBrowserVersions(version1: PCWSTR, version2: PCWSTR, result: *mut i32) -> HRESULT;
}
/// Contains the information packed into the `LPARAM` sent to a Win32 key
/// event.  For more information about `WM_KEYDOWN`, navigate to
/// \[WM_KEYDOWN message\]\[WindowsWin32InputdevWmKeydown\].
///
/// \[WindowsWin32InputdevWmKeydown\]: /windows/win32/inputdev/wm-keydown "WM_KEYDOWN message | Microsoft Docs"
#[repr(C)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub struct PhysicalKeyStatus {
    /// Specifies the repeat count for the current message.
    pub repeat_count: u32,
    /// Specifies the scan code.
    pub scan_code: u32,
    /// Indicates that the key is an extended key.
    pub is_extended_key: BOOL,
    /// Indicates that a menu key is held down (context code).
    pub is_menu_key_down: BOOL,
    /// Indicates that the key was held down.
    pub was_key_down: BOOL,
    /// Indicates that the key was released.
    pub is_key_released: BOOL,
}

/// A value representing RGBA color (Red, Green, Blue, Alpha) for WebView2.
/// Each component takes a value from 0 to 255, with 0 being no intensity
/// and 255 being the highest intensity.
#[repr(C)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub struct Color {
    /// Specifies the intensity of the Alpha ie. opacity value. 0 is transparent,
    /// 255 is opaque.
    pub a: BYTE,
    /// Specifies the intensity of the Red color.
    pub r: BYTE,
    /// Specifies the intensity of the Green color.
    pub g: BYTE,
    /// Specifies the intensity of the Blue color.
    pub b: BYTE,
}

/// Specifies the image format for the `ICoreWebView2::CapturePreview` method.
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum CapturePreviewImageFormat {
    /// Indicates that the PNG image format is used.
    PNG,
    /// Indicates the JPEG image format is used.
    JPEG,
}

/// Kind of cookie SameSite status used in the ICoreWebView2Cookie interface.
/// These fields match those as specified in https://developer.mozilla.org/docs/Web/HTTP/Cookies#.
/// Learn more about SameSite cookies here: https://tools.ietf.org/html/draft-west-first-party-cookies-07
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum CookieSameSiteKind {
    /// None SameSite type. No restrictions on cross-site requests.
    None,
    /// Lax SameSite type. The cookie will be sent with "same-site" requests, and with "cross-site" top level navigation.
    Lax,
    /// Strict SameSite type. The cookie will only be sent along with "same-site" requests.
    Strict,
}

/// Kind of cross origin resource access allowed for host resources during download.
/// Note that other normal access checks like same origin DOM access check and [Content
/// Security Policy](https://developer.mozilla.org/docs/Web/HTTP/CSP) still apply.
/// The following table illustrates the host resource cross origin access according to
/// access context and `COREWEBVIEW2_HOST_RESOURCE_ACCESS_KIND`.
///
/// Cross Origin Access Context | DENY | ALLOW | DENY_CORS
/// --- | --- | --- | ---
/// From DOM like src of img, script or iframe element| Deny | Allow | Allow
/// From Script like Fetch or XMLHttpRequest| Deny | Allow | Deny
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum HostResourceAccessKind {
    /// All cross origin resource access is denied, including normal sub resource access
    /// as src of a script or image element.
    Deny,
    /// All cross origin resource access is allowed, including accesses that are
    /// subject to Cross-Origin Resource Sharing(CORS) check. The behavior is similar to
    /// a web site sends back http header Access-Control-Allow-Origin: *.
    Allow,
    /// Cross origin resource access is allowed for normal sub resource access like
    /// as src of a script or image element, while any access that subjects to CORS check
    /// will be denied.
    /// See [Cross-Origin Resource Sharing](https://developer.mozilla.org/docs/Web/HTTP/CORS)
    /// for more information.
    DenyCors,
}

/// Specifies the JavaScript dialog type used in the
/// `ICoreWebView2ScriptDialogOpeningEventHandler` interface.
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum ScriptDialogKind {
    /// Indicates that the dialog uses the `window.alert` JavaScript function.
    Alert,
    /// Indicates that the dialog uses the `window.confirm` JavaScript function.
    Confirm,
    /// Indicates that the dialog uses the `window.prompt` JavaScript function.
    Prompt,
    /// Indicates that the dialog uses the `beforeunload` JavaScript event.
    Beforeunload,
}

/// Specifies the process failure type used in the
/// `ICoreWebView2ProcessFailedEventHandler` interface. The values in this enum
/// make reference to the process kinds in the Chromium architecture. For more
/// information about what these processes are and what they do, see
/// [Browser Architecture - Inside look at modern web browser](https://developers.google.com/web/updates/2018/09/inside-browser-part1).
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum ProcessFailedKind {
    /// Indicates that the browser process ended unexpectedly.  The WebView
    /// automatically moves to the Closed state.  The app has to recreate a new
    /// WebView to recover from this failure.
    BrowserProcessExited,
    /// Indicates that the main frame's render process ended unexpectedly.  A new
    /// render process is created automatically and navigated to an error page.
    /// You can use the `Reload` method to try to reload the page that failed.
    RenderProcessExited,
    /// Indicates that the main frame's render process is unresponsive.
    RenderProcessUnresponsive,
    /// Indicates that a frame-only render process ended unexpectedly. The process
    /// exit does not affect the top-level document, only a subset of the
    /// subframes within it. The content in these frames is replaced with an error
    /// page in the frame.
    FrameRenderProcessExited,
    /// Indicates that a utility process ended unexpectedly.
    UtilityProcessExited,
    /// Indicates that a sandbox helper process ended unexpectedly.
    SandboxHelperProcessExited,
    /// Indicates that the GPU process ended unexpectedly.
    GpuProcessExited,
    /// Indicates that a PPAPI plugin process ended unexpectedly.
    PpapiPluginProcessExited,
    /// Indicates that a PPAPI plugin broker process ended unexpectedly.
    PpapiBrokerProcessExited,
    /// Indicates that a process of unspecified kind ended unexpectedly.
    UnknownProcessExited,
}

/// Specifies the process failure reason used in the
/// `ICoreWebView2ProcessFailedEventHandler` interface.
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum ProcessFailedReason {
    /// An unexpected process failure occurred.
    Unexpected,
    /// The process became unresponsive.
    /// This only applies to the main frame's render process.
    Unresponsive,
    /// The process was terminated. For example, from Task Manager.
    Terminated,
    /// The process crashed.
    Crashed,
    /// The process failed to launch.
    LaunchFailed,
    /// The process died due to running out of memory.
    OutOfMemory,
}

/// Indicates the type of a permission request.
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum PermissionKind {
    /// Indicates an unknown permission.
    UnknownPermission,
    /// Indicates permission to capture audio.
    Microphone,
    /// Indicates permission to capture video.
    Camera,
    /// Indicates permission to access geolocation.
    Geolocation,
    /// Indicates permission to send web notifications.  This permission request
    /// is currently auto-rejected and no event is run for it.
    Notifications,
    /// Indicates permission to access generic sensor.  Generic Sensor covering
    /// ambient-light-sensor, accelerometer, gyroscope, and magnetometer.
    OtherSensors,
    /// Indicates permission to read the system clipboard without a user gesture.
    ClipboardRead,
}

/// Specifies the response to a permission request.
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum PermissionState {
    /// Specifies that the default browser behavior is used, which normally
    /// prompt users for decision.
    Default,
    /// Specifies that the permission request is granted.
    Allow,
    /// Specifies that the permission request is denied.
    Deny,
}

/// Indicates the error status values for web navigations.
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum WebErrorStatus {
    /// Indicates that an unknown error occurred.
    Unknown,
    /// Indicates that the SSL certificate common name does not match the web
    /// address.
    CertificateCommonNameIsIncorrect,
    /// Indicates that the SSL certificate has expired.
    CertificateExpired,
    /// Indicates that the SSL client certificate contains errors.
    ClientCertificateContainsErrors,
    /// Indicates that the SSL certificate has been revoked.
    CertificateRevoked,
    /// Indicates that the SSL certificate is not valid.  The certificate may not
    /// match the public key pins for the host name, the certificate is signed
    /// by an untrusted authority or using a weak sign algorithm, the certificate
    /// claimed DNS names violate name constraints, the certificate contains a
    /// weak key, the validity period of the certificate is too long, lack of
    /// revocation information or revocation mechanism, non-unique host name,
    /// lack of certificate transparency information, or the certificate is
    /// chained to a
    /// \[legacy Symantec root\]\[GoogleblogSecurity201803DistrustSymantecPkiImmediateHtml\].
    ///
    /// \[GoogleblogSecurity201803DistrustSymantecPkiImmediateHtml\]: https://security.googleblog.com/2018/03/distrust-of-symantec-pki-immediate.html "Distrust of the Symantec PKI: Immediate action needed by site operators | Google Security Blog"
    CertificateIsInvalid,
    /// Indicates that the host is unreachable.
    ServerUnreachable,
    /// Indicates that the connection has timed out.
    Timeout,
    /// Indicates that the server returned an invalid or unrecognized response.
    ErrorHttpInvalidServerResponse,
    /// Indicates that the connection was stopped.
    ConnectionAborted,
    /// Indicates that the connection was reset.
    ConnectionReset,
    /// Indicates that the Internet connection has been lost.
    Disconnected,
    /// Indicates that a connection to the destination was not established.
    CannotConnect,
    /// Indicates that the provided host name was not able to be resolved.
    HostNameNotResolved,
    /// Indicates that the operation was canceled.
    OperationCanceled,
    /// Indicates that the request redirect failed.
    RedirectFailed,
    /// Indicates that an unexpected error occurred.
    UnexpectedError,
}

/// Specifies the web resource request contexts.
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum WebResourceContext {
    /// Specifies all resources.
    All,
    /// Specifies a document resource.
    Document,
    /// Specifies a CSS resource.
    Stylesheet,
    /// Specifies an image resource.
    Image,
    /// Specifies another media resource such as a video.
    Media,
    /// Specifies a font resource.
    Font,
    /// Specifies a script resource.
    Script,
    /// Specifies an XML HTTP request.
    XmlHttpRequest,
    /// Specifies a Fetch API communication.
    Fetch,
    /// Specifies a TextTrack resource.
    TextTrack,
    /// Specifies an EventSource API communication.
    EventSource,
    /// Specifies a WebSocket API communication.
    Websocket,
    /// Specifies a Web App Manifest.
    Manifest,
    /// Specifies a Signed HTTP Exchange.
    SignedExchange,
    /// Specifies a Ping request.
    Ping,
    /// Specifies a CSP Violation Report.
    CspViolationReport,
    /// Specifies an other resource.
    Other,
}

/// Specifies the reason for moving focus.
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum MoveFocusReason {
    /// Specifies that the code is setting focus into WebView.
    Programmatic,
    /// Specifies that the focus is moving due to Tab traversal forward.
    Next,
    /// Specifies that the focus is moving due to Tab traversal backward.
    Previous,
}

/// Specifies the key event type that triggered an `AcceleratorKeyPressed`
/// event.
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum KeyEventKind {
    /// Specifies that the key event type corresponds to window message
    /// `WM_KEYDOWN`.
    KeyDown,
    /// Specifies that the key event type corresponds to window message
    /// `WM_KEYUP`.
    KeyUp,
    /// Specifies that the key event type corresponds to window message
    /// `WM_SYSKEYDOWN`.
    SystemKeyDown,
    /// Specifies that the key event type corresponds to window message
    /// `WM_SYSKEYUP`.
    SystemKeyUp,
}

/// Mouse event type used by SendMouseInput to convey the type of mouse event
/// being sent to WebView. The values of this enum align with the matching
/// WM_* window messages.
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum MouseEventKind {
    /// Mouse horizontal wheel scroll event, WM_MOUSEHWHEEL.
    HorizontalWheel = 0x020E,
    /// Left button double click mouse event, WM_LBUTTONDBLCLK.
    LeftButtonDoubleClick = 0x0203,
    /// Left button down mouse event, WM_LBUTTONDOWN.
    LeftButtonDown = 0x0201,
    /// Left button up mouse event, WM_LBUTTONUP.
    LeftButtonUp = 0x0202,
    /// Mouse leave event, WM_MOUSELEAVE.
    Leave = 0x02A3,
    /// Middle button double click mouse event, WM_MBUTTONDBLCLK.
    MiddleButtonDoubleClick = 0x0209,
    /// Middle button down mouse event, WM_MBUTTONDOWN.
    MiddleButtonDown = 0x0207,
    /// Middle button up mouse event, WM_MBUTTONUP.
    MiddleButtonUp = 0x0208,
    /// Mouse move event, WM_MOUSEMOVE.
    Move = 0x0200,
    /// Right button double click mouse event, WM_RBUTTONDBLCLK.
    RightButtonDoubleClick = 0x0206,
    /// Right button down mouse event, WM_RBUTTONDOWN.
    RightButtonDown = 0x0204,
    /// Right button up mouse event, WM_RBUTTONUP.
    RightButtonUp = 0x0205,
    /// Mouse wheel scroll event, WM_MOUSEWHEEL.
    Wheel = 0x020A,
    /// First or second X button double click mouse event, WM_XBUTTONDBLCLK.
    XButtonDoubleClick = 0x020D,
    /// First or second X button down mouse event, WM_XBUTTONDOWN.
    XButtonDown = 0x020B,
    /// First or second X button up mouse event, WM_XBUTTONUP.
    XButtonUp = 0x020C,
}

/// Mouse event virtual keys associated with a COREWEBVIEW2_MOUSE_EVENT_KIND for
/// SendMouseInput. These values can be combined into a bit flag if more than
/// one virtual key is pressed for the event. The values of this enum align
/// with the matching MK_* mouse keys.
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum MouseEventVirtualKeys {
    /// No additional keys pressed.
    None = 0x0,
    /// Left mouse button is down, MK_LBUTTON.
    LeftButton = 0x0001,
    /// Right mouse button is down, MK_RBUTTON.
    RightButton = 0x0002,
    /// SHIFT key is down, MK_SHIFT.
    Shift = 0x0004,
    /// CTRL key is down, MK_CONTROL.
    Control = 0x0008,
    /// Middle mouse button is down, MK_MBUTTON.
    MiddleButton = 0x0010,
    /// First X button is down, MK_XBUTTON1
    XButton1 = 0x0020,
    /// Second X button is down, MK_XBUTTON2
    XButton2 = 0x0040,
}

/// Pointer event type used by SendPointerInput to convey the type of pointer
/// event being sent to WebView. The values of this enum align with the
/// matching WM_POINTER* window messages.
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum PointerEventKind {
    /// Corresponds to WM_POINTERACTIVATE.
    Activate = 0x024B,
    /// Corresponds to WM_POINTERDOWN.
    Down = 0x0246,
    /// Corresponds to WM_POINTERENTER.
    Enter = 0x0249,
    /// Corresponds to WM_POINTERLEAVE.
    Leave = 0x024A,
    /// Corresponds to WM_POINTERUP.
    Up = 0x0247,
    /// Corresponds to WM_POINTERUPDATE.
    Update = 0x0245,
}

/// Mode for how the Bounds property is interpreted in relation to the RasterizationScale property.
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum BoundsMode {
    /// Bounds property represents raw pixels. Physical size of Webview is not impacted by RasterizationScale.
    UseRawPixels,
    /// Bounds property represents logicl pixels and the RasterizationScale property is used to get the physical size of the WebView.
    UseRasterizationScale,
}

#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum ClientCertificateKind {
    /// Specifies smart card certificate.
    SmartCard,
    /// Specifies PIN certificate.
    Pin,
    /// Specifies other certificate.
    Other,
}

/// State of the download operation.
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum DownloadState {
    /// The download is in progress.
    InProgress,
    /// The connection with the file host was broken. The `InterruptReason` property
    /// can be accessed from `ICoreWebView2DownloadOperation`. See
    /// `COREWEBVIEW2_DOWNLOAD_INTERRUPT_REASON` for descriptions of kinds of
    /// interrupt reasons. Host can check whether an interrupted download can be
    /// resumed with the `CanResume` property on the `ICoreWebView2DownloadOperation`.
    /// Once resumed, a download is in the `COREWEBVIEW2_DOWNLOAD_STATE_IN_PROGRESS` state.
    Interrupted,
    /// The download completed successfully.
    Completed,
}

/// Reason why a download was interrupted.
#[repr(u32)]
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum DownloadInterruptReason {
    None,
    /// Generic file error.
    FileFailed,
    /// Access denied due to security restrictions.
    FileAccessDenied,
    /// Disk full. User should free some space or choose a different location to
    /// store the file.
    FileNoSpace,
    /// Result file path with file name is too long.
    FileNameTooLong,
    /// File is too large for file system.
    FileTooLarge,
    /// Microsoft Defender Smartscreen detected a virus in the file.
    FileMalicious,
    /// File was in use, too many files opened, or out of memory.
    FileTransientError,
    /// File blocked by local policy.
    FileBlockedByPolicy,
    /// Security check failed unexpectedly. Microsoft Defender SmartScreen could
    /// not scan this file.
    FileSecurityCheckFailed,
    /// Seeking past the end of a file in opening a file, as part of resuming an
    /// interrupted download. The file did not exist or was not as large as
    /// expected. Partially downloaded file was truncated or deleted, and download
    /// will be restarted automatically.
    FileTooShort,
    /// Partial file did not match the expected hash and was deleted. Download
    /// will be restarted automatically.
    FileHashMismatch,
    /// Generic network error. User can retry the download manually.
    NetworkFailed,
    /// Network operation timed out.
    NetworkTimeout,
    /// Network connection lost. User can retry the download manually.
    NetworkDisconnected,
    /// Server has gone down. User can retry the download manually.
    NetworkServerDown,
    /// Network request invalid because original or redirected URI is invalid, has
    /// an unsupported scheme, or is disallowed by network policy.
    NetworkInvalidRequest,
    /// Generic server error. User can retry the download manually.
    ServerFailed,
    /// Server does not support range requests.
    ServerNoRange,
    /// Server does not have the requested data.
    ServerBadContent,
    /// Server did not authorize access to resource.
    ServerUnauthorized,
    /// Server certificate problem.
    ServerCertificateProblem,
    /// Server access forbidden.
    ServerForbidden,
    /// Unexpected server response. Responding server may not be intended server.
    /// User can retry the download manually.
    ServerUnexpectedResponse,
    /// Server sent fewer bytes than the Content-Length header. Content-length
    /// header may be invalid or connection may have closed. Download is treated
    /// as complete unless there are
    /// [strong validators](https://tools.ietf.org/html/rfc7232#section-2) present
    /// to interrupt the download.
    ServerContentLengthMismatch,
    /// Unexpected cross-origin redirect.
    ServerCrossOriginRedirect,
    /// User canceled the download.
    UserCanceled,
    /// User shut down the WebView. Resuming downloads that were interrupted
    /// during shutdown is not yet supported.
    UserShutdown,
    /// User paused the download.
    UserPaused,
    /// WebView crashed.
    DownloadProcessCrashed,
}

/// WebView2 enables you to host web content using the latest Microsoft Edge
/// browser and web technology.
#[com_interface("76eceacb-0462-4d94-ac83-423a6793775e")]
pub trait ICoreWebView2: IUnknown {
    /// The `ICoreWebView2Settings` object contains various modifiable settings
    /// for the running WebView.
    unsafe fn get_settings(
        &self,
        /* out, retval */ settings: *mut *mut *mut ICoreWebView2SettingsVTable,
    ) -> HRESULT;

    /// The URI of the current top level document.  This value potentially
    /// changes as a part of the `SourceChanged` event that runs for some cases
    /// such as navigating to a different site or fragment navigations.  It
    /// remains the same for other types of navigations such as page refreshes
    /// or `history.pushState` with the same URL as the current page.
    ///
    /// \snippet ControlComponent.cpp SourceChanged
    unsafe fn get_source(&self, /* out, retval */ uri: *mut LPWSTR) -> HRESULT;

    /// Cause a navigation of the top-level document to run to the specified URI.
    /// For more information, navigate to \[Navigation events\]\[MicrosoftEdgeWebview2ConceptsNavigationevents\].
    ///
    /// \[MicrosoftEdgeWebview2ConceptsNavigationevents\]: /microsoft-edge/webview2/concepts/navigation-events "Navigation events | Microsoft Docs"
    ///
    /// \> [!NOTE]\n\> This operation starts a navigation and the corresponding
    /// `NavigationStarting` event triggers sometime after `Navigate` runs.
    ///
    /// \snippet ControlComponent.cpp Navigate
    unsafe fn navigate(&self, /* in */ uri: LPCWSTR) -> HRESULT;

    /// Initiates a navigation to htmlContent as source HTML of a new document.
    /// The `htmlContent` parameter may not be larger than 2 MB (2 * 1024 * 1024 bytes) in total size.
    /// The origin of the new page is `about:blank`.
    ///
    /// \snippet SettingsComponent.cpp NavigateToString
    unsafe fn navigate_to_string(&self, /* in */ html_content: LPCWSTR) -> HRESULT;

    /// Add an event handler for the `NavigationStarting` event.
    /// `NavigationStarting` runs when the WebView main frame is requesting
    /// permission to navigate to a different URI.  Redirects trigger this
    /// operation as well, and the navigation id is the same as the original
    /// one.
    ///
    /// You may block corresponding navigations until the event handler returns.
    ///
    /// \snippet SettingsComponent.cpp NavigationStarting
    unsafe fn add_navigation_starting(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2NavigationStartingEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_NavigationStarting`.
    unsafe fn remove_navigation_starting(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// Add an event handler for the `ContentLoading` event.  `ContentLoading`
    /// triggers before any content is loaded, including scripts added with
    /// `AddScriptToExecuteOnDocumentCreated`.  `ContentLoading` does not trigger
    /// if a same page navigation occurs (such as through `fragment`
    /// navigations or `history.pushState` navigations).  This operation
    /// follows the `NavigationStarting` and `SourceChanged` events and precedes
    /// the `HistoryChanged` and `NavigationCompleted` events.
    unsafe fn add_content_loading(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2ContentLoadingEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_ContentLoading`.
    unsafe fn remove_content_loading(&self, /* in */ token: EventRegistrationToken) -> HRESULT;

    /// Add an event handler for the `SourceChanged` event.  `SourceChanged`
    /// triggers when the `Source` property changes.  `SourceChanged` runs when
    /// navigating to a different site or fragment navigations.  It does not
    /// trigger for other types of navigations such as page refreshes or
    /// `history.pushState` with the same URL as the current page.
    /// `SourceChanged` runs before `ContentLoading` for navigation to a new
    /// document.
    ///
    /// \snippet ControlComponent.cpp SourceChanged
    unsafe fn add_source_changed(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2SourceChangedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_SourceChanged`.
    unsafe fn remove_source_changed(&self, /* in */ token: EventRegistrationToken) -> HRESULT;

    /// Add an event handler for the `HistoryChanged` event.  `HistoryChanged`
    /// listens to the change of navigation history for the top level document.
    /// Use `HistoryChanged` to verify that the `CanGoBack` or `CanGoForward`
    /// value has changed.  `HistoryChanged` also runs for using `GoBack`or
    /// `GoForward`.  `HistoryChanged` runs after `SourceChanged` and
    /// `ContentLoading`.
    ///
    /// \snippet ControlComponent.cpp HistoryChanged
    unsafe fn add_history_changed(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2HistoryChangedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_HistoryChanged`.
    unsafe fn remove_history_changed(&self, /* in */ token: EventRegistrationToken) -> HRESULT;

    /// Add an event handler for the `NavigationCompleted` event.
    /// `NavigationCompleted` runs when the WebView has completely loaded
    /// (`body.onload` runs) or loading stopped with error.
    ///
    /// \snippet ControlComponent.cpp NavigationCompleted
    unsafe fn add_navigation_completed(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2NavigationCompletedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_NavigationCompleted`.
    unsafe fn remove_navigation_completed(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// Add an event handler for the `FrameNavigationStarting` event.
    /// `FrameNavigationStarting` triggers when a child frame in the WebView
    /// requests permission to navigate to a different URI.  Redirects trigger
    /// this operation as well, and the navigation id is the same as the original
    /// one.
    ///
    /// You may block corresponding navigations until the event handler returns.
    ///
    /// \snippet SettingsComponent.cpp FrameNavigationStarting
    unsafe fn add_frame_navigation_starting(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2NavigationStartingEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with
    /// `add_FrameNavigationStarting`.
    unsafe fn remove_frame_navigation_starting(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// Add an event handler for the `FrameNavigationCompleted` event.
    /// `FrameNavigationCompleted` triggers when a child frame has completely
    /// loaded (`body.onload` has triggered) or loading stopped with error.
    ///
    /// \snippet ControlComponent.cpp FrameNavigationCompleted
    unsafe fn add_frame_navigation_completed(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2NavigationCompletedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with
    /// `add_FrameNavigationCompleted`.
    unsafe fn remove_frame_navigation_completed(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// Add an event handler for the `ScriptDialogOpening` event.
    /// `ScriptDialogOpening` runs when a JavaScript dialog (`alert`, `confirm`,
    /// `prompt`, or `beforeunload`) displays for the webview.  This event only
    /// triggers if the `ICoreWebView2Settings::AreDefaultScriptDialogsEnabled`
    /// property is set to `FALSE`.  The `ScriptDialogOpening` event suppresses
    /// dialogs or replaces default dialogs with custom dialogs.
    ///
    /// If a deferral is not taken on the event args, the subsequent scripts are
    /// blocked until the event handler returns.  If a deferral is taken, the
    /// scripts are blocked until the deferral is completed.
    ///
    /// \snippet SettingsComponent.cpp ScriptDialogOpening
    unsafe fn add_script_dialog_opening(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2ScriptDialogOpeningEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_ScriptDialogOpening`.
    unsafe fn remove_script_dialog_opening(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// Add an event handler for the `PermissionRequested` event.
    /// `PermissionRequested` runs when content in a WebView requests permission
    /// to access some privileged resources.
    ///
    /// If a deferral is not taken on the event args, the subsequent scripts are
    /// blocked until the event handler returns.  If a deferral is taken, the
    /// scripts are blocked until the deferral is completed.
    ///
    /// \snippet SettingsComponent.cpp PermissionRequested
    unsafe fn add_permission_requested(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2PermissionRequestedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_PermissionRequested`.
    unsafe fn remove_permission_requested(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// Add an event handler for the `ProcessFailed` event.  `ProcessFailed` runs
    /// when a WebView process ends unexpectedly or becomes unresponsive.
    ///
    /// \snippet ProcessComponent.cpp ProcessFailed
    unsafe fn add_process_failed(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2ProcessFailedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with add_ProcessFailed.
    unsafe fn remove_process_failed(&self, /* in */ token: EventRegistrationToken) -> HRESULT;

    /// Add the provided JavaScript to a list of scripts that should be run after
    /// the global object has been created, but before the HTML document has
    /// been parsed and before any other script included by the HTML document is
    /// run.  This method injects a script that runs on all top-level document
    /// and child frame page navigations.  This method runs asynchronously, and
    /// you must wait for the completion handler to finish before the injected
    /// script is ready to run.  When this method completes, the `Invoke` method
    /// of the handler is run with the `id` of the injected script.  `id` is a
    /// string.  To remove the injected script, use
    /// `RemoveScriptToExecuteOnDocumentCreated`.
    ///
    /// If the method is run in add_NewWindowRequested handler it should be called
    /// after the new window is set. For more details see `ICoreWebView2NewWindowRequestedEventArgs::put_NewWindow`.
    ///
    /// \> [!NOTE]\n\> If an HTML document is running in a sandbox of some kind using
    /// \[sandbox\]\[MdnDocsWebHtmlElementIframeAttrSandbox\]
    /// properties or the
    /// \[Content-Security-Policy\]\[MdnDocsWebHttpHeadersContentSecurityPolicy\]
    /// HTTP header affects the script that runs.  For example, if the
    /// `allow-modals` keyword is not set then requests to run the `alert`
    /// function are ignored.
    ///
    /// \snippet ScriptComponent.cpp AddScriptToExecuteOnDocumentCreated
    ///
    /// \[MdnDocsWebHtmlElementIframeAttrSandbox\]: https://developer.mozilla.org/docs/Web/HTML/Element/iframe#attr-sandbox "sandbox - \<iframe\>: The Inline Frame element | MDN"
    ///
    /// \[MdnDocsWebHttpHeadersContentSecurityPolicy\]: https://developer.mozilla.org/docs/Web/HTTP/Headers/Content-Security-Policy "Content-Security-Policy | MDN"
    unsafe fn add_script_to_execute_on_document_created(
        &self,
        /* in */ java_script: LPCWSTR,
        /* in */
        handler: *mut *mut ICoreWebView2AddScriptToExecuteOnDocumentCreatedCompletedHandlerVTable,
    ) -> HRESULT;

    /// Remove the corresponding JavaScript added using
    /// `AddScriptToExecuteOnDocumentCreated` with the specified script ID.
    unsafe fn remove_script_to_execute_on_document_created(
        &self,
        /* in */ id: LPCWSTR,
    ) -> HRESULT;

    /// Run JavaScript code from the javascript parameter in the current
    /// top-level document rendered in the WebView.  The result of evaluating
    /// the provided JavaScript is used in this parameter.  The result value is
    /// a JSON encoded string.  If the result is undefined, contains a reference
    /// cycle, or otherwise is not able to be encoded into JSON, the JSON `null`
    /// value is returned as the `null` string.
    ///
    /// \> [!NOTE]\n\> A function that has no explicit return value returns undefined.  If the
    /// script that was run throws an unhandled exception, then the result is
    /// also `null`.  This method is applied asynchronously.  If the method is
    /// run after the `NavigationStarting` event during a navigation, the script
    /// runs in the new document when loading it, around the time
    /// `ContentLoading` is run.  This operation works even if
    /// `ICoreWebView2Settings::IsScriptEnabled` is set to `FALSE`.
    ///
    /// \snippet ScriptComponent.cpp ExecuteScript
    unsafe fn execute_script(
        &self,
        /* in */ java_script: LPCWSTR,
        /* in */ handler: *mut *mut ICoreWebView2ExecuteScriptCompletedHandlerVTable,
    ) -> HRESULT;

    /// Capture an image of what WebView is displaying.  Specify the format of
    /// the image with the `imageFormat` parameter.  The resulting image binary
    /// data is written to the provided `imageStream` parameter.  When
    /// `CapturePreview` finishes writing to the stream, the `Invoke` method on
    /// the provided `handler` parameter is run.  This method fails if called
    /// before the first ContentLoading event.  For example if this is called in
    /// the NavigationStarting event for the first navigation it will fail.
    /// For subsequent navigations, the method may not fail, but will not capture
    /// an image of a given webpage until the ContentLoading event has been fired
    /// for it.  Any call to this method prior to that will result in a capture of
    /// the page being navigated away from.
    ///
    /// \snippet FileComponent.cpp CapturePreview
    unsafe fn capture_preview(
        &self,
        /* in */ image_format: CapturePreviewImageFormat,
        /* in */ image_stream: *mut *mut IStreamVTable,
        /* in */ handler: *mut *mut ICoreWebView2CapturePreviewCompletedHandlerVTable,
    ) -> HRESULT;

    /// Reload the current page.  This is similar to navigating to the URI of
    /// current top level document including all navigation events firing and
    /// respecting any entries in the HTTP cache.  But, the back or forward
    /// history are not modified.
    unsafe fn reload(&self) -> HRESULT;

    /// Post the specified webMessage to the top level document in this WebView.
    /// Runs the message event of the `window.chrome.webview` of the top-level
    /// document.  JavaScript in that document may subscribe and unsubscribe to
    /// the event using the following code.
    ///
    /// ```cpp
    /// window.chrome.webview.addEventListener('message', handler)
    /// window.chrome.webview.removeEventListener('message', handler)
    /// ```
    ///
    /// The event args is an instance of `MessageEvent`.  The
    /// `ICoreWebView2Settings::IsWebMessageEnabled` setting must be `TRUE` or
    /// this method fails with `E_INVALIDARG`.  The `data` property of the event
    /// arg is the `webMessage` string parameter parsed as a JSON string into a
    /// JavaScript object.  The `source` property of the event arg is a reference
    ///  to the `window.chrome.webview` object.  For information about sending
    /// messages from the HTML document in the WebView to the host, navigate to
    /// \[add_WebMessageReceived].  The message is sent asynchronously.  If a
    /// navigation occurs before the message is posted to the page, the message
    /// is not sent.
    ///
    /// \snippet ScenarioWebMessage.cpp WebMessageReceived
    unsafe fn post_web_message_as_json(
        &self,
        /* in */ web_message_as_json: LPCWSTR,
    ) -> HRESULT;

    /// Posts a message that is a simple string rather than a JSON string
    /// representation of a JavaScript object.  This behaves in exactly the same
    /// manner as `PostWebMessageAsJson`, but the `data` property of the event
    /// arg of the `window.chrome.webview` message is a string with the same
    /// value as `webMessageAsString`.  Use this instead of
    /// `PostWebMessageAsJson` if you want to communicate using simple strings
    /// rather than JSON objects.
    unsafe fn post_web_message_as_string(
        &self,
        /* in */ web_message_as_string: LPCWSTR,
    ) -> HRESULT;

    /// Add an event handler for the `WebMessageReceived` event.
    /// `WebMessageReceived` runs when the
    /// `ICoreWebView2Settings::IsWebMessageEnabled` setting is set and the
    /// top-level document of the WebView runs
    /// `window.chrome.webview.postMessage`.  The `postMessage` function is
    /// `void postMessage(object)` where object is any object supported by JSON
    /// conversion.
    ///
    /// \snippet assets\ScenarioWebMessage.html chromeWebView
    ///
    /// When `postMessage` is run, the `Invoke` method of the `handler` is run
    /// with the `object` parameter of the `postMessage` converted to a JSON
    /// string.
    ///
    /// \snippet ScenarioWebMessage.cpp WebMessageReceived
    unsafe fn add_web_message_received(
        &self,
        /* in */ handler: *mut *mut ICoreWebView2WebMessageReceivedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_WebMessageReceived`.
    unsafe fn remove_web_message_received(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// Runs an asynchronous `DevToolsProtocol` method.  For more information
    /// about available methods, navigate to
    /// \[DevTools Protocol Viewer\]\[GithubChromedevtoolsDevtoolsProtocolTot\]
    /// .  The `methodName` parameter is the full name of the method in the
    /// `{domain}.{method}` format.  The `parametersAsJson` parameter is a JSON
    /// formatted string containing the parameters for the corresponding method.
    /// The `Invoke` method of the `handler` is run when the method
    /// asynchronously completes.  `Invoke` is run with the return object of the
    /// method as a JSON string.
    ///
    /// \snippet ScriptComponent.cpp CallDevToolsProtocolMethod
    ///
    /// \[GithubChromedevtoolsDevtoolsProtocolTot\]: https://chromedevtools.github.io/devtools-protocol/tot "latest (tip-of-tree) protocol - Chrome DevTools Protocol | GitHub"
    unsafe fn call_dev_tools_protocol_method(
        &self,
        /* in */ method_name: LPCWSTR,
        /* in */ parameters_as_json: LPCWSTR,
        /* in */
        handler: *mut *mut ICoreWebView2CallDevToolsProtocolMethodCompletedHandlerVTable,
    ) -> HRESULT;

    /// The process ID of the browser process that hosts the WebView.
    unsafe fn get_browser_process_id(&self, /* out, retval */ value: *mut u32) -> HRESULT;

    /// `TRUE` if the WebView is able to navigate to a previous page in the
    /// navigation history.  If `CanGoBack` changes value, the `HistoryChanged`
    /// event runs.
    unsafe fn get_can_go_back(&self, /* out, retval */ can_go_back: *mut BOOL) -> HRESULT;

    /// `TRUE` if the WebView is able to navigate to a next page in the
    /// navigation history.  If `CanGoForward` changes value, the
    /// `HistoryChanged` event runs.
    unsafe fn get_can_go_forward(
        &self,
        /* out, retval */ can_go_forward: *mut BOOL,
    ) -> HRESULT;

    /// Navigates the WebView to the previous page in the navigation history.
    unsafe fn go_back(&self) -> HRESULT;

    /// Navigates the WebView to the next page in the navigation history.
    unsafe fn go_forward(&self) -> HRESULT;

    /// Get a DevTools Protocol event receiver that allows you to subscribe to a
    /// DevTools Protocol event.  The `eventName` parameter is the full name of
    /// the event in the format `{domain}.{event}`.  For more information about
    /// DevTools Protocol events description and event args, navigate to
    /// \[DevTools Protocol Viewer\]\[GithubChromedevtoolsDevtoolsProtocolTot\].
    ///
    /// \snippet ScriptComponent.cpp DevToolsProtocolEventReceived
    ///
    /// \[GithubChromedevtoolsDevtoolsProtocolTot\]: https://chromedevtools.github.io/devtools-protocol/tot "latest (tip-of-tree) protocol - Chrome DevTools Protocol | GitHub"
    unsafe fn get_dev_tools_protocol_event_receiver(
        &self,
        /* in */ event_name: LPCWSTR,
        /* out, retval */
        receiver: *mut *mut *mut ICoreWebView2DevToolsProtocolEventReceiverVTable,
    ) -> HRESULT;

    /// Stop all navigations and pending resource fetches.  Does not stop scripts.
    unsafe fn stop(&self) -> HRESULT;

    /// Add an event handler for the `NewWindowRequested` event.
    /// `NewWindowRequested` runs when content inside the WebView requests to
    /// open a new window, such as through `window.open`.  The app passes a
    /// target WebView that is considered the opened window.
    ///
    /// If a deferral is not taken on the event args, scripts that resulted in
    /// the new window that are requested are blocked until the event handler
    /// returns.  If a deferral is taken, then scripts are blocked until the
    /// deferral is completed or new window is set.
    ///
    /// For more details and considerations on the target WebView to be supplied
    /// at the opened window, see `ICoreWebView2NewWindowRequestedEventArgs::put_NewWindow`.
    ///
    /// \snippet AppWindow.cpp NewWindowRequested
    unsafe fn add_new_window_requested(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2NewWindowRequestedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_NewWindowRequested`.
    unsafe fn remove_new_window_requested(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// Add an event handler for the `DocumentTitleChanged` event.
    /// `DocumentTitleChanged` runs when the `DocumentTitle` property of the
    /// WebView changes and may run before or after the `NavigationCompleted`
    /// event.
    ///
    /// \snippet FileComponent.cpp DocumentTitleChanged
    unsafe fn add_document_title_changed(
        &self,
        /* in */
        event_handler: *mut *mut ICoreWebView2DocumentTitleChangedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_DocumentTitleChanged`.
    unsafe fn remove_document_title_changed(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// The title for the current top-level document.  If the document has no
    /// explicit title or is otherwise empty, a default that may or may not match
    ///  the URI of the document is used.
    unsafe fn get_document_title(&self, /* out, retval */ title: *mut LPWSTR) -> HRESULT;

    /// Add the provided host object to script running in the WebView with the
    /// specified name.  Host objects are exposed as host object proxies using
    /// `window.chrome.webview.hostObjects.{name}`.  Host object proxies are
    /// promises and resolves to an object representing the host object.  The
    /// promise is rejected if the app has not added an object with the name.
    /// When JavaScript code access a property or method of the object, a promise
    ///  is return, which resolves to the value returned from the host for the
    /// property or method, or rejected in case of error, for example, no
    /// property or method on the object or parameters are not valid.
    ///
    /// \> [!NOTE]\n\> While simple types, `IDispatch` and array are supported, generic
    /// `IUnknown`, `VT_DECIMAL`, or `VT_RECORD` variant is not supported.
    /// Remote JavaScript objects like callback functions are represented as an
    /// `VT_DISPATCH` `VARIANT` with the object implementing `IDispatch`.  The
    /// JavaScript callback method may be invoked using `DISPID_VALUE` for the
    /// `DISPID`.  Nested arrays are supported up to a depth of 3.  Arrays of by
    /// reference types are not supported. `VT_EMPTY` and `VT_NULL` are mapped
    /// into JavaScript as `null`.  In JavaScript, `null` and undefined are
    /// mapped to `VT_EMPTY`.
    ///
    /// Additionally, all host objects are exposed as
    /// `window.chrome.webview.hostObjects.sync.{name}`.  Here the host objects
    /// are exposed as synchronous host object proxies. These are not promises
    /// and function runtimes or property access synchronously block running
    /// script waiting to communicate cross process for the host code to run.
    /// Accordingly the result may have reliability issues and it is recommended
    /// that you use the promise-based asynchronous
    /// `window.chrome.webview.hostObjects.{name}` API.
    ///
    /// Synchronous host object proxies and asynchronous host object proxies may
    /// both use a proxy to the same host object.  Remote changes made by one
    /// proxy propagates to any other proxy of that same host object whether
    /// the other proxies and synchronous or asynchronous.
    ///
    /// While JavaScript is blocked on a synchronous run to native code, that
    /// native code is unable to run back to JavaScript.  Attempts to do so fail
    ///  with `HRESULT_FROM_WIN32(ERROR_POSSIBLE_DEADLOCK)`.
    ///
    /// Host object proxies are JavaScript Proxy objects that intercept all
    /// property get, property set, and method invocations. Properties or methods
    ///  that are a part of the Function or Object prototype are run locally.
    /// Additionally any property or method in the
    /// `chrome.webview.hostObjects.options.forceLocalProperties`
    /// array are also run locally.  This defaults to including optional methods
    /// that have meaning in JavaScript like `toJSON` and `Symbol.toPrimitive`.
    /// Add more to the array as required.
    ///
    /// The `chrome.webview.hostObjects.cleanupSome` method performs a best
    /// effort garbage collection on host object proxies.
    ///
    /// Host object proxies additionally have the following methods which run
    /// locally.
    ///
    /// Method name | Details
    /// ---|---
    ///`applyHostFunction`, `getHostProperty`, `setHostProperty` | Perform a method invocation, property get, or property set on the host object. Use the methods to explicitly force a method or property to run remotely if a conflicting local method or property exists.  For instance, `proxy.toString()` runs the local `toString` method on the proxy object. But proxy.applyHostFunction('toString') runs `toString` on the host proxied object instead.
    ///`getLocalProperty`, `setLocalProperty` | Perform property get, or property set locally.  Use the methods to force getting or setting a property on the host object proxy rather than on the host object it represents. For instance, `proxy.unknownProperty` gets the property named `unknownProperty` from the host proxied object.  But proxy.getLocalProperty('unknownProperty') gets the value of the property `unknownProperty` on the proxy object.
    ///`sync` | Asynchronous host object proxies expose a sync method which returns a promise for a synchronous host object proxy for the same host object.  For example, `chrome.webview.hostObjects.sample.methodCall()` returns an asynchronous host object proxy.  Use the `sync` method to obtain a synchronous host object proxy instead: `const syncProxy = await chrome.webview.hostObjects.sample.methodCall().sync()`.
    ///`async` | Synchronous host object proxies expose an async method which blocks and returns an asynchronous host object proxy for the same host object.  For example, `chrome.webview.hostObjects.sync.sample.methodCall()` returns a synchronous host object proxy.  Running the `async` method on this blocks and then returns an asynchronous host object proxy for the same host object: `const asyncProxy = chrome.webview.hostObjects.sync.sample.methodCall().async()`.
    ///`then` | Asynchronous host object proxies have a `then` method.  Allows proxies to be awaitable.  `then` returns a promise that resolves with a representation of the host object.  If the proxy represents a JavaScript literal, a copy of that is returned locally.  If the proxy represents a function, a non-awaitable proxy is returned.  If the proxy represents a JavaScript object with a mix of literal properties and function properties, the a copy of the object is returned with some properties as host object proxies.
    ///
    /// All other property and method invocations (other than the above Remote
    /// object proxy methods, `forceLocalProperties` list, and properties on
    /// Function and Object prototypes) are run remotely.  Asynchronous host
    /// object proxies return a promise representing asynchronous completion of
    /// remotely invoking the method, or getting the property.  The promise
    /// resolves after the remote operations complete and the promises resolve to
    ///  the resulting value of the operation.  Synchronous host object proxies
    /// work similarly, but block running JavaScript and wait for the remote
    /// operation to complete.
    ///
    /// Setting a property on an asynchronous host object proxy works slightly
    /// differently.  The set returns immediately and the return value is the
    /// value that is set.  This is a requirement of the JavaScript Proxy object.
    /// If you need to asynchronously wait for the property set to complete, use
    /// the `setHostProperty` method which returns a promise as described above.
    /// Synchronous object property set property synchronously blocks until the
    /// property is set.
    ///
    /// For example, suppose you have a COM object with the following interface.
    ///
    /// \snippet HostObjectSample.idl AddHostObjectInterface
    ///
    /// Add an instance of this interface into your JavaScript with
    /// `AddHostObjectToScript`.  In this case, name it `sample`.
    ///
    /// \snippet ScenarioAddHostObject.cpp AddHostObjectToScript
    ///
    /// In the HTML document, use the COM object using
    /// `chrome.webview.hostObjects.sample`.
    ///
    /// \snippet assets\ScenarioAddHostObject.html HostObjectUsage
    ///
    /// Exposing host objects to script has security risk.  For more information
    /// about best practices, navigate to
    /// \[Best practices for developing secure WebView2 applications\]\[MicrosoftEdgeWebview2ConceptsSecurity\].
    ///
    /// \[MicrosoftEdgeWebview2ConceptsSecurity\]: /microsoft-edge/webview2/concepts/security "Best practices for developing secure WebView2 applications | Microsoft Docs"
    unsafe fn add_host_object_to_script(
        &self,
        /* in */ name: LPCWSTR,
        /* in */ object: *mut VARIANT,
    ) -> HRESULT;

    /// Remove the host object specified by the name so that it is no longer
    /// accessible from JavaScript code in the WebView.  While new access
    /// attempts are denied, if the object is already obtained by JavaScript code
    /// in the WebView, the JavaScript code continues to have access to that
    /// object.   Run this method for a name that is already removed or never
    /// added fails.
    unsafe fn remove_host_object_from_script(&self, /* in */ name: LPCWSTR) -> HRESULT;

    /// Opens the DevTools window for the current document in the WebView. Does
    /// nothing if run when the DevTools window is already open.
    unsafe fn open_dev_tools_window(&self) -> HRESULT;

    /// Add an event handler for the `ContainsFullScreenElementChanged` event.
    /// `ContainsFullScreenElementChanged` triggers when the
    /// `ContainsFullScreenElement` property changes.  An HTML element inside the
    /// WebView may enter fullscreen to the size of the WebView or leave
    /// fullscreen.  This event is useful when, for example, a video element
    /// requests to go fullscreen.  The listener of
    /// `ContainsFullScreenElementChanged` may resize the WebView in response.
    ///
    /// \snippet AppWindow.cpp ContainsFullScreenElementChanged
    unsafe fn add_contains_full_screen_element_changed(
        &self,
        /* in */
        event_handler: *mut *mut ICoreWebView2ContainsFullScreenElementChangedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with
    /// `add_ContainsFullScreenElementChanged`.
    unsafe fn remove_contains_full_screen_element_changed(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// Indicates if the WebView contains a fullscreen HTML element.
    unsafe fn get_contains_full_screen_element(
        &self,
        /* out, retval */ contains_full_screen_element: *mut BOOL,
    ) -> HRESULT;

    /// Add an event handler for the `WebResourceRequested` event.
    /// `WebResourceRequested` runs when the WebView is performing a URL request
    /// to a matching URL and resource context filter that was added with
    /// `AddWebResourceRequestedFilter`.  At least one filter must be added for
    /// the event to run.
    ///
    /// The web resource requested may be blocked until the event handler returns
    /// if a deferral is not taken on the event args.  If a deferral is taken,
    /// then the web resource requested is blocked until the deferral is
    /// completed.
    ///
    /// If the method is run in add_NewWindowRequested handler it should be called
    /// after the new window is set. For more details see `ICoreWebView2NewWindowRequestedEventArgs::put_NewWindow`.
    ///
    /// Currently this only supports file, http, and https URI schemes.
    ///
    /// \snippet SettingsComponent.cpp WebResourceRequested0
    /// \snippet SettingsComponent.cpp WebResourceRequested1
    unsafe fn add_web_resource_requested(
        &self,
        /* in */
        event_handler: *mut *mut ICoreWebView2WebResourceRequestedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_WebResourceRequested`.
    unsafe fn remove_web_resource_requested(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// Adds a URI and resource context filter for the `WebResourceRequested`
    /// event.  A web resource request with a resource context that matches this
    /// filter's resource context and a URI that matches this filter's URI
    /// wildcard string will be raised via the `WebResourceRequested` event.
    ///
    /// The `uri` parameter value is a wildcard string matched against the URI
    /// of the web resource request. This is a glob style
    /// wildcard string in which a `*` matches zero or more characters and a `?`
    /// matches exactly one character.
    /// These wildcard characters can be escaped using a backslash just before
    /// the wildcard character in order to represent the literal `*` or `?`.
    ///
    /// The matching occurs over the URI as a whole string and not limiting
    /// wildcard matches to particular parts of the URI.
    /// The wildcard filter is compared to the URI after the URI has been
    /// normalized, any URI fragment has been removed, and non-ASCII hostnames
    /// have been converted to punycode.
    ///
    /// Specifying a `nullptr` for the uri is equivalent to an empty string which
    /// matches no URIs.
    ///
    /// For more information about resource context filters, navigate to
    /// [COREWEBVIEW2_WEB_RESOURCE_CONTEXT].
    ///
    /// | URI Filter String | Request URI | Match | Notes |
    /// | ---- | ---- | ---- | ---- |
    /// | `*` | https://contoso.com/a/b/c | Yes | A single * will match all URIs |
    /// | `*://contoso.com/*` | https://contoso.com/a/b/c | Yes | Matches everything in contoso.com across all schemes |
    /// | `*://contoso.com/*` | https://example.com/?https://contoso.com/ | Yes | But also matches a URI with just the same text anywhere in the URI |
    /// | `example` | https://contoso.com/example | No | The filter does not perform partial matches |
    /// | `*example` | https://contoso.com/example | Yes | The filter matches across URI parts  |
    /// | `*example` | https://contoso.com/path/?example | Yes | The fitler matches across URI parts |
    /// | `*example` | https://contoso.com/path/?query#example | No | The filter is matched against the URI with no fragment |
    /// | `*example` | https://example | No | The URI is normalzied before filter matching so the actual URI used for comparison is https://example.com/ |
    /// | `*example/` | https://example | Yes | Just like above, but this time the filter ends with a / just like the normalized URI |
    /// | https://xn--qei.example/ | https://&#x2764;.example/ | Yes | Non-ASCII hostnames are normalized to punycode before wildcard comparison |
    /// | https://&#x2764;.example/ | https://xn--qei.example/ | No | Non-ASCII hostnames are normalized to punycode before wildcard comparison |
    unsafe fn add_web_resource_requested_filter(
        &self,
        /* in */ uri: LPCWSTR,
        /* in */ resource_context: WebResourceContext,
    ) -> HRESULT;

    /// Removes a matching WebResource filter that was previously added for the
    /// `WebResourceRequested` event.  If the same filter was added multiple
    /// times, then it must be removed as many times as it was added for the
    /// removal to be effective.  Returns `E_INVALIDARG` for a filter that was
    /// never added.
    unsafe fn remove_web_resource_requested_filter(
        &self,
        /* in */ uri: LPCWSTR,
        /* in */ resource_context: WebResourceContext,
    ) -> HRESULT;

    /// Add an event handler for the `WindowCloseRequested` event.
    /// `WindowCloseRequested` triggers when content inside the WebView
    /// requested to close the window, such as after `window.close` is run.  The
    /// app should close the WebView and related app window if that makes sense
    /// to the app.
    ///
    /// \snippet AppWindow.cpp WindowCloseRequested
    unsafe fn add_window_close_requested(
        &self,
        /* in */
        event_handler: *mut *mut ICoreWebView2WindowCloseRequestedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_WindowCloseRequested`.
    unsafe fn remove_window_close_requested(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;
}

/// A continuation of the ICoreWebView2 interface.
#[com_interface("9E8F0CF8-E670-4B5E-B2BC-73E061E3184C")]
pub trait ICoreWebView2_2: ICoreWebView2 {
    /// Add an event handler for the WebResourceResponseReceived event.
    /// WebResourceResponseReceived is raised when the WebView receives the
    /// response for a request for a web resource (any URI resolution performed by
    /// the WebView; such as HTTP/HTTPS, file and data requests from redirects,
    /// navigations, declarations in HTML, implicit favicon lookups, and fetch API
    /// usage in the document). The host app can use this event to view the actual
    /// request and response for a web resource. There is no guarantee about the
    /// order in which the WebView processes the response and the host app's
    /// handler runs. The app's handler will not block the WebView from processing
    /// the response.
    /// \snippet ScenarioAuthentication.cpp WebResourceResponseReceived
    unsafe fn add_web_resource_response_received(
        &self,
        /* in */
        event_handler: *mut *mut ICoreWebView2WebResourceResponseReceivedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with
    /// add_WebResourceResponseReceived.
    unsafe fn remove_web_resource_response_received(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// Navigates using a constructed WebResourceRequest object. This lets you
    /// provide post data or additional request headers during navigation.
    /// The headers in the WebResourceRequest override headers
    /// added by WebView2 runtime except for Cookie headers.
    /// Method can only be either "GET" or "POST". Provided post data will only
    /// be sent only if the method is "POST" and the uri scheme is HTTP(S).
    /// \snippet ScenarioNavigateWithWebResourceRequest.cpp NavigateWithWebResourceRequest
    unsafe fn navigate_with_web_resource_request(
        &self,
        /* in */ request: *mut *mut ICoreWebView2WebResourceRequestVTable,
    ) -> HRESULT;

    /// Add an event handler for the DOMContentLoaded event.
    /// DOMContentLoaded is raised when the initial html document has been parsed.
    /// This aligns with the the document's DOMContentLoaded event in html.
    ///
    /// \snippet ScenarioDOMContentLoaded.cpp DOMContentLoaded
    unsafe fn add_domcontent_loaded(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2DOMContentLoadedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with add_DOMContentLoaded.
    unsafe fn remove_domcontent_loaded(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// Gets the cookie manager object associated with this ICoreWebView2.
    /// See ICoreWebView2CookieManager.
    ///
    /// \snippet ScenarioCookieManagement.cpp CookieManager
    unsafe fn get_cookie_manager(
        &self,
        /* out, retval */ cookie_manager: *mut *mut *mut ICoreWebView2CookieManagerVTable,
    ) -> HRESULT;

    /// Exposes the CoreWebView2Environment used to create this CoreWebView2.
    unsafe fn get_environment(
        &self,
        /* out, retval */ environment: *mut *mut *mut ICoreWebView2EnvironmentVTable,
    ) -> HRESULT;
}

/// A continuation of the ICoreWebView2_2 interface.
#[com_interface("A0D6DF20-3B92-416D-AA0C-437A9C727857")]
pub trait ICoreWebView2_3: ICoreWebView2_2 {
    /// An app may call the `TrySuspend` API to have the WebView2 consume less memory.
    /// This is useful when a Win32 app becomes invisible, or when a Universal Windows
    /// Platform app is being suspended, during the suspended event handler before completing
    /// the suspended event.
    /// The CoreWebView2Controller's IsVisible property must be false when the API is called.
    /// Otherwise, the API fails with `HRESULT_FROM_WIN32(ERROR_INVALID_STATE)`.
    /// Suspending is similar to putting a tab to sleep in the Edge browser. Suspending pauses
    /// WebView script timers and animations, minimizes CPU usage for the associated
    /// browser renderer process and allows the operating system to reuse the memory that was
    /// used by the renderer process for other processes.
    /// Note that Suspend is best effort and considered completed successfully once the request
    /// is sent to browser renderer process. If there is a running script, the script will continue
    /// to run and the renderer process will be suspended after that script is done.
    /// See [Sleeping Tabs FAQ](https://techcommunity.microsoft.com/t5/articles/sleeping-tabs-faq/m-p/1705434)
    /// for conditions that might prevent WebView from being suspended. In those situations,
    /// the completed handler will be invoked with isSuccessful as false and errorCode as S_OK.
    /// The WebView will be automatically resumed when it becomes visible. Therefore, the
    /// app normally does not have to call `Resume` explicitly.
    /// The app can call `Resume` and then `TrySuspend` periodically for an invisible WebView so that
    /// the invisible WebView can sync up with latest data and the page ready to show fresh content
    /// when it becomes visible.
    /// All WebView APIs can still be accessed when a WebView is suspended. Some APIs like Navigate
    /// will auto resume the WebView. To avoid unexpected auto resume, check `IsSuspended` property
    /// before calling APIs that might change WebView state.
    ///
    /// \snippet ViewComponent.cpp ToggleIsVisibleOnMinimize
    ///
    /// \snippet ViewComponent.cpp Suspend
    ///
    unsafe fn try_suspend(
        &self,
        /* in */ handler: *mut *mut ICoreWebView2TrySuspendCompletedHandlerVTable,
    ) -> HRESULT;

    /// Resumes the WebView so that it resumes activities on the web page.
    /// This API can be called while the WebView2 controller is invisible.
    /// The app can interact with the WebView immediately after `Resume`.
    /// WebView will be automatically resumed when it becomes visible.
    ///
    /// \snippet ViewComponent.cpp ToggleIsVisibleOnMinimize
    ///
    /// \snippet ViewComponent.cpp Resume
    ///
    unsafe fn resume(&self) -> HRESULT;

    /// Whether WebView is suspended.
    /// `TRUE` when WebView is suspended, from the time when TrySuspend has completed
    ///  successfully until WebView is resumed.
    unsafe fn get_is_suspended(&self, /* out, retval */ is_suspended: *mut BOOL) -> HRESULT;

    /// Sets a mapping between a virtual host name and a folder path to make available to web sites
    /// via that host name.
    ///
    /// After setting the mapping, documents loaded in the WebView can use HTTP or HTTPS URLs at
    /// the specified host name specified by hostName to access files in the local folder specified
    /// by folderPath.
    ///
    /// This mapping applies to both top-level document and iframe navigations as well as subresource
    /// references from a document. This also applies to web workers including dedicated/shared worker
    /// and service worker, for loading either worker scripts or subresources
    /// (importScripts(), fetch(), XHR, etc.) issued from within a worker.
    /// For virtual host mapping to work with service worker, please keep the virtual host name
    /// mappings consistent among all WebViews sharing the same browser instance. As service worker
    /// works independently of WebViews, we merge mappings from all WebViews when resolving virutal
    /// host name, inconsistent mappings between WebViews would lead unexpected bevavior.
    ///
    /// Due to a current implementation limitation, media files accessed using virtual host name can be
    /// very slow to load.
    /// As the resource loaders for the current page might have already been created and running,
    /// changes to the mapping might not be applied to the current page and a reload of the page is
    /// needed to apply the new mapping.
    ///
    /// Both absolute and relative paths are supported for folderPath. Relative paths are interpreted
    /// as relative to the folder where the exe of the app is in.
    ///
    /// accessKind specifies the level of access to resources under the virtual host from other sites.
    ///
    /// For example, after calling
    /// ```cpp
    ///    SetVirtualHostNameToFolderMapping(
    ///        L"appassets.example", L"assets",
    ///        COREWEBVIEW2_HOST_RESOURCE_ACCESS_KIND_DENY);
    /// ```
    /// navigating to `https://appassets.example/my-local-file.html` will
    /// show content from my-local-file.html in the assets subfolder located on disk under the same
    /// path as the app's executable file.
    ///
    /// You should typically choose virtual host names that are never used by real sites.
    /// If you own a domain such as example.com, another option is to use a subdomain reserved for
    /// the app (like my-app.example.com).
    ///
    /// [RFC 6761](https://tools.ietf.org/html/rfc6761) has reserved several special-use domain
    /// names that are guaranteed to not be used by real sites (for example, .example, .test, and
    /// .invalid.)
    ///
    /// Apps should use distinct domain names when mapping folder from different sources that
    /// should be isolated from each other. For instance, the app might use app-file.example for
    /// files that ship as part of the app, and book1.example might be used for files containing
    /// books from a less trusted source that were previously downloaded and saved to the disk by
    /// the app.
    ///
    /// The host name used in the APIs is canonicalized using Chromium's host name parsing logic
    /// before being used internally.
    ///
    /// All host names that are canonicalized to the same string are considered identical.
    /// For example, `EXAMPLE.COM` and `example.com` are treated as the same host name.
    /// An international host name and its Punycode-encoded host name are considered the same host
    /// name. There is no DNS resolution for host name and the trailing '.' is not normalized as
    /// part of canonicalization.
    ///
    /// Therefore `example.com` and `example.com.` are treated as different host names. Similarly,
    /// `virtual-host-name` and `virtual-host-name.example.com` are treated as different host names
    /// even if the machine has a DNS suffix of `example.com`.
    ///
    /// Specify the minimal cross-origin access necessary to run the app. If there is not a need to
    /// access local resources from other origins, use COREWEBVIEW2_HOST_RESOURCE_ACCESS_KIND_DENY.
    ///
    /// \snippet AppWindow.cpp AddVirtualHostNameToFolderMapping
    ///
    /// \snippet AppWindow.cpp LocalUrlUsage
    unsafe fn set_virtual_host_name_to_folder_mapping(
        &self,
        /* in */ host_name: LPCWSTR,
        /* in */ folder_path: LPCWSTR,
        /* in */ access_kind: HostResourceAccessKind,
    ) -> HRESULT;

    /// Clears a host name mapping for local folder that was added by `SetVirtualHostNameToFolderMapping`.
    unsafe fn clear_virtual_host_name_to_folder_mapping(
        &self,
        /* in */ host_name: LPCWSTR,
    ) -> HRESULT;
}

/// A continuation of the ICoreWebView2_3 interface to support FrameCreated and
/// DownloadStarting events.
#[com_interface("20d02d59-6df2-42dc-bd06-f98a694b1302")]
pub trait ICoreWebView2_4: ICoreWebView2_3 {
    /// Raised when a new iframe is created. Use the
    /// CoreWebView2Frame.add_Destroyed to listen for when this iframe goes
    /// away.
    unsafe fn add_frame_created(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2FrameCreatedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with add_FrameCreated.
    unsafe fn remove_frame_created(&self, /* in */ token: EventRegistrationToken) -> HRESULT;

    /// Add an event handler for the `DownloadStarting` event. This event is
    /// raised when a download has begun, blocking the default download dialog,
    /// but not blocking the progress of the download.
    ///
    /// The host can choose to cancel a download, change the result file path,
    /// and hide the default download dialog.
    /// If the host chooses to cancel the download, the download is not saved, no
    /// dialog is shown, and the state is changed to
    /// COREWEBVIEW2_DOWNLOAD_STATE_INTERRUPTED with interrupt reason
    /// COREWEBVIEW2_DOWNLOAD_INTERRUPT_REASON_USER_CANCELED. Otherwise, the
    /// download is saved to the default path after the event completes,
    /// and default download dialog is shown if the host did not choose to hide it.
    /// The host can change the visibility of the download dialog using the
    /// `Handled` property. If the event is not handled, downloads complete
    /// normally with the default dialog shown.
    ///
    /// \snippet ScenarioCustomDownloadExperience.cpp DownloadStarting
    unsafe fn add_download_starting(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2DownloadStartingEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_DownloadStarting`.
    unsafe fn remove_download_starting(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;
}

/// A continuation of the ICoreWebView2_4 interface to support ClientCertificateRequested
/// event.
#[com_interface("bedb11b8-d63c-11eb-b8bc-0242ac130003")]
pub trait ICoreWebView2_5: ICoreWebView2_4 {
    /// Add an event handler for the ClientCertificateRequested event.
    /// The ClientCertificateRequested event is raised when the WebView2
    /// is making a request to an HTTP server that needs a client certificate
    /// for HTTP authentication.
    /// Read more about HTTP client certificates at
    /// [RFC 8446 The Transport Layer Security (TLS) Protocol Version 1.3](https://tools.ietf.org/html/rfc8446).
    ///
    /// With this event you have several options for responding to client certificate requests:
    ///
    /// Scenario                                                   | Handled | Cancel | SelectedCertificate
    /// ---------------------------------------------------------- | ------- | ------ | -------------------
    /// Respond to server with a certificate                       | True    | False  | MutuallyTrustedCertificate value
    /// Respond to server without certificate                      | True    | False  | null
    /// Display default client certificate selection dialog prompt | False   | False  | n/a
    /// Cancel the request                                         | n/a     | True   | n/a
    ///
    /// If you don't handle the event, WebView2 will
    /// show the default client certificate selection dialog prompt to user.
    ///
    /// \snippet SettingsComponent.cpp ClientCertificateRequested1
    /// \snippet ScenarioClientCertificateRequested.cpp ClientCertificateRequested2
    unsafe fn add_client_certificate_requested(
        &self,
        /* in */
        event_handler: *mut *mut ICoreWebView2ClientCertificateRequestedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with add_ClientCertificateRequested.
    unsafe fn remove_client_certificate_requested(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;
}

/// The caller implements this interface to receive the TrySuspend result.
#[com_interface("00F206A7-9D17-4605-91F6-4E8E4DE192E3")]
pub trait ICoreWebView2TrySuspendCompletedHandler: IUnknown {
    /// Provides the result of the TrySuspend operation.
    /// See [Sleeping Tabs FAQ](https://techcommunity.microsoft.com/t5/articles/sleeping-tabs-faq/m-p/1705434)
    /// for conditions that might prevent WebView from being suspended. In those situations,
    /// isSuccessful will be false and errorCode is S_OK.
    unsafe fn invoke(
        &self,
        /* in */ error_code: HRESULT,
        /* in */ is_successful: BOOL,
    ) -> HRESULT;
}

/// The owner of the `CoreWebView2` object that provides support for resizing,
/// showing and hiding, focusing, and other functionality related to
/// windowing and composition.  The `CoreWebView2Controller` owns the
/// `CoreWebView2`, and if all references to the `CoreWebView2Controller` go
/// away, the WebView is closed.
#[com_interface("4d00c0d1-9434-4eb6-8078-8697a560334f")]
pub trait ICoreWebView2Controller: IUnknown {
    /// The `IsVisible` property determines whether to show or hide the WebView2.
    ///   If `IsVisible` is set to `FALSE`, the WebView2 is transparent and is
    /// not rendered.   However, this does not affect the window containing the
    /// WebView2 (the `HWND` parameter that was passed to
    /// `CreateCoreWebView2Controller`).  If you want that window to disappear
    /// too, run `ShowWindow` on it directly in addition to modifying the
    /// `IsVisible` property.  WebView2 as a child window does not get window
    /// messages when the top window is minimized or restored.  For performance
    /// reason, developer should set `IsVisible` property of the WebView to
    /// `FALSE` when the app window is minimized and back to `TRUE` when app
    /// window is restored.  App window does this by handling
    /// `SC_MINIMIZE and SC_RESTORE` command upon receiving `WM_SYSCOMMAND`
    /// message.
    ///
    /// \snippet ViewComponent.cpp ToggleIsVisible
    unsafe fn get_is_visible(&self, /* out, retval */ is_visible: *mut BOOL) -> HRESULT;

    /// Sets the `IsVisible` property.
    ///
    /// \snippet ViewComponent.cpp ToggleIsVisibleOnMinimize
    unsafe fn put_is_visible(&self, /* in */ is_visible: BOOL) -> HRESULT;

    /// The WebView bounds. Bounds are relative to the parent `HWND`.  The app
    /// has two ways to position a WebView.
    ///
    /// *   Create a child `HWND` that is the WebView parent `HWND`.  Position
    ///     the window where the WebView should be.  Use `(0, 0)` for the
    ///     top-left corner (the offset) of the `Bounds` of the WebView.
    /// *   Use the top-most window of the app as the WebView parent HWND.  For
    ///     example, to position WebView correctly in the app, set the top-left
    ///     corner of the Bound of the WebView.
    ///
    /// The values of `Bounds` are limited by the coordinate space of the host.
    unsafe fn get_bounds(&self, /* out, retval */ bounds: *mut RECT) -> HRESULT;

    /// Sets the `Bounds` property.
    ///
    /// \snippet ViewComponent.cpp ResizeWebView
    unsafe fn put_bounds(&self, /* in */ bounds: RECT) -> HRESULT;

    /// The zoom factor for the WebView.
    ///
    /// \> [!NOTE]\n\> Changing zoom factor may cause `window.innerWidth`,
    /// `window.innerHeight`, both, and page layout to change.  A zoom factor
    /// that is applied by the host by running `ZoomFactor` becomes the new
    /// default zoom for the WebView.  The zoom factor applies across navigations
    /// and is the zoom factor WebView is returned to when the user chooses
    /// Ctrl+0.  When the zoom factor is changed by the user (resulting in
    /// the app receiving `ZoomFactorChanged`), that zoom applies only for the
    /// current page.  Any user applied zoom is only for the current page and is
    /// reset on a navigation.  Specifying a `zoomFactor` less than or equal to
    /// `0` is not allowed.  WebView also has an internal supported zoom factor
    /// range.  When a specified zoom factor is out of that range, it is
    /// normalized to be within the range, and a `ZoomFactorChanged` event is
    /// triggered for the real applied zoom factor.  When the range normalization
    /// happens, the `ZoomFactor` property reports the zoom factor specified
    /// during the previous modification of the `ZoomFactor` property until the
    /// `ZoomFactorChanged` event is received after WebView applies the
    /// normalized zoom factor.
    unsafe fn get_zoom_factor(&self, /* out, retval */ zoom_factor: *mut f64) -> HRESULT;

    /// Sets the `ZoomFactor` property.
    unsafe fn put_zoom_factor(&self, /* in */ zoom_factor: f64) -> HRESULT;

    /// Adds an event handler for the `ZoomFactorChanged` event.
    /// `ZoomFactorChanged` runs when the `ZoomFactor` property of the WebView
    /// changes.  The event may run because the `ZoomFactor` property was
    /// modified, or due to the user manually modifying the zoom.  When it is
    /// modified using the `ZoomFactor` property, the internal zoom factor is
    /// updated immediately and no `ZoomFactorChanged` event is triggered.
    /// WebView associates the last used zoom factor for each site.  It is
    /// possible for the zoom factor to change when navigating to a different
    /// page.  When the zoom factor changes due to a navigation change, the
    /// `ZoomFactorChanged` event runs right after the `ContentLoading` event.
    ///
    /// \snippet ViewComponent.cpp ZoomFactorChanged
    unsafe fn add_zoom_factor_changed(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2ZoomFactorChangedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_ZoomFactorChanged`.
    unsafe fn remove_zoom_factor_changed(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// Updates `Bounds` and `ZoomFactor` properties at the same time.  This
    /// operation is atomic from the perspective of the host.  After returning
    /// from this function, the `Bounds` and `ZoomFactor` properties are both
    /// updated if the function is successful, or neither is updated if the
    /// function fails.  If `Bounds` and `ZoomFactor` are both updated by the
    /// same scale (for example, `Bounds` and `ZoomFactor` are both doubled),
    /// then the page does not display a change in `window.innerWidth` or
    /// `window.innerHeight` and the WebView renders the content at the new size
    /// and zoom without intermediate renderings.  This function also updates
    /// just one of `ZoomFactor` or `Bounds` by passing in the new value for one
    /// and the current value for the other.
    ///
    /// \snippet ViewComponent.cpp SetBoundsAndZoomFactor
    unsafe fn set_bounds_and_zoom_factor(
        &self,
        /* in */ bounds: RECT,
        /* in */ zoom_factor: f64,
    ) -> HRESULT;

    /// Moves focus into WebView.  WebView gets focus and focus is set to
    /// correspondent element in the page hosted in the WebView.  For
    /// Programmatic reason, focus is set to previously focused element or the
    /// default element if no previously focused element exists.  For `Next`
    /// reason, focus is set to the first element.  For `Previous` reason, focus
    /// is set to the last element.  WebView changes focus through user
    /// interaction including selecting into a WebView or Tab into it.  For
    /// tabbing, the app runs MoveFocus with Next or Previous to align with Tab
    /// and Shift+Tab respectively when it decides the WebView is the next
    /// element that may exist in a tab.  Or, the app runs `IsDialogMessage`
    /// as part of the associated message loop to allow the platform to auto
    /// handle tabbing.  The platform rotates through all windows with
    /// `WS_TABSTOP`.  When the WebView gets focus from `IsDialogMessage`, it is
    /// internally put the focus on the first or last element for tab and
    /// Shift+Tab respectively.
    ///
    /// \snippet App.cpp MoveFocus0
    ///
    /// \snippet ControlComponent.cpp MoveFocus1
    ///
    /// \snippet ControlComponent.cpp MoveFocus2
    unsafe fn move_focus(&self, /* in */ reason: MoveFocusReason) -> HRESULT;

    /// Adds an event handler for the `MoveFocusRequested` event.
    /// `MoveFocusRequested` runs when user tries to tab out of the WebView.  The
    /// focus of the WebView has not changed when this event is run.
    ///
    /// \snippet ControlComponent.cpp MoveFocusRequested
    unsafe fn add_move_focus_requested(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2MoveFocusRequestedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Removes an event handler previously added with `add_MoveFocusRequested`.
    unsafe fn remove_move_focus_requested(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// Adds an event handler for the `GotFocus` event.  `GotFocus` runs when
    /// WebView has focus.
    unsafe fn add_got_focus(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2FocusChangedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Removes an event handler previously added with `add_GotFocus`.
    unsafe fn remove_got_focus(&self, /* in */ token: EventRegistrationToken) -> HRESULT;

    /// Adds an event handler for the `LostFocus` event.  `LostFocus` runs when
    /// WebView loses focus.  In the case where `MoveFocusRequested` event is
    /// run, the focus is still on WebView when `MoveFocusRequested` event runs.
    /// `LostFocus` only runs afterwards when code of the app or default action
    /// of `MoveFocusRequested` event set focus away from WebView.
    unsafe fn add_lost_focus(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2FocusChangedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Removes an event handler previously added with `add_LostFocus`.
    unsafe fn remove_lost_focus(&self, /* in */ token: EventRegistrationToken) -> HRESULT;

    /// Adds an event handler for the `AcceleratorKeyPressed` event.
    /// `AcceleratorKeyPressed` runs when an accelerator key or key combo is
    /// pressed or released while the WebView is focused.  A key is considered an
    ///  accelerator if either of the following conditions are true.
    ///
    /// *   Ctrl or Alt is currently being held.
    /// *   The pressed key does not map to a character.
    ///
    /// A few specific keys are never considered accelerators, such as Shift.
    /// The `Escape` key is always considered an accelerator.
    ///
    /// Auto-repeated key events caused by holding the key down also triggers
    /// this event.  Filter out the auto-repeated key events by verifying the
    /// `KeyEventLParam` or `PhysicalKeyStatus` event args.
    ///
    /// In windowed mode, the event handler is run synchronously.  Until you
    /// run `Handled()` on the event args or the event handler returns, the
    /// browser process is blocked and outgoing cross-process COM requests fail
    /// with `RPC_E_CANTCALLOUT_ININPUTSYNCCALL`.  All `CoreWebView2` API methods
    /// work, however.
    ///
    /// In windowless mode, the event handler is run asynchronously.  Further
    /// input do not reach the browser until the event handler returns or
    /// `Handled()` is run, but the browser process is not blocked, and outgoing
    /// COM requests work normally.
    ///
    /// It is recommended to run `Handled(TRUE)` as early as are able to know
    /// that you want to handle the accelerator key.
    ///
    /// \snippet ControlComponent.cpp AcceleratorKeyPressed
    unsafe fn add_accelerator_key_pressed(
        &self,
        /* in */
        event_handler: *mut *mut ICoreWebView2AcceleratorKeyPressedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Removes an event handler previously added with
    /// `add_AcceleratorKeyPressed`.
    unsafe fn remove_accelerator_key_pressed(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// The parent window provided by the app that this WebView is using to
    /// render content.  This API initially returns the window passed into
    /// `CreateCoreWebView2Controller`.
    unsafe fn get_parent_window(&self, /* out, retval */ parent_window: *mut HWND) -> HRESULT;

    /// Sets the parent window for the WebView.  This causes the WebView to
    /// re-parent the main WebView window to the newly provided window.
    unsafe fn put_parent_window(&self, /* in */ parent_window: HWND) -> HRESULT;

    /// This is a notification separate from `Bounds` that tells WebView that the
    ///  main WebView parent (or any ancestor) `HWND` moved.  This is needed
    /// for accessibility and certain dialogs in WebView to work correctly.
    ///
    /// \snippet ViewComponent.cpp NotifyParentWindowPositionChanged
    unsafe fn notify_parent_window_position_changed(&self) -> HRESULT;

    /// Closes the WebView and cleans up the underlying browser instance.
    /// Cleaning up the browser instance releases the resources powering the
    /// WebView.  The browser instance is shut down if no other WebViews are
    /// using it.
    ///
    /// After running `Close`, most methods will fail and event handlers stop
    /// running.  Specifically, the WebView releases the associated references to
    /// any associated event handlers when `Close` is run.
    ///
    /// `Close` is implicitly run when the `CoreWebView2Controller` loses the
    /// final reference and is destructed.  But it is best practice to
    /// explicitly run `Close` to avoid any accidental cycle of references
    /// between the WebView and the app code.  Specifically, if you capture a
    /// reference to the WebView in an event handler you create a reference cycle
    /// between the WebView and the event handler.  Run `Close` to break the
    /// cycle by releasing all event handlers.  But to avoid the situation, it is
    /// best to both explicitly run `Close` on the WebView and to not capture a
    /// reference to the WebView to ensure the WebView is cleaned up correctly.
    ///
    /// \snippet AppWindow.cpp Close
    unsafe fn close(&self) -> HRESULT;

    /// Gets the `CoreWebView2` associated with this `CoreWebView2Controller`.
    unsafe fn get_core_web_view2(
        &self,
        /* out, retval */ core_web_view2: *mut *mut *mut ICoreWebView2VTable,
    ) -> HRESULT;
}

/// A continuation of the ICoreWebView2Controller interface.
#[com_interface("c979903e-d4ca-4228-92eb-47ee3fa96eab")]
pub trait ICoreWebView2Controller2: ICoreWebView2Controller {
    /// The `DefaultBackgroundColor` property is the color WebView renders
    /// underneath all web content. This means WebView renders this color when
    /// there is no web content loaded such as before the initial navigation or
    /// between navigations. This also means web pages with undefined css
    /// background properties or background properties containing transparent
    /// pixels will render their contents over this color. Web pages with defined
    /// and opaque background properties that span the page will obscure the
    /// `DefaultBackgroundColor` and display normally. The default value for this
    /// property is white to resemble the native browser experience.
    ///
    /// The Color is specified by the COREWEBVIEW2_COLOR that represents an RGBA
    /// value. The `A` represents an Alpha value, meaning
    /// `DefaultBackgroundColor` can be transparent. In the case of a transparent
    /// `DefaultBackgroundColor` WebView will render hosting app content as the
    /// background. This Alpha value is not supported on Windows 7. Any `A` value
    /// other than 255 will result in E_INVALIDARG on Windows 7.
    /// It is supported on all other WebView compatible platforms.
    ///
    /// Semi-transparent colors are not currently supported by this API and
    /// setting `DefaultBackgroundColor` to a semi-transparent color will fail
    /// with E_INVALIDARG. The only supported alpha values are 0 and 255, all
    /// other values will result in E_INVALIDARG.
    /// `DefaultBackgroundColor` can only be an opaque color or transparent.
    ///
    /// \snippet ViewComponent.cpp DefaultBackgroundColor
    unsafe fn get_default_background_color(
        &self,
        /* out, retval */ background_color: *mut Color,
    ) -> HRESULT;

    /// Sets the `DefaultBackgroundColor` property.
    unsafe fn put_default_background_color(&self, /* in */ background_color: Color) -> HRESULT;
}

/// A continuation of the ICoreWebView2Controller2 interface.
#[com_interface("f9614724-5d2b-41dc-aef7-73d62b51543b")]
pub trait ICoreWebView2Controller3: ICoreWebView2Controller2 {
    /// The rasterization scale for the WebView. The rasterization scale is the
    /// combination of the monitor DPI scale and text scaling set by the user.
    /// This value should be updated when the DPI scale of the app's top level
    /// window changes (i.e. monitor DPI scale changes or window changes monitor)
    /// or when the text scale factor of the system changes.
    ///
    /// \snippet AppWindow.cpp DPIChanged
    ///
    /// \snippet AppWindow.cpp TextScaleChanged1
    ///
    /// \snippet AppWindow.cpp TextScaleChanged2
    ///
    /// Rasterization scale applies to the WebView content, as well as
    /// popups, context menus, scroll bars, and so on. Normal app scaling
    /// scenarios should use the ZoomFactor property or SetBoundsAndZoomFactor
    /// API which only scale the rendered HTML content and not popups, context
    /// menus, scroll bars, and so on.
    ///
    /// \snippet ViewComponent.cpp RasterizationScale
    unsafe fn get_rasterization_scale(&self, /* out, retval */ scale: *mut f64) -> HRESULT;

    /// Set the rasterization scale property.
    unsafe fn put_rasterization_scale(&self, /* in */ scale: f64) -> HRESULT;

    /// ShouldDetectMonitorScaleChanges property determines whether the WebView
    /// attempts to track monitor DPI scale changes. When true, the WebView will
    /// track monitor DPI scale changes, update the RasterizationScale property,
    /// and raises RasterizationScaleChanged event. When false, the WebView will
    /// not track monitor DPI scale changes, and the app must update the
    /// RasterizationScale property itself. RasterizationScaleChanged event will
    /// never raise when ShouldDetectMonitorScaleChanges is false.
    unsafe fn get_should_detect_monitor_scale_changes(
        &self,
        /* out, retval */ value: *mut BOOL,
    ) -> HRESULT;

    /// Set the ShouldDetectMonitorScaleChanges property.
    unsafe fn put_should_detect_monitor_scale_changes(&self, /* in */ value: BOOL) -> HRESULT;

    /// Add an event handler for the RasterizationScaleChanged event.
    /// The event is raised when the WebView detects that the monitor DPI scale
    /// has changed, ShouldDetectMonitorScaleChanges is true, and the WebView has
    /// changed the RasterizationScale property.
    ///
    /// \snippet ViewComponent.cpp RasterizationScaleChanged
    unsafe fn add_rasterization_scale_changed(
        &self,
        /* in */
        event_handler: *mut *mut ICoreWebView2RasterizationScaleChangedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with
    /// add_RasterizationScaleChanged.
    unsafe fn remove_rasterization_scale_changed(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// BoundsMode affects how setting the Bounds and RasterizationScale
    /// properties work. Bounds mode can either be in COREWEBVIEW2_BOUNDS_MODE_USE_RAW_PIXELS
    /// mode or COREWEBVIEW2_BOUNDS_MODE_USE_RASTERIZATION_SCALE mode.
    ///
    /// When the mode is in COREWEBVIEW2_BOUNDS_MODE_USE_RAW_PIXELS, setting the bounds
    /// property will set the size of the WebView in raw screen pixels. Changing
    /// the rasterization scale in this mode won't change the raw pixel size of
    /// the WebView and will only change the rasterization scale.
    ///
    /// When the mode is in COREWEBVIEW2_BOUNDS_MODE_USE_RASTERIZATION_SCALE, setting the
    /// bounds property will change the logical size of the WebView which can be
    /// described by the following equation:
    /// ```text
    /// Logical size * rasterization scale = Raw Pixel size
    /// ```
    /// In this case, changing the rasterization scale will keep the logical size
    /// the same and change the raw pixel size.
    ///
    /// \snippet ViewComponent.cpp BoundsMode
    unsafe fn get_bounds_mode(
        &self,
        /* out, retval */ bounds_mode: *mut BoundsMode,
    ) -> HRESULT;

    /// Set the BoundsMode property.
    unsafe fn put_bounds_mode(&self, /* in */ bounds_mode: BoundsMode) -> HRESULT;
}

/// This interface is an extension of the ICoreWebView2Controller interface to
/// support visual hosting. An object implementing the
/// ICoreWebView2CompositionController interface will also implement
/// ICoreWebView2Controller. Callers are expected to use
/// ICoreWebView2Controller for resizing, visibility, focus, and so on, and
/// then use ICoreWebView2CompositionController to connect to a composition
/// tree and provide input meant for the WebView.
#[com_interface("3df9b733-b9ae-4a15-86b4-eb9ee9826469")]
pub trait ICoreWebView2CompositionController: IUnknown {
    /// The RootVisualTarget is a visual in the hosting app's visual tree. This
    /// visual is where the WebView will connect its visual tree. The app uses
    /// this visual to position the WebView within the app. The app still needs
    /// to use the Bounds property to size the WebView. The RootVisualTarget
    /// property can be an IDCompositionVisual or a
    /// Windows::UI::Composition::ContainerVisual. WebView will connect its visual
    /// tree to the provided visual before returning from the property setter. The
    /// app needs to commit on its device setting the RootVisualTarget property.
    /// The RootVisualTarget property supports being set to nullptr to disconnect
    /// the WebView from the app's visual tree.
    /// \snippet ViewComponent.cpp SetRootVisualTarget
    /// \snippet ViewComponent.cpp BuildDCompTree
    unsafe fn get_root_visual_target(
        &self,
        /* out, retval */ target: *mut *mut *mut IUnknownVTable,
    ) -> HRESULT;

    /// Set the RootVisualTarget property.
    unsafe fn put_root_visual_target(
        &self,
        /* in */ target: *mut *mut IUnknownVTable,
    ) -> HRESULT;

    /// If eventKind is COREWEBVIEW2_MOUSE_EVENT_KIND_HORIZONTAL_WHEEL or
    /// COREWEBVIEW2_MOUSE_EVENT_KIND_WHEEL, then mouseData specifies the amount of
    /// wheel movement. A positive value indicates that the wheel was rotated
    /// forward, away from the user; a negative value indicates that the wheel was
    /// rotated backward, toward the user. One wheel click is defined as
    /// WHEEL_DELTA, which is 120.
    /// If eventKind is COREWEBVIEW2_MOUSE_EVENT_KIND_X_BUTTON_DOUBLE_CLICK
    /// COREWEBVIEW2_MOUSE_EVENT_KIND_X_BUTTON_DOWN, or
    /// COREWEBVIEW2_MOUSE_EVENT_KIND_X_BUTTON_UP, then mouseData specifies which X
    /// buttons were pressed or released. This value should be 1 if the first X
    /// button is pressed/released and 2 if the second X button is
    /// pressed/released.
    /// If eventKind is COREWEBVIEW2_MOUSE_EVENT_KIND_LEAVE, then virtualKeys,
    /// mouseData, and point should all be zero.
    /// If eventKind is any other value, then mouseData should be zero.
    /// Point is expected to be in the client coordinate space of the WebView.
    /// To track mouse events that start in the WebView and can potentially move
    /// outside of the WebView and host application, calling SetCapture and
    /// ReleaseCapture is recommended.
    /// To dismiss hover popups, it is also recommended to send
    /// COREWEBVIEW2_MOUSE_EVENT_KIND_LEAVE messages.
    /// \snippet ViewComponent.cpp SendMouseInput
    unsafe fn send_mouse_input(
        &self,
        /* in */ event_kind: MouseEventKind,
        /* in */ virtual_keys: MouseEventVirtualKeys,
        /* in */ mouse_data: u32,
        /* in */ point: POINT,
    ) -> HRESULT;

    /// SendPointerInput accepts touch or pen pointer input of types defined in
    /// COREWEBVIEW2_POINTER_EVENT_KIND. Any pointer input from the system must be
    /// converted into an ICoreWebView2PointerInfo first.
    unsafe fn send_pointer_input(
        &self,
        /* in */ event_kind: PointerEventKind,
        /* in */ pointer_info: *mut *mut ICoreWebView2PointerInfoVTable,
    ) -> HRESULT;

    /// The current cursor that WebView thinks it should be. The cursor should be
    /// set in WM_SETCURSOR through \::SetCursor or set on the corresponding
    /// parent/ancestor HWND of the WebView through \::SetClassLongPtr. The HCURSOR
    /// can be freed so CopyCursor/DestroyCursor is recommended to keep your own
    /// copy if you are doing more than immediately setting the cursor.
    unsafe fn get_cursor(&self, /* out, retval */ cursor: *mut HCURSOR) -> HRESULT;

    /// The current system cursor ID reported by the underlying rendering engine
    /// for WebView. For example, most of the time, when the cursor is over text,
    /// this will return the int value for IDC_IBEAM. The systemCursorId is only
    /// valid if the rendering engine reports a default Windows cursor resource
    /// value. Navigate to
    /// /[LoadCursorW/]/[WindowsWin32ApiWinuserNfwinuserloadcursorw/] for more
    /// details. Otherwise, if custom CSS cursors are being used, this will return
    /// 0. To actually use systemCursorId in LoadCursor or LoadImage,
    /// MAKEINTRESOURCE must be called on it first.
    ///
    /// [WindowsWin32ApiWinuserNfwinuserloadcursorw]: /windows/win32/api/winuser/nf-winuser-loadcursorw "LoadCursorW function (winuser.h) - Win32 apps | Microsoft Docs"
    ///
    /// \snippet ViewComponent.cpp SystemCursorId
    unsafe fn get_system_cursor_id(
        &self,
        /* out, retval */ system_cursor_id: *mut u32,
    ) -> HRESULT;

    /// Add an event handler for the CursorChanged event.
    /// The event is raised when WebView thinks the cursor should be changed. For
    /// example, when the mouse cursor is currently the default cursor but is then
    /// moved over text, it may try to change to the IBeam cursor.
    ///
    /// It is expected for the developer to send
    /// COREWEBVIEW2_MOUSE_EVENT_KIND_LEAVE messages (in addition to
    /// COREWEBVIEW2_MOUSE_EVENT_KIND_MOVE messages) through the SendMouseInput
    /// API. This is to ensure that the mouse is actually within the WebView that
    /// sends out CursorChanged events.
    ///
    /// \snippet ViewComponent.cpp CursorChanged
    unsafe fn add_cursor_changed(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2CursorChangedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with add_CursorChanged.
    unsafe fn remove_cursor_changed(&self, /* in */ token: EventRegistrationToken) -> HRESULT;
}

/// A continuation of the ICoreWebView2CompositionController interface.
#[com_interface("0b6a3d24-49cb-4806-ba20-b5e0734a7b26")]
pub trait ICoreWebView2CompositionController2: ICoreWebView2CompositionController {
    /// Returns the UI Automation Provider for the WebView.
    unsafe fn get_uiaprovider(
        &self,
        /* out, retval */ provider: *mut *mut *mut IUnknownVTable,
    ) -> HRESULT;
}

/// This interface is used to complete deferrals on event args that support
/// getting deferrals using the `GetDeferral` method.
#[com_interface("c10e7f7b-b585-46f0-a623-8befbf3e4ee0")]
pub trait ICoreWebView2Deferral: IUnknown {
    /// Completes the associated deferred event.  Complete should only be run
    /// once for each deferral taken.
    unsafe fn complete(&self) -> HRESULT;
}

/// Defines properties that enable, disable, or modify WebView features.
/// Setting changes made after `NavigationStarting` event does not apply
/// until the next top-level navigation.
#[com_interface("e562e4f0-d7fa-43ac-8d71-c05150499f00")]
pub trait ICoreWebView2Settings: IUnknown {
    /// Controls if running JavaScript is enabled in all future navigations in
    /// the WebView.  This only affects scripts in the document.  Scripts
    /// injected with `ExecuteScript` runs even if script is disabled.
    /// The default value is `TRUE`.
    ///
    /// \snippet SettingsComponent.cpp IsScriptEnabled
    unsafe fn get_is_script_enabled(
        &self,
        /* out, retval */ is_script_enabled: *mut BOOL,
    ) -> HRESULT;

    /// Sets the `IsScriptEnabled` property.
    unsafe fn put_is_script_enabled(&self, /* in */ is_script_enabled: BOOL) -> HRESULT;

    /// The `IsWebMessageEnabled` property is used when loading a new HTML
    /// document.  If set to `TRUE`, communication from the host to the top-level
    ///  HTML document of the WebView is allowed using `PostWebMessageAsJson`,
    /// `PostWebMessageAsString`, and message event of `window.chrome.webview`.
    /// For more information, navigate to [PostWebMessageAsJson].  Communication
    /// from the top-level HTML document of the WebView to the host is allowed
    /// using the postMessage function of `window.chrome.webview` and
    /// `add_WebMessageReceived` method.  For more information, navigate to
    /// [add_WebMessageReceived].  If set to false, then communication is
    /// disallowed.  `PostWebMessageAsJson` and `PostWebMessageAsString` fails
    /// with `E_ACCESSDENIED` and `window.chrome.webview.postMessage` fails by
    /// throwing an instance of an `Error` object.
    /// The default value is `TRUE`.
    ///
    /// \snippet ScenarioWebMessage.cpp IsWebMessageEnabled
    unsafe fn get_is_web_message_enabled(
        &self,
        /* out, retval */ is_web_message_enabled: *mut BOOL,
    ) -> HRESULT;

    /// Sets the `IsWebMessageEnabled` property.
    unsafe fn put_is_web_message_enabled(
        &self,
        /* in */ is_web_message_enabled: BOOL,
    ) -> HRESULT;

    /// `AreDefaultScriptDialogsEnabled` is used when loading a new HTML
    /// document.  If set to `FALSE`, WebView2 does not render the default JavaScript
    /// dialog box (Specifically those displayed by the JavaScript alert,
    /// confirm, prompt functions and `beforeunload` event).  Instead, if an
    /// event handler is set using `add_ScriptDialogOpening`, WebView sends an
    /// event that contains all of the information for the dialog and allow the
    /// host app to show a custom UI.
    /// The default value is `TRUE`.
    unsafe fn get_are_default_script_dialogs_enabled(
        &self,
        /* out, retval */ are_default_script_dialogs_enabled: *mut BOOL,
    ) -> HRESULT;

    /// Sets the `AreDefaultScriptDialogsEnabled` property.
    unsafe fn put_are_default_script_dialogs_enabled(
        &self,
        /* in */ are_default_script_dialogs_enabled: BOOL,
    ) -> HRESULT;

    /// `IsStatusBarEnabled` controls whether the status bar is displayed.  The
    /// status bar is usually displayed in the lower left of the WebView and
    /// shows things such as the URI of a link when the user hovers over it and
    /// other information.
    /// The default value is `TRUE`.
    /// The status bar UI can be altered by web content and should not be considered secure.
    unsafe fn get_is_status_bar_enabled(
        &self,
        /* out, retval */ is_status_bar_enabled: *mut BOOL,
    ) -> HRESULT;

    /// Sets the `IsStatusBarEnabled` property.
    unsafe fn put_is_status_bar_enabled(
        &self,
        /* in */ is_status_bar_enabled: BOOL,
    ) -> HRESULT;

    /// `AreDevToolsEnabled` controls whether the user is able to use the context
    /// menu or keyboard shortcuts to open the DevTools window.
    /// The default value is `TRUE`.
    unsafe fn get_are_dev_tools_enabled(
        &self,
        /* out, retval */ are_dev_tools_enabled: *mut BOOL,
    ) -> HRESULT;

    /// Sets the `AreDevToolsEnabled` property.
    unsafe fn put_are_dev_tools_enabled(
        &self,
        /* in */ are_dev_tools_enabled: BOOL,
    ) -> HRESULT;

    /// The `AreDefaultContextMenusEnabled` property is used to prevent default
    /// context menus from being shown to user in WebView.
    /// The default value is `TRUE`.
    ///
    /// \snippet SettingsComponent.cpp DisableContextMenu
    unsafe fn get_are_default_context_menus_enabled(
        &self,
        /* out, retval */ enabled: *mut BOOL,
    ) -> HRESULT;

    /// Sets the `AreDefaultContextMenusEnabled` property.
    unsafe fn put_are_default_context_menus_enabled(&self, /* in */ enabled: BOOL) -> HRESULT;

    /// The `AreHostObjectsAllowed` property is used to control whether host
    /// objects are accessible from the page in WebView.
    /// The default value is `TRUE`.
    ///
    /// \snippet SettingsComponent.cpp HostObjectsAccess
    unsafe fn get_are_host_objects_allowed(
        &self,
        /* out, retval */ allowed: *mut BOOL,
    ) -> HRESULT;

    /// Sets the `AreHostObjectsAllowed` property.
    unsafe fn put_are_host_objects_allowed(&self, /* in */ allowed: BOOL) -> HRESULT;

    /// The `IsZoomControlEnabled` property is used to prevent the user from
    /// impacting the zoom of the WebView.  When disabled, the user is not able
    /// to zoom using Ctrl++, Ctrl+-, or Ctrl+mouse wheel, but the zoom
    /// is set using `ZoomFactor` API.  The default value is `TRUE`.
    ///
    /// \snippet SettingsComponent.cpp DisableZoomControl
    unsafe fn get_is_zoom_control_enabled(
        &self,
        /* out, retval */ enabled: *mut BOOL,
    ) -> HRESULT;

    /// Sets the `IsZoomControlEnabled` property.
    unsafe fn put_is_zoom_control_enabled(&self, /* in */ enabled: BOOL) -> HRESULT;

    /// The `IsBuiltInErrorPageEnabled` property is used to disable built in
    /// error page for navigation failure and render process failure.  When
    /// disabled, a blank page is displayed when the related error happens.
    /// The default value is `TRUE`.
    ///
    /// \snippet SettingsComponent.cpp BuiltInErrorPageEnabled
    unsafe fn get_is_built_in_error_page_enabled(
        &self,
        /* out, retval */ enabled: *mut BOOL,
    ) -> HRESULT;

    /// Sets the `IsBuiltInErrorPageEnabled` property.
    unsafe fn put_is_built_in_error_page_enabled(&self, /* in */ enabled: BOOL) -> HRESULT;
}

/// A continuation of the ICoreWebView2Settings interface that manages the user agent.
#[com_interface("ee9a0f68-f46c-4e32-ac23-ef8cac224d2a")]
pub trait ICoreWebView2Settings2: ICoreWebView2Settings {
    /// Returns the User Agent. The default value is the default User Agent of the
    /// Microsoft Edge browser.
    ///
    /// \snippet SettingsComponent.cpp UserAgent
    unsafe fn get_user_agent(&self, /* out, retval */ user_agent: *mut LPWSTR) -> HRESULT;

    /// Sets the `UserAgent` property. This property may be overridden if
    /// the User-Agent header is set in a request. If the parameter is empty
    /// the User Agent will not be updated and the current User Agent will remain.
    unsafe fn put_user_agent(&self, /* in */ user_agent: LPCWSTR) -> HRESULT;
}

/// A continuation of the ICoreWebView2Settings interface that manages whether
/// browser accelerator keys are enabled.
#[com_interface("fdb5ab74-af33-4854-84f0-0a631deb5eba")]
pub trait ICoreWebView2Settings3: ICoreWebView2Settings2 {
    /// When this setting is set to FALSE, it disables all accelerator keys that
    /// access features specific to a web browser, including but not limited to:
    ///  - Ctrl-F and F3 for Find on Page
    ///  - Ctrl-P for Print
    ///  - Ctrl-R and F5 for Reload
    ///  - Ctrl-Plus and Ctrl-Minus for zooming
    ///  - Ctrl-Shift-C and F12 for DevTools
    ///  - Special keys for browser functions, such as Back, Forward, and Search
    ///
    /// It does not disable accelerator keys related to movement and text editing,
    /// such as:
    ///  - Home, End, Page Up, and Page Down
    ///  - Ctrl-X, Ctrl-C, Ctrl-V
    ///  - Ctrl-A for Select All
    ///  - Ctrl-Z for Undo
    ///
    /// Those accelerator keys will always be enabled unless they are handled in
    /// the `AcceleratorKeyPressed` event.
    ///
    /// This setting has no effect on the `AcceleratorKeyPressed` event.  The event
    /// will be fired for all accelerator keys, whether they are enabled or not.
    ///
    /// The default value for `AreBrowserAcceleratorKeysEnabled` is TRUE.
    ///
    /// \snippet SettingsComponent.cpp AreBrowserAcceleratorKeysEnabled
    unsafe fn get_are_browser_accelerator_keys_enabled(
        &self,
        /* out, retval */ are_browser_accelerator_keys_enabled: *mut BOOL,
    ) -> HRESULT;

    /// Sets the `AreBrowserAcceleratorKeysEnabled` property.
    unsafe fn put_are_browser_accelerator_keys_enabled(
        &self,
        /* in */ are_browser_accelerator_keys_enabled: BOOL,
    ) -> HRESULT;
}

/// A continuation of the ICoreWebView2Settings interface to manage autofill.
#[com_interface("cb56846c-4168-4d53-b04f-03b6d6796ff2")]
pub trait ICoreWebView2Settings4: ICoreWebView2Settings3 {
    /// IsPasswordAutosaveEnabled controls whether autosave for password
    /// information is enabled. The IsPasswordAutosaveEnabled property behaves
    /// independently of the IsGeneralAutofillEnabled property. When IsPasswordAutosaveEnabled is
    /// false, no new password data is saved and no Save/Update Password prompts are displayed.
    /// However, if there was password data already saved before disabling this setting,
    /// then that password information is auto-populated, suggestions are shown and clicking on
    /// one will populate the fields.
    /// When IsPasswordAutosaveEnabled is true, password information is auto-populated,
    /// suggestions are shown and clicking on one will populate the fields, new data
    /// is saved, and a Save/Update Password prompt is displayed.
    /// The default value is `FALSE`.
    ///
    /// \snippet SettingsComponent.cpp PasswordAutosaveEnabled
    unsafe fn get_is_password_autosave_enabled(
        &self,
        /* out, retval */ value: *mut BOOL,
    ) -> HRESULT;

    /// Set the IsPasswordAutosaveEnabled property.
    unsafe fn put_is_password_autosave_enabled(&self, /* in */ value: BOOL) -> HRESULT;

    /// IsGeneralAutofillEnabled controls whether autofill for information
    /// like names, street and email addresses, phone numbers, and arbitrary input
    /// is enabled. This excludes password and credit card information. When
    /// IsGeneralAutofillEnabled is false, no suggestions appear, and no new information
    /// is saved. When IsGeneralAutofillEnabled is true, information is saved, suggestions
    /// appear and clicking on one will populate the form fields.
    /// The default value is `TRUE`.
    ///
    /// \snippet SettingsComponent.cpp GeneralAutofillEnabled
    unsafe fn get_is_general_autofill_enabled(
        &self,
        /* out, retval */ value: *mut BOOL,
    ) -> HRESULT;

    /// Set the IsGeneralAutofillEnabled property.
    unsafe fn put_is_general_autofill_enabled(&self, /* in */ value: BOOL) -> HRESULT;
}

/// A continuation of the ICoreWebView2Settings interface to manage pinch zoom.
#[com_interface("183e7052-1d03-43a0-ab99-98e043b66b39")]
pub trait ICoreWebView2Settings5: ICoreWebView2Settings4 {
    /// Pinch-zoom, referred to as "Page Scale" zoom, is performed as a post-rendering step,
    /// it changes the page scale factor property and scales the surface the web page is
    /// rendered onto when user performs a pinch zooming action. It does not change the layout
    /// but rather changes the viewport and clips the web content, the content outside of the
    /// viewport isn't visible onscreen and users can't reach this content using mouse.
    ///
    /// The `IsPinchZoomEnabled` property enables or disables the ability of
    /// the end user to use a pinching motion on touch input enabled devices
    /// to scale the web content in the WebView2. It defaults to `TRUE`.
    /// When set to `FALSE`, the end user cannot pinch zoom after the next navigation.
    /// Disabling/Enabling `IsPinchZoomEnabled` only affects the end user's ability to use
    /// pinch motions and does not change the page scale factor.
    /// This API only affects the Page Scale zoom and has no effect on the
    /// existing browser zoom properties (`IsZoomControlEnabled` and `ZoomFactor`)
    /// or other end user mechanisms for zooming.
    ///
    /// \snippet SettingsComponent.cpp TogglePinchZoomEnabled
    unsafe fn get_is_pinch_zoom_enabled(
        &self,
        /* out, retval */ enabled: *mut BOOL,
    ) -> HRESULT;

    /// Set the `IsPinchZoomEnabled` property
    unsafe fn put_is_pinch_zoom_enabled(&self, /* in */ enabled: BOOL) -> HRESULT;
}

/// Event args for the `ProcessFailed` event.
#[com_interface("8155a9a4-1474-4a86-8cae-151b0fa6b8ca")]
pub trait ICoreWebView2ProcessFailedEventArgs: IUnknown {
    /// The kind of process failure that has occurred. `processFailedKind` is
    /// `COREWEBVIEW2_PROCESS_FAILED_KIND_RENDER_PROCESS_EXITED` if the
    /// failed process is the main frame's renderer, even if there were subframes
    /// rendered by such process; all frames are gone when this happens.
    unsafe fn get_process_failed_kind(
        &self,
        /* out, retval */ process_failed_kind: *mut ProcessFailedKind,
    ) -> HRESULT;
}

/// Receives `ProcessFailed` events.
#[com_interface("79e0aea4-990b-42d9-aa1d-0fcc2e5bc7f1")]
pub trait ICoreWebView2ProcessFailedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut ICoreWebView2ProcessFailedEventArgsVTable,
    ) -> HRESULT;
}

/// Implements the interface to receive `ZoomFactorChanged` events.  Use the
/// `ICoreWebView2Controller.ZoomFactor` property to get the modified zoom
/// factor.
#[com_interface("b52d71d6-c4df-4543-a90c-64a3e60f38cb")]
pub trait ICoreWebView2ZoomFactorChangedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.  No event args exist
    /// and the `args` parameter is set to `null`.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2ControllerVTable,
        /* in */ args: *mut *mut IUnknownVTable,
    ) -> HRESULT;
}

/// Iterator for a collection of HTTP headers.  For more information, navigate
/// to ICoreWebView2HttpRequestHeaders and ICoreWebView2HttpResponseHeaders.
///
/// \snippet ScenarioWebViewEventMonitor.cpp HttpRequestHeaderIterator
#[com_interface("0702fc30-f43b-47bb-ab52-a42cb552ad9f")]
pub trait ICoreWebView2HttpHeadersCollectionIterator: IUnknown {
    /// Get the name and value of the current HTTP header of the iterator.  If
    /// the previous `MoveNext` operation set the `hasNext` parameter to `FALSE`,
    /// this method fails.
    unsafe fn get_current_header(
        &self,
        /* out */ name: *mut LPWSTR,
        /* out */ value: *mut LPWSTR,
    ) -> HRESULT;

    /// `TRUE` when the iterator has not run out of headers.  If the collection
    /// over which the iterator is iterating is empty or if the iterator has gone
    ///  past the end of the collection then this is `FALSE`.
    unsafe fn get_has_current_header(
        &self,
        /* out, retval */ has_current: *mut BOOL,
    ) -> HRESULT;

    /// Move the iterator to the next HTTP header in the collection.
    ///
    /// \> [!NOTE]\n \> If no more HTTP headers exist, the `hasNext` parameter is set to
    /// `FALSE`.  After this occurs the `GetCurrentHeader` method fails.
    unsafe fn move_next(&self, /* out, retval */ has_next: *mut BOOL) -> HRESULT;
}

/// HTTP request headers.  Used to inspect the HTTP request on
/// `WebResourceRequested` event and `NavigationStarting` event.
///
/// \> [!NOTE]\n\> It is possible to modify the HTTP request from a `WebResourceRequested`
/// event, but not from a `NavigationStarting` event.
#[com_interface("e86cac0e-5523-465c-b536-8fb9fc8c8c60")]
pub trait ICoreWebView2HttpRequestHeaders: IUnknown {
    /// Gets the header value matching the name.
    unsafe fn get_header(
        &self,
        /* in */ name: LPCWSTR,
        /* out, retval */ value: *mut LPWSTR,
    ) -> HRESULT;

    /// Gets the header value matching the name using an iterator.
    unsafe fn get_headers(
        &self,
        /* in */ name: LPCWSTR,
        /* out, retval */
        iterator: *mut *mut *mut ICoreWebView2HttpHeadersCollectionIteratorVTable,
    ) -> HRESULT;

    /// Verifies that the headers contain an entry that matches the header name.
    unsafe fn contains(
        &self,
        /* in */ name: LPCWSTR,
        /* out, retval */ contains: *mut BOOL,
    ) -> HRESULT;

    /// Adds or updates header that matches the name.
    unsafe fn set_header(
        &self,
        /* in */ name: LPCWSTR,
        /* in */ value: LPCWSTR,
    ) -> HRESULT;

    /// Removes header that matches the name.
    unsafe fn remove_header(&self, /* in */ name: LPCWSTR) -> HRESULT;

    /// Gets an iterator over the collection of request headers.
    unsafe fn get_iterator(
        &self,
        /* out, retval */
        iterator: *mut *mut *mut ICoreWebView2HttpHeadersCollectionIteratorVTable,
    ) -> HRESULT;
}

/// HTTP response headers.  Used to construct a `WebResourceResponse` for the
/// `WebResourceRequested` event.
#[com_interface("03c5ff5a-9b45-4a88-881c-89a9f328619c")]
pub trait ICoreWebView2HttpResponseHeaders: IUnknown {
    /// Appends header line with name and value.
    unsafe fn append_header(
        &self,
        /* in */ name: LPCWSTR,
        /* in */ value: LPCWSTR,
    ) -> HRESULT;

    /// Verifies that the headers contain entries that match the header name.
    unsafe fn contains(
        &self,
        /* in */ name: LPCWSTR,
        /* out, retval */ contains: *mut BOOL,
    ) -> HRESULT;

    /// Gets the first header value in the collection matching the name.
    unsafe fn get_header(
        &self,
        /* in */ name: LPCWSTR,
        /* out, retval */ value: *mut LPWSTR,
    ) -> HRESULT;

    /// Gets the header values matching the name.
    unsafe fn get_headers(
        &self,
        /* in */ name: LPCWSTR,
        /* out, retval */
        iterator: *mut *mut *mut ICoreWebView2HttpHeadersCollectionIteratorVTable,
    ) -> HRESULT;

    /// Gets an iterator over the collection of entire response headers.
    unsafe fn get_iterator(
        &self,
        /* out, retval */
        iterator: *mut *mut *mut ICoreWebView2HttpHeadersCollectionIteratorVTable,
    ) -> HRESULT;
}

/// An HTTP request used with the `WebResourceRequested` event.
#[com_interface("97055cd4-512c-4264-8b5f-e3f446cea6a5")]
pub trait ICoreWebView2WebResourceRequest: IUnknown {
    /// The request URI.
    unsafe fn get_uri(&self, /* out, retval */ uri: *mut LPWSTR) -> HRESULT;

    /// Sets the `Uri` property.
    unsafe fn put_uri(&self, /* in */ uri: LPCWSTR) -> HRESULT;

    /// The HTTP request method.
    unsafe fn get_method(&self, /* out, retval */ method: *mut LPWSTR) -> HRESULT;

    /// Sets the `Method` property.
    unsafe fn put_method(&self, /* in */ method: LPCWSTR) -> HRESULT;

    /// The HTTP request message body as stream.  POST data should be here.  If a
    /// stream is set, which overrides the message body, the stream must have
    /// all the content data available by the time the `WebResourceRequested`
    /// event deferral of this response is completed.  Stream should be agile or
    /// be created from a background STA to prevent performance impact to the UI
    /// thread.  `Null` means no content data.  `IStream` semantics apply
    /// (return `S_OK` to `Read` runs until all data is exhausted).
    unsafe fn get_content(
        &self,
        /* out, retval */ content: *mut *mut *mut IStreamVTable,
    ) -> HRESULT;

    /// Sets the `Content` property.
    unsafe fn put_content(&self, /* in */ content: *mut *mut IStreamVTable) -> HRESULT;

    /// The mutable HTTP request headers
    unsafe fn get_headers(
        &self,
        /* out, retval */ headers: *mut *mut *mut ICoreWebView2HttpRequestHeadersVTable,
    ) -> HRESULT;
}

/// An HTTP response used with the `WebResourceRequested` event.
#[com_interface("aafcc94f-fa27-48fd-97df-830ef75aaec9")]
pub trait ICoreWebView2WebResourceResponse: IUnknown {
    /// HTTP response content as stream.  Stream must have all the content data
    /// available by the time the `WebResourceRequested` event deferral of this
    /// response is completed.  Stream should be agile or be created from a
    /// background thread to prevent performance impact to the UI thread.  `Null`
    ///  means no content data.  `IStream` semantics apply (return `S_OK` to
    /// `Read` runs until all data is exhausted).
    unsafe fn get_content(
        &self,
        /* out, retval */ content: *mut *mut *mut IStreamVTable,
    ) -> HRESULT;

    /// Sets the `Content` property.
    unsafe fn put_content(&self, /* in */ content: *mut *mut IStreamVTable) -> HRESULT;

    /// Overridden HTTP response headers.
    unsafe fn get_headers(
        &self,
        /* out, retval */ headers: *mut *mut *mut ICoreWebView2HttpResponseHeadersVTable,
    ) -> HRESULT;

    /// The HTTP response status code.
    unsafe fn get_status_code(&self, /* out, retval */ status_code: *mut i32) -> HRESULT;

    /// Sets the `StatusCode` property.
    unsafe fn put_status_code(&self, /* in */ status_code: i32) -> HRESULT;

    /// The HTTP response reason phrase.
    unsafe fn get_reason_phrase(
        &self,
        /* out, retval */ reason_phrase: *mut LPWSTR,
    ) -> HRESULT;

    /// Sets the `ReasonPhrase` property.
    unsafe fn put_reason_phrase(&self, /* in */ reason_phrase: LPCWSTR) -> HRESULT;
}

/// Event args for the `NavigationStarting` event.
#[com_interface("5b495469-e119-438a-9b18-7604f25f2e49")]
pub trait ICoreWebView2NavigationStartingEventArgs: IUnknown {
    /// The uri of the requested navigation.
    unsafe fn get_uri(&self, /* out, retval */ uri: *mut LPWSTR) -> HRESULT;

    /// `TRUE` when the navigation was initiated through a user gesture as
    /// opposed to programmatic navigation by page script. Navigations initiated
    /// via WebView2 APIs are considered as user initiated.
    unsafe fn get_is_user_initiated(
        &self,
        /* out, retval */ is_user_initiated: *mut BOOL,
    ) -> HRESULT;

    /// `TRUE` when the navigation is redirected.
    unsafe fn get_is_redirected(&self, /* out, retval */ is_redirected: *mut BOOL) -> HRESULT;

    /// The HTTP request headers for the navigation.
    ///
    /// \> [!NOTE]\n\> You are not able to modify the HTTP request headers in a
    /// `NavigationStarting` event.
    unsafe fn get_request_headers(
        &self,
        /* out, retval */
        request_headers: *mut *mut *mut ICoreWebView2HttpRequestHeadersVTable,
    ) -> HRESULT;

    /// The host may set this flag to cancel the navigation.  If set, the
    /// navigation is not longer present and the content of the current page is
    /// intact.  For performance reasons, `GET` HTTP requests may happen, while
    /// the host is responding.  You may set cookies and use part of a request
    /// for the navigation.  Cancellation for navigation to `about:blank` or
    /// frame navigation to `srcdoc` is not supported.  Such attempts are
    /// ignored.
    unsafe fn get_cancel(&self, /* out, retval */ cancel: *mut BOOL) -> HRESULT;

    /// Sets the `Cancel` property.
    unsafe fn put_cancel(&self, /* in */ cancel: BOOL) -> HRESULT;

    /// The ID of the navigation.
    unsafe fn get_navigation_id(&self, /* out, retval */ navigation_id: *mut u64) -> HRESULT;
}

/// Receives `NavigationStarting` events.
#[com_interface("9adbe429-f36d-432b-9ddc-f8881fbd76e3")]
pub trait ICoreWebView2NavigationStartingEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut ICoreWebView2NavigationStartingEventArgsVTable,
    ) -> HRESULT;
}

/// Event args for the `ContentLoading` event.
#[com_interface("0c8a1275-9b6b-4901-87ad-70df25bafa6e")]
pub trait ICoreWebView2ContentLoadingEventArgs: IUnknown {
    /// `TRUE` if the loaded content is an error page.
    unsafe fn get_is_error_page(&self, /* out, retval */ is_error_page: *mut BOOL) -> HRESULT;

    /// The ID of the navigation.
    unsafe fn get_navigation_id(&self, /* out, retval */ navigation_id: *mut u64) -> HRESULT;
}

/// Receives `ContentLoading` events.
#[com_interface("364471e7-f2be-4910-bdba-d72077d51c4b")]
pub trait ICoreWebView2ContentLoadingEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut ICoreWebView2ContentLoadingEventArgsVTable,
    ) -> HRESULT;
}

/// Event args for the `SourceChanged` event.
#[com_interface("31e0e545-1dba-4266-8914-f63848a1f7d7")]
pub trait ICoreWebView2SourceChangedEventArgs: IUnknown {
    /// `TRUE` if the page being navigated to is a new document.
    unsafe fn get_is_new_document(
        &self,
        /* out, retval */ is_new_document: *mut BOOL,
    ) -> HRESULT;
}

/// Receives `SourceChanged` events.
#[com_interface("3c067f9f-5388-4772-8b48-79f7ef1ab37c")]
pub trait ICoreWebView2SourceChangedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut ICoreWebView2SourceChangedEventArgsVTable,
    ) -> HRESULT;
}

/// Receives `HistoryChanged` events.
#[com_interface("c79a420c-efd9-4058-9295-3e8b4bcab645")]
pub trait ICoreWebView2HistoryChangedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.  No event args exist
    /// and the `args` parameter is set to `null`.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut IUnknownVTable,
    ) -> HRESULT;
}

/// Event args for the `ScriptDialogOpening` event.
#[com_interface("7390bb70-abe0-4843-9529-f143b31b03d6")]
pub trait ICoreWebView2ScriptDialogOpeningEventArgs: IUnknown {
    /// The URI of the page that requested the dialog box.
    unsafe fn get_uri(&self, /* out, retval */ uri: *mut LPWSTR) -> HRESULT;

    /// The kind of JavaScript dialog box.  `alert`, `confirm`, `prompt`, or
    /// `beforeunload`.
    unsafe fn get_kind(&self, /* out, retval */ kind: *mut ScriptDialogKind) -> HRESULT;

    /// The message of the dialog box.  From JavaScript this is the first
    /// parameter passed to `alert`, `confirm`, and `prompt` and is empty for
    /// `beforeunload`.
    unsafe fn get_message(&self, /* out, retval */ message: *mut LPWSTR) -> HRESULT;

    /// The host may run this to respond with **OK** to `confirm`, `prompt`, and
    /// `beforeunload` dialogs.  Do not run this method to indicate cancel.
    /// From JavaScript, this means that the `confirm` and `beforeunload` function
    /// returns `TRUE` if `Accept` is run.  And for the prompt function it returns
    /// the value of `ResultText` if `Accept` is run and otherwise returns
    /// `FALSE`.
    unsafe fn accept(&self) -> HRESULT;

    /// The second parameter passed to the JavaScript prompt dialog.
    /// The result of the prompt JavaScript function uses this value as the
    /// default value.
    unsafe fn get_default_text(&self, /* out, retval */ default_text: *mut LPWSTR) -> HRESULT;

    /// The return value from the JavaScript prompt function if `Accept` is run.
    ///  This value is ignored for dialog kinds other than prompt.  If `Accept`
    /// is not run, this value is ignored and `FALSE` is returned from prompt.
    unsafe fn get_result_text(&self, /* out, retval */ result_text: *mut LPWSTR) -> HRESULT;

    /// Sets the `ResultText` property.
    unsafe fn put_result_text(&self, /* in */ result_text: LPCWSTR) -> HRESULT;

    /// Returns an `ICoreWebView2Deferral` object.  Use this operation to
    /// complete the event at a later time.
    unsafe fn get_deferral(
        &self,
        /* out, retval */ deferral: *mut *mut *mut ICoreWebView2DeferralVTable,
    ) -> HRESULT;
}

/// Receives `ScriptDialogOpening` events.
#[com_interface("ef381bf9-afa8-4e37-91c4-8ac48524bdfb")]
pub trait ICoreWebView2ScriptDialogOpeningEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut ICoreWebView2ScriptDialogOpeningEventArgsVTable,
    ) -> HRESULT;
}

/// Event args for the `NavigationCompleted` event.
#[com_interface("30d68b7d-20d9-4752-a9ca-ec8448fbb5c1")]
pub trait ICoreWebView2NavigationCompletedEventArgs: IUnknown {
    /// `TRUE` when the navigation is successful.  `FALSE` for a navigation that
    /// ended up in an error page (failures due to no network, DNS lookup
    /// failure, HTTP server responds with 4xx), but may also be `FALSE` for
    /// additional scenarios such as `window.stop()` run on navigated page.
    unsafe fn get_is_success(&self, /* out, retval */ is_success: *mut BOOL) -> HRESULT;

    /// The error code if the navigation failed.
    unsafe fn get_web_error_status(
        &self,
        /* out, retval */ web_error_status: *mut WebErrorStatus,
    ) -> HRESULT;

    /// The ID of the navigation.
    unsafe fn get_navigation_id(&self, /* out, retval */ navigation_id: *mut u64) -> HRESULT;
}

/// Receives `NavigationCompleted` events.
#[com_interface("d33a35bf-1c49-4f98-93ab-006e0533fe1c")]
pub trait ICoreWebView2NavigationCompletedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut ICoreWebView2NavigationCompletedEventArgsVTable,
    ) -> HRESULT;
}

/// Event args for the `PermissionRequested` event.
#[com_interface("973ae2ef-ff18-4894-8fb2-3c758f046810")]
pub trait ICoreWebView2PermissionRequestedEventArgs: IUnknown {
    /// The origin of the web content that requests the permission.
    unsafe fn get_uri(&self, /* out, retval */ uri: *mut LPWSTR) -> HRESULT;

    /// The type of the permission that is requested.
    unsafe fn get_permission_kind(
        &self,
        /* out, retval */ permission_kind: *mut PermissionKind,
    ) -> HRESULT;

    /// `TRUE` when the permission request was initiated through a user gesture.
    ///
    /// \> [!NOTE]\n\> Being initiated through a user gesture does not mean that user intended
    /// to access the associated resource.
    unsafe fn get_is_user_initiated(
        &self,
        /* out, retval */ is_user_initiated: *mut BOOL,
    ) -> HRESULT;

    /// The status of a permission request, (for example is the request is granted).
    /// The default value is `COREWEBVIEW2_PERMISSION_STATE_DEFAULT`.
    unsafe fn get_state(&self, /* out, retval */ state: *mut PermissionState) -> HRESULT;

    /// Sets the `State` property.
    unsafe fn put_state(&self, /* in */ state: PermissionState) -> HRESULT;

    /// Gets an `ICoreWebView2Deferral` object.  Use the deferral object to make
    /// the permission decision at a later time.
    unsafe fn get_deferral(
        &self,
        /* out, retval */ deferral: *mut *mut *mut ICoreWebView2DeferralVTable,
    ) -> HRESULT;
}

/// Receives `PermissionRequested` events.
#[com_interface("15e1c6a3-c72a-4df3-91d7-d097fbec6bfd")]
pub trait ICoreWebView2PermissionRequestedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut ICoreWebView2PermissionRequestedEventArgsVTable,
    ) -> HRESULT;
}

/// Receives the result of the `AddScriptToExecuteOnDocumentCreated` method.
#[com_interface("b99369f3-9b11-47b5-bc6f-8e7895fcea17")]
pub trait ICoreWebView2AddScriptToExecuteOnDocumentCreatedCompletedHandler: IUnknown {
    /// Provide the completion status and result of the corresponding
    /// asynchronous method.
    unsafe fn invoke(
        &self,
        /* in */ error_code: HRESULT,
        /* in */ id: LPCWSTR,
    ) -> HRESULT;
}

/// Receives the result of the `ExecuteScript` method.
#[com_interface("49511172-cc67-4bca-9923-137112f4c4cc")]
pub trait ICoreWebView2ExecuteScriptCompletedHandler: IUnknown {
    /// Provide the implementer with the completion status and result of the
    /// corresponding asynchronous method.
    unsafe fn invoke(
        &self,
        /* in */ error_code: HRESULT,
        /* in */ result_object_as_json: LPCWSTR,
    ) -> HRESULT;
}

/// Event args for the `WebResourceRequested` event.
#[com_interface("453e667f-12c7-49d4-be6d-ddbe7956f57a")]
pub trait ICoreWebView2WebResourceRequestedEventArgs: IUnknown {
    /// The Web resource request.  The request object may be missing some headers
    /// that are added by network stack at a later time.
    unsafe fn get_request(
        &self,
        /* out, retval */ request: *mut *mut *mut ICoreWebView2WebResourceRequestVTable,
    ) -> HRESULT;

    /// A placeholder for the web resource response object.  If this object is
    /// set, the web resource request is completed with the specified response.
    unsafe fn get_response(
        &self,
        /* out, retval */ response: *mut *mut *mut ICoreWebView2WebResourceResponseVTable,
    ) -> HRESULT;

    /// Sets the `Response` property.  Create an empty web resource response
    /// object with `CreateWebResourceResponse` and then modify it to construct
    /// the response.
    unsafe fn put_response(
        &self,
        /* in */ response: *mut *mut ICoreWebView2WebResourceResponseVTable,
    ) -> HRESULT;

    /// Obtain an `ICoreWebView2Deferral` object and put the event into a
    /// deferred state.  Use the `ICoreWebView2Deferral` object to complete the
    /// request at a later time.
    unsafe fn get_deferral(
        &self,
        /* out, retval */ deferral: *mut *mut *mut ICoreWebView2DeferralVTable,
    ) -> HRESULT;

    /// The web resource request context.
    unsafe fn get_resource_context(
        &self,
        /* out, retval */ context: *mut WebResourceContext,
    ) -> HRESULT;
}

/// Runs when a URL request (through network, file, and so on) is made in
/// the webview for a Web resource matching resource context filter and URL
/// specified in `AddWebResourceRequestedFilter`.  The host views and modifies
/// the request or provide a response in a similar pattern to HTTP, in which
/// case the request immediately completed.  This may not contain any request
/// headers that are added by the network stack, such as an `Authorization`
/// header.
#[com_interface("ab00b74c-15f1-4646-80e8-e76341d25d71")]
pub trait ICoreWebView2WebResourceRequestedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut ICoreWebView2WebResourceRequestedEventArgsVTable,
    ) -> HRESULT;
}

/// Receives the result of the `CapturePreview` method.  The result is written
/// to the stream provided in the `CapturePreview` method.
#[com_interface("697e05e9-3d8f-45fa-96f4-8ffe1ededaf5")]
pub trait ICoreWebView2CapturePreviewCompletedHandler: IUnknown {
    /// Provides the completion status of the corresponding asynchronous method.
    unsafe fn invoke(&self, /* in */ error_code: HRESULT) -> HRESULT;
}

/// Receives `GotFocus` and `LostFocus` events.
#[com_interface("05ea24bd-6452-4926-9014-4b82b498135d")]
pub trait ICoreWebView2FocusChangedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.  No event args exist
    /// and the `args` parameter is set to `null`.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2ControllerVTable,
        /* in */ args: *mut *mut IUnknownVTable,
    ) -> HRESULT;
}

/// Event args for the `MoveFocusRequested` event.
#[com_interface("2d6aa13b-3839-4a15-92fc-d88b3c0d9c9d")]
pub trait ICoreWebView2MoveFocusRequestedEventArgs: IUnknown {
    /// The reason for WebView to run the `MoveFocusRequested` event.
    unsafe fn get_reason(&self, /* out, retval */ reason: *mut MoveFocusReason) -> HRESULT;

    /// Indicates whether the event has been handled by the app.  If the app has
    /// moved the focus to another desired location, it should set the `Handled`
    /// property to `TRUE`.  When the `Handled` property is `FALSE` after the
    /// event handler returns, default action is taken.  The default action is to
    /// try to find the next tab stop child window in the app and try to move
    /// focus to that window.  If no other window exists to move focus, focus is
    /// cycled within the web content of the WebView.
    unsafe fn get_handled(&self, /* out, retval */ value: *mut BOOL) -> HRESULT;

    /// Sets the `Handled` property.
    unsafe fn put_handled(&self, /* in */ value: BOOL) -> HRESULT;
}

/// Receives `MoveFocusRequested` events.
#[com_interface("69035451-6dc7-4cb8-9bce-b2bd70ad289f")]
pub trait ICoreWebView2MoveFocusRequestedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2ControllerVTable,
        /* in */ args: *mut *mut ICoreWebView2MoveFocusRequestedEventArgsVTable,
    ) -> HRESULT;
}

/// Event args for the `WebMessageReceived` event.
#[com_interface("0f99a40c-e962-4207-9e92-e3d542eff849")]
pub trait ICoreWebView2WebMessageReceivedEventArgs: IUnknown {
    /// The URI of the document that sent this web message.
    unsafe fn get_source(&self, /* out, retval */ source: *mut LPWSTR) -> HRESULT;

    /// The message posted from the WebView content to the host converted to a
    /// JSON string.  Run this operation to communicate using JavaScript objects.
    ///
    /// For example, the following `postMessage` runs result in the following
    /// `WebMessageAsJson` values.
    ///
    /// ```json
    /// postMessage({'a': 'b'})      L"{\"a\": \"b\"}"
    /// postMessage(1.2)             L"1.2"
    /// postMessage('example')       L"\"example\""
    /// ```
    unsafe fn get_web_message_as_json(
        &self,
        /* out, retval */ web_message_as_json: *mut LPWSTR,
    ) -> HRESULT;

    /// If the message posted from the WebView content to the host is a string
    /// type, this method returns the value of that string.  If the message
    /// posted is some other kind of JavaScript type this method fails with the
    /// following error.
    ///
    /// ```text
    /// E_INVALIDARG
    /// ```
    ///
    /// Run this operation to communicate using simple strings.
    ///
    /// For example, the following `postMessage` runs result in the following
    /// `WebMessageAsString` values.
    ///
    /// ```json
    /// postMessage({'a': 'b'})      E_INVALIDARG
    /// postMessage(1.2)             E_INVALIDARG
    /// postMessage('example')       L"example"
    /// ```
    unsafe fn try_get_web_message_as_string(
        &self,
        /* out, retval */ web_message_as_string: *mut LPWSTR,
    ) -> HRESULT;
}

/// Receives `WebMessageReceived` events.
#[com_interface("57213f19-00e6-49fa-8e07-898ea01ecbd2")]
pub trait ICoreWebView2WebMessageReceivedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut ICoreWebView2WebMessageReceivedEventArgsVTable,
    ) -> HRESULT;
}

/// Event args for the `DevToolsProtocolEventReceived` event.
#[com_interface("653c2959-bb3a-4377-8632-b58ada4e66c4")]
pub trait ICoreWebView2DevToolsProtocolEventReceivedEventArgs: IUnknown {
    /// The parameter object of the corresponding `DevToolsProtocol` event
    /// represented as a JSON string.
    unsafe fn get_parameter_object_as_json(
        &self,
        /* out, retval */ parameter_object_as_json: *mut LPWSTR,
    ) -> HRESULT;
}

/// Receives `DevToolsProtocolEventReceived` events from the WebView.
#[com_interface("e2fda4be-5456-406c-a261-3d452138362c")]
pub trait ICoreWebView2DevToolsProtocolEventReceivedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut ICoreWebView2DevToolsProtocolEventReceivedEventArgsVTable,
    ) -> HRESULT;
}

/// Receives `CallDevToolsProtocolMethod` completion results.
#[com_interface("5c4889f0-5ef6-4c5a-952c-d8f1b92d0574")]
pub trait ICoreWebView2CallDevToolsProtocolMethodCompletedHandler: IUnknown {
    /// Provides the completion status and result of the corresponding
    /// asynchronous method.
    unsafe fn invoke(
        &self,
        /* in */ error_code: HRESULT,
        /* in */ return_object_as_json: LPCWSTR,
    ) -> HRESULT;
}

/// Receives the `CoreWebView2Controller` created using `CreateCoreWebView2Controller`.
#[com_interface("6c4819f3-c9b7-4260-8127-c9f5bde7f68c")]
pub trait ICoreWebView2CreateCoreWebView2ControllerCompletedHandler: IUnknown {
    /// Provides the completion status and result of the corresponding
    /// asynchronous method.
    unsafe fn invoke(
        &self,
        error_code: HRESULT,
        created_controller: *mut *mut ICoreWebView2ControllerVTable,
    ) -> HRESULT;
}

/// The caller implements this interface to receive the CoreWebView2Controller
/// created via CreateCoreWebView2CompositionController.
#[com_interface("02fab84b-1428-4fb7-ad45-1b2e64736184")]
pub trait ICoreWebView2CreateCoreWebView2CompositionControllerCompletedHandler: IUnknown {
    /// Called to provide the implementer with the completion status and result
    /// of the corresponding asynchronous method call.
    unsafe fn invoke(
        &self,
        error_code: HRESULT,
        web_view: *mut *mut ICoreWebView2CompositionControllerVTable,
    ) -> HRESULT;
}

/// Event args for the `NewWindowRequested` event.  The event is run when
/// content inside webview requested to a open a new window (through
/// `window.open()` and so on).
#[com_interface("34acb11c-fc37-4418-9132-f9c21d1eafb9")]
pub trait ICoreWebView2NewWindowRequestedEventArgs: IUnknown {
    /// The target uri of the new window requested.
    unsafe fn get_uri(&self, /* out, retval */ uri: *mut LPWSTR) -> HRESULT;

    /// Sets a CoreWebView2 as a result of the NewWindowRequested event. If the
    /// `NewWindow` is set, the top-level window returns as the opened `WindowProxy`.
    /// The NewWindow property should be set to a CoreWebView2 that has not been
    /// navigated previously. Don't use methods that cause navigation or interact
    /// with the DOM on this CoreWebView2. Setting event handlers, changing
    /// Settings properties, or other methods are fine to call. Once the
    /// NewWindow is set the underlying web contents of this CoreWebView2 will be
    /// replaced and navigated as appropriate for the new window.
    /// After setting new window it cannot be changed and error will be return otherwise.
    ///
    /// The methods which should affect the new web contents like
    /// AddScriptToExecuteOnDocumentCreated and add_WebResourceRequested
    /// have to be called after setting NewWindow.
    unsafe fn put_new_window(
        &self,
        /* in */ new_window: *mut *mut ICoreWebView2VTable,
    ) -> HRESULT;

    /// Gets the new window.
    unsafe fn get_new_window(
        &self,
        /* out, retval */ new_window: *mut *mut *mut ICoreWebView2VTable,
    ) -> HRESULT;

    /// Sets whether the `NewWindowRequested` event is handled by host.  If this
    /// is `FALSE` and no `NewWindow` is set, the WebView opens a popup window
    /// and it returns as opened `WindowProxy`.  If set to `TRUE` and no
    /// `NewWindow` is set for `window.open`, the opened `WindowProxy` is for an
    /// testing window object and no window loads.
    /// The default value is `FALSE`.
    unsafe fn put_handled(&self, /* in */ handled: BOOL) -> HRESULT;

    /// Gets whether the `NewWindowRequested` event is handled by host.
    unsafe fn get_handled(&self, /* out, retval */ handled: *mut BOOL) -> HRESULT;

    /// `TRUE` when the new window request was initiated through a user gesture
    /// such as selecting an anchor tag with target.  The Microsoft Edge popup
    /// blocker is disabled for WebView so the app is able to use this flag to
    /// block non-user initiated popups.
    unsafe fn get_is_user_initiated(
        &self,
        /* out, retval */ is_user_initiated: *mut BOOL,
    ) -> HRESULT;

    /// Obtain an `ICoreWebView2Deferral` object and put the event into a
    /// deferred state.  Use the `ICoreWebView2Deferral` object to complete the
    /// window open request at a later time.  While this event is deferred the
    /// opener window returns a `WindowProxy` to an un-navigated window, which
    /// navigates when the deferral is complete.
    unsafe fn get_deferral(
        &self,
        /* out, retval */ deferral: *mut *mut *mut ICoreWebView2DeferralVTable,
    ) -> HRESULT;

    /// Window features specified by the `window.open`.  The features should be
    /// considered for positioning and sizing of new webview windows.
    unsafe fn get_window_features(
        &self,
        /* out, retval */ value: *mut *mut *mut ICoreWebView2WindowFeaturesVTable,
    ) -> HRESULT;
}

/// The window features for a WebView popup window.  The fields match the
/// `windowFeatures` passed to `window.open` as specified in
/// \[Window features\]\[MdnDocsWebApiWindowOpenWindowFeatures\]
/// on MDN.
/// There is no requirement for you to respect the values.  If your app does
/// not have corresponding UI features (for example, no toolbar) or if all
/// instance of WebView are opened in tabs and do not have distinct size or
/// positions, then your app does not respect the values.  You may want to
/// respect values, but perhaps only some apply to the UI of you app.
/// Accordingly, you may respect all, some, or none of the properties as
/// appropriate for your app.  For all numeric properties, if the value that is
/// passed to `window.open` is outside the range of an unsigned 32bit int, the
/// resulting value is the absolute value of the maximum for unsigned 32bit
/// integer.  If you are not able to parse the value an integer, it is
/// considered `0`.  If the value is a floating point value, it is rounded down
/// to an integer.
///
/// \[MdnDocsWebApiWindowOpenWindowFeatures\]: https://developer.mozilla.org/docs/Web/API/Window/open#Window_features "Window features - Window.open() | MDN"
#[com_interface("5eaf559f-b46e-4397-8860-e422f287ff1e")]
pub trait ICoreWebView2WindowFeatures: IUnknown {
    /// Specifies left and top values.
    unsafe fn get_has_position(&self, /* out, retval */ value: *mut BOOL) -> HRESULT;

    /// Specifiesheight and width values.
    unsafe fn get_has_size(&self, /* out, retval */ value: *mut BOOL) -> HRESULT;

    /// Specifies the left position of the window.   If `HasPosition` is set to
    /// `FALSE`, this field is ignored.
    unsafe fn get_left(&self, /* out, retval */ value: *mut u32) -> HRESULT;

    /// Specifies the top position of the window.   If `HasPosition` is set to
    /// `FALSE`, this field is ignored.
    unsafe fn get_top(&self, /* out, retval */ value: *mut u32) -> HRESULT;

    /// Specifies the height of the window.  Minimum value is `100`.  If
    /// `HasSize` is set to `FALSE`, this field is ignored.
    unsafe fn get_height(&self, /* out, retval */ value: *mut u32) -> HRESULT;

    /// Specifies the width of the window.  Minimum value is `100`.  If `HasSize`
    /// is set to `FALSE`, this field is ignored.
    unsafe fn get_width(&self, /* out, retval */ value: *mut u32) -> HRESULT;

    /// Indicates that the menu bar is displayed.
    unsafe fn get_should_display_menu_bar(
        &self,
        /* out, retval */ value: *mut BOOL,
    ) -> HRESULT;

    /// Indicates that the status bar is displayed.
    unsafe fn get_should_display_status(&self, /* out, retval */ value: *mut BOOL) -> HRESULT;

    /// Indicates that the browser toolbar is displayed.
    unsafe fn get_should_display_toolbar(&self, /* out, retval */ value: *mut BOOL) -> HRESULT;

    /// Indicates that the scroll bars are displayed.
    unsafe fn get_should_display_scroll_bars(
        &self,
        /* out, retval */ value: *mut BOOL,
    ) -> HRESULT;
}

/// Receives `NewWindowRequested` events.
#[com_interface("d4c185fe-c81c-4989-97af-2d3fa7ab5651")]
pub trait ICoreWebView2NewWindowRequestedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut ICoreWebView2NewWindowRequestedEventArgsVTable,
    ) -> HRESULT;
}

/// Receives `DocumentTitleChanged` events.  Use the `DocumentTitle` property
/// to get the modified title.
#[com_interface("f5f2b923-953e-4042-9f95-f3a118e1afd4")]
pub trait ICoreWebView2DocumentTitleChangedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.  No event args exist
    /// and the `args` parameter is set to `null`.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut IUnknownVTable,
    ) -> HRESULT;
}

/// Event args for the `AcceleratorKeyPressed` event.
#[com_interface("9f760f8a-fb79-42be-9990-7b56900fa9c7")]
pub trait ICoreWebView2AcceleratorKeyPressedEventArgs: IUnknown {
    /// The key event type that caused the event to run.
    unsafe fn get_key_event_kind(
        &self,
        /* out, retval */ key_event_kind: *mut KeyEventKind,
    ) -> HRESULT;

    /// The Win32 virtual key code of the key that was pressed or released.  It
    /// is one of the Win32 virtual key constants such as `VK_RETURN` or an
    /// (uppercase) ASCII value such as `A`.  Verify whether Ctrl or Alt
    /// are pressed by running `GetKeyState(VK_CONTROL)` or
    /// `GetKeyState(VK_MENU)`.
    unsafe fn get_virtual_key(&self, /* out, retval */ virtual_key: *mut u32) -> HRESULT;

    /// The `LPARAM` value that accompanied the window message.  For more
    /// information, navigate to
    /// \[WM_KEYDOWN\]\[WindowsWin32InputdevWmKeydown\]
    /// and
    /// \[WM_KEYUP\]\[WindowsWin32InputdevWmKeyup\].
    ///
    /// \[WindowsWin32InputdevWmKeydown\]: /windows/win32/inputdev/wm-keydown "WM_KEYDOWN message | Microsoft Docs"
    ///
    /// \[WindowsWin32InputdevWmKeyup\]: /windows/win32/inputdev/wm-keyup "WM_KEYUP message | Microsoft Docs"
    unsafe fn get_key_event_lparam(&self, /* out, retval */ l_param: *mut i32) -> HRESULT;

    /// A structure representing the information passed in the `LPARAM` of the
    /// window message.
    unsafe fn get_physical_key_status(
        &self,
        /* out, retval */ physical_key_status: *mut PhysicalKeyStatus,
    ) -> HRESULT;

    /// During `AcceleratorKeyPressedEvent` handler invocation the WebView is
    /// blocked waiting for the decision of if the accelerator is handled by the
    /// host (or not).  If the `Handled` property is set to `TRUE` then this
    /// prevents the WebView from performing the default action for this
    /// accelerator key.  Otherwise the WebView performs the default action for
    /// the accelerator key.
    unsafe fn get_handled(&self, /* out, retval */ handled: *mut BOOL) -> HRESULT;

    /// Sets the `Handled` property.
    unsafe fn put_handled(&self, /* in */ handled: BOOL) -> HRESULT;
}

/// Receives `AcceleratorKeyPressed` events.
#[com_interface("b29c7e28-fa79-41a8-8e44-65811c76dcb2")]
pub trait ICoreWebView2AcceleratorKeyPressedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2ControllerVTable,
        /* in */ args: *mut *mut ICoreWebView2AcceleratorKeyPressedEventArgsVTable,
    ) -> HRESULT;
}

/// Receives `NewBrowserVersionAvailable` events.
#[com_interface("f9a2976e-d34e-44fc-adee-81b6b57ca914")]
pub trait ICoreWebView2NewBrowserVersionAvailableEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2EnvironmentVTable,
        /* in */ args: *mut *mut IUnknownVTable,
    ) -> HRESULT;
}

/// Receives `ContainsFullScreenElementChanged` events.
#[com_interface("e45d98b1-afef-45be-8baf-6c7728867f73")]
pub trait ICoreWebView2ContainsFullScreenElementChangedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.  No event args exist
    /// and the `args` parameter is set to `null`.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut IUnknownVTable,
    ) -> HRESULT;
}

/// Receives `WindowCloseRequested` events.
#[com_interface("5c19e9e0-092f-486b-affa-ca8231913039")]
pub trait ICoreWebView2WindowCloseRequestedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.  No event args exist
    /// and the `args` parameter is set to `null`.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut IUnknownVTable,
    ) -> HRESULT;
}

/// Receives `WebResourceResponseReceived` events.
#[com_interface("7DE9898A-24F5-40C3-A2DE-D4F458E69828")]
pub trait ICoreWebView2WebResourceResponseReceivedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut ICoreWebView2WebResourceResponseReceivedEventArgsVTable,
    ) -> HRESULT;
}

/// Event args for the WebResourceResponseReceived event.
#[com_interface("D1DB483D-6796-4B8B-80FC-13712BB716F4")]
pub trait ICoreWebView2WebResourceResponseReceivedEventArgs: IUnknown {
    /// The request object for the web resource, as committed. This includes
    /// headers added by the network stack that were not be included during the
    /// associated WebResourceRequested event, such as Authentication headers.
    /// Modifications to this object have no effect on how the request is
    /// processed as it has already been sent.
    unsafe fn get_request(
        &self,
        /* out, retval */ request: *mut *mut *mut ICoreWebView2WebResourceRequestVTable,
    ) -> HRESULT;

    /// View of the response object received for the web resource.
    unsafe fn get_response(
        &self,
        /* out, retval */
        response: *mut *mut *mut ICoreWebView2WebResourceResponseViewVTable,
    ) -> HRESULT;
}

/// View of the HTTP representation for a web resource response. The properties
/// of this object are not mutable. This response view is used with the
/// WebResourceResponseReceived event.
#[com_interface("79701053-7759-4162-8F7D-F1B3F084928D")]
pub trait ICoreWebView2WebResourceResponseView: IUnknown {
    /// The HTTP response headers as received.
    unsafe fn get_headers(
        &self,
        /* out, retval */ headers: *mut *mut *mut ICoreWebView2HttpResponseHeadersVTable,
    ) -> HRESULT;

    /// The HTTP response status code.
    unsafe fn get_status_code(&self, /* out, retval */ status_code: *mut i32) -> HRESULT;

    /// The HTTP response reason phrase.
    unsafe fn get_reason_phrase(
        &self,
        /* out, retval */ reason_phrase: *mut LPWSTR,
    ) -> HRESULT;

    /// Get the response content asynchronously. The handler will receive the
    /// response content stream.
    /// If this method is being called again before a first call has completed,
    /// the handler will be invoked at the same time the handlers from prior calls
    /// are invoked.
    /// If this method is being called after a first call has completed, the
    /// handler will be invoked immediately.
    /// \snippet ScenarioWebViewEventMonitor.cpp GetContent
    unsafe fn get_content(
        &self,
        /* in */
        handler: *mut *mut ICoreWebView2WebResourceResponseViewGetContentCompletedHandlerVTable,
    ) -> HRESULT;
}

/// Receives the result of the
/// `ICoreWebView2WebResourceResponseView::GetContent` method.
#[com_interface("875738E1-9FA2-40E3-8B74-2E8972DD6FE7")]
pub trait ICoreWebView2WebResourceResponseViewGetContentCompletedHandler: IUnknown {
    /// Provides the completion status and result of the corresponding
    /// asynchronous method call. A failure `errorCode` will be passed if the
    /// content failed to load. `E_ABORT` means the response loading was blocked
    /// (e.g., by CORS policy); `ERROR_CANCELLED` means the response loading was
    /// cancelled. `ERROR_NO_DATA` means the response has no content data,
    /// `content` is `null` in this case. Note content (if any) is ignored for
    /// redirects, 204 No Content, 205 Reset Content, and HEAD-request responses.
    unsafe fn invoke(
        &self,
        /* in */ error_code: HRESULT,
        /* in */ content: *mut *mut IStreamVTable,
    ) -> HRESULT;
}

/// Event args for the DOMContentLoaded event.
#[com_interface("16B1E21A-C503-44F2-84C9-70ABA5031283")]
pub trait ICoreWebView2DOMContentLoadedEventArgs: IUnknown {
    /// The ID of the navigation which corresponds to other navigation ID properties on other navigation events.
    unsafe fn get_navigation_id(&self, /* out, retval */ navigation_id: *mut u64) -> HRESULT;
}

/// Receives `DOMContentLoaded` events.
#[com_interface("4BAC7E9C-199E-49ED-87ED-249303ACF019")]
pub trait ICoreWebView2DOMContentLoadedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut ICoreWebView2DOMContentLoadedEventArgsVTable,
    ) -> HRESULT;
}

/// Provides a set of properties that are used to manage an
/// ICoreWebView2Cookie.
///
/// \snippet ScenarioCookieManagement.cpp CookieObject
#[com_interface("AD26D6BE-1486-43E6-BF87-A2034006CA21")]
pub trait ICoreWebView2Cookie: IUnknown {
    /// Cookie name.
    unsafe fn get_name(&self, /* out, retval */ name: *mut LPWSTR) -> HRESULT;

    /// Cookie value.
    unsafe fn get_value(&self, /* out, retval */ value: *mut LPWSTR) -> HRESULT;

    /// Set the cookie value property.
    unsafe fn put_value(&self, /* in */ value: LPCWSTR) -> HRESULT;

    /// The domain for which the cookie is valid.
    /// The default is the host that this cookie has been received from.
    /// Note that, for instance, ".bing.com", "bing.com", and "www.bing.com" are
    /// considered different domains.
    unsafe fn get_domain(&self, /* out, retval */ domain: *mut LPWSTR) -> HRESULT;

    /// The path for which the cookie is valid. The default is "/", which means
    /// this cookie will be sent to all pages on the Domain.
    unsafe fn get_path(&self, /* out, retval */ path: *mut LPWSTR) -> HRESULT;

    /// The expiration date and time for the cookie as the number of seconds since the UNIX epoch.
    /// The default is -1.0, which means cookies are session cookies by default.
    unsafe fn get_expires(&self, /* out, retval */ expires: *mut f64) -> HRESULT;

    /// Set the Expires property. Cookies are session cookies and will not be
    /// persistent if Expires is set to -1.0. NaN, infinity, and any negative
    /// value set other than -1.0 is disallowed.
    unsafe fn put_expires(&self, /* in */ expires: f64) -> HRESULT;

    /// Whether this cookie is http-only.
    /// True if a page script or other active content cannot access this
    /// cookie. The default is false.
    unsafe fn get_is_http_only(&self, /* out, retval */ is_http_only: *mut BOOL) -> HRESULT;

    /// Set the IsHttpOnly property.
    unsafe fn put_is_http_only(&self, /* in */ is_http_only: BOOL) -> HRESULT;

    /// SameSite status of the cookie which represents the enforcement mode of the cookie.
    /// The default is COREWEBVIEW2_COOKIE_SAME_SITE_KIND_LAX.
    unsafe fn get_same_site(
        &self,
        /* out, retval */ same_site: *mut CookieSameSiteKind,
    ) -> HRESULT;

    /// Set the SameSite property.
    unsafe fn put_same_site(&self, /* in */ same_site: CookieSameSiteKind) -> HRESULT;

    /// The security level of this cookie. True if the client is only to return
    /// the cookie in subsequent requests if those requests use HTTPS.
    /// The default is false.
    /// Note that cookie that requests COREWEBVIEW2_COOKIE_SAME_SITE_KIND_NONE but
    /// is not marked Secure will be rejected.
    unsafe fn get_is_secure(&self, /* out, retval */ is_secure: *mut BOOL) -> HRESULT;

    /// Set the IsSecure property.
    unsafe fn put_is_secure(&self, /* in */ is_secure: BOOL) -> HRESULT;

    /// Whether this is a session cookie. The default is false.
    unsafe fn get_is_session(&self, /* out, retval */ is_session: *mut BOOL) -> HRESULT;
}

/// Creates, adds or updates, gets, or or view the cookies. The changes would
/// apply to the context of the user profile. That is, other WebViews under the
/// same user profile could be affected.
#[com_interface("177CD9E7-B6F5-451A-94A0-5D7A3A4C4141")]
pub trait ICoreWebView2CookieManager: IUnknown {
    /// Create a cookie object with a specified name, value, domain, and path.
    /// One can set other optional properties after cookie creation.
    /// This only creates a cookie object and it is not added to the cookie
    /// manager until you call AddOrUpdateCookie.
    /// Leading or trailing whitespace(s), empty string, and special characters
    /// are not allowed for name.
    /// See ICoreWebView2Cookie for more details.
    unsafe fn create_cookie(
        &self,
        /* in */ name: LPCWSTR,
        /* in */ value: LPCWSTR,
        /* in */ domain: LPCWSTR,
        /* in */ path: LPCWSTR,
        /* out, retval */ cookie: *mut *mut *mut ICoreWebView2CookieVTable,
    ) -> HRESULT;

    /// Creates a cookie whose params matches those of the specified cookie.
    unsafe fn copy_cookie(
        &self,
        /* in */ cookie_param: *mut *mut ICoreWebView2CookieVTable,
        /* out, retval */ cookie: *mut *mut *mut ICoreWebView2CookieVTable,
    ) -> HRESULT;

    /// Gets a list of cookies matching the specific URI.
    /// If uri is empty string or null, all cookies under the same profile are
    /// returned.
    /// You can modify the cookie objects by calling
    /// ICoreWebView2CookieManager::AddOrUpdateCookie, and the changes
    /// will be applied to the webview.
    /// \snippet ScenarioCookieManagement.cpp GetCookies
    unsafe fn get_cookies(
        &self,
        /* in */ uri: LPCWSTR,
        /* in */ handler: *mut *mut ICoreWebView2GetCookiesCompletedHandlerVTable,
    ) -> HRESULT;

    /// Adds or updates a cookie with the given cookie data; may overwrite
    /// cookies with matching name, domain, and path if they exist.
    /// This method will fail if the domain of the given cookie is not specified.
    /// \snippet ScenarioCookieManagement.cpp AddOrUpdateCookie
    unsafe fn add_or_update_cookie(
        &self,
        /* in */ cookie: *mut *mut ICoreWebView2CookieVTable,
    ) -> HRESULT;

    /// Deletes a cookie whose name and domain/path pair
    /// match those of the specified cookie.
    unsafe fn delete_cookie(
        &self,
        /* in */ cookie: *mut *mut ICoreWebView2CookieVTable,
    ) -> HRESULT;

    /// Deletes cookies with matching name and uri.
    /// Cookie name is required.
    /// All cookies with the given name where domain
    /// and path match provided URI are deleted.
    unsafe fn delete_cookies(
        &self,
        /* in */ name: LPCWSTR,
        /* in */ uri: LPCWSTR,
    ) -> HRESULT;

    /// Deletes cookies with matching name and domain/path pair.
    /// Cookie name is required.
    /// If domain is specified, deletes only cookies with the exact domain.
    /// If path is specified, deletes only cookies with the exact path.
    unsafe fn delete_cookies_with_domain_and_path(
        &self,
        /* in */ name: LPCWSTR,
        /* in */ domain: LPCWSTR,
        /* in */ path: LPCWSTR,
    ) -> HRESULT;

    /// Deletes all cookies under the same profile.
    /// This could affect other WebViews under the same user profile.
    unsafe fn delete_all_cookies(&self) -> HRESULT;
}

/// A list of cookie objects. See `ICoreWebView2Cookie`.
/// \snippet ScenarioCookieManagement.cpp GetCookies
#[com_interface("F7F6F714-5D2A-43C6-9503-346ECE02D186")]
pub trait ICoreWebView2CookieList: IUnknown {
    /// The number of cookies contained in the ICoreWebView2CookieList.
    unsafe fn get_count(&self, /* out, retval */ count: *mut u32) -> HRESULT;

    /// Gets the cookie object at the given index.
    unsafe fn get_value_at_index(
        &self,
        /* in */ index: u32,
        /* out, retval */ cookie: *mut *mut *mut ICoreWebView2CookieVTable,
    ) -> HRESULT;
}

/// Receives the result of the `GetCookies` method.  The result is written to
/// the cookie list provided in the `GetCookies` method call.
#[com_interface("5A4F5069-5C15-47C3-8646-F4DE1C116670")]
pub trait ICoreWebView2GetCookiesCompletedHandler: IUnknown {
    /// Provides the completion status of the corresponding asynchronous method
    /// call.
    unsafe fn invoke(
        &self,
        result: HRESULT,
        cookie_list: *mut *mut ICoreWebView2CookieListVTable,
    ) -> HRESULT;
}

/// Provides access to the certificate metadata
#[com_interface("e7188076-bcc3-11eb-8529-0242ac130003")]
pub trait ICoreWebView2ClientCertificate: IUnknown {
    /// Subject of the certificate.
    unsafe fn get_subject(&self, /* out, retval */ value: *mut LPWSTR) -> HRESULT;

    /// Name of the certificate authority that issued the certificate.
    unsafe fn get_issuer(&self, /* out, retval */ value: *mut LPWSTR) -> HRESULT;

    /// The valid start date and time for the certificate as the number of seconds since
    /// the UNIX epoch.
    unsafe fn get_valid_from(&self, /* out, retval */ value: *mut f64) -> HRESULT;

    /// The valid expiration date and time for the certificate as the number of seconds since
    /// the UNIX epoch.
    unsafe fn get_valid_to(&self, /* out, retval */ value: *mut f64) -> HRESULT;

    /// DER encoded serial number of the certificate.
    /// Read more about DER at [RFC 7468 DER]
    /// (https://tools.ietf.org/html/rfc7468#appendix-B).
    unsafe fn get_der_encoded_serial_number(
        &self,
        /* out, retval */ value: *mut LPWSTR,
    ) -> HRESULT;

    /// Display name for a certificate.
    unsafe fn get_display_name(&self, /* out, retval */ value: *mut LPWSTR) -> HRESULT;

    /// PEM encoded data for the certificate.
    /// Returns Base64 encoding of DER encoded certificate.
    /// Read more about PEM at [RFC 1421 Privacy Enhanced Mail]
    /// (https://tools.ietf.org/html/rfc1421).
    unsafe fn to_pem_encoding(
        &self,
        /* out, retval */ pem_encoded_data: *mut LPWSTR,
    ) -> HRESULT;

    /// Collection of PEM encoded client certificate issuer chain.
    /// In this collection first element is the current certificate followed by
    /// intermediate1, intermediate2...intermediateN-1. Root certificate is the
    /// last element in collection.
    unsafe fn get_pem_encoded_issuer_certificate_chain(
        &self,
        /* out, retval */ value: *mut *mut *mut ICoreWebView2StringCollectionVTable,
    ) -> HRESULT;

    /// Kind of a certificate (eg., smart card, pin, other).
    unsafe fn get_kind(&self, /* out, retval */ value: *mut ClientCertificateKind) -> HRESULT;
}

/// A collection of client certificate object.
#[com_interface("ef5674d2-bcc3-11eb-8529-0242ac130003")]
pub trait ICoreWebView2ClientCertificateCollection: IUnknown {
    /// The number of client certificates contained in the
    /// ICoreWebView2ClientCertificateCollection.
    unsafe fn get_count(&self, /* out, retval */ value: *mut u32) -> HRESULT;

    /// Gets the certificate object at the given index.
    unsafe fn get_value_at_index(
        &self,
        /* in */ index: u32,
        /* out, retval */ certificate: *mut *mut *mut ICoreWebView2ClientCertificateVTable,
    ) -> HRESULT;
}

/// A collection of strings.
#[com_interface("f41f3f8a-bcc3-11eb-8529-0242ac130003")]
pub trait ICoreWebView2StringCollection: IUnknown {
    /// The number of strings contained in ICoreWebView2StringCollection.
    unsafe fn get_count(&self, /* out, retval */ value: *mut u32) -> HRESULT;

    /// Gets the value at a given index.
    unsafe fn get_value_at_index(
        &self,
        /* in */ index: u32,
        /* out, retval */ value: *mut LPWSTR,
    ) -> HRESULT;
}

/// An event handler for the `ClientCertificateRequested` event.
#[com_interface("d7175ba2-bcc3-11eb-8529-0242ac130003")]
pub trait ICoreWebView2ClientCertificateRequestedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut ICoreWebView2ClientCertificateRequestedEventArgsVTable,
    ) -> HRESULT;
}

/// Event args for the `ClientCertificateRequested` event.
#[com_interface("bc59db28-bcc3-11eb-8529-0242ac130003")]
pub trait ICoreWebView2ClientCertificateRequestedEventArgs: IUnknown {
    /// Host name of the server that requested client certificate authentication.
    /// Normalization rules applied to the hostname are:
    /// * Convert to lowercase characters for ascii characters.
    /// * Punycode is used for representing non ascii characters.
    /// * Strip square brackets for IPV6 address.
    unsafe fn get_host(&self, /* out, retval */ value: *mut LPWSTR) -> HRESULT;

    /// Port of the server that requested client certificate authentication.
    unsafe fn get_port(&self, /* out, retval */ value: *mut i32) -> HRESULT;

    /// Returns true if the server that issued this request is an http proxy.
    /// Returns false if the server is the origin server.
    unsafe fn get_is_proxy(&self, /* out, retval */ value: *mut BOOL) -> HRESULT;

    /// Returns the `ICoreWebView2StringCollection`.
    /// The collection contains distinguished names of certificate authorities
    /// allowed by the server.
    unsafe fn get_allowed_certificate_authorities(
        &self,
        /* out, retval */ value: *mut *mut *mut ICoreWebView2StringCollectionVTable,
    ) -> HRESULT;

    /// Returns the `ICoreWebView2ClientCertificateCollection` when client
    /// certificate authentication is requested. The collection contains mutually
    /// trusted CA certificates.
    unsafe fn get_mutually_trusted_certificates(
        &self,
        /* out, retval */
        value: *mut *mut *mut ICoreWebView2ClientCertificateCollectionVTable,
    ) -> HRESULT;

    /// Returns the selected certificate.
    unsafe fn get_selected_certificate(
        &self,
        /* out, retval */ value: *mut *mut *mut ICoreWebView2ClientCertificateVTable,
    ) -> HRESULT;

    /// Sets the certificate to respond to the server.
    unsafe fn put_selected_certificate(
        &self,
        /* in */ value: *mut *mut ICoreWebView2ClientCertificateVTable,
    ) -> HRESULT;

    /// You�may�set�this�flag�to�cancel�the�certificate selection. If canceled,
    /// the request is aborted regardless of the `Handled` property. By default the
    /// value is `FALSE`.
    unsafe fn get_cancel(&self, /* out, retval */ value: *mut BOOL) -> HRESULT;

    ///�Sets�the�`Cancel`�property.
    unsafe fn put_cancel(&self, /* in */ value: BOOL) -> HRESULT;

    /// You�may�set�this�flag�to�`TRUE` to respond to the server with or
    /// without a certificate. If this flag is `TRUE` with a `SelectedCertificate`
    /// it responds to the server with the selected certificate otherwise respond to the
    /// server without a certificate. By default the value of `Handled` and `Cancel` are `FALSE`
    /// and display default client certificate selection dialog prompt to allow the user to
    /// choose a certificate. The `SelectedCertificate` is ignored unless `Handled` is set `TRUE`.
    unsafe fn get_handled(&self, /* out, retval */ value: *mut BOOL) -> HRESULT;

    ///�Sets�the�`Handled`�property.
    unsafe fn put_handled(&self, /* in */ value: BOOL) -> HRESULT;

    /// Returns an `ICoreWebView2Deferral` object. Use this operation to
    /// complete the event at a later time.
    unsafe fn get_deferral(
        &self,
        /* out, retval */ deferral: *mut *mut *mut ICoreWebView2DeferralVTable,
    ) -> HRESULT;
}

/// This mostly represents a combined win32
/// POINTER_INFO/POINTER_TOUCH_INFO/POINTER_PEN_INFO object. It takes fields
/// from all three and excludes some win32 specific data types like HWND and
/// HANDLE. Note, sourceDevice is taken out but we expect the PointerDeviceRect
/// and DisplayRect to cover the existing use cases of sourceDevice.
/// Another big difference is that any of the point or rect locations are
/// expected to be in WebView physical coordinates. That is, coordinates
/// relative to the WebView and no DPI scaling applied.
#[com_interface("e6995887-d10d-4f5d-9359-4ce46e4f96b9")]
pub trait ICoreWebView2PointerInfo: IUnknown {
    /// Get the PointerKind of the pointer event. This corresponds to the
    /// pointerKind property of the POINTER_INFO struct. The values are defined by
    /// the POINTER_INPUT_KIND enum in the Windows SDK (winuser.h). Supports
    /// PT_PEN and PT_TOUCH.
    unsafe fn get_pointer_kind(&self, /* out, retval */ pointer_kind: *mut DWORD) -> HRESULT;

    /// Set the PointerKind of the pointer event. This corresponds to the
    /// pointerKind property of the POINTER_INFO struct. The values are defined by
    /// the POINTER_INPUT_KIND enum in the Windows SDK (winuser.h). Supports
    /// PT_PEN and PT_TOUCH.
    unsafe fn put_pointer_kind(&self, /* in */ pointer_kind: DWORD) -> HRESULT;

    /// Get the PointerId of the pointer event. This corresponds to the pointerId
    /// property of the POINTER_INFO struct as defined in the Windows SDK
    /// (winuser.h).
    unsafe fn get_pointer_id(&self, /* out, retval */ pointer_id: *mut u32) -> HRESULT;

    /// Set the PointerId of the pointer event. This corresponds to the pointerId
    /// property of the POINTER_INFO struct as defined in the Windows SDK
    /// (winuser.h).
    unsafe fn put_pointer_id(&self, /* in */ pointer_id: u32) -> HRESULT;

    /// Get the FrameID of the pointer event. This corresponds to the frameId
    /// property of the POINTER_INFO struct as defined in the Windows SDK
    /// (winuser.h).
    unsafe fn get_frame_id(&self, /* out, retval */ frame_id: *mut u32) -> HRESULT;

    /// Set the FrameID of the pointer event. This corresponds to the frameId
    /// property of the POINTER_INFO struct as defined in the Windows SDK
    /// (winuser.h).
    unsafe fn put_frame_id(&self, /* in */ frame_id: u32) -> HRESULT;

    /// Get the PointerFlags of the pointer event. This corresponds to the
    /// pointerFlags property of the POINTER_INFO struct. The values are defined
    /// by the POINTER_FLAGS constants in the Windows SDK (winuser.h).
    unsafe fn get_pointer_flags(&self, /* out, retval */ pointer_flags: *mut u32) -> HRESULT;

    /// Set the PointerFlags of the pointer event. This corresponds to the
    /// pointerFlags property of the POINTER_INFO struct. The values are defined
    /// by the POINTER_FLAGS constants in the Windows SDK (winuser.h).
    unsafe fn put_pointer_flags(&self, /* in */ pointer_flags: u32) -> HRESULT;

    /// Get the PointerDeviceRect of the sourceDevice property of the
    /// POINTER_INFO struct as defined in the Windows SDK (winuser.h).
    unsafe fn get_pointer_device_rect(
        &self,
        /* out, retval */ pointer_device_rect: *mut RECT,
    ) -> HRESULT;

    /// Set the PointerDeviceRect of the sourceDevice property of the
    /// POINTER_INFO struct as defined in the Windows SDK (winuser.h).
    unsafe fn put_pointer_device_rect(&self, /* in */ pointer_device_rect: RECT) -> HRESULT;

    /// Get the DisplayRect of the sourceDevice property of the POINTER_INFO
    /// struct as defined in the Windows SDK (winuser.h).
    unsafe fn get_display_rect(&self, /* out, retval */ display_rect: *mut RECT) -> HRESULT;

    /// Set the DisplayRect of the sourceDevice property of the POINTER_INFO
    /// struct as defined in the Windows SDK (winuser.h).
    unsafe fn put_display_rect(&self, /* in */ display_rect: RECT) -> HRESULT;

    /// Get the PixelLocation of the pointer event. This corresponds to the
    /// ptPixelLocation property of the POINTER_INFO struct as defined in the
    /// Windows SDK (winuser.h).
    unsafe fn get_pixel_location(
        &self,
        /* out, retval */ pixel_location: *mut POINT,
    ) -> HRESULT;

    /// Set the PixelLocation of the pointer event. This corresponds to the
    /// ptPixelLocation property of the POINTER_INFO struct as defined in the
    /// Windows SDK (winuser.h).
    unsafe fn put_pixel_location(&self, /* in */ pixel_location: POINT) -> HRESULT;

    /// Get the HimetricLocation of the pointer event. This corresponds to the
    /// ptHimetricLocation property of the POINTER_INFO struct as defined in the
    /// Windows SDK (winuser.h).
    unsafe fn get_himetric_location(
        &self,
        /* out, retval */ himetric_location: *mut POINT,
    ) -> HRESULT;

    /// Set the HimetricLocation of the pointer event. This corresponds to the
    /// ptHimetricLocation property of the POINTER_INFO struct as defined in the
    /// Windows SDK (winuser.h).
    unsafe fn put_himetric_location(&self, /* in */ himetric_location: POINT) -> HRESULT;

    /// Get the PixelLocationRaw of the pointer event. This corresponds to the
    /// ptPixelLocationRaw property of the POINTER_INFO struct as defined in the
    /// Windows SDK (winuser.h).
    unsafe fn get_pixel_location_raw(
        &self,
        /* out, retval */ pixel_location_raw: *mut POINT,
    ) -> HRESULT;

    /// Set the PixelLocationRaw of the pointer event. This corresponds to the
    /// ptPixelLocationRaw property of the POINTER_INFO struct as defined in the
    /// Windows SDK (winuser.h).
    unsafe fn put_pixel_location_raw(&self, /* in */ pixel_location_raw: POINT) -> HRESULT;

    /// Get the HimetricLocationRaw of the pointer event. This corresponds to the
    /// ptHimetricLocationRaw property of the POINTER_INFO struct as defined in
    /// the Windows SDK (winuser.h).
    unsafe fn get_himetric_location_raw(
        &self,
        /* out, retval */ himetric_location_raw: *mut POINT,
    ) -> HRESULT;

    /// Set the HimetricLocationRaw of the pointer event. This corresponds to the
    /// ptHimetricLocationRaw property of the POINTER_INFO struct as defined in
    /// the Windows SDK (winuser.h).
    unsafe fn put_himetric_location_raw(
        &self,
        /* in */ himetric_location_raw: POINT,
    ) -> HRESULT;

    /// Get the Time of the pointer event. This corresponds to the dwTime property
    /// of the POINTER_INFO struct as defined in the Windows SDK (winuser.h).
    unsafe fn get_time(&self, /* out, retval */ time: *mut DWORD) -> HRESULT;

    /// Set the Time of the pointer event. This corresponds to the dwTime property
    /// of the POINTER_INFO struct as defined in the Windows SDK (winuser.h).
    unsafe fn put_time(&self, /* in */ time: DWORD) -> HRESULT;

    /// Get the HistoryCount of the pointer event. This corresponds to the
    /// historyCount property of the POINTER_INFO struct as defined in the Windows
    /// SDK (winuser.h).
    unsafe fn get_history_count(&self, /* out, retval */ history_count: *mut u32) -> HRESULT;

    /// Set the HistoryCount of the pointer event. This corresponds to the
    /// historyCount property of the POINTER_INFO struct as defined in the Windows
    /// SDK (winuser.h).
    unsafe fn put_history_count(&self, /* in */ history_count: u32) -> HRESULT;

    /// Get the InputData of the pointer event. This corresponds to the InputData
    /// property of the POINTER_INFO struct as defined in the Windows SDK
    /// (winuser.h).
    unsafe fn get_input_data(&self, /* out, retval */ input_data: *mut i32) -> HRESULT;

    /// Set the InputData of the pointer event. This corresponds to the InputData
    /// property of the POINTER_INFO struct as defined in the Windows SDK
    /// (winuser.h).
    unsafe fn put_input_data(&self, /* in */ input_data: i32) -> HRESULT;

    /// Get the KeyStates of the pointer event. This corresponds to the
    /// dwKeyStates property of the POINTER_INFO struct as defined in the Windows
    /// SDK (winuser.h).
    unsafe fn get_key_states(&self, /* out, retval */ key_states: *mut DWORD) -> HRESULT;

    /// Set the KeyStates of the pointer event. This corresponds to the
    /// dwKeyStates property of the POINTER_INFO struct as defined in the Windows
    /// SDK (winuser.h).
    unsafe fn put_key_states(&self, /* in */ key_states: DWORD) -> HRESULT;

    /// Get the PerformanceCount of the pointer event. This corresponds to the
    /// PerformanceCount property of the POINTER_INFO struct as defined in the
    /// Windows SDK (winuser.h).
    unsafe fn get_performance_count(
        &self,
        /* out, retval */ performance_count: *mut u64,
    ) -> HRESULT;

    /// Set the PerformanceCount of the pointer event. This corresponds to the
    /// PerformanceCount property of the POINTER_INFO struct as defined in the
    /// Windows SDK (winuser.h).
    unsafe fn put_performance_count(&self, /* in */ performance_count: u64) -> HRESULT;

    /// Get the ButtonChangeKind of the pointer event. This corresponds to the
    /// ButtonChangeKind property of the POINTER_INFO struct. The values are
    /// defined by the POINTER_BUTTON_CHANGE_KIND enum in the Windows SDK
    /// (winuser.h).
    unsafe fn get_button_change_kind(
        &self,
        /* out, retval */ button_change_kind: *mut i32,
    ) -> HRESULT;

    /// Set the ButtonChangeKind of the pointer event. This corresponds to the
    /// ButtonChangeKind property of the POINTER_INFO struct. The values are
    /// defined by the POINTER_BUTTON_CHANGE_KIND enum in the Windows SDK
    /// (winuser.h).
    unsafe fn put_button_change_kind(&self, /* in */ button_change_kind: i32) -> HRESULT;

    /// Get the PenFlags of the pointer event. This corresponds to the penFlags
    /// property of the POINTER_PEN_INFO struct. The values are defined by the
    /// PEN_FLAGS constants in the Windows SDK (winuser.h).
    unsafe fn get_pen_flags(&self, /* out, retval */ pen_flags: *mut u32) -> HRESULT;

    /// Set the PenFlags of the pointer event. This corresponds to the penFlags
    /// property of the POINTER_PEN_INFO struct. The values are defined by the
    /// PEN_FLAGS constants in the Windows SDK (winuser.h).
    unsafe fn put_pen_flags(&self, /* in */ pen_flags: u32) -> HRESULT;

    /// Get the PenMask of the pointer event. This corresponds to the penMask
    /// property of the POINTER_PEN_INFO struct. The values are defined by the
    /// PEN_MASK constants in the Windows SDK (winuser.h).
    unsafe fn get_pen_mask(&self, /* out, retval */ pen_mask: *mut u32) -> HRESULT;

    /// Set the PenMask of the pointer event. This corresponds to the penMask
    /// property of the POINTER_PEN_INFO struct. The values are defined by the
    /// PEN_MASK constants in the Windows SDK (winuser.h).
    unsafe fn put_pen_mask(&self, /* in */ pen_mask: u32) -> HRESULT;

    /// Get the PenPressure of the pointer event. This corresponds to the pressure
    /// property of the POINTER_PEN_INFO struct as defined in the Windows SDK
    /// (winuser.h).
    unsafe fn get_pen_pressure(&self, /* out, retval */ pen_pressure: *mut u32) -> HRESULT;

    /// Set the PenPressure of the pointer event. This corresponds to the pressure
    /// property of the POINTER_PEN_INFO struct as defined in the Windows SDK
    /// (winuser.h).
    unsafe fn put_pen_pressure(&self, /* in */ pen_pressure: u32) -> HRESULT;

    /// Get the PenRotation of the pointer event. This corresponds to the rotation
    /// property of the POINTER_PEN_INFO struct as defined in the Windows SDK
    /// (winuser.h).
    unsafe fn get_pen_rotation(&self, /* out, retval */ pen_rotation: *mut u32) -> HRESULT;

    /// Set the PenRotation of the pointer event. This corresponds to the rotation
    /// property of the POINTER_PEN_INFO struct as defined in the Windows SDK
    /// (winuser.h).
    unsafe fn put_pen_rotation(&self, /* in */ pen_rotation: u32) -> HRESULT;

    /// Get the PenTiltX of the pointer event. This corresponds to the tiltX
    /// property of the POINTER_PEN_INFO struct as defined in the Windows SDK
    /// (winuser.h).
    unsafe fn get_pen_tilt_x(&self, /* out, retval */ pen_tilt_x: *mut i32) -> HRESULT;

    /// Set the PenTiltX of the pointer event. This corresponds to the tiltX
    /// property of the POINTER_PEN_INFO struct as defined in the Windows SDK
    /// (winuser.h).
    unsafe fn put_pen_tilt_x(&self, /* in */ pen_tilt_x: i32) -> HRESULT;

    /// Get the PenTiltY of the pointer event. This corresponds to the tiltY
    /// property of the POINTER_PEN_INFO struct as defined in the Windows SDK
    /// (winuser.h).
    unsafe fn get_pen_tilt_y(&self, /* out, retval */ pen_tilt_y: *mut i32) -> HRESULT;

    /// Set the PenTiltY of the pointer event. This corresponds to the tiltY
    /// property of the POINTER_PEN_INFO struct as defined in the Windows SDK
    /// (winuser.h).
    unsafe fn put_pen_tilt_y(&self, /* in */ pen_tilt_y: i32) -> HRESULT;

    /// Get the TouchFlags of the pointer event. This corresponds to the
    /// touchFlags property of the POINTER_TOUCH_INFO struct. The values are
    /// defined by the TOUCH_FLAGS constants in the Windows SDK (winuser.h).
    unsafe fn get_touch_flags(&self, /* out, retval */ touch_flags: *mut u32) -> HRESULT;

    /// Set the TouchFlags of the pointer event. This corresponds to the
    /// touchFlags property of the POINTER_TOUCH_INFO struct. The values are
    /// defined by the TOUCH_FLAGS constants in the Windows SDK (winuser.h).
    unsafe fn put_touch_flags(&self, /* in */ touch_flags: u32) -> HRESULT;

    /// Get the TouchMask of the pointer event. This corresponds to the
    /// touchMask property of the POINTER_TOUCH_INFO struct. The values are
    /// defined by the TOUCH_MASK constants in the Windows SDK (winuser.h).
    unsafe fn get_touch_mask(&self, /* out, retval */ touch_mask: *mut u32) -> HRESULT;

    /// Set the TouchMask of the pointer event. This corresponds to the
    /// touchMask property of the POINTER_TOUCH_INFO struct. The values are
    /// defined by the TOUCH_MASK constants in the Windows SDK (winuser.h).
    unsafe fn put_touch_mask(&self, /* in */ touch_mask: u32) -> HRESULT;

    /// Get the TouchContact of the pointer event. This corresponds to the
    /// rcContact property of the POINTER_TOUCH_INFO struct as defined in the
    /// Windows SDK (winuser.h).
    unsafe fn get_touch_contact(&self, /* out, retval */ touch_contact: *mut RECT) -> HRESULT;

    /// Set the TouchContact of the pointer event. This corresponds to the
    /// rcContact property of the POINTER_TOUCH_INFO struct as defined in the
    /// Windows SDK (winuser.h).
    unsafe fn put_touch_contact(&self, /* in */ touch_contact: RECT) -> HRESULT;

    /// Get the TouchContactRaw of the pointer event. This corresponds to the
    /// rcContactRaw property of the POINTER_TOUCH_INFO struct as defined in the
    /// Windows SDK (winuser.h).
    unsafe fn get_touch_contact_raw(
        &self,
        /* out, retval */ touch_contact_raw: *mut RECT,
    ) -> HRESULT;

    /// Set the TouchContactRaw of the pointer event. This corresponds to the
    /// rcContactRaw property of the POINTER_TOUCH_INFO struct as defined in the
    /// Windows SDK (winuser.h).
    unsafe fn put_touch_contact_raw(&self, /* in */ touch_contact_raw: RECT) -> HRESULT;

    /// Get the TouchOrientation of the pointer event. This corresponds to the
    /// orientation property of the POINTER_TOUCH_INFO struct as defined in the
    /// Windows SDK (winuser.h).
    unsafe fn get_touch_orientation(
        &self,
        /* out, retval */ touch_orientation: *mut u32,
    ) -> HRESULT;

    /// Set the TouchOrientation of the pointer event. This corresponds to the
    /// orientation property of the POINTER_TOUCH_INFO struct as defined in the
    /// Windows SDK (winuser.h).
    unsafe fn put_touch_orientation(&self, /* in */ touch_orientation: u32) -> HRESULT;

    /// Get the TouchPressure of the pointer event. This corresponds to the
    /// pressure property of the POINTER_TOUCH_INFO struct as defined in the
    /// Windows SDK (winuser.h).
    unsafe fn get_touch_pressure(&self, /* out, retval */ touch_pressure: *mut u32) -> HRESULT;

    /// Set the TouchPressure of the pointer event. This corresponds to the
    /// pressure property of the POINTER_TOUCH_INFO struct as defined in the
    /// Windows SDK (winuser.h).
    unsafe fn put_touch_pressure(&self, /* in */ touch_pressure: u32) -> HRESULT;
}

/// The caller implements this interface to receive CursorChanged events. Use
/// the Cursor property to get the new cursor.
#[com_interface("9da43ccc-26e1-4dad-b56c-d8961c94c571")]
pub trait ICoreWebView2CursorChangedEventHandler: IUnknown {
    /// Called to provide the implementer with the event args for the
    /// corresponding event. There are no event args and the args
    /// parameter will be null.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2CompositionControllerVTable,
        /* in */ args: *mut *mut IUnknownVTable,
    ) -> HRESULT;
}

/// Receives `RasterizationScaleChanged` events.  Use the `RasterizationScale`
/// property to get the modified scale.
#[com_interface("9c98c8b1-ac53-427e-a345-3049b5524bbe")]
pub trait ICoreWebView2RasterizationScaleChangedEventHandler: IUnknown {
    /// Called to provide the implementer with the event args for the
    /// corresponding event. There are no event args and the args
    /// parameter will be null.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2ControllerVTable,
        /* in */ args: *mut *mut IUnknownVTable,
    ) -> HRESULT;
}

/// Represents the WebView2 Environment.  WebViews created from an environment
/// run on the browser process specified with environment parameters and
/// objects created from an environment should be used in the same
/// environment.  Using it in different environments are not guaranteed to be
///  compatible and may fail.
#[com_interface("b96d755e-0319-4e92-a296-23436f46a1fc")]
pub trait ICoreWebView2Environment: IUnknown {
    /// Asynchronously create a new WebView.
    ///
    /// `parentWindow` is the `HWND` in which the WebView should be displayed and
    /// from which receive input.  The WebView adds a child window to the
    /// provided window during WebView creation.  Z-order and other things
    /// impacted by sibling window order are affected accordingly.
    ///
    /// It is recommended that the app set Application User Model ID for the
    /// process or the app window.  If none is set, during WebView creation a
    /// generated Application User Model ID is set to root window of
    /// `parentWindow`.
    ///
    /// \snippet AppWindow.cpp CreateCoreWebView2Controller
    ///
    /// It is recommended that the app handles restart manager messages, to
    /// gracefully restart it in the case when the app is using the WebView2
    /// Runtime from a certain installation and that installation is being
    /// uninstalled.  For example, if a user installs a version of the WebView2
    /// Runtime and opts to use another version of the WebView2 Runtime for
    /// testing the app, and then uninstalls the 1st version of the WebView2
    /// Runtime without closing the app, the app restarts to allow
    /// un-installation to succeed.
    ///
    /// \snippet AppWindow.cpp RestartManager
    ///
    /// When the app retries `CreateCoreWebView2Controller` upon failure, it is
    /// recommended that the app restarts from creating a new WebView2
    /// Environment.  If an WebView2 Runtime update happens, the version
    /// associated with a WebView2 Environment may have been removed and causing
    /// the object to no longer work.  Creating a new WebView2 Environment works
    /// since it uses the latest version.
    ///
    /// WebView creation fails if a running instance using the same user data
    /// folder exists, and the Environment objects have different
    /// `EnvironmentOptions`.  For example, if a WebView was created with one
    /// language, an attempt to create a WebView with a different language using
    /// the same user data folder fails.
    unsafe fn create_core_web_view2_controller(
        &self,
        parent_window: HWND,
        handler: *mut *mut ICoreWebView2CreateCoreWebView2ControllerCompletedHandlerVTable,
    ) -> HRESULT;

    /// Create a new web resource response object.  The `headers` parameter is
    /// the raw response header string delimited by newline.  It is also possible
    /// to create this object with null headers string and then use the
    /// `ICoreWebView2HttpResponseHeaders` to construct the headers line by line.
    /// For more information about other parameters, navigate to
    /// \[ICoreWebView2WebResourceResponse\]\[MicrosoftEdgeWebview2ReferenceWin32Icorewebview2webresourceresponse\].
    ///
    /// \snippet SettingsComponent.cpp WebResourceRequested0
    /// \snippet SettingsComponent.cpp WebResourceRequested1
    ///
    /// \[MicrosoftEdgeWebview2ReferenceWin32Icorewebview2webresourceresponse\]: /microsoft-edge/webview2/reference/win32/icorewebview2webresourceresponse "interface ICoreWebView2WebResourceResponse | Microsoft Docs"
    unsafe fn create_web_resource_response(
        &self,
        /* in */ content: *mut *mut IStreamVTable,
        /* in */ status_code: i32,
        /* in */ reason_phrase: LPCWSTR,
        /* in */ headers: LPCWSTR,
        /* out, retval */ response: *mut *mut *mut ICoreWebView2WebResourceResponseVTable,
    ) -> HRESULT;

    /// The browser version info of the current `ICoreWebView2Environment`,
    /// including channel name if it is not the WebView2 Runtime.  It matches the
    /// format of the `GetAvailableCoreWebView2BrowserVersionString` API.
    /// Channel names are `beta`, `dev`, and `canary`.
    ///
    /// \snippet AppWindow.cpp GetBrowserVersionString
    unsafe fn get_browser_version_string(
        &self,
        /* out, retval */ version_info: *mut LPWSTR,
    ) -> HRESULT;

    /// Add an event handler for the `NewBrowserVersionAvailable` event.
    /// `NewBrowserVersionAvailable` runs when a newer version of the WebView2
    /// Runtime is installed and available using WebView2.  To use the newer
    /// version of the browser you must create a new environment and WebView.
    /// The event only runs for new version from the same WebView2 Runtime from
    /// which the code is running. When not running with installed WebView2
    /// Runtime, no event is run.
    ///
    /// Because a user data folder is only able to be used by one browser
    /// process at a time, if you want to use the same user data folder in the
    /// WebView using the new version of the browser, you must close the
    /// environment and instance of WebView that are using the older version of
    /// the browser first.  Or simply prompt the user to restart the app.
    ///
    /// \snippet AppWindow.cpp NewBrowserVersionAvailable
    unsafe fn add_new_browser_version_available(
        &self,
        /* in */
        event_handler: *mut *mut ICoreWebView2NewBrowserVersionAvailableEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_NewBrowserVersionAvailable`.
    unsafe fn remove_new_browser_version_available(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;
}

/// A continuation of the ICoreWebView2Environment interface.
#[com_interface("41F3632B-5EF4-404F-AD82-2D606C5A9A21")]
pub trait ICoreWebView2Environment2: ICoreWebView2Environment {
    /// Create a new web resource request object.
    /// URI parameter must be absolute URI.
    /// The headers string is the raw request header string delimited by CRLF
    /// (optional in last header).
    /// It's also possible to create this object with null headers string
    /// and then use the ICoreWebView2HttpRequestHeaders to construct the headers
    /// line by line.
    /// For information on other parameters see ICoreWebView2WebResourceRequest.
    ///
    /// \snippet ScenarioNavigateWithWebResourceRequest.cpp NavigateWithWebResourceRequest
    unsafe fn create_web_resource_request(
        &self,
        /* in */ uri: LPCWSTR,
        /* in */ method: LPCWSTR,
        /* in */ post_data: *mut *mut IStreamVTable,
        /* in */ headers: LPCWSTR,
        /* out, retval */ request: *mut *mut *mut ICoreWebView2WebResourceRequestVTable,
    ) -> HRESULT;
}

/// A continuation of the ICoreWebView2Environment2 interface.
#[com_interface("80a22ae3-be7c-4ce2-afe1-5a50056cdeeb")]
pub trait ICoreWebView2Environment3: ICoreWebView2Environment2 {
    /// Asynchronously create a new WebView for use with visual hosting.
    ///
    /// parentWindow is the HWND in which the app will connect the visual tree of
    /// the WebView. This will be the HWND that the app will receive pointer/
    /// mouse input meant for the WebView (and will need to use SendMouseInput/
    /// SendPointerInput to forward). If the app moves the WebView visual tree to
    /// underneath a different window, then it needs to call put_ParentWindow to
    /// update the new parent HWND of the visual tree.
    ///
    /// Use put_RootVisualTarget on the created CoreWebView2CompositionController to
    /// provide a visual to host the browser's visual tree.
    ///
    /// It is recommended that the application set Application User Model ID for
    /// the process or the application window. If none is set, during WebView
    /// creation a generated Application User Model ID is set to root window of
    /// parentWindow.
    /// \snippet AppWindow.cpp CreateCoreWebView2Controller
    ///
    /// It is recommended that the application handles restart manager messages
    /// so that it can be restarted gracefully in the case when the app is using
    /// Edge for WebView from a certain installation and that installation is
    /// being uninstalled. For example, if a user installs Edge from Dev channel
    /// and opts to use Edge from that channel for testing the app, and then
    /// uninstalls Edge from that channel without closing the app, the app will
    /// be restarted to allow uninstallation of the dev channel to succeed.
    /// \snippet AppWindow.cpp RestartManager
    unsafe fn create_core_web_view2_composition_controller(
        &self,
        parent_window: HWND,
        handler: *mut *mut ICoreWebView2CreateCoreWebView2CompositionControllerCompletedHandlerVTable,
    ) -> HRESULT;

    /// Create an empty ICoreWebView2PointerInfo. The returned
    /// ICoreWebView2PointerInfo needs to be populated with all of the relevant
    /// info before calling SendPointerInput.
    unsafe fn create_core_web_view2_pointer_info(
        &self,
        /* out, retval */ pointer_info: *mut *mut *mut ICoreWebView2PointerInfoVTable,
    ) -> HRESULT;
}

/// A continuation of the ICoreWebView2Environment3 interface.
#[com_interface("20944379-6dcf-41d6-a0a0-abc0fc50de0d")]
pub trait ICoreWebView2Environment4: ICoreWebView2Environment3 {
    /// Returns the UI Automation Provider for the
    /// ICoreWebView2CompositionController that corresponds with the given HWND.
    unsafe fn get_provider_for_hwnd(
        &self,
        /* in */ hwnd: HWND,
        /* out, retval */ provider: *mut *mut *mut IUnknownVTable,
    ) -> HRESULT;
}

/// Options used to create WebView2 Environment.  A default implementation is
/// provided in `WebView2EnvironmentOptions.h`.
///
/// \snippet AppWindow.cpp CreateCoreWebView2EnvironmentWithOptions
#[com_interface("2fde08a8-1e9a-4766-8c05-95a9ceb9d1c5")]
pub trait ICoreWebView2EnvironmentOptions: IUnknown {
    /// Changes the behavior of the WebView.  The arguments are passed to the
    /// browser process as part of the command.  For more information about
    /// using command-line switches with Chromium browser processes, navigate to
    /// \[Run Chromium with Flags\]\[ChromiumDevelopersHowTosRunWithFlags\].
    /// The value appended to a switch is appended to the browser process, for
    /// example, in `--edge-webview-switches=xxx` the value is `xxx`.  If you
    /// specify a switch that is important to WebView functionality, it is
    /// ignored, for example, `--user-data-dir`.  Specific features are disabled
    /// internally and blocked from being enabled.  If a switch is specified
    /// multiple times, only the last instance is used.
    ///
    /// \> [!NOTE]\n\> A merge of the different values of the same switch is not attempted,
    /// except for disabled and enabled features. The features specified by
    /// `--enable-features` and `--disable-features` are merged with simple
    /// logic.\n\> *   The features is the union of the specified features
    /// and built-in features.  If a feature is disabled, it is removed from the
    /// enabled features list.
    ///
    /// If you specify command-line switches and use the
    /// `additionalBrowserArguments` parameter, the `--edge-webview-switches`
    /// value takes precedence and is processed last.  If a switch fails to
    /// parse, the switched is ignored.  The default state for the operation is
    /// to run the browser process with no extra flags.
    ///
    /// <!--TODO: replace https://aka.ms/RunChromiumWithFlags link with go link in the following line and delete this line.  -->
    /// \[ChromiumDevelopersHowTosRunWithFlags\]: https://www.chromium.org/developers/how-tos/run-chromium-with-flags "Run Chromium with flags | The Chromium Projects"
    unsafe fn get_additional_browser_arguments(
        &self,
        /* out, retval */ value: *mut LPWSTR,
    ) -> HRESULT;

    /// Sets the `AdditionalBrowserArguments` property.
    unsafe fn put_additional_browser_arguments(&self, /* in */ value: LPCWSTR) -> HRESULT;

    /// The default display language for WebView.  It applies to browser UI such as
    /// context menu and dialogs.  It also applies to the `accept-languages` HTTP
    ///  header that WebView sends to websites.  It is in the format of
    /// `language[-country]` where `language` is the 2-letter code from
    /// <!--TODO: Update the link in the following line and delete this line.  -->
    /// \[ISO 639\]\[ISO639LanguageCodesHtml\]
    /// and `country` is the
    /// 2-letter code from
    /// \[ISO 3166\]\[ISOStandard72482Html\].
    ///
    /// \[ISO639LanguageCodesHtml\]: https://www.iso.org/iso-639-language-codes.html "ISO 639 | ISO"
    ///
    /// \[ISOStandard72482Html\]: https://www.iso.org/standard/72482.html "ISO 3166-1:2020 | ISO"
    unsafe fn get_language(&self, /* out, retval */ value: *mut LPWSTR) -> HRESULT;

    /// Sets the `Language` property.
    unsafe fn put_language(&self, /* in */ value: LPCWSTR) -> HRESULT;

    /// Specifies the version of the WebView2 Runtime binaries required to be
    /// compatible with your app.  This defaults to the WebView2 Runtime version
    /// that corresponds with the version of the SDK the app is using.  The
    /// format of this value is the same as the format of the
    /// `BrowserVersionString` property and other `BrowserVersion` values.  Only
    /// the version part of the `BrowserVersion` value is respected.  The channel
    ///  suffix, if it exists, is ignored.  The version of the WebView2 Runtime
    /// binaries actually used may be different from the specified
    /// `TargetCompatibleBrowserVersion`.  The binaries are only guaranteed to be
    /// compatible.  Verify the actual version on the `BrowserVersionString`
    /// property on the `ICoreWebView2Environment`.
    unsafe fn get_target_compatible_browser_version(
        &self,
        /* out, retval */ value: *mut LPWSTR,
    ) -> HRESULT;

    /// Sets the `TargetCompatibleBrowserVersion` property.
    unsafe fn put_target_compatible_browser_version(&self, /* in */ value: LPCWSTR) -> HRESULT;

    /// The `AllowSingleSignOnUsingOSPrimaryAccount` property is used to enable
    /// single sign on with Azure Active Directory (AAD) and personal Microsoft
    /// Account (MSA) resources inside WebView. All AAD accounts, connected to
    /// Windows and shared for all apps, are supported. For MSA, SSO is only enabled
    /// for the account associated for Windows account login, if any.
    /// Default is disabled. Universal Windows Platform apps must also declare
    /// `enterpriseCloudSSO`
    /// \[Restricted capabilities\]\[WindowsUwpPackagingAppCapabilityDeclarationsRestrictedCapabilities\]
    /// for the single sign on (SSO) to work.
    ///
    /// \[WindowsUwpPackagingAppCapabilityDeclarationsRestrictedCapabilities\]: /windows/uwp/packaging/app-capability-declarations\#restricted-capabilities "Restricted capabilities - App capability declarations | Microsoft Docs"
    unsafe fn get_allow_single_sign_on_using_osprimary_account(
        &self,
        /* out, retval */ allow: *mut BOOL,
    ) -> HRESULT;

    /// Sets the `AllowSingleSignOnUsingOSPrimaryAccount` property.
    unsafe fn put_allow_single_sign_on_using_osprimary_account(
        &self,
        /* in */ allow: BOOL,
    ) -> HRESULT;
}

/// Receives the `WebView2Environment` created using
/// `CreateCoreWebView2Environment`.
#[com_interface("4e8a3389-c9d8-4bd2-b6b5-124fee6cc14d")]
pub trait ICoreWebView2CreateCoreWebView2EnvironmentCompletedHandler: IUnknown {
    /// Provides the completion status and result of the corresponding
    /// asynchronous method.
    unsafe fn invoke(
        &self,
        error_code: HRESULT,
        created_environment: *mut *mut ICoreWebView2EnvironmentVTable,
    ) -> HRESULT;
}

/// A Receiver is created for a particular DevTools Protocol event and allows
/// you to subscribe and unsubscribe from that event.  Obtained from the
/// WebView object using `GetDevToolsProtocolEventReceiver`.
#[com_interface("b32ca51a-8371-45e9-9317-af021d080367")]
pub trait ICoreWebView2DevToolsProtocolEventReceiver: IUnknown {
    /// Subscribe to a `DevToolsProtocol` event.  The `Invoke` method of the
    /// `handler` runs whenever the corresponding `DevToolsProtocol` event runs.
    /// `Invoke` runs with an event args object containing the parameter object
    /// of the DevTools Protocol event as a JSON string.
    ///
    /// \snippet ScriptComponent.cpp DevToolsProtocolEventReceived
    unsafe fn add_dev_tools_protocol_event_received(
        &self,
        /* in */
        handler: *mut *mut ICoreWebView2DevToolsProtocolEventReceivedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with
    /// `add_DevToolsProtocolEventReceived`.
    unsafe fn remove_dev_tools_protocol_event_received(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;
}

/// WebView2Frame provides direct access to the iframes information.
#[com_interface("f1131a5e-9ba9-11eb-a8b3-0242ac130003")]
pub trait ICoreWebView2Frame: IUnknown {
    /// The name of the iframe from the iframe html tag declaring it.
    /// Calling this method fails if it is called after the iframe is destroyed.
    unsafe fn get_name(&self, /* out, retval  */ name: *mut LPWSTR) -> HRESULT;

    /// Raised when the iframe changes its window.name property.
    unsafe fn add_name_changed(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2FrameNameChangedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with add_NameChanged.
    unsafe fn remove_name_changed(&self, /* in */ token: EventRegistrationToken) -> HRESULT;

    /// Add the provided host object to script running in the iframe with the
    /// specified name for the list of the specified origins. The host object
    /// will be accessible for this iframe only if the iframe's origin during
    /// access matches one of the origins which are passed. The provided origins
    /// will be normalized before comparing to the origin of the document.
    /// So the scheme name is made lower case, the host will be punycode decoded
    /// as appropriate, default port values will be removed, and so on.
    /// This means the origin's host may be punycode encoded or not and will match
    /// regardless. If list contains malformed origin the call will fail.
    /// The method can be called multiple times in a row without calling
    /// RemoveHostObjectFromScript for the same object name. It will replace
    /// the previous object with the new object and new list of origins.
    /// List of origins will be treated as following:
    /// 1. empty list - call will succeed and object will be added for the iframe
    /// but it will not be exposed to any origin;
    /// 2. list with origins - during access to host object from iframe the
    /// origin will be checked that it belongs to this list;
    /// 3. list with "*" element - host object will be available for iframe for
    /// all origins. We suggest not to use this feature without understanding
    /// security implications of giving access to host object from from iframes
    /// with unknown origins.
    /// Calling this method fails if it is called after the iframe is destroyed.
    /// \snippet ScenarioAddHostObject.cpp AddHostObjectToScriptWithOrigins
    /// For more information about host objects navigate to
    /// ICoreWebView2::AddHostObjectToScript.
    unsafe fn add_host_object_to_script_with_origins(
        &self,
        /* in */ name: LPCWSTR,
        /* in */ object: *mut VARIANT,
        /* in */ origins_count: u32,
        /* in, size_is(originsCount) */ origins: *mut LPCWSTR,
    ) -> HRESULT;

    /// Remove the host object specified by the name so that it is no longer
    /// accessible from JavaScript code in the iframe. While new access
    /// attempts are denied, if the object is already obtained by JavaScript code
    /// in the iframe, the JavaScript code continues to have access to that
    /// object. Calling this method for a name that is already removed or was
    /// never added fails. If the iframe is destroyed this method will return fail
    /// also.
    unsafe fn remove_host_object_from_script(&self, /* in */ name: LPCWSTR) -> HRESULT;

    /// The Destroyed event is raised when the iframe corresponding
    /// to this CoreWebView2Frame object is removed or the document
    /// containing that iframe is destroyed.
    unsafe fn add_destroyed(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2FrameDestroyedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with add_Destroyed.
    unsafe fn remove_destroyed(&self, /* in */ token: EventRegistrationToken) -> HRESULT;

    /// Check whether a frame is destroyed. Returns true during
    /// the Destroyed event.
    unsafe fn is_destroyed(&self, /* out, retval  */ destroyed: *mut BOOL) -> HRESULT;
}

/// Receives `FrameCreated` event.
#[com_interface("38059770-9baa-11eb-a8b3-0242ac130003")]
pub trait ICoreWebView2FrameCreatedEventHandler: IUnknown {
    /// Provides the result for the iframe created event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut ICoreWebView2FrameCreatedEventArgsVTable,
    ) -> HRESULT;
}

/// Receives `FrameNameChanged` event.
#[com_interface("435c7dc8-9baa-11eb-a8b3-0242ac130003")]
pub trait ICoreWebView2FrameNameChangedEventHandler: IUnknown {
    /// Provides the result for the iframe name changed event.
    /// No event args exist and the `args` parameter is set to `null`.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2FrameVTable,
        /* in */ args: *mut *mut IUnknownVTable,
    ) -> HRESULT;
}

/// Event args for the `FrameCreated` events.
#[com_interface("4d6e7b5e-9baa-11eb-a8b3-0242ac130003")]
pub trait ICoreWebView2FrameCreatedEventArgs: IUnknown {
    /// The frame which was created.
    unsafe fn get_frame(
        &self,
        /* out, retval  */ frame: *mut *mut *mut ICoreWebView2FrameVTable,
    ) -> HRESULT;
}

/// Receives `FrameDestroyed` event.
#[com_interface("59dd7b4c-9baa-11eb-a8b3-0242ac130003")]
pub trait ICoreWebView2FrameDestroyedEventHandler: IUnknown {
    /// Provides the result for the iframe destroyed event.
    /// No event args exist and the `args` parameter is set to `null`.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2FrameVTable,
        /* in */ args: *mut *mut IUnknownVTable,
    ) -> HRESULT;
}

/// Add an event handler for the `DownloadStarting` event.
#[com_interface("efedc989-c396-41ca-83f7-07f845a55724")]
pub trait ICoreWebView2DownloadStartingEventHandler: IUnknown {
    /// Provides the event args for the corresponding event.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2VTable,
        /* in */ args: *mut *mut ICoreWebView2DownloadStartingEventArgsVTable,
    ) -> HRESULT;
}

/// Event args for the `DownloadStarting` event.
#[com_interface("e99bbe21-43e9-4544-a732-282764eafa60")]
pub trait ICoreWebView2DownloadStartingEventArgs: IUnknown {
    /// Returns the `ICoreWebView2DownloadOperation` for the download that
    /// has started.
    unsafe fn get_download_operation(
        &self,
        /* out, retval */
        download_operation: *mut *mut *mut ICoreWebView2DownloadOperationVTable,
    ) -> HRESULT;

    /// The host may set this flag to cancel the download. If canceled, the
    /// download save dialog is not displayed regardless of the
    /// `Handled` property.
    unsafe fn get_cancel(&self, /* out, retval */ cancel: *mut BOOL) -> HRESULT;

    /// Sets the `Cancel` property.
    unsafe fn put_cancel(&self, /* in */ cancel: BOOL) -> HRESULT;

    /// The path to the file. If setting the path, the host should ensure that it
    /// is an absolute path, including the file name, and that the path does not
    /// point to an existing file. If the path points to an existing file, the
    /// file will be overwritten. If the directory does not exist, it is created.
    unsafe fn get_result_file_path(
        &self,
        /* out, retval */ result_file_path: *mut LPWSTR,
    ) -> HRESULT;

    /// Sets the `ResultFilePath` property.
    unsafe fn put_result_file_path(&self, /* in */ result_file_path: LPCWSTR) -> HRESULT;

    /// The host may set this flag to `TRUE` to hide the default download dialog
    /// for this download. The download will progress as normal if it is not
    /// canceled, there will just be no default UI shown. By default the value is
    /// `FALSE` and the default download dialog is shown.
    unsafe fn get_handled(&self, /* out, retval */ handled: *mut BOOL) -> HRESULT;

    /// Sets the `Handled` property.
    unsafe fn put_handled(&self, /* in */ handled: BOOL) -> HRESULT;

    /// Returns an `ICoreWebView2Deferral` object.  Use this operation to
    /// complete the event at a later time.
    unsafe fn get_deferral(
        &self,
        /* out, retval */ deferral: *mut *mut *mut ICoreWebView2DeferralVTable,
    ) -> HRESULT;
}

/// Implements the interface to receive `BytesReceivedChanged` event.  Use the
/// `ICoreWebView2DownloadOperation.BytesReceived` property to get the received
/// bytes count.
#[com_interface("828e8ab6-d94c-4264-9cef-5217170d6251")]
pub trait ICoreWebView2BytesReceivedChangedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event. No event args exist
    /// and the `args` parameter is set to `null`.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2DownloadOperationVTable,
        /* in */ args: *mut *mut IUnknownVTable,
    ) -> HRESULT;
}

/// Implements the interface to receive `EstimatedEndTimeChanged` event. Use the
/// `ICoreWebView2DownloadOperation.EstimatedEndTime` property to get the new
/// estimated end time.
#[com_interface("28f0d425-93fe-4e63-9f8d-2aeec6d3ba1e")]
pub trait ICoreWebView2EstimatedEndTimeChangedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event. No event args exist
    /// and the `args` parameter is set to `null`.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2DownloadOperationVTable,
        /* in */ args: *mut *mut IUnknownVTable,
    ) -> HRESULT;
}

/// Implements the interface to receive `StateChanged` event. Use the
/// `ICoreWebView2DownloadOperation.State` property to get the current state,
/// which can be in progress, interrupted, or completed. Use the
/// `ICoreWebView2DownloadOperation.InterruptReason` property to get the
/// interrupt reason if the download is interrupted.
#[com_interface("81336594-7ede-4ba9-bf71-acf0a95b58dd")]
pub trait ICoreWebView2StateChangedEventHandler: IUnknown {
    /// Provides the event args for the corresponding event. No event args exist
    /// and the `args` parameter is set to `null`.
    unsafe fn invoke(
        &self,
        /* in */ sender: *mut *mut ICoreWebView2DownloadOperationVTable,
        /* in */ args: *mut *mut IUnknownVTable,
    ) -> HRESULT;
}

/// Represents a download operation. Gives access to the download's metadata
/// and supports a user canceling, pausing, or resuming the download.
#[com_interface("3d6b6cf2-afe1-44c7-a995-c65117714336")]
pub trait ICoreWebView2DownloadOperation: IUnknown {
    /// Add an event handler for the `BytesReceivedChanged` event.
    ///
    /// \snippet ScenarioCustomDownloadExperience.cpp BytesReceivedChanged
    unsafe fn add_bytes_received_changed(
        &self,
        /* in */
        event_handler: *mut *mut ICoreWebView2BytesReceivedChangedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_BytesReceivedChanged`.
    unsafe fn remove_bytes_received_changed(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// Add an event handler for the `EstimatedEndTimeChanged` event.
    unsafe fn add_estimated_end_time_changed(
        &self,
        /* in */
        event_handler: *mut *mut ICoreWebView2EstimatedEndTimeChangedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_EstimatedEndTimeChanged`.
    unsafe fn remove_estimated_end_time_changed(
        &self,
        /* in */ token: EventRegistrationToken,
    ) -> HRESULT;

    /// Add an event handler for the `StateChanged` event.
    ///
    /// \snippet ScenarioCustomDownloadExperience.cpp StateChanged
    unsafe fn add_state_changed(
        &self,
        /* in */ event_handler: *mut *mut ICoreWebView2StateChangedEventHandlerVTable,
        /* out */ token: *mut EventRegistrationToken,
    ) -> HRESULT;

    /// Remove an event handler previously added with `add_StateChanged`.
    unsafe fn remove_state_changed(&self, /* in */ token: EventRegistrationToken) -> HRESULT;

    /// The URI of the download.
    unsafe fn get_uri(&self, /* out, retval */ uri: *mut LPWSTR) -> HRESULT;

    /// The Content-Disposition header value from the download's HTTP response.
    unsafe fn get_content_disposition(
        &self,
        /* out, retval */ content_disposition: *mut LPWSTR,
    ) -> HRESULT;

    /// MIME type of the downloaded content.
    unsafe fn get_mime_type(&self, /* out, retval */ mime_type: *mut LPWSTR) -> HRESULT;

    /// The expected size of the download in total number of bytes based on the
    /// HTTP Content-Length header. Returns -1 if the size is unknown.
    unsafe fn get_total_bytes_to_receive(
        &self,
        /* out, retval */ total_bytes_to_receive: *mut i64,
    ) -> HRESULT;

    /// The number of bytes that have been written to the download file.
    unsafe fn get_bytes_received(&self, /* out, retval */ bytes_received: *mut i64) -> HRESULT;

    /// The estimated end time in [ISO 8601 Date and Time Format](https://www.iso.org/iso-8601-date-and-time-format.html).
    unsafe fn get_estimated_end_time(
        &self,
        /* out, retval */ estimated_end_time: *mut LPWSTR,
    ) -> HRESULT;

    /// The absolute path to the download file, including file name. Host can change
    /// this from `ICoreWebView2DownloadStartingEventArgs`.
    unsafe fn get_result_file_path(
        &self,
        /* out, retval */ result_file_path: *mut LPWSTR,
    ) -> HRESULT;

    /// The state of the download. A download can be in progress, interrupted, or
    /// completed. See `COREWEBVIEW2_DOWNLOAD_STATE` for descriptions of states.
    unsafe fn get_state(
        &self,
        /* out, retval */ download_state: *mut DownloadState,
    ) -> HRESULT;

    /// The reason why connection with file host was broken.
    unsafe fn get_interrupt_reason(
        &self,
        /* out, retval */ interrupt_reason: *mut DownloadInterruptReason,
    ) -> HRESULT;

    /// Cancels the download. If canceled, the default download dialog shows
    /// that the download was canceled. Host should set the `Cancel` property from
    /// `ICoreWebView2SDownloadStartingEventArgs` if the download should be
    /// canceled without displaying the default download dialog.
    unsafe fn cancel(&self) -> HRESULT;

    /// Pauses the download. If paused, the default download dialog shows that the
    /// download is paused. No effect if download is already paused. Pausing a
    /// download changes the state to `COREWEBVIEW2_DOWNLOAD_STATE_INTERRUPTED`
    /// with `InterruptReason` set to `COREWEBVIEW2_DOWNLOAD_INTERRUPT_REASON_USER_PAUSED`.
    unsafe fn pause(&self) -> HRESULT;

    /// Resumes a paused download. May also resume a download that was interrupted
    /// for another reason, if `CanResume` returns true. Resuming a download changes
    /// the state from `COREWEBVIEW2_DOWNLOAD_STATE_INTERRUPTED` to
    /// `COREWEBVIEW2_DOWNLOAD_STATE_IN_PROGRESS`.
    unsafe fn resume(&self) -> HRESULT;

    /// Returns true if an interrupted download can be resumed. Downloads with
    /// the following interrupt reasons may automatically resume without you
    /// calling any methods:
    /// `COREWEBVIEW2_DOWNLOAD_INTERRUPT_REASON_SERVER_NO_RANGE`,
    /// `COREWEBVIEW2_DOWNLOAD_INTERRUPT_REASON_FILE_HASH_MISMATCH`,
    /// `COREWEBVIEW2_DOWNLOAD_INTERRUPT_REASON_FILE_TOO_SHORT`.
    /// In these cases download progress may be restarted with `BytesReceived`
    /// reset to 0.
    unsafe fn get_can_resume(&self, /* out, retval */ can_resume: *mut BOOL) -> HRESULT;
}

/// A continuation of the ICoreWebView2ProcessFailedEventArgs interface.
#[com_interface("4dab9422-46fa-4c3e-a5d2-41d2071d3680")]
pub trait ICoreWebView2ProcessFailedEventArgs2: ICoreWebView2ProcessFailedEventArgs {
    /// The reason for the process failure. The reason is always
    /// `COREWEBVIEW2_PROCESS_FAILED_REASON_UNEXPECTED` when `ProcessFailedKind`
    /// is `COREWEBVIEW2_PROCESS_FAILED_KIND_BROWSER_PROCESS_EXITED`, and
    /// `COREWEBVIEW2_PROCESS_FAILED_REASON_UNRESPONSIVE` when `ProcessFailedKind`
    /// is `COREWEBVIEW2_PROCESS_FAILED_KIND_RENDER_PROCESS_UNRESPONSIVE`.
    /// For other process failure kinds, the reason may be any of the reason
    /// values.
    unsafe fn get_reason(&self, /* out, retval */ reason: *mut ProcessFailedReason) -> HRESULT;

    /// The exit code of the failing process, for telemetry purposes. The exit
    /// code is always `1` when `ProcessFailedKind` is
    /// `COREWEBVIEW2_PROCESS_FAILED_KIND_BROWSER_PROCESS_EXITED`, and
    /// `STILL_ACTIVE` (`259`) when `ProcessFailedKind` is
    /// `COREWEBVIEW2_PROCESS_FAILED_KIND_RENDER_PROCESS_UNRESPONSIVE`.
    unsafe fn get_exit_code(&self, /* out, retval */ exit_code: *mut i32) -> HRESULT;

    /// Description of the process assigned by the WebView2 Runtime. This is a
    /// technical English term appropriate for logging or development purposes,
    /// and not localized for the end user. It applies to utility processes (for
    /// example, "Audio Service", "Video Capture") and plugin processes (for
    /// example, "Flash"). The returned `processDescription` is empty if the
    /// WebView2 Runtime did not assign a description to the process.
    unsafe fn get_process_description(
        &self,
        /* out, retval */ process_description: *mut LPWSTR,
    ) -> HRESULT;

    /// The collection of `FrameInfo`s for frames in the `ICoreWebView2` that were
    /// being rendered by the failed process. The content in these frames is
    /// replaced with an error page.
    /// This is only available when `ProcessFailedKind` is
    /// `COREWEBVIEW2_PROCESS_FAILED_KIND_FRAME_RENDER_PROCESS_EXITED`;
    /// `frames` is `null` for all other process failure kinds, including the case
    /// in which the failed process was the renderer for the main frame and
    /// subframes within it, for which the failure kind is
    /// `COREWEBVIEW2_PROCESS_FAILED_KIND_RENDER_PROCESS_EXITED`.
    unsafe fn get_frame_infos_for_failed_process(
        &self,
        /* out, retval */ frames: *mut *mut *mut ICoreWebView2FrameInfoCollectionVTable,
    ) -> HRESULT;
}

/// Collection of `FrameInfo`s (name and source). Used to list the affected
/// frames' info when a frame-only render process failure occurs in the
/// `ICoreWebView2`.
#[com_interface("8f834154-d38e-4d90-affb-6800a7272839")]
pub trait ICoreWebView2FrameInfoCollection: IUnknown {
    /// Gets an iterator over the collection of `FrameInfo`s.
    unsafe fn get_iterator(
        &self,
        /* out, retval */
        iterator: *mut *mut *mut ICoreWebView2FrameInfoCollectionIteratorVTable,
    ) -> HRESULT;
}

/// Iterator for a collection of `FrameInfo`s. For more info, see
/// `ICoreWebView2ProcessFailedEventArgs2` and
/// `ICoreWebView2FrameInfoCollection`.
#[com_interface("1bf89e2d-1b2b-4629-b28f-05099b41bb03")]
pub trait ICoreWebView2FrameInfoCollectionIterator: IUnknown {
    /// `TRUE` when the iterator has not run out of `FrameInfo`s.  If the
    /// collection over which the iterator is iterating is empty or if the
    /// iterator has gone past the end of the collection, then this is `FALSE`.
    unsafe fn get_has_current(&self, /* out, retval */ has_current: *mut BOOL) -> HRESULT;

    /// Get the current `ICoreWebView2FrameInfo` of the iterator.
    unsafe fn get_current(
        &self,
        /* out, retval */ frame_info: *mut *mut *mut ICoreWebView2FrameInfoVTable,
    ) -> HRESULT;

    /// Move the iterator to the next `FrameInfo` in the collection.
    unsafe fn move_next(&self, /* out, retval */ has_next: *mut BOOL) -> HRESULT;
}

/// Provides a set of properties for a frame in the `ICoreWebView2`.
#[com_interface("da86b8a1-bdf3-4f11-9955-528cefa59727")]
pub trait ICoreWebView2FrameInfo: IUnknown {
    /// The name attribute of the frame, as in `<iframe name="frame-name" ...>`.
    /// The returned string is empty when the frame has no name attribute.
    unsafe fn get_name(&self, /* out, retval */ name: *mut LPWSTR) -> HRESULT;

    /// The URI of the document in the frame.
    unsafe fn get_source(&self, /* out, retval */ source: *mut LPWSTR) -> HRESULT;
}

/// This is the ICoreWebView2Interop interface.
/// Interop interface for the CoreWebView2 WinRT object to allow WinRT end
/// developers to be able to use COM interfaces as parameters for some methods.
#[com_interface("912b34a7-d10b-49c4-af18-7cb7e604e01a")]
pub trait ICoreWebView2Interop: IUnknown {
    /// Add the provided host object to script running in the WebView with the
    /// specified name.
    /// See the documentation for ICoreWebView2::AddHostObjectToScript for more
    /// information.
    unsafe fn add_host_object_to_script(
        &self,
        /* in */ name: LPCWSTR,
        /* in */ object: *mut VARIANT,
    ) -> HRESULT;
}

/// This is the ICoreWebView2CompositionControllerInterop interface.
/// Interop interface for the CoreWebView2CompositionController WinRT object to
/// allow WinRT end developers to be able to use the COM interfaces as parameters
/// for some methods.
#[com_interface("8e9922ce-9c80-42e6-bad7-fcebf291a495")]
pub trait ICoreWebView2CompositionControllerInterop: IUnknown {
    /// Returns the UI Automation Provider for the WebView.
    unsafe fn get_uiaprovider(
        &self,
        /* out, retval */ provider: *mut *mut *mut IUnknownVTable,
    ) -> HRESULT;

    /// The RootVisualTarget is a visual in the hosting app's visual tree. This
    /// visual is where the WebView will connect its visual tree. The app uses
    /// this visual to position the WebView within the app. The app still needs
    /// to use the Bounds property to size the WebView. The RootVisualTarget
    /// property can be an IDCompositionVisual or a
    /// Windows::UI::Composition::ContainerVisual. WebView will connect its visual
    /// tree to the provided visual before returning from the property setter. The
    /// app needs to commit on its device setting the RootVisualTarget property.
    /// The RootVisualTarget property supports being set to nullptr to disconnect
    /// the WebView from the app's visual tree.
    /// \snippet ViewComponent.cpp SetRootVisualTarget
    /// \snippet ViewComponent.cpp BuildDCompTree
    unsafe fn get_root_visual_target(
        &self,
        /* out, retval */ target: *mut *mut *mut IUnknownVTable,
    ) -> HRESULT;

    /// Set the RootVisualTarget property.
    unsafe fn put_root_visual_target(
        &self,
        /* in */ target: *mut *mut IUnknownVTable,
    ) -> HRESULT;
}

/// This is the ICoreWebView2EnvironmentInterop interface.
/// Interop interface for the CoreWebView2Environment WinRT object to allow
/// WinRT end developers to be able to use COM interfaces as parameters for some
/// methods.
#[com_interface("ee503a63-c1e2-4fbf-8a4d-824e95f8bb13")]
pub trait ICoreWebView2EnvironmentInterop: IUnknown {
    /// Returns the UI Automation Provider for the
    /// ICoreWebView2CompositionController that corresponds with the given HWND.
    unsafe fn get_provider_for_hwnd(
        &self,
        /* in */ hwnd: HWND,
        /* out, retval */ provider: *mut *mut *mut IUnknownVTable,
    ) -> HRESULT;
}
