use cs_trace::{create_trace_listener, TraceListenerOptions, create_trace, SubscriberInitExt, child};
use helpers::handle_channel_messages;
use tokio::{net::TcpListener, join, time};
use webrtc::{peer_connection::configuration::RTCConfiguration, ice_transport::ice_server::RTCIceServer};
use webrtc_connection::RtcConnection;

use chrono::Utc;

pub mod helpers;

const PORT_NUMBER: u128 = 6000;

#[tokio::main]
async fn main() {
    let server_address = &format!("127.0.0.1:{}", PORT_NUMBER);
    let server = TcpListener::bind(server_address).await.unwrap();

    // make GH Codespaces extension to autoforward the port
    // println!("\nTCP server started at http://{}\n", server_address);

    let (stream, _) = server.accept().await
        .expect("Cannot establish TCP connection.");

    let logs_file_path = format!(
        "/tmp/codespaces_logs/rtc-connection/{}_tcp-server.log",
        Utc::now()
            .date()
            .to_string()
            .replace(" ", "_")
            .replace("UTC", "-UTC"),
    );

    create_trace_listener(
        TraceListenerOptions::new()
                .with_stdout(true)
                .with_file_path(logs_file_path, true),
    ).init();

    let trace = create_trace!("tcp-server");

    let rtc_config = RTCConfiguration {
        ice_servers: vec![RTCIceServer {
            urls: vec!["stun:stun.l.google.com:19302".to_owned()],
            ..Default::default()
        }],
        ..Default::default()
    };

    let server = RtcConnection::new(
        Box::new(stream),
        rtc_config,
    )
    .await
    .expect("Cannot create RTC connection.");

    trace.trace("RTC server created.");

    let mut connection = server
        .listen()
        .await
        .expect("Failed to accept remote connection.");

    trace.info("connected");

    let mut on_data_channel = connection.on_remote_channel()
        .expect("Cannot get \"on_remote_channel\" stream.");

    let on_data_channel_trace = child!(trace, "channel");
    let on_data_channel_trace2 = child!(trace, "channel");

    let _res = join!(
        // create new channel
        tokio::spawn(async move {
            let mut count = 0;
            loop {
                time::sleep(time::Duration::from_secs(7)).await;
                count += 1;

                let channel_name = format!("server-channel{}", &count);
                let source_name = format!("server{}", &count);

                let channel = connection
                    .channel(channel_name)
                    .await
                    .expect("Cannot create channel.");

                let channel_trace = child!(on_data_channel_trace, "");

                tokio::spawn(async move {
                    handle_channel_messages(&channel_trace, channel, &source_name, 2).await;
                });
            }
        }),
        // get remote channels and subscribe to their events
        tokio::spawn(async move {
            while let Some(channel) = on_data_channel.recv().await {
                let channel_trace = child!(on_data_channel_trace2, "");

                tokio::spawn(async move {
                    handle_channel_messages(
                        &channel_trace,
                        channel,
                        "server2",
                        1,
                    ).await;
                });
            } 
        },
    ));

    trace.warn("🛑 TCP server stopped");
}
