use chrono::Utc;
use tokio::{net::TcpStream, time, join};
use webrtc::{peer_connection::configuration::RTCConfiguration, ice_transport::ice_server::RTCIceServer};

use webrtc_connection::RtcConnection;

use cs_trace::{create_trace_listener, TraceListenerOptions, SubscriberInitExt, create_trace, child};

use crate::helpers::handle_channel_messages;

pub mod helpers;

const PORT_NUMBER: u128 = 6000;

#[tokio::main]
async fn main() {
    let server_address = &format!("127.0.0.1:{}", PORT_NUMBER);
    let stream = TcpStream::connect(server_address)
        .await
        .expect(
            &format!("Cannot connect to the server \"{}\"", server_address),
        );
  
    let logs_file_path = format!(
        "/tmp/codespaces_logs/rtc-connection/{}_tcp-client.log",
        Utc::now()
            .date()
            .to_string()
            .replace(" ", "_")
            .replace("UTC", "-UTC"),
    );

    create_trace_listener(
        TraceListenerOptions::new()
                .with_stdout(true)
                .with_file_path(logs_file_path, true),
    ).init();

    let trace = create_trace!("tcp-client");

    let rtc_config = RTCConfiguration {
        ice_servers: vec![RTCIceServer {
            urls: vec!["stun:stun.l.google.com:19302".to_owned()],
            ..Default::default()
        }],
        ..Default::default()
    };

    let client = RtcConnection::new(
        Box::new(stream),
        rtc_config,
    )
    .await
    .expect("Cannot create RTC connection.");

    trace.trace("RTC client created.");

    let mut connection = client
        .connect()
        .await
        .expect("Failed to connect.");

    trace.info("connected");

    let mut on_remote_channel = connection.on_remote_channel()
        .expect("Cannot get \"on_remote_channel\" stream.");

    let on_remote_channel_trace = child!(trace, "channel");
    let on_remote_channel_trace2 = child!(trace, "channel");

    let _res = join!(
        // create new channel
        tokio::spawn(async move {
            let mut count = 0;
            loop {
                time::sleep(time::Duration::from_secs(5)).await;
                count += 1;

                let channel_name = format!("client-channel{}", &count);
                let source_name = format!("client{}", &count);

                let channel = connection
                    .channel(channel_name)
                    .await
                    .expect("Cannot create channel.");

                let channel_trace = child!(on_remote_channel_trace, "");

                tokio::spawn(async move {
                    handle_channel_messages(&channel_trace, channel, &source_name, 1).await;
                });
            }
        }),
        // get remote channels and subscribe to their events
        tokio::spawn(async move {
            while let Some(channel) = on_remote_channel.recv().await {
                let channel_trace = child!(on_remote_channel_trace2, "");

                tokio::spawn(async move {
                    handle_channel_messages(
                        &channel_trace,
                        channel,
                        "client2",
                        1,
                    ).await;
                });
            } 
        },
    ));

    trace.warn("🛑 TCP client stopped");
}
