use crate::error::WebauthnCError;
use crate::U2FToken;
use crate::{U2FRegistrationData, U2FSignData};
use webauthn_rs::proto::AllowCredentials;

use pcsc::*;
use std::ffi::{CStr, CString};

pub mod apdu;

use self::apdu::*;

pub struct NFC {
    ctx: Context,
    rdr: CString,
    card: Card,
}

impl Default for NFC {
    fn default() -> Self {
        let ctx = Context::establish(Scope::User).expect("Failed to establish pcsc context");

        let mut readers_buf = [0; 2048];
        let mut readers = ctx
            .list_readers(&mut readers_buf)
            .expect("Failed to list pcsc readers");

        let rdr = readers
            .next()
            .map(|s| s.to_owned())
            .expect("No pcsc readers are connected.");
        info!("Using reader: {:?}", rdr);

        let card = ctx
            .connect(&rdr, ShareMode::Shared, Protocols::ANY)
            .expect("Failed to access NFC card");

        // We need to setuup to talk to ctap2

        debug!("Sending APDU: {:x?}", &APPLET_SELECT_CMD);
        let mut rapdu_buf = [0; MAX_SHORT_BUFFER_SIZE];
        let rapdu = card
            .transmit(&APPLET_SELECT_CMD, &mut rapdu_buf)
            .expect("Failed to select CTAP2.1 applet");

        if rapdu == &APPLET_U2F_V2 {
            info!("Selected U2F_V2 applet");
        } else {
            panic!("Invalid response from CTAP2.1 request");
        };

        NFC { ctx, rdr, card }
    }
}

impl NFC {
    pub fn new() -> Self {
        unimplemented!();
    }

    pub fn authenticator_get_info(
        &mut self,
    ) -> Result<AuthenticatorGetInfoResponse, WebauthnCError> {
        let mut rapdu_buf = [0; MAX_SHORT_BUFFER_SIZE];
        debug!("Sending APDU: {:x?}", &AUTHENTICATOR_GET_INFO_APDU);
        let rapdu = self
            .card
            .transmit(&AUTHENTICATOR_GET_INFO_APDU, &mut rapdu_buf)
            .map_err(|e| {
                error!("Failed to transmit APDU command to card: {}", e);
                WebauthnCError::ApduTransmission
            })?;

        trace!("got raw APDU response: {:?}", rapdu);

        AuthenticatorGetInfoResponse::try_from(rapdu).map_err(|e| {
            error!(?e);
            WebauthnCError::Cbor
        })
    }
}

impl U2FToken for NFC {
    fn perform_u2f_register(
        &mut self,
        // This is rp.id_hash
        app_bytes: Vec<u8>,
        // This is client_data_json_hash
        chal_bytes: Vec<u8>,
        // timeout from options
        timeout_ms: u64,
        //
        platform_attached: bool,
        resident_key: bool,
        user_verification: bool,
    ) -> Result<U2FRegistrationData, WebauthnCError> {
        unimplemented!();
    }

    fn perform_u2f_sign(
        &mut self,
        // This is rp.id_hash
        app_bytes: Vec<u8>,
        // This is client_data_json_hash
        chal_bytes: Vec<u8>,
        // timeout from options
        timeout_ms: u64,
        // list of creds
        allowed_credentials: &[AllowCredentials],
        user_verification: bool,
    ) -> Result<U2FSignData, WebauthnCError> {
        unimplemented!();
    }
}

#[cfg(test)]
mod tests {
    use crate::nfc::NFC;
    use crate::WebauthnAuthenticator;
    use webauthn_rs::ephemeral::WebauthnEphemeralConfig;
    use webauthn_rs::Webauthn;

    #[test]
    fn webauthn_authenticator_wan_nfc_interact() {
        let _ = tracing_subscriber::fmt()
            .with_max_level(tracing::Level::DEBUG)
            .try_init();

        let wan_c = WebauthnEphemeralConfig::new(
            "https://localhost:8080/auth",
            "https://localhost:8080",
            "localhost",
            None,
        );

        let wan = Webauthn::new(wan_c);

        let username = "william".to_string();

        let (chal, reg_state) = wan.generate_challenge_register(&username, false).unwrap();

        println!("🍿 challenge -> {:x?}", chal);

        // We can vie the nfc info.
        let mut nfc = NFC::default();
        println!("{:?}", nfc.authenticator_get_info());

        let mut wa = WebauthnAuthenticator::new(nfc);
        let r = wa
            .do_registration("https://localhost:8080", chal)
            .map_err(|e| {
                error!("Error -> {:x?}", e);
                e
            })
            .expect("Failed to register");

        let (cred, _reg_data) = wan
            .register_credential(&r, &reg_state, |_| Ok(false))
            .unwrap();

        let (chal, auth_state) = wan.generate_challenge_authenticate(vec![cred]).unwrap();

        let r = wa
            .do_authentication("https://localhost:8080", chal)
            .map_err(|e| {
                error!("Error -> {:x?}", e);
                e
            })
            .expect("Failed to auth");

        let auth_res = wan
            .authenticate_credential(&r, &auth_state)
            .expect("webauth authentication denied");
        info!("auth_res -> {:x?}", auth_res);
    }
}
