use std::borrow::Borrow;
use std::cell::UnsafeCell;
use std::hash::{Hash, Hasher};
use std::marker;

use std::ops::{Deref, Range, RangeFrom, RangeFull, RangeInclusive, RangeTo, RangeToInclusive};
use std::sync::Arc;

use web_sys::WebGl2RenderingContext as Gl;

use crate::image::format::{
    Filterable, FloatSamplable, IntegerSamplable, PixelUnpack, ShadowSamplable, TextureFormat,
    UnsignedIntegerSamplable,
};
use crate::image::image_source::Image2DSourceInternal;
use crate::image::sampler::{CompatibleSampler, Sampler, SamplerData, ShadowSampler};
use crate::image::texture_object_dropper::TextureObjectDropper;
use crate::image::util::{
    max_mipmap_levels, mipmap_size, region_2d_overlap_height, region_2d_overlap_width,
    region_2d_sub_image, texture_data_as_js_buffer,
};
use crate::image::{Image2DSource, MaxMipmapLevelsExceeded, MipmapLevels, Region2D};
use crate::runtime::state::ContextUpdate;
use crate::runtime::{Connection, RenderingContext};
use crate::task::{ContextId, GpuTask, Progress};
use crate::util::JsId;

/// Provides the information necessary for the creation of a [TextureCube].
///
/// See [RenderingContext::create_texture_cube] for details.
pub struct TextureCubeDescriptor<F>
where
    F: TextureFormat + 'static,
{
    /// The format type the [Texture2D] will use to store its image data.
    ///
    /// Must implement [TextureFormat].
    pub format: F,

    /// The width of the [Texture2D].
    pub width: u32,

    /// The height of the [Texture2D].
    pub height: u32,

    /// The number of levels in the mipmap for the [Texture2D].
    ///
    /// See [MipmapLevels] for details.
    pub levels: MipmapLevels,
}

/// Image storage for the (partial or complete) mipmap chain of a cube map.
///
/// See [RenderingContext::create_texture_cube] for details on how a [TextureCube] is created.
///
/// # Cube map
///
/// A [TextureCube] stores 6 2-dimensional images (one for each face of a cube) of the same size
/// (all six images share the same `width`, and all 6 images share the same `height`) and the same
/// [InternalFormat]. The six images in the [TextureCube] are also referred to as its "faces",
/// specifically they are:
///
/// - The "positive x" face (see [Level::positive_x]).
/// - The "negative x" face (see [Level::negative_x]).
/// - The "positive y" face (see [Level::positive_y]).
/// - The "negative y" face (see [Level::negative_y]).
/// - The "positive z" face (see [Level::positive_z]).
/// - The "negative z" face (see [Level::negative_z]).
///
/// # Mipmap
///
/// A [TextureCube] stores a partial or complete mipmap chain for the base image for each of its 6
/// faces. See the module documentation for [web_glitz::image] for more information on mipmaps.
///
/// Note that a [TextureCube] does not necessarily have to store a complete mipmap chain, it may
/// only store a partial mipmap chain. For example, it may only store the first three levels (see
/// [MipmapLevels] for details). However, it must store at least the first level: level `0`, the
/// base level (see [TextureCube::base_level]).
///
/// Each image in the chain initially starts out in a "cleared" state, where each bit is set to `0`
/// (note that "zeroed data" is valid data for all [TextureFormat]s).
///
/// Mipmapping is typically used with minification filtering, in which case each level in the chain
/// is typically a down-filtered version of the previous level (see [MinificationFilter] for
/// details). If the texture format implements [Filterable], then the image data for such a chain
/// may be generated by first uploading data for the base levels of the 6 faces, and then generating
/// the subsequent levels with [TextureCube::generate_mipmap] (see [TextureCube::generate_mipmap]
/// for details). Image data may also be uploaded to each level of each face individually.
///
/// # Sampling
///
/// The GPU may access the data in a [TextureCube] through a [Sampler] or [ShadowSampler], see
/// [TextureCube::sampled_float], [TextureCube::sampled_integer],
/// [TextureCube::sampled_unsigned_integer] and [TextureCube::sampled_shadow]. A sampled
/// [TextureCube] may be bound to a pipeline as a resource, see
/// [web_glitz::pipeline::resources::Resources].
///
/// # Example
///
/// The following example creates a cube map texture with a width of 256 pixels and a height of 256
/// pixels stored in the [RGB8] format, with a complete mipmap chain. Different colors are uploaded
/// to each of the faces in the texture's base level and then image data for the rest of the mipmap
/// levels is generated:
///
/// ```rust
/// # use web_glitz::runtime::RenderingContext;
/// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext + Clone + 'static {
/// use web_glitz::image::{Image2DSource, MipmapLevels};
/// use web_glitz::image::format::RGB8;
/// use web_glitz::image::texture_cube::TextureCubeDescriptor;
/// use web_glitz::{join_all, sequence_all};
///
/// let texture = context.try_create_texture_cube(&TextureCubeDescriptor {
///     format: RGB8,
///     width: 256,
///     height: 256,
///     levels: MipmapLevels::Complete
/// }).unwrap();
///
/// let positive_x_pixels: Vec<[u8; 3]> = vec![[255, 0, 0]; 256 * 256];
/// let positive_x_data = Image2DSource::from_pixels(positive_x_pixels, 256, 256).unwrap();
/// let negative_x_pixels: Vec<[u8; 3]> = vec![[0, 255, 0]; 256 * 256];
/// let negative_x_data = Image2DSource::from_pixels(negative_x_pixels, 256, 256).unwrap();
/// let positive_y_pixels: Vec<[u8; 3]> = vec![[0, 0, 255]; 256 * 256];
/// let positive_y_data = Image2DSource::from_pixels(positive_y_pixels, 256, 256).unwrap();
/// let negative_y_pixels: Vec<[u8; 3]> = vec![[255, 255, 0]; 256 * 256];
/// let negative_y_data = Image2DSource::from_pixels(negative_y_pixels, 256, 256).unwrap();
/// let positive_z_pixels: Vec<[u8; 3]> = vec![[255, 0, 255]; 256 * 256];
/// let positive_z_data = Image2DSource::from_pixels(positive_z_pixels, 256, 256).unwrap();
/// let negative_z_pixels: Vec<[u8; 3]> = vec![[0, 255, 255]; 256 * 256];
/// let negative_z_data = Image2DSource::from_pixels(negative_z_pixels, 256, 256).unwrap();
///
/// context.submit(sequence_all![
///     join_all![
///         texture.base_level().positive_x().upload_command(positive_x_data),
///         texture.base_level().negative_x().upload_command(negative_x_data),
///         texture.base_level().positive_y().upload_command(positive_y_data),
///         texture.base_level().negative_y().upload_command(negative_y_data),
///         texture.base_level().positive_z().upload_command(positive_z_data),
///         texture.base_level().negative_z().upload_command(negative_z_data),
///     ],
///     texture.generate_mipmap_command()
/// ]);
/// # }
/// ```
pub struct TextureCube<F> {
    object_id: u64,
    data: Arc<TextureCubeData>,
    format: F,
}

impl<F> TextureCube<F> {
    pub(crate) fn data(&self) -> &Arc<TextureCubeData> {
        &self.data
    }
}

impl<F> TextureCube<F>
where
    F: TextureFormat + 'static,
{
    pub(crate) fn new<Rc>(
        context: &Rc,
        object_id: u64,
        descriptor: &TextureCubeDescriptor<F>,
    ) -> Result<Self, MaxMipmapLevelsExceeded>
    where
        Rc: RenderingContext + Clone + 'static,
    {
        let TextureCubeDescriptor {
            format,
            width,
            height,
            levels,
            ..
        } = descriptor;
        let max_mipmap_levels = max_mipmap_levels(*width, *height);

        let levels = match levels {
            MipmapLevels::Complete => max_mipmap_levels,
            MipmapLevels::Partial(levels) => {
                if *levels > max_mipmap_levels {
                    return Err(MaxMipmapLevelsExceeded {
                        given: *levels,
                        max: max_mipmap_levels,
                    });
                }

                *levels
            }
        };

        let data = Arc::new(TextureCubeData {
            id: UnsafeCell::new(None),
            context_id: context.id(),
            dropper: Box::new(context.clone()),
            width: *width,
            height: *height,
            levels,
        });

        context.submit(AllocateCommand::<F> {
            data: data.clone(),
            _marker: marker::PhantomData,
        });

        Ok(TextureCube {
            object_id,
            data,
            format: *format,
        })
    }

    /// Returns a reference to the base mipmap level for this [TextureCube] (level 0).
    pub fn base_level(&self) -> Level<F> {
        Level {
            handle: self,
            level: 0,
        }
    }

    /// Returns a reference to the base mipmap level for this [TextureCube] (level 0).
    pub fn base_level_mut(&mut self) -> LevelMut<F> {
        LevelMut {
            inner: Level {
                handle: self,
                level: 0,
            },
        }
    }

    /// Returns a reference to the levels of this [TextureCube].
    ///
    /// See also [TextureCube::levels_mut].
    ///
    /// # Examples
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # use web_glitz::image::MipmapLevels;
    /// # use web_glitz::image::format::RGB8;
    /// # use web_glitz::image::texture_cube::TextureCubeDescriptor;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext + Clone + 'static {
    /// # let texture = context.try_create_texture_cube(&TextureCubeDescriptor {
    /// #     format: RGB8,
    /// #     width: 256,
    /// #     height: 256,
    /// #     levels: MipmapLevels::Complete
    /// # }).unwrap();
    /// // Returns a reference to mipmap level 2 if the texture has a level 2, or None otherwise:
    /// let level_2 = texture.levels().get(2);
    ///
    /// // We can also iterate over references to all levels that exist for the texture:
    /// for level in texture.levels().iter() {
    ///     let index = level.level();
    ///     let width = level.width();
    ///     let height = level.height();
    ///
    ///     println!("Level {} has a width of {} and height of {}!", index, width, height);
    /// }
    /// # }
    /// ```
    pub fn levels(&self) -> Levels<F> {
        Levels {
            handle: self,
            offset: 0,
            len: self.data.levels,
        }
    }

    /// Returns a mutable reference to the levels of this [TextureCube].
    ///
    /// See also [TextureCube::levels].
    ///
    /// # Examples
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # use web_glitz::image::MipmapLevels;
    /// # use web_glitz::image::format::RGB8;
    /// # use web_glitz::image::texture_cube::TextureCubeDescriptor;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext + Clone + 'static {
    /// # let mut texture = context.try_create_texture_cube(&TextureCubeDescriptor {
    /// #     format: RGB8,
    /// #     width: 256,
    /// #     height: 256,
    /// #     levels: MipmapLevels::Complete
    /// # }).unwrap();
    /// // Returns a mutable reference to mipmap level 2 if the texture has a level 2, or None
    /// // otherwise:
    /// let level_2 = texture.levels_mut().get_mut(2);
    /// # }
    /// ```
    pub fn levels_mut(&mut self) -> LevelsMut<F> {
        LevelsMut {
            inner: Levels {
                handle: self,
                offset: 0,
                len: self.data.levels,
            },
        }
    }

    /// The texture format for this [TextureCube]
    pub fn format(&self) -> F {
        self.format
    }

    /// The width of this [TextureCube].
    pub fn width(&self) -> u32 {
        self.data.width
    }

    /// The height of this [TextureCube].
    pub fn height(&self) -> u32 {
        self.data.height
    }
}

impl<F> TextureCube<F>
where
    F: TextureFormat + Filterable + 'static,
{
    /// Returns a command which, when executed, will generate new mipmap data for the [TextureCube].
    ///
    /// This will overwrite the image data for each face in every mipmap level except the base level.
    /// Starting at level 1, an image is generated that is half the width and height of the previous
    /// level (rounded down), by linear minification filtering of the previous level (see also
    /// [MinificationFilter::Linear]); this stops when the maximum level for which storage was
    /// allocated when the texture was created (see [RenderingContext::create_texture_cube]) has
    /// been overwritten. Note that the base level (level 0) is not modified (rather, it serves as
    /// the input for this process).
    ///
    /// This operation is only available to a texture if the texture format implements [Filterable].
    pub fn generate_mipmap_command(&self) -> GenerateMipmapCommand {
        GenerateMipmapCommand {
            texture_data: self.data.clone(),
        }
    }
}

impl<F> TextureCube<F>
where
    F: TextureFormat + FloatSamplable + 'static,
{
    /// Combines this [TextureCube] with the `sampler` as a [FloatSampledTextureCube], which can be
    /// bound to a pipeline as a texture resource.
    ///
    /// Returns an [IncompatibleSampler] error if the `sampler` is not compatible with this
    /// texture's format.
    ///
    /// See also [web_glitz::pipeline::resources::Resources].
    ///
    /// # Panics
    ///
    /// Panics if this texture and the `sampler` do not belong to the same [RenderingContext].
    pub fn float_sampled<S, Min, Mag>(&self, sampler: S) -> FloatSampledTextureCube
    where
        S: Borrow<Sampler<Min, Mag>> + CompatibleSampler<F>,
    {
        let sampler = sampler.borrow();

        if self.data().context_id() != sampler.data().context_id() {
            panic!("Texture and sampler do not belong to the same context.");
        }

        FloatSampledTextureCube {
            sampler_data: sampler.data().clone(),
            texture_data: self.data().clone(),
            _marker: marker::PhantomData,
        }
    }
}

/// A texture-sampler combination that can bound to a pipeline as a resource for a cube-map floating
/// point sampler.
#[derive(Clone)]
pub struct FloatSampledTextureCube<'a> {
    pub(crate) sampler_data: Arc<SamplerData>,
    pub(crate) texture_data: Arc<TextureCubeData>,
    _marker: marker::PhantomData<&'a ()>,
}

impl<F> TextureCube<F>
where
    F: TextureFormat + IntegerSamplable + 'static,
{
    /// Combines this [TextureCube] with the `sampler` as a [IntegerSampledTextureCube], which can
    /// be bound to a pipeline as a texture resource.
    ///
    /// Returns an [IncompatibleSampler] error if the `sampler` is not compatible with this
    /// texture's format.
    ///
    /// See also [web_glitz::pipeline::resources::Resources].
    ///
    /// # Panics
    ///
    /// Panics if this texture and the `sampler` do not belong to the same [RenderingContext].
    pub fn integer_sampled<S, Min, Mag>(&self, sampler: S) -> IntegerSampledTextureCube
    where
        S: Borrow<Sampler<Min, Mag>> + CompatibleSampler<F>,
    {
        let sampler = sampler.borrow();

        if self.data().context_id() != sampler.data().context_id() {
            panic!("Texture and sampler do not belong to the same context.");
        }

        IntegerSampledTextureCube {
            sampler_data: sampler.data().clone(),
            texture_data: self.data().clone(),
            _marker: marker::PhantomData,
        }
    }
}

/// A texture-sampler combination that can bound to a pipeline as a resource for a cube-map integer
/// sampler.
#[derive(Clone)]
pub struct IntegerSampledTextureCube<'a> {
    pub(crate) sampler_data: Arc<SamplerData>,
    pub(crate) texture_data: Arc<TextureCubeData>,
    _marker: marker::PhantomData<&'a ()>,
}

impl<F> TextureCube<F>
where
    F: TextureFormat + UnsignedIntegerSamplable + 'static,
{
    /// Combines this [TextureCube] with the `sampler` as a [UnsignedIntegerSampledTextureCube],
    /// which can be bound to a pipeline as a texture resource.
    ///
    /// Returns an [IncompatibleSampler] error if the `sampler` is not compatible with this
    /// texture's format.
    ///
    /// See also [web_glitz::pipeline::resources::Resources].
    ///
    /// # Panics
    ///
    /// Panics if this texture and the `sampler` do not belong to the same [RenderingContext].
    pub fn unsigned_integer_sampled<S, Min, Mag>(
        &self,
        sampler: S,
    ) -> UnsignedIntegerSampledTextureCube
    where
        S: Borrow<Sampler<Min, Mag>> + CompatibleSampler<F>,
    {
        let sampler = sampler.borrow();

        if self.data().context_id() != sampler.data().context_id() {
            panic!("Texture and sampler do not belong to the same context.");
        }

        UnsignedIntegerSampledTextureCube {
            sampler_data: sampler.data().clone(),
            texture_data: self.data().clone(),
            _marker: marker::PhantomData,
        }
    }
}

/// A texture-sampler combination that can bound to a pipeline as a resource for a cube-map unsigned
/// integer sampler.
#[derive(Clone)]
pub struct UnsignedIntegerSampledTextureCube<'a> {
    pub(crate) sampler_data: Arc<SamplerData>,
    pub(crate) texture_data: Arc<TextureCubeData>,
    _marker: marker::PhantomData<&'a ()>,
}

impl<F> TextureCube<F>
where
    F: TextureFormat + ShadowSamplable + 'static,
{
    /// Combines this [TextureCube] with the `shadow_sampler` as a [ShadowSampledTextureCube], which
    /// can be bound to a pipeline as a texture resource.
    ///
    /// See also [web_glitz::pipeline::resources::Resources].
    ///
    /// # Panics
    ///
    /// Panics if this texture and the `shadow_sampler` do not belong to the same
    /// [RenderingContext].
    pub fn shadow_sampled(&self, shadow_sampler: &ShadowSampler) -> ShadowSampledTextureCube {
        if self.data().context_id() != shadow_sampler.data().context_id() {
            panic!("Texture and sampler do not belong to the same context.");
        }

        ShadowSampledTextureCube {
            sampler_data: shadow_sampler.data().clone(),
            texture_data: self.data().clone(),
            _marker: marker::PhantomData,
        }
    }
}

/// A texture-sampler combination that can bound to a pipeline as a resource for a cube-map shadow
/// sampler.
#[derive(Clone)]
pub struct ShadowSampledTextureCube<'a> {
    pub(crate) sampler_data: Arc<SamplerData>,
    pub(crate) texture_data: Arc<TextureCubeData>,
    _marker: marker::PhantomData<&'a ()>,
}

impl<F> PartialEq for TextureCube<F> {
    fn eq(&self, other: &Self) -> bool {
        self.object_id == other.object_id
    }
}

impl<F> Hash for TextureCube<F> {
    fn hash<H: Hasher>(&self, state: &mut H) {
        self.object_id.hash(state);
    }
}

pub(crate) struct TextureCubeData {
    id: UnsafeCell<Option<JsId>>,
    context_id: u64,
    dropper: Box<dyn TextureObjectDropper>,
    width: u32,
    height: u32,
    levels: usize,
}

impl TextureCubeData {
    pub(crate) fn id(&self) -> Option<JsId> {
        unsafe { *self.id.get() }
    }

    pub(crate) fn context_id(&self) -> u64 {
        self.context_id
    }
}

impl PartialEq for TextureCubeData {
    fn eq(&self, other: &Self) -> bool {
        self.id() == other.id()
    }
}

impl Hash for TextureCubeData {
    fn hash<H>(&self, state: &mut H)
    where
        H: Hasher,
    {
        self.id().hash(state);
    }
}

impl Drop for TextureCubeData {
    fn drop(&mut self) {
        if let Some(id) = self.id() {
            self.dropper.drop_texture_object(id);
        }
    }
}

/// Returned from [TextureCube::levels], a reference to the levels of a [TextureCube].
///
/// See [TextureCube::levels] for details.
#[derive(PartialEq, Hash)]
pub struct Levels<'a, F> {
    handle: &'a TextureCube<F>,
    offset: usize,
    len: usize,
}

impl<'a, F> Levels<'a, F>
where
    F: TextureFormat,
{
    /// The number of levels defined for the [TextureCube].
    pub fn len(&self) -> usize {
        self.len
    }

    /// Returns a reference to level at the `index`, or `None` if the `index` is out of bounds.
    ///
    /// See also [get_unchecked] for an unsafe variant of this method that does not do any bounds
    /// checks.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::MipmapLevels;
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_cube::TextureCubeDescriptor;
    ///
    /// let texture = context.try_create_texture_cube(&TextureCubeDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     levels: MipmapLevels::Partial(3)
    /// }).unwrap();
    ///
    /// let levels = texture.levels();
    ///
    /// assert_eq!(levels.get(1).map(|l| (l.width(), l.height())), Some((128, 128)));
    /// assert_eq!(levels.get(4).map(|l| (l.width(), l.height())), None);
    /// # }
    /// ```
    pub fn get<'b, I>(&'b self, index: I) -> Option<I::Output>
    where
        I: LevelsIndex<'b, F>,
    {
        index.get(self)
    }

    /// Returns a reference to level at the `index`, without doing any bounds checks.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::MipmapLevels;
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_cube::TextureCubeDescriptor;
    ///
    /// let texture = context.try_create_texture_cube(&TextureCubeDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     levels: MipmapLevels::Partial(3)
    /// }).unwrap();
    ///
    /// let levels = texture.levels();
    ///
    /// let level = unsafe { levels.get_unchecked(1) };
    ///
    /// assert_eq!((level.width(), level.height()), (128, 128));
    /// # }
    /// ```
    ///
    /// # Unsafe
    ///
    /// The `index` must be in bounds. See also [get] for a safe variant of this method that does
    /// bounds checks.
    pub unsafe fn get_unchecked<'b, I>(&'b self, index: I) -> I::Output
    where
        I: LevelsIndex<'b, F>,
    {
        index.get_unchecked(self)
    }

    /// Returns an iterator over the levels.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::MipmapLevels;
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_cube::TextureCubeDescriptor;
    ///
    /// let texture = context.try_create_texture_cube(&TextureCubeDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     levels: MipmapLevels::Partial(3)
    /// }).unwrap();
    ///
    /// let levels = texture.levels();
    /// let mut iter = levels.iter();
    ///
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), Some((128, 128)));
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), Some((64, 64)));
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), Some((32, 32)));
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), None);
    /// # }
    /// ```
    pub fn iter(&self) -> LevelsIter<F> {
        LevelsIter {
            handle: self.handle,
            current_level: self.offset,
            end_level: self.offset + self.len,
        }
    }
}

impl<'a, F> IntoIterator for Levels<'a, F>
where
    F: TextureFormat,
{
    type Item = Level<'a, F>;

    type IntoIter = LevelsIter<'a, F>;

    fn into_iter(self) -> Self::IntoIter {
        LevelsIter {
            handle: self.handle,
            current_level: self.offset,
            end_level: self.offset + self.len,
        }
    }
}

/// An iterator over [Levels].
///
/// See [Levels::iter] for details.
pub struct LevelsIter<'a, F> {
    handle: &'a TextureCube<F>,
    current_level: usize,
    end_level: usize,
}

impl<'a, F> Iterator for LevelsIter<'a, F>
where
    F: TextureFormat,
{
    type Item = Level<'a, F>;

    fn next(&mut self) -> Option<Self::Item> {
        let level = self.current_level;

        if level < self.end_level {
            self.current_level += 1;

            Some(Level {
                handle: self.handle,
                level,
            })
        } else {
            None
        }
    }
}

/// A helper trait for indexing [Levels].
///
/// See [Levels::get] and [Levels::get_unchecked].
pub trait LevelsIndex<'a, F> {
    /// The output type returned by the indexing operations.
    type Output;

    /// Returns the output for this operation if in bounds, or `None` otherwise.
    fn get(self, levels: &'a Levels<F>) -> Option<Self::Output>;

    /// Returns the output for this operation, without performing any bounds checking.
    unsafe fn get_unchecked(self, levels: &'a Levels<F>) -> Self::Output;
}

impl<'a, F> LevelsIndex<'a, F> for usize
where
    F: 'a,
{
    type Output = Level<'a, F>;

    fn get(self, levels: &'a Levels<F>) -> Option<Self::Output> {
        if self < levels.len {
            Some(Level {
                handle: levels.handle,
                level: levels.offset + self,
            })
        } else {
            None
        }
    }

    unsafe fn get_unchecked(self, levels: &'a Levels<F>) -> Self::Output {
        Level {
            handle: levels.handle,
            level: levels.offset + self,
        }
    }
}

impl<'a, F> LevelsIndex<'a, F> for RangeFull
where
    F: 'a,
{
    type Output = Levels<'a, F>;

    fn get(self, levels: &'a Levels<F>) -> Option<Self::Output> {
        Some(Levels {
            handle: levels.handle,
            offset: levels.offset,
            len: levels.len,
        })
    }

    unsafe fn get_unchecked(self, levels: &'a Levels<F>) -> Self::Output {
        Levels {
            handle: levels.handle,
            offset: levels.offset,
            len: levels.len,
        }
    }
}

impl<'a, F> LevelsIndex<'a, F> for Range<usize>
where
    F: 'a,
{
    type Output = Levels<'a, F>;

    fn get(self, levels: &'a Levels<F>) -> Option<Self::Output> {
        let Range { start, end } = self;

        if start > end || end > levels.len {
            None
        } else {
            Some(Levels {
                handle: levels.handle,
                offset: levels.offset + start,
                len: end - start,
            })
        }
    }

    unsafe fn get_unchecked(self, levels: &'a Levels<F>) -> Self::Output {
        let Range { start, end } = self;

        Levels {
            handle: levels.handle,
            offset: levels.offset + start,
            len: end - start,
        }
    }
}

impl<'a, F> LevelsIndex<'a, F> for RangeInclusive<usize>
where
    F: 'a,
{
    type Output = Levels<'a, F>;

    fn get(self, levels: &'a Levels<F>) -> Option<Self::Output> {
        if *self.end() == usize::max_value() {
            None
        } else {
            (*self.start()..self.end() + 1).get(levels)
        }
    }

    unsafe fn get_unchecked(self, levels: &'a Levels<F>) -> Self::Output {
        (*self.start()..self.end() + 1).get_unchecked(levels)
    }
}

impl<'a, F> LevelsIndex<'a, F> for RangeFrom<usize>
where
    F: 'a,
{
    type Output = Levels<'a, F>;

    fn get(self, levels: &'a Levels<F>) -> Option<Self::Output> {
        (self.start..levels.len).get(levels)
    }

    unsafe fn get_unchecked(self, levels: &'a Levels<F>) -> Self::Output {
        (self.start..levels.len).get_unchecked(levels)
    }
}

impl<'a, F> LevelsIndex<'a, F> for RangeTo<usize>
where
    F: 'a,
{
    type Output = Levels<'a, F>;

    fn get(self, levels: &'a Levels<F>) -> Option<Self::Output> {
        (0..self.end).get(levels)
    }

    unsafe fn get_unchecked(self, levels: &'a Levels<F>) -> Self::Output {
        (0..self.end).get_unchecked(levels)
    }
}

impl<'a, F> LevelsIndex<'a, F> for RangeToInclusive<usize>
where
    F: 'a,
{
    type Output = Levels<'a, F>;

    fn get(self, levels: &'a Levels<F>) -> Option<Self::Output> {
        (0..=self.end).get(levels)
    }

    unsafe fn get_unchecked(self, levels: &'a Levels<F>) -> Self::Output {
        (0..=self.end).get_unchecked(levels)
    }
}

/// A reference to a mipmap level of a [TextureCube].
///
/// A reference to the base level of a [TextureCube] can be obtained through
/// [TextureCube::base_level]. References to other levels of a [TextureCube] can be obtained via
/// [Levels].
#[derive(PartialEq, Hash)]
pub struct Level<'a, F> {
    handle: &'a TextureCube<F>,
    level: usize,
}

impl<'a, F> Level<'a, F>
where
    F: TextureFormat,
{
    /// Returns the integer that identifies this level.
    ///
    /// For example, if this [Level] is the texture's base level, returns `0`; if it is the second
    /// level, returns `1`; etc.
    pub fn level(&self) -> usize {
        self.level
    }

    /// Returns the width of this level.
    pub fn width(&self) -> u32 {
        mipmap_size(self.handle.data.width, self.level)
    }

    /// Returns the height of this level.
    pub fn height(&self) -> u32 {
        mipmap_size(self.handle.data.height, self.level)
    }

    /// Returns a reference to the "positive x" face for this level.
    pub fn positive_x(&self) -> LevelFace<F> {
        LevelFace {
            handle: self.handle,
            level: self.level,
            face: CubeFace::PositiveX,
        }
    }

    /// Returns a reference to the "negative x" face for this level.
    pub fn negative_x(&self) -> LevelFace<F> {
        LevelFace {
            handle: self.handle,
            level: self.level,
            face: CubeFace::NegativeX,
        }
    }

    /// Returns a reference to the "positive y" face for this level.
    pub fn positive_y(&self) -> LevelFace<F> {
        LevelFace {
            handle: self.handle,
            level: self.level,
            face: CubeFace::PositiveY,
        }
    }

    /// Returns a reference to the "negative y" face for this level.
    pub fn negative_y(&self) -> LevelFace<F> {
        LevelFace {
            handle: self.handle,
            level: self.level,
            face: CubeFace::NegativeY,
        }
    }

    /// Returns a reference to the "positive z" face for this level.
    pub fn positive_z(&self) -> LevelFace<F> {
        LevelFace {
            handle: self.handle,
            level: self.level,
            face: CubeFace::PositiveZ,
        }
    }

    /// Returns a reference to the "negative z" face for this level.
    pub fn negative_z(&self) -> LevelFace<F> {
        LevelFace {
            handle: self.handle,
            level: self.level,
            face: CubeFace::NegativeZ,
        }
    }
}

/// Enumerates the faces of a [TextureCube].
#[derive(Clone, Copy, PartialEq, Debug, Hash)]
pub enum CubeFace {
    PositiveX,
    NegativeX,
    PositiveY,
    NegativeY,
    PositiveZ,
    NegativeZ,
}

impl CubeFace {
    pub(crate) fn id(&self) -> u32 {
        match self {
            CubeFace::PositiveX => Gl::TEXTURE_CUBE_MAP_POSITIVE_X,
            CubeFace::NegativeX => Gl::TEXTURE_CUBE_MAP_NEGATIVE_X,
            CubeFace::PositiveY => Gl::TEXTURE_CUBE_MAP_POSITIVE_Y,
            CubeFace::NegativeY => Gl::TEXTURE_CUBE_MAP_NEGATIVE_Y,
            CubeFace::PositiveZ => Gl::TEXTURE_CUBE_MAP_POSITIVE_Z,
            CubeFace::NegativeZ => Gl::TEXTURE_CUBE_MAP_NEGATIVE_Z,
        }
    }
}

/// A reference to a face of a [Level].
///
/// See [Level::positive_x], [Level::negative_x], [Level::positive_y], [Level::negative_y],
/// [Level::positive_z] and [Level::negative_z].
#[derive(PartialEq, Hash, Clone)]
pub struct LevelFace<'a, F> {
    handle: &'a TextureCube<F>,
    level: usize,
    face: CubeFace,
}

impl<'a, F> LevelFace<'a, F>
where
    F: TextureFormat,
{
    pub(crate) fn texture_data(&self) -> &Arc<TextureCubeData> {
        &self.handle.data
    }

    /// Returns the index that identifies the level this face belongs to.
    pub fn level(&self) -> usize {
        self.level
    }

    /// Returns the face referenced by this [LevelFace].
    pub fn face(&self) -> CubeFace {
        self.face
    }

    /// Returns the width of this [LevelFace].
    pub fn width(&self) -> u32 {
        mipmap_size(self.handle.data.width, self.level)
    }

    /// Returns the height of this [LevelFace].
    pub fn height(&self) -> u32 {
        mipmap_size(self.handle.data.height, self.level)
    }

    /// Returns a reference to the sub-region of this [LevelFace]'s image described by `region`.
    ///
    /// # Example
    ///
    /// This may for example be used to upload data to only a sub-region of an image, rather than
    /// the complete image:
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext + Clone + 'static {
    /// use web_glitz::image::{Image2DSource, MipmapLevels, Region2D};
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_cube::TextureCubeDescriptor;
    ///
    /// let texture = context.try_create_texture_cube(&TextureCubeDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     levels: MipmapLevels::Complete
    /// }).unwrap();
    ///
    /// let level = texture.base_level();
    /// let face = level.positive_x();
    /// let sub_image = face.sub_image(Region2D::Area((0, 0), 128, 128));
    ///
    /// let pixels: Vec<[u8; 3]> = vec![[0, 0, 255]; 128 * 128];
    /// let data = Image2DSource::from_pixels(pixels, 128, 128).unwrap();
    ///
    /// context.submit(sub_image.upload_command(data));
    /// # }
    /// ```
    ///
    /// The lower left quadrant of the "positive x" face of the texture's base level now contains
    /// blue pixels.
    pub fn sub_image(&self, region: Region2D) -> LevelFaceSubImage<F> {
        LevelFaceSubImage {
            handle: self.handle,
            level: self.level,
            face: self.face,
            region,
        }
    }

    /// Returns a command which, when executed, replaces the image data in this [LevelFace]'s image
    /// with the image data provided in `data`.
    ///
    /// The image data must be stored as a [PixelUnpack] type that is suitable for the texture's
    /// [TextureFormat].
    ///
    /// If the dimensions of the image provided in `data` are not sufficient to cover the
    /// [LevelFace]'s image entirely, starting from the origin, then only the region of overlap is
    /// updated (note that the origin of an image is the lower left corner). For example, given a
    /// [LevelFace] with a width of 256 pixels and a height of 256 pixels, and `data` with a width
    /// of 128 pixels and a height of 128 pixels, then only the lower left quadrant of the
    /// [LevelFace] is updated. If the dimensions of the image provided in `data` would, when
    /// starting from the origin, cover more than the [Level]'s image (the width and/or height of
    /// `data` is/are greater than the width and/or height of the [Level]'s image), then any pixels
    /// that would fall outside of the [LevelFace] are ignored. For example, given a [LevelFace]
    /// with a width of 256 pixels and a height of 256 pixels, and `data` with a width of 256 pixels
    /// and a height of 512 pixels, then only the lower half of the image in `data` is used to
    /// update the [LevelFace] and the upper half is ignored.
    ///
    /// # Example
    ///
    /// This example uploads red pixels to the "positive x" face of the base mipmap level a
    /// [TextureCube]:
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext + Clone + 'static {
    /// use web_glitz::image::{Image2DSource, MipmapLevels};
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_cube::TextureCubeDescriptor;
    ///
    /// let texture = context.try_create_texture_cube(&TextureCubeDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     levels: MipmapLevels::Complete
    /// }).unwrap();
    ///
    /// let pixels: Vec<[u8; 3]> = vec![[255, 0, 0]; 256 * 256];
    /// let data = Image2DSource::from_pixels(pixels, 256, 256).unwrap();
    ///
    /// context.submit(texture.base_level().positive_x().upload_command(data));
    /// # }
    /// ```
    pub fn upload_command<D, T>(&self, data: Image2DSource<D, T>) -> UploadCommand<D, T, F>
    where
        T: PixelUnpack<F>,
    {
        UploadCommand {
            data,
            texture_data: self.handle.data.clone(),
            level: self.level,
            face: self.face,
            region: Region2D::Fill,
            _marker: marker::PhantomData,
        }
    }
}

/// Returned from [LevelFace::sub_image], a reference to a sub-region of a [LevelFace]'s image.
///
/// See [Level::sub_image] for details.
#[derive(PartialEq, Hash)]
pub struct LevelFaceSubImage<'a, F> {
    handle: &'a TextureCube<F>,
    level: usize,
    face: CubeFace,
    region: Region2D,
}

impl<'a, F> LevelFaceSubImage<'a, F>
where
    F: TextureFormat,
{
    pub(crate) fn level_face_ref(&self) -> LevelFace<F> {
        LevelFace {
            handle: self.handle,
            level: self.level,
            face: self.face,
        }
    }

    pub(crate) fn texture_data(&self) -> &Arc<TextureCubeData> {
        &self.handle.data
    }

    /// Returns the index of the mipmap level this [LevelFaceSubImage] references.
    pub fn level(&self) -> usize {
        self.level
    }

    /// Returns the face referenced by this [LevelFaceSubImage].
    pub fn face(&self) -> CubeFace {
        self.face
    }

    /// Returns the sub-region of the level-face this [LevelFaceSubImage] references.
    pub fn region(&self) -> Region2D {
        self.region
    }

    /// Returns the width of this [LevelFaceSubImage].
    pub fn width(&self) -> u32 {
        region_2d_overlap_width(self.handle.data.width, self.level, &self.region)
    }

    /// Returns the height of this [LevelFaceSubImage].
    pub fn height(&self) -> u32 {
        region_2d_overlap_height(self.handle.data.height, self.level, &self.region)
    }

    /// Returns a reference to the sub-region of this [LevelFaceSubImage]'s image described by
    /// `region`.
    ///
    /// See also [LevelFace::sub_image].
    pub fn sub_image(&self, region: Region2D) -> LevelFaceSubImage<F> {
        LevelFaceSubImage {
            handle: self.handle,
            level: self.level,
            face: self.face,
            region: region_2d_sub_image(self.region, region),
        }
    }

    /// Returns a command which, when executed, replaces the image data in this
    /// [LevelFaceSubImage]'s image region with the image data provided in `data`.
    ///
    /// The image data must be stored as a [PixelUnpack] type that is suitable for the texture's
    /// [TextureFormat].
    ///
    /// If the dimensions of the image provided in `data` are not sufficient to cover the
    /// [LevelFaceSubImage] region entirely, starting from the region's origin, then only the region
    /// of overlap is updated (note that the origin of a region is at its lower left corner). For
    /// example, given a [LevelFaceSubImage] with a width of 256 pixels and a height of 256 pixels,
    /// and `data` with a width of 128 pixels and a height of 128 pixels, then only the lower left
    /// quadrant of the [LevelFaceSubImage]'s region is updated. If the dimensions of the image
    /// provided in `data` would, when starting from the [LevelSubImage]'s origin, cover more than
    /// the [LevelFaceSubImage]'s region (the width and/or height of `data` is/are greater than the
    /// width and/or height of the [LevelFaceSubImage]'s region), then any pixels that would fall
    /// outside of the [LevelFaceSubImage]'s region are ignored. For example, given a
    /// [LevelFaceSubImage] with a width of 256 pixels and a height of 256 pixels, and `data` with a
    /// width of 256 pixels and a height of 512 pixels, then only the lower half of the image in
    /// `data` is used to update the [LevelFaceSubImage] and the upper half is ignored.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext + Clone + 'static {
    /// use web_glitz::image::{Image2DSource, MipmapLevels, Region2D};
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_cube::TextureCubeDescriptor;
    ///
    /// let texture = context.try_create_texture_cube(&TextureCubeDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     levels: MipmapLevels::Complete
    /// }).unwrap();
    ///
    /// let level = texture.base_level();
    /// let face = level.positive_x();
    /// let sub_image = face.sub_image(Region2D::Area((0, 0), 128, 128));
    ///
    /// let pixels: Vec<[u8; 3]> = vec![[255, 0, 0]; 128 * 128];
    /// let data = Image2DSource::from_pixels(pixels, 128, 128).unwrap();
    ///
    /// context.submit(sub_image.upload_command(data));
    /// # }
    /// ```
    pub fn upload_command<D, T>(&self, data: Image2DSource<D, T>) -> UploadCommand<D, T, F>
    where
        T: PixelUnpack<F>,
    {
        UploadCommand {
            data,
            texture_data: self.handle.data.clone(),
            level: self.level,
            face: self.face,
            region: self.region,
            _marker: marker::PhantomData,
        }
    }
}

/// Returned from [TextureCube::levels_mut], a mutable reference to the levels of a [TextureCube].
///
/// See [TextureCube::levels_mut] for details.
///
/// [Deref]s to [Levels].
#[derive(PartialEq, Hash)]
pub struct LevelsMut<'a, F> {
    inner: Levels<'a, F>,
}

impl<'a, F> LevelsMut<'a, F>
where
    F: TextureFormat,
{
    /// Returns a mutable reference to level at the `index`, or `None` if the `index` is out of
    /// bounds.
    ///
    /// See also [get_unchecked_mut] for an unsafe variant or this method that does not do any
    /// bounds checks.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::MipmapLevels;
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_cube::TextureCubeDescriptor;
    ///
    /// let mut texture = context.try_create_texture_cube(&TextureCubeDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     levels: MipmapLevels::Partial(3)
    /// }).unwrap();
    ///
    /// let mut levels = texture.levels_mut();
    ///
    /// assert_eq!(levels.get_mut(1).map(|l| (l.width(), l.height())), Some((128, 128)));
    /// assert_eq!(levels.get_mut(4).map(|l| (l.width(), l.height())), None);
    /// # }
    /// ```
    pub fn get_mut<'b, I>(&'b mut self, index: I) -> Option<I::Output>
    where
        I: LevelsMutIndex<'b, F>,
    {
        index.get_mut(self)
    }

    /// Returns a mutable reference to level at the `index`, without doing any bounds checks.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::MipmapLevels;
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_cube::TextureCubeDescriptor;
    ///
    /// let mut texture = context.try_create_texture_cube(&TextureCubeDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     levels: MipmapLevels::Partial(3)
    /// }).unwrap();
    ///
    /// let mut levels = texture.levels_mut();
    ///
    /// let level = unsafe { levels.get_unchecked_mut(1) };
    ///
    /// assert_eq!((level.width(), level.height()), (128, 128));
    /// # }
    /// ```
    ///
    /// # Unsafe
    ///
    /// The `index` must be in bounds. See also [get_mut] for a safe variant of this method that
    /// does bounds checks.
    pub unsafe fn get_unchecked_mut<'b, I>(&'b mut self, index: I) -> I::Output
    where
        I: LevelsMutIndex<'b, F>,
    {
        index.get_unchecked_mut(self)
    }

    /// Returns an iterator over mutable references to the levels.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::MipmapLevels;
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_cube::TextureCubeDescriptor;
    ///
    /// let mut texture = context.try_create_texture_cube(&TextureCubeDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     levels: MipmapLevels::Partial(3)
    /// }).unwrap();
    ///
    /// let mut levels = texture.levels_mut();
    /// let mut iter = levels.iter_mut();
    ///
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), Some((128, 128)));
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), Some((64, 64)));
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), Some((32, 32)));
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), None);
    /// # }
    /// ```
    pub fn iter_mut(&mut self) -> LevelsMutIter<F> {
        LevelsMutIter {
            current_level: self.offset,
            end_level: self.offset + self.len,
            handle: &self.inner.handle,
        }
    }
}

impl<'a, F> IntoIterator for LevelsMut<'a, F>
where
    F: TextureFormat,
{
    type Item = LevelMut<'a, F>;

    type IntoIter = LevelsMutIter<'a, F>;

    fn into_iter(self) -> Self::IntoIter {
        LevelsMutIter {
            current_level: self.offset,
            end_level: self.offset + self.len,
            handle: &self.inner.handle,
        }
    }
}

impl<'a, F> Deref for LevelsMut<'a, F>
where
    F: TextureFormat,
{
    type Target = Levels<'a, F>;

    fn deref(&self) -> &Self::Target {
        &self.inner
    }
}

/// An iterator over [LevelsMut].
///
/// See [LevelsMut::iter_mut] for details.
pub struct LevelsMutIter<'a, F> {
    handle: &'a TextureCube<F>,
    current_level: usize,
    end_level: usize,
}

impl<'a, F> Iterator for LevelsMutIter<'a, F>
where
    F: TextureFormat,
{
    type Item = LevelMut<'a, F>;

    fn next(&mut self) -> Option<Self::Item> {
        let level = self.current_level;

        if level < self.end_level {
            self.current_level += 1;

            Some(LevelMut {
                inner: Level {
                    handle: &self.handle,
                    level,
                },
            })
        } else {
            None
        }
    }
}

/// A helper trait for indexing a [LevelsMut].
///
/// See [LevelsMut::get_mut] and [LevelsMut::get_unchecked_mut].
pub trait LevelsMutIndex<'a, F> {
    /// The output type returned by the indexing operations.
    type Output;

    /// Returns the output for this operation if in bounds, or `None` otherwise.
    fn get_mut(self, levels: &'a mut LevelsMut<F>) -> Option<Self::Output>;

    /// Returns the output for this operation, without performing any bounds checking.
    unsafe fn get_unchecked_mut(self, levels: &'a mut LevelsMut<F>) -> Self::Output;
}

impl<'a, F> LevelsMutIndex<'a, F> for usize
where
    F: 'a,
{
    type Output = LevelMut<'a, F>;

    fn get_mut(self, levels: &'a mut LevelsMut<F>) -> Option<Self::Output> {
        if self < levels.inner.len {
            Some(LevelMut {
                inner: Level {
                    handle: levels.inner.handle,
                    level: levels.inner.offset + self,
                },
            })
        } else {
            None
        }
    }

    unsafe fn get_unchecked_mut(self, levels: &'a mut LevelsMut<F>) -> Self::Output {
        LevelMut {
            inner: Level {
                handle: levels.inner.handle,
                level: levels.inner.offset + self,
            },
        }
    }
}

impl<'a, F> LevelsMutIndex<'a, F> for RangeFull
where
    F: 'a,
{
    type Output = LevelsMut<'a, F>;

    fn get_mut(self, levels: &'a mut LevelsMut<F>) -> Option<Self::Output> {
        Some(LevelsMut {
            inner: Levels {
                handle: levels.inner.handle,
                offset: levels.inner.offset,
                len: levels.inner.len,
            },
        })
    }

    unsafe fn get_unchecked_mut(self, levels: &'a mut LevelsMut<F>) -> Self::Output {
        LevelsMut {
            inner: Levels {
                handle: levels.inner.handle,
                offset: levels.inner.offset,
                len: levels.inner.len,
            },
        }
    }
}

impl<'a, F> LevelsMutIndex<'a, F> for Range<usize>
where
    F: 'a,
{
    type Output = LevelsMut<'a, F>;

    fn get_mut(self, levels: &'a mut LevelsMut<F>) -> Option<Self::Output> {
        let Range { start, end } = self;

        if start > end || end > levels.inner.len {
            None
        } else {
            Some(LevelsMut {
                inner: Levels {
                    handle: levels.inner.handle,
                    offset: levels.inner.offset + start,
                    len: end - start,
                },
            })
        }
    }

    unsafe fn get_unchecked_mut(self, levels: &'a mut LevelsMut<F>) -> Self::Output {
        let Range { start, end } = self;

        LevelsMut {
            inner: Levels {
                handle: levels.inner.handle,
                offset: levels.inner.offset + start,
                len: end - start,
            },
        }
    }
}

impl<'a, F> LevelsMutIndex<'a, F> for RangeInclusive<usize>
where
    F: 'a,
{
    type Output = LevelsMut<'a, F>;

    fn get_mut(self, levels: &'a mut LevelsMut<F>) -> Option<Self::Output> {
        if *self.end() == usize::max_value() {
            None
        } else {
            (*self.start()..self.end() + 1).get_mut(levels)
        }
    }

    unsafe fn get_unchecked_mut(self, levels: &'a mut LevelsMut<F>) -> Self::Output {
        (*self.start()..self.end() + 1).get_unchecked_mut(levels)
    }
}

impl<'a, F> LevelsMutIndex<'a, F> for RangeFrom<usize>
where
    F: 'a,
{
    type Output = LevelsMut<'a, F>;

    fn get_mut(self, levels: &'a mut LevelsMut<F>) -> Option<Self::Output> {
        (self.start..levels.inner.len).get_mut(levels)
    }

    unsafe fn get_unchecked_mut(self, levels: &'a mut LevelsMut<F>) -> Self::Output {
        (self.start..levels.inner.len).get_unchecked_mut(levels)
    }
}

impl<'a, F> LevelsMutIndex<'a, F> for RangeTo<usize>
where
    F: 'a,
{
    type Output = LevelsMut<'a, F>;

    fn get_mut(self, levels: &'a mut LevelsMut<F>) -> Option<Self::Output> {
        (0..self.end).get_mut(levels)
    }

    unsafe fn get_unchecked_mut(self, levels: &'a mut LevelsMut<F>) -> Self::Output {
        (0..self.end).get_unchecked_mut(levels)
    }
}

impl<'a, F> LevelsMutIndex<'a, F> for RangeToInclusive<usize>
where
    F: 'a,
{
    type Output = LevelsMut<'a, F>;

    fn get_mut(self, levels: &'a mut LevelsMut<F>) -> Option<Self::Output> {
        (0..=self.end).get_mut(levels)
    }

    unsafe fn get_unchecked_mut(self, levels: &'a mut LevelsMut<F>) -> Self::Output {
        (0..=self.end).get_unchecked_mut(levels)
    }
}

/// A mutable reference to a [Texture2D] mipmap level.
///
/// [Deref]s to [Level].
#[derive(PartialEq, Hash)]
pub struct LevelMut<'a, F> {
    inner: Level<'a, F>,
}

impl<'a, F> LevelMut<'a, F> {
    /// Returns a mutable reference to the "positive x" face of the level.
    pub fn positive_x_mut(&mut self) -> LevelFaceMut<F> {
        LevelFaceMut {
            inner: LevelFace {
                handle: self.handle,
                level: self.level,
                face: CubeFace::PositiveX,
            },
        }
    }

    /// Returns a mutable reference to the "negative x" face of the level.
    pub fn negative_x_mut(&mut self) -> LevelFaceMut<F> {
        LevelFaceMut {
            inner: LevelFace {
                handle: self.handle,
                level: self.level,
                face: CubeFace::NegativeX,
            },
        }
    }

    /// Returns a mutable reference to the "positive y" face of the level.
    pub fn positive_y_mut(&mut self) -> LevelFaceMut<F> {
        LevelFaceMut {
            inner: LevelFace {
                handle: self.handle,
                level: self.level,
                face: CubeFace::PositiveY,
            },
        }
    }

    /// Returns a mutable reference to the "negative y" face of the level.
    pub fn negative_y_mut(&mut self) -> LevelFaceMut<F> {
        LevelFaceMut {
            inner: LevelFace {
                handle: self.handle,
                level: self.level,
                face: CubeFace::NegativeY,
            },
        }
    }

    /// Returns a mutable reference to the "positive z" face of the level.
    pub fn positive_z_mut(&mut self) -> LevelFaceMut<F> {
        LevelFaceMut {
            inner: LevelFace {
                handle: self.handle,
                level: self.level,
                face: CubeFace::PositiveZ,
            },
        }
    }

    /// Returns a mutable reference to the "negative z" face of the level.
    pub fn negative_z_mut(&mut self) -> LevelFaceMut<F> {
        LevelFaceMut {
            inner: LevelFace {
                handle: self.handle,
                level: self.level,
                face: CubeFace::NegativeZ,
            },
        }
    }
}

impl<'a, F> Deref for LevelMut<'a, F> {
    type Target = Level<'a, F>;

    fn deref(&self) -> &Self::Target {
        &self.inner
    }
}

/// A mutable reference to a face of a [TextureCube] mipmap level.
///
/// [Deref]s to [LevelFace].
#[derive(PartialEq, Hash)]
pub struct LevelFaceMut<'a, F> {
    inner: LevelFace<'a, F>,
}

impl<'a, F> Deref for LevelFaceMut<'a, F> {
    type Target = LevelFace<'a, F>;

    fn deref(&self) -> &Self::Target {
        &self.inner
    }
}

struct AllocateCommand<F> {
    data: Arc<TextureCubeData>,
    _marker: marker::PhantomData<[F]>,
}

unsafe impl<F> GpuTask<Connection> for AllocateCommand<F>
where
    F: TextureFormat,
{
    type Output = ();

    fn context_id(&self) -> ContextId {
        ContextId::Id(self.data.context_id)
    }

    fn progress(&mut self, connection: &mut Connection) -> Progress<Self::Output> {
        let (gl, state) = unsafe { connection.unpack_mut() };
        let data = &self.data;

        let texture_object = gl.create_texture().unwrap();

        state.set_active_texture_lru().apply(gl).unwrap();
        state
            .bind_texture_2d(Some(&texture_object))
            .apply(gl)
            .unwrap();

        let levels = data.levels as i32;

        gl.tex_storage_2d(
            Gl::TEXTURE_CUBE_MAP,
            levels,
            F::ID,
            data.width as i32,
            data.height as i32,
        );

        gl.tex_parameteri(Gl::TEXTURE_CUBE_MAP, Gl::TEXTURE_MAX_LEVEL, levels);

        unsafe {
            *data.id.get() = Some(JsId::from_value(texture_object.into()));
        }

        Progress::Finished(())
    }
}

/// Uploads data to a [LevelFace] or [LevelFaceSubImage].
///
/// See [LevelFace::upload_command] and [LevelFaceSubImage::upload_command] for details.
pub struct UploadCommand<D, T, F> {
    data: Image2DSource<D, T>,
    texture_data: Arc<TextureCubeData>,
    level: usize,
    face: CubeFace,
    region: Region2D,
    _marker: marker::PhantomData<[F]>,
}

unsafe impl<D, T, F> GpuTask<Connection> for UploadCommand<D, T, F>
where
    D: Borrow<[T]>,
    T: PixelUnpack<F>,
    F: TextureFormat,
{
    type Output = ();

    fn context_id(&self) -> ContextId {
        ContextId::Id(self.texture_data.context_id)
    }

    fn progress(&mut self, connection: &mut Connection) -> Progress<Self::Output> {
        let mut width = region_2d_overlap_width(self.texture_data.width, self.level, &self.region);
        let height = region_2d_overlap_height(self.texture_data.height, self.level, &self.region);

        if width == 0 || height == 0 {
            return Progress::Finished(());
        }

        let (gl, state) = unsafe { connection.unpack_mut() };

        match &self.data.internal {
            Image2DSourceInternal::PixelData {
                data,
                row_length,
                alignment,
                ..
            } => {
                state.set_active_texture_lru().apply(gl).unwrap();

                unsafe {
                    self.texture_data
                        .id()
                        .unwrap()
                        .with_value_unchecked(|texture_object| {
                            state
                                .bind_texture_cube_map(Some(texture_object))
                                .apply(gl)
                                .unwrap();
                        });
                }

                state
                    .set_pixel_unpack_alignment((*alignment).into())
                    .apply(gl)
                    .unwrap();

                if width < *row_length {
                    state
                        .set_pixel_unpack_row_length(*row_length as i32)
                        .apply(gl)
                        .unwrap();
                } else {
                    width = *row_length;

                    state.set_pixel_unpack_row_length(0).apply(gl).unwrap();
                }

                let (offset_x, offset_y) = match self.region {
                    Region2D::Fill => (0, 0),
                    Region2D::Area(offset, ..) => offset,
                };

                let elements = *row_length as usize * height as usize;
                let data_buffer = texture_data_as_js_buffer(data.borrow(), elements);

                gl.tex_sub_image_2d_with_i32_and_i32_and_u32_and_type_and_opt_array_buffer_view(
                    self.face.id(),
                    self.level as i32,
                    offset_x as i32,
                    offset_y as i32,
                    width as i32,
                    height as i32,
                    T::FORMAT_ID,
                    T::TYPE_ID,
                    Some(&data_buffer),
                )
                .unwrap();
            }
        }

        Progress::Finished(())
    }
}

/// Returned from [TextureCube::generate_mipmap_command], generates the image data for a the mipmap
/// chains for each of the [TextureCube]'s faces.
///
/// See [TextureCube::generate_mipmap_command] for details.
pub struct GenerateMipmapCommand {
    texture_data: Arc<TextureCubeData>,
}

unsafe impl GpuTask<Connection> for GenerateMipmapCommand {
    type Output = ();

    fn context_id(&self) -> ContextId {
        ContextId::Id(self.texture_data.context_id)
    }

    fn progress(&mut self, connection: &mut Connection) -> Progress<Self::Output> {
        let (gl, state) = unsafe { connection.unpack_mut() };

        unsafe {
            self.texture_data
                .id()
                .unwrap()
                .with_value_unchecked(|texture_object| {
                    state.bind_texture_cube_map(Some(texture_object));
                });
        }

        gl.generate_mipmap(Gl::TEXTURE_CUBE_MAP);

        Progress::Finished(())
    }
}
