use std::borrow::Borrow;
use std::cell::UnsafeCell;
use std::cmp;
use std::hash::{Hash, Hasher};
use std::marker;
use std::ops::{Deref, Range, RangeFrom, RangeFull, RangeInclusive, RangeTo, RangeToInclusive};
use std::sync::Arc;

use web_sys::WebGl2RenderingContext as Gl;

use crate::image::format::{
    Filterable, FloatSamplable, IntegerSamplable, PixelUnpack, ShadowSamplable, TextureFormat,
    UnsignedIntegerSamplable,
};
use crate::image::image_source::{Image2DSourceInternal, LayeredImageSourceInternal};
use crate::image::sampler::{CompatibleSampler, SamplerData, ShadowSampler};
use crate::image::texture_object_dropper::TextureObjectDropper;
use crate::image::util::{
    max_mipmap_levels, mipmap_size, region_2d_overlap_height, region_2d_overlap_width,
    region_2d_sub_image, region_3d_overlap_depth, region_3d_overlap_height,
    region_3d_overlap_width, region_3d_sub_image, texture_data_as_js_buffer,
};
use crate::image::{
    Image2DSource, LayeredImageSource, MaxMipmapLevelsExceeded, MipmapLevels, Region2D, Region3D,
};
use crate::runtime::state::ContextUpdate;
use crate::runtime::{Connection, RenderingContext};
use crate::task::{ContextId, GpuTask, Progress};
use crate::util::JsId;

/// Provides the information necessary for the creation of a [Texture2DArray].
///
/// See [RenderingContext::create_texture_2d_array] for details.
pub struct Texture2DArrayDescriptor<F>
where
    F: TextureFormat + 'static,
{
    /// The format type the [Texture2DArray] will use to store its image data.
    ///
    /// Must implement [TextureFormat].
    pub format: F,

    /// The width of the [Texture2DArray].
    pub width: u32,

    /// The height of the [Texture2DArray].
    pub height: u32,

    /// The depth of the [Texture2DArray].
    pub depth: u32,

    /// The number of levels in the mipmap for the [Texture2DArray].
    ///
    /// See [MipmapLevels] for details.
    pub levels: MipmapLevels,
}

/// Layered image storage for the (partial or complete) mipmap chain of an array of 2-dimensional
/// images.
///
/// A [Texture2DArray] is a layered image and has a width, a height and a depth; depth here
/// corresponds to the length of the array (see also the "Layered Image" section below). A
/// [Texture2DArray] defines a [TextureFormat] `F`: all image data stored in the texture is stored
/// in this format.
///
/// See [RenderingContext::create_texture_2d_array] for details on how a [Texture2DArray] is
/// created.
///
/// # Layered Image
///
/// Although the name might imply that a [Texture2DArray] should be thought of as an array of
/// 2-dimensional images, the semantics chosen for this API instead consider a [Texture2DArray] to
/// be a single "layered" image (much like a [Texture3D], with which the [Texture2DArray] API shares
/// many commonalities). The module documentation for [web_glitz::image] goes into more detail on
/// layered image storage.
///
/// These semantics make certain operations more intuitive. It is for example possible to upload the
/// image data for an entire mipmap [Level] at once from a single [LayeredImageSource] (see
/// [Level::upload_command]). It is also possible to first select a sub-image of a mipmap [Level]
/// (see [Level::sub_image]) and then upload data to only a sub-region of each layer (see
/// [LevelSubImage::upload_command]).
///
/// # Mipmap
///
/// A [Texture2DArray] stores a partial or complete mipmap chains its layered base image (or, if you
/// prefer, it stores an array of mipmap chains for each of its 2-dimensional base images). See the
/// module documentation for [web_glitz::image] for more information on mipmaps.
///
/// Note that a [Texture2DArray] does not necessarily have to store a complete mipmap chain, it may
/// only store a partial mipmap chain. For example, it may only store the first three levels (see
/// [MipmapLevels] for details). However, it must store at least the first level: level `0`, the
/// base level (see [Texture2DArray::base_level]).
///
/// Each image in the chain initially starts out in a "cleared" state, where each bit is set to `0`
/// (note that "zeroed data" is valid data for all [TextureFormat]s).
///
/// Mipmapping is typically used with minification filtering, in which case each level in the chain
/// is typically a down-filtered version of the previous level (see [MinificationFilter] for
/// details). If the texture format implements [Filterable], then the image data for such a chain
/// may be generated by first uploading data for the base level, and then generating the subsequent
/// levels with [Texture2DArray::generate_mipmap] (see [Texture2DArray::generate_mipmap] for
/// details).
///
/// # Sampling
///
/// The GPU may access the data in a [Texture2DArray], typically through a [Sampler] or
/// [ShadowSampler], see [Texture2DArray::sampled_float], [Texture2DArray::sampled_integer],
/// [Texture2DArray::sampled_unsigned_integer] and [Texture2DArray::sampled_shadow]. A sampled
/// [Texture2DArray] may be bound to a pipeline as a resource, see
/// [web_glitz::pipeline::resources::Resources].
///
/// # Example
///
/// The following example creates a 2d array texture with a width of 256 pixels, a height of 256 and
/// a depth of 16 layers stored in the [RGB8] format, with a complete mipmap chain. All pixels in
/// the base image are set to `[255, 0, 0]` (red) with an "upload" command and then the pixel data
/// for all other levels is generated with a "generate mipmap" command:
///
/// ```rust
/// # use web_glitz::runtime::RenderingContext;
/// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext + Clone + 'static {
/// use web_glitz::image::{LayeredImageSource, MipmapLevels};
/// use web_glitz::image::format::RGB8;
/// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
/// use web_glitz::sequence_all;
///
/// let texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
///     format: RGB8,
///     width: 256,
///     height: 256,
///     depth: 16,
///     levels: MipmapLevels::Complete
/// }).unwrap();
///
/// let pixels: Vec<[u8; 3]> = vec![[255, 0, 0]; 256 * 256 * 16];
/// let data = LayeredImageSource::from_pixels(pixels, 256, 256, 16).unwrap();
///
/// context.submit(sequence_all![
///     texture.base_level().upload_command(data),
///     texture.generate_mipmap_command()
/// ]);
/// # }
/// ```
pub struct Texture2DArray<F> {
    object_id: u64,
    data: Arc<Texture2DArrayData>,
    format: F,
}

impl<F> Texture2DArray<F> {
    pub(crate) fn data(&self) -> &Arc<Texture2DArrayData> {
        &self.data
    }
}

impl<F> Texture2DArray<F>
where
    F: TextureFormat + 'static,
{
    pub(crate) fn new<Rc>(
        context: &Rc,
        object_id: u64,
        descriptor: &Texture2DArrayDescriptor<F>,
    ) -> Result<Self, MaxMipmapLevelsExceeded>
    where
        Rc: RenderingContext + Clone + 'static,
    {
        let Texture2DArrayDescriptor {
            format,
            width,
            height,
            depth,
            levels,
            ..
        } = descriptor;
        let max_mipmap_levels = max_mipmap_levels(*width, *height);

        let levels = match levels {
            MipmapLevels::Complete => max_mipmap_levels,
            MipmapLevels::Partial(levels) => {
                if *levels > max_mipmap_levels {
                    return Err(MaxMipmapLevelsExceeded {
                        given: *levels,
                        max: max_mipmap_levels,
                    });
                }

                *levels
            }
        };

        let data = Arc::new(Texture2DArrayData {
            id: UnsafeCell::new(None),
            context_id: context.id(),
            dropper: Box::new(context.clone()),
            width: *width,
            height: *height,
            depth: *depth,
            levels,
        });

        context.submit(AllocateCommand::<F> {
            data: data.clone(),
            _marker: marker::PhantomData,
        });

        Ok(Texture2DArray {
            object_id,
            data,
            format: *format,
        })
    }

    /// Returns a reference to the base mipmap level for this [Texture2DArray] (level 0).
    pub fn base_level(&self) -> Level<F> {
        Level {
            handle: self,
            level: 0,
        }
    }

    /// Returns a mutable reference to the base mipmap level for this [Texture2DArray] (level 0).
    pub fn base_level_mut(&mut self) -> LevelMut<F> {
        LevelMut {
            inner: Level {
                handle: self,
                level: 0,
            },
        }
    }

    /// Returns a reference to the levels of this [Texture2DArray].
    ///
    /// See also [Texture2DArray::levels_mut].
    ///
    /// # Examples
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # use web_glitz::image::MipmapLevels;
    /// # use web_glitz::image::format::RGB8;
    /// # use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext + Clone + 'static {
    /// # let texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    /// #     format: RGB8,
    /// #     width: 256,
    /// #     height: 256,
    /// #     depth: 16,
    /// #     levels: MipmapLevels::Complete
    /// # }).unwrap();
    /// // Returns a reference to mipmap level 2 if the texture has a level 2, or None otherwise:
    /// let level_2 = texture.levels().get(2);
    ///
    /// // We can also iterate over references to all levels that exist for the texture:
    /// for level in texture.levels().iter() {
    ///     let index = level.level();
    ///     let width = level.width();
    ///     let height = level.height();
    ///
    ///     println!("Level {} has a width of {} and height of {}!", index, width, height);
    /// }
    /// # }
    /// ```
    pub fn levels(&self) -> Levels<F> {
        Levels {
            handle: self,
            offset: 0,
            len: self.data.levels,
        }
    }

    /// Returns a mutable reference to the levels of this [Texture2DArray].
    ///
    /// See also [Texture2DArray::levels].
    ///
    /// # Examples
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # use web_glitz::image::MipmapLevels;
    /// # use web_glitz::image::format::RGB8;
    /// # use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext + Clone + 'static {
    /// # let mut texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    /// #     format: RGB8,
    /// #     width: 256,
    /// #     height: 256,
    /// #     depth: 16,
    /// #     levels: MipmapLevels::Complete
    /// # }).unwrap();
    /// // Returns a mutable reference to mipmap level 2 if the texture has a level 2, or None
    /// // otherwise:
    /// let level_2 = texture.levels_mut().get_mut(2);
    /// # }
    /// ```
    pub fn levels_mut(&mut self) -> LevelsMut<F> {
        LevelsMut {
            inner: Levels {
                handle: self,
                offset: 0,
                len: self.data.levels,
            },
        }
    }

    /// The texture format for this [Texture2DArray]
    pub fn format(&self) -> F {
        self.format
    }

    /// The width of this [Texture2DArray].
    pub fn width(&self) -> u32 {
        self.data.width
    }

    /// The height of this [Texture2DArray].
    pub fn height(&self) -> u32 {
        self.data.height
    }

    /// The depth of this [Texture2DArray].
    pub fn depth(&self) -> u32 {
        self.data.depth
    }
}

impl<F> Texture2DArray<F>
where
    F: TextureFormat + Filterable + 'static,
{
    /// Returns a command which, when executed, will generate new mipmap data for the
    /// [Texture2DArray].
    ///
    /// This will overwrite the image data for each mipmap level except the base level. Starting at
    /// level 1, an image is generated that is half the width and height of the previous level
    /// (rounded down), by linear minification filtering of the previous level (see also
    /// [MinificationFilter::Linear]); this stops when the maximum level for which storage was
    /// allocated when the texture was created (see [RenderingContext::create_texture_2d_array]) has
    /// been overwritten. Note that the base level (level 0) is not modified (rather, it serves as
    /// the input for this process).
    ///
    /// This operation is only available to a texture if the texture format implements [Filterable].
    pub fn generate_mipmap_command(&self) -> GenerateMipmapCommand {
        GenerateMipmapCommand {
            texture_data: self.data.clone(),
        }
    }
}

impl<F> Texture2DArray<F>
where
    F: TextureFormat + FloatSamplable + 'static,
{
    /// Combines this [Texture2DArray] with the `sampler` as a [FloatSampledTexture2DArray], which
    /// can be bound to a pipeline as a texture resource.
    ///
    /// Returns an [IncompatibleSampler] error if the `sampler` is not compatible with this
    /// texture's format.
    ///
    /// See also [web_glitz::pipeline::resources::Resources].
    ///
    /// # Panics
    ///
    /// Panics if this texture and the `sampler` do not belong to the same [RenderingContext].
    pub fn float_sampled<S>(&self, sampler: S) -> FloatSampledTexture2DArray
    where
        S: CompatibleSampler<F>,
    {
        let sampler = sampler.get_ref();

        if self.data().context_id() != sampler.data().context_id() {
            panic!("Texture and sampler do not belong to the same context.");
        }

        FloatSampledTexture2DArray {
            sampler_data: sampler.data().clone(),
            texture_data: self.data().clone(),
            _marker: marker::PhantomData,
        }
    }
}

/// A texture-sampler combination that can bound to a pipeline as a resource for a 2D array floating
/// point sampler.
#[derive(Clone)]
pub struct FloatSampledTexture2DArray<'a> {
    pub(crate) sampler_data: Arc<SamplerData>,
    pub(crate) texture_data: Arc<Texture2DArrayData>,
    _marker: marker::PhantomData<&'a ()>,
}

impl<F> Texture2DArray<F>
where
    F: TextureFormat + IntegerSamplable + 'static,
{
    /// Combines this [Texture2DArray] with the `sampler` as a [IntegerSampledTexture2DArray], which
    /// can be bound to a pipeline as a texture resource.
    ///
    /// Returns an [IncompatibleSampler] error if the `sampler` is not compatible with this
    /// texture's format.
    ///
    /// See also [web_glitz::pipeline::resources::Resources].
    ///
    /// # Panics
    ///
    /// Panics if this texture and the `sampler` do not belong to the same [RenderingContext].
    pub fn integer_sampled<S>(&self, sampler: S) -> IntegerSampledTexture2DArray
    where
        S: CompatibleSampler<F>,
    {
        let sampler = sampler.get_ref();

        if self.data().context_id() != sampler.data().context_id() {
            panic!("Texture and sampler do not belong to the same context.");
        }

        IntegerSampledTexture2DArray {
            sampler_data: sampler.data().clone(),
            texture_data: self.data().clone(),
            _marker: marker::PhantomData,
        }
    }
}

/// A texture-sampler combination that can bound to a pipeline as a resource for a 2D array integer
/// sampler.
#[derive(Clone)]
pub struct IntegerSampledTexture2DArray<'a> {
    pub(crate) sampler_data: Arc<SamplerData>,
    pub(crate) texture_data: Arc<Texture2DArrayData>,
    _marker: marker::PhantomData<&'a ()>,
}

impl<F> Texture2DArray<F>
where
    F: TextureFormat + UnsignedIntegerSamplable + 'static,
{
    /// Combines this [Texture2DArray] with the `sampler` as a
    /// [UnsignedIntegerSampledTexture2DArray], which can be bound to a pipeline as a texture
    /// resource.
    ///
    /// Returns an [IncompatibleSampler] error if the `sampler` is not compatible with this
    /// texture's format.
    ///
    /// See also [web_glitz::pipeline::resources::Resources].
    ///
    /// # Panics
    ///
    /// Panics if this texture and the `sampler` do not belong to the same [RenderingContext].
    pub fn unsigned_integer_sampled<S>(&self, sampler: S) -> UnsignedIntegerSampledTexture2DArray
    where
        S: CompatibleSampler<F>,
    {
        let sampler = sampler.get_ref();

        if self.data().context_id() != sampler.data().context_id() {
            panic!("Texture and sampler do not belong to the same context.");
        }

        UnsignedIntegerSampledTexture2DArray {
            sampler_data: sampler.data().clone(),
            texture_data: self.data().clone(),
            _marker: marker::PhantomData,
        }
    }
}

/// A texture-sampler combination that can bound to a pipeline as a resource for a 2D array unsigned
/// integer sampler.
#[derive(Clone)]
pub struct UnsignedIntegerSampledTexture2DArray<'a> {
    pub(crate) sampler_data: Arc<SamplerData>,
    pub(crate) texture_data: Arc<Texture2DArrayData>,
    _marker: marker::PhantomData<&'a ()>,
}

impl<F> Texture2DArray<F>
where
    F: TextureFormat + ShadowSamplable + 'static,
{
    /// Combines this [Texture2DArray] with the `shadow_sampler` as a [ShadowSampledTexture2DArray],
    /// which can be bound to a pipeline as a texture resource.
    ///
    /// See also [web_glitz::pipeline::resources::Resources].
    ///
    /// # Panics
    ///
    /// Panics if this texture and the `shadow_sampler` do not belong to the same
    /// [RenderingContext].
    pub fn shadow_sampled(&self, shadow_sampler: &ShadowSampler) -> ShadowSampledTexture2DArray {
        if self.data().context_id() != shadow_sampler.data().context_id() {
            panic!("Texture and sampler do not belong to the same context.");
        }

        ShadowSampledTexture2DArray {
            sampler_data: shadow_sampler.data().clone(),
            texture_data: self.data().clone(),
            _marker: marker::PhantomData,
        }
    }
}

/// A texture-sampler combination that can bound to a pipeline as a resource for a 2D array shadow
/// sampler.
#[derive(Clone)]
pub struct ShadowSampledTexture2DArray<'a> {
    pub(crate) sampler_data: Arc<SamplerData>,
    pub(crate) texture_data: Arc<Texture2DArrayData>,
    _marker: marker::PhantomData<&'a ()>,
}

pub(crate) struct Texture2DArrayData {
    id: UnsafeCell<Option<JsId>>,
    context_id: u64,
    dropper: Box<dyn TextureObjectDropper>,
    width: u32,
    height: u32,
    depth: u32,
    levels: usize,
}

impl<F> PartialEq for Texture2DArray<F> {
    fn eq(&self, other: &Self) -> bool {
        self.object_id == other.object_id
    }
}

impl<F> Hash for Texture2DArray<F> {
    fn hash<H: Hasher>(&self, state: &mut H) {
        self.object_id.hash(state);
    }
}

impl Texture2DArrayData {
    pub(crate) fn id(&self) -> Option<JsId> {
        unsafe { *self.id.get() }
    }

    pub(crate) fn context_id(&self) -> u64 {
        self.context_id
    }
}

impl PartialEq for Texture2DArrayData {
    fn eq(&self, other: &Self) -> bool {
        self.id() == other.id()
    }
}

impl Hash for Texture2DArrayData {
    fn hash<H>(&self, state: &mut H)
    where
        H: Hasher,
    {
        self.id().hash(state);
    }
}

impl Drop for Texture2DArrayData {
    fn drop(&mut self) {
        if let Some(id) = self.id() {
            self.dropper.drop_texture_object(id);
        }
    }
}

/// Returned from [Texture2DArray::levels], a reference to the levels of a [Texture2DArray].
///
/// See [Texture2DArray::levels] for details.
#[derive(PartialEq, Hash)]
pub struct Levels<'a, F> {
    handle: &'a Texture2DArray<F>,
    offset: usize,
    len: usize,
}

impl<'a, F> Levels<'a, F>
where
    F: TextureFormat,
{
    /// The number of levels defined for the [Texture2DArray].
    pub fn len(&self) -> usize {
        self.len
    }

    /// Returns a reference to level at the `index`, or `None` if the `index` is out of bounds.
    ///
    /// See also [get_unchecked] for an unsafe variant of this method that does not do any bounds
    /// checks.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::MipmapLevels;
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    ///
    /// let texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     depth: 16,
    ///     levels: MipmapLevels::Partial(3)
    /// }).unwrap();
    ///
    /// let levels = texture.levels();
    ///
    /// assert_eq!(levels.get(1).map(|l| (l.width(), l.height())), Some((128, 128)));
    /// assert_eq!(levels.get(4).map(|l| (l.width(), l.height())), None);
    /// # }
    /// ```
    pub fn get<'b, I>(&'b self, index: I) -> Option<I::Output>
    where
        I: LevelsIndex<'b, F>,
    {
        index.get(self)
    }

    /// Returns a reference to level at the `index`, without doing any bounds checks.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::MipmapLevels;
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    ///
    /// let texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     depth: 16,
    ///     levels: MipmapLevels::Partial(3)
    /// }).unwrap();
    ///
    /// let levels = texture.levels();
    ///
    /// let level = unsafe { levels.get_unchecked(1) };
    ///
    /// assert_eq!((level.width(), level.height()), (128, 128));
    /// # }
    /// ```
    ///
    /// # Unsafe
    ///
    /// The `index` must be in bounds. See also [get] for a safe variant of this method that does
    /// bounds checks.
    pub unsafe fn get_unchecked<'b, I>(&'b self, index: I) -> I::Output
    where
        I: LevelsIndex<'b, F>,
    {
        index.get_unchecked(self)
    }

    /// Returns an iterator over the levels.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::MipmapLevels;
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    ///
    /// let texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     depth: 16,
    ///     levels: MipmapLevels::Partial(3)
    /// }).unwrap();
    ///
    /// let levels = texture.levels();
    /// let mut iter = levels.iter();
    ///
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), Some((128, 128)));
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), Some((64, 64)));
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), Some((32, 32)));
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), None);
    /// # }
    /// ```
    pub fn iter(&self) -> LevelsIter<F> {
        LevelsIter {
            handle: self.handle,
            current_level: self.offset,
            end_level: self.offset + self.len,
        }
    }
}

impl<'a, F> IntoIterator for Levels<'a, F>
where
    F: TextureFormat,
{
    type Item = Level<'a, F>;

    type IntoIter = LevelsIter<'a, F>;

    fn into_iter(self) -> Self::IntoIter {
        LevelsIter {
            handle: self.handle,
            current_level: self.offset,
            end_level: self.offset + self.len,
        }
    }
}

/// An iterator over [Levels].
///
/// See [Levels::iter] for details.
pub struct LevelsIter<'a, F> {
    handle: &'a Texture2DArray<F>,
    current_level: usize,
    end_level: usize,
}

impl<'a, F> Iterator for LevelsIter<'a, F>
where
    F: TextureFormat,
{
    type Item = Level<'a, F>;

    fn next(&mut self) -> Option<Self::Item> {
        let level = self.current_level;

        if level < self.end_level {
            self.current_level += 1;

            Some(Level {
                handle: self.handle,
                level,
            })
        } else {
            None
        }
    }
}

/// A helper trait for indexing [Levels].
///
/// See [Levels::get] and [Levels::get_unchecked].
pub trait LevelsIndex<'a, F> {
    /// The output type returned by the indexing operations.
    type Output;

    /// Returns the output for this operation if in bounds, or `None` otherwise.
    fn get(self, levels: &'a Levels<F>) -> Option<Self::Output>;

    /// Returns the output for this operation, without performing any bounds checking.
    unsafe fn get_unchecked(self, levels: &'a Levels<F>) -> Self::Output;
}

impl<'a, F> LevelsIndex<'a, F> for usize
where
    F: 'a,
{
    type Output = Level<'a, F>;

    fn get(self, levels: &'a Levels<F>) -> Option<Self::Output> {
        if self < levels.len {
            Some(Level {
                handle: levels.handle,
                level: levels.offset + self,
            })
        } else {
            None
        }
    }

    unsafe fn get_unchecked(self, levels: &'a Levels<F>) -> Self::Output {
        Level {
            handle: levels.handle,
            level: levels.offset + self,
        }
    }
}

impl<'a, F> LevelsIndex<'a, F> for RangeFull
where
    F: 'a,
{
    type Output = Levels<'a, F>;

    fn get(self, levels: &'a Levels<F>) -> Option<Self::Output> {
        Some(Levels {
            handle: levels.handle,
            offset: levels.offset,
            len: levels.len,
        })
    }

    unsafe fn get_unchecked(self, levels: &'a Levels<F>) -> Self::Output {
        Levels {
            handle: levels.handle,
            offset: levels.offset,
            len: levels.len,
        }
    }
}

impl<'a, F> LevelsIndex<'a, F> for Range<usize>
where
    F: 'a,
{
    type Output = Levels<'a, F>;

    fn get(self, levels: &'a Levels<F>) -> Option<Self::Output> {
        let Range { start, end } = self;

        if start > end || end > levels.len {
            None
        } else {
            Some(Levels {
                handle: levels.handle,
                offset: levels.offset + start,
                len: end - start,
            })
        }
    }

    unsafe fn get_unchecked(self, levels: &'a Levels<F>) -> Self::Output {
        let Range { start, end } = self;

        Levels {
            handle: levels.handle,
            offset: levels.offset + start,
            len: end - start,
        }
    }
}

impl<'a, F> LevelsIndex<'a, F> for RangeInclusive<usize>
where
    F: 'a,
{
    type Output = Levels<'a, F>;

    fn get(self, levels: &'a Levels<F>) -> Option<Self::Output> {
        if *self.end() == usize::max_value() {
            None
        } else {
            <Range<usize> as LevelsIndex<'a, F>>::get(*self.start()..self.end() + 1, levels)
        }
    }

    unsafe fn get_unchecked(self, levels: &'a Levels<F>) -> Self::Output {
        <Range<usize> as LevelsIndex<'a, F>>::get_unchecked(*self.start()..self.end() + 1, levels)
    }
}

impl<'a, F> LevelsIndex<'a, F> for RangeFrom<usize>
where
    F: 'a,
{
    type Output = Levels<'a, F>;

    fn get(self, levels: &'a Levels<F>) -> Option<Self::Output> {
        <Range<usize> as LevelsIndex<'a, F>>::get(self.start..levels.len, levels)
    }

    unsafe fn get_unchecked(self, levels: &'a Levels<F>) -> Self::Output {
        <Range<usize> as LevelsIndex<'a, F>>::get_unchecked(self.start..levels.len, levels)
    }
}

impl<'a, F> LevelsIndex<'a, F> for RangeTo<usize>
where
    F: 'a,
{
    type Output = Levels<'a, F>;

    fn get(self, levels: &'a Levels<F>) -> Option<Self::Output> {
        <Range<usize> as LevelsIndex<'a, F>>::get(0..self.end, levels)
    }

    unsafe fn get_unchecked(self, levels: &'a Levels<F>) -> Self::Output {
        <Range<usize> as LevelsIndex<'a, F>>::get_unchecked(0..self.end, levels)
    }
}

impl<'a, F> LevelsIndex<'a, F> for RangeToInclusive<usize>
where
    F: 'a,
{
    type Output = Levels<'a, F>;

    fn get(self, levels: &'a Levels<F>) -> Option<Self::Output> {
        <RangeInclusive<usize> as LevelsIndex<'a, F>>::get(0..=self.end, levels)
    }

    unsafe fn get_unchecked(self, levels: &'a Levels<F>) -> Self::Output {
        <RangeInclusive<usize> as LevelsIndex<'a, F>>::get_unchecked(0..=self.end, levels)
    }
}

/// A reference to a mipmap level of a [Texture2DArray].
///
/// A reference to the base level of a [Texture2DArray] can be obtained through
/// [Texture2DArray::base_level]. References to other levels of a [Texture2DArray] can be obtained
/// via [Levels].
#[derive(PartialEq, Hash)]
pub struct Level<'a, F> {
    handle: &'a Texture2DArray<F>,
    level: usize,
}

impl<'a, F> Level<'a, F>
where
    F: TextureFormat,
{
    /// Returns the integer that identifies this level.
    ///
    /// For example, if this [Level] is the texture's base level, returns `0`; if it is the second
    /// level, returns `1`; etc.
    pub fn level(&self) -> usize {
        self.level
    }

    /// Returns the width of this level.
    pub fn width(&self) -> u32 {
        mipmap_size(self.handle.data.width, self.level)
    }

    /// Returns the height of this level.
    pub fn height(&self) -> u32 {
        mipmap_size(self.handle.data.height, self.level)
    }

    /// Returns the depth of the layered image.
    ///
    /// Note that the depth is the same for all levels of a [Texture2DArray].
    pub fn depth(&self) -> u32 {
        self.handle.data.depth
    }

    /// Returns a reference to the layers of this [Level].
    ///
    /// # Examples
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # use web_glitz::image::MipmapLevels;
    /// # use web_glitz::image::format::RGB8;
    /// # use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext + Clone + 'static {
    /// # let texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    /// #     format: RGB8,
    /// #     width: 256,
    /// #     height: 256,
    /// #     depth: 16,
    /// #     levels: MipmapLevels::Complete
    /// # }).unwrap();
    /// // Returns a reference to layer 2 of the texture's base level if the texture has a 2 or more
    /// // layers, or None otherwise:
    /// let layer_2 = texture.base_level().layers().get(2);
    ///
    /// // We can also iterate over references to all layers that exist for the texture's base
    /// // level:
    /// for layer in texture.base_level().layers().iter() {
    ///     // ...
    /// }
    /// # }
    /// ```
    pub fn layers(&self) -> LevelLayers<F> {
        LevelLayers {
            handle: self.handle,
            level: self.level,
            offset: 0,
            len: self.handle.data.depth as usize,
        }
    }

    /// Returns a reference to the sub-region of this [Level]'s layered image described by `region`.
    ///
    /// # Example
    ///
    /// This may for example be used to upload data to only a sub-region of a layered image, rather
    /// than the complete image:
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext + Clone + 'static {
    /// use web_glitz::image::{LayeredImageSource, MipmapLevels, Region3D};
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    ///
    /// let texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     depth: 16,
    ///     levels: MipmapLevels::Complete
    /// }).unwrap();
    ///
    /// let base_level = texture.base_level();
    /// let sub_image = base_level.sub_image(Region3D::Area((0, 0, 0), 128, 128, 8));
    ///
    /// let pixels: Vec<[u8; 3]> = vec![[0, 0, 255]; 128 * 128 * 8];
    /// let data = LayeredImageSource::from_pixels(pixels, 128, 128, 8).unwrap();
    ///
    /// context.submit(sub_image.upload_command(data));
    /// # }
    /// ```
    ///
    /// The lower left quadrants of the first 8 layers of texture's base level now contain blue
    /// pixels.
    pub fn sub_image(&self, region: Region3D) -> LevelSubImage<F> {
        LevelSubImage {
            handle: self.handle,
            level: self.level,
            region,
        }
    }

    /// Returns a command which, when executed, replaces the image data in this [Level]'s image
    /// with the image data provided in `data`.
    ///
    /// The image data must be stored as a [PixelUnpack] type that is suitable for the texture's
    /// [TextureFormat].
    ///
    /// If the dimensions of the image provided in `data` are not sufficient to cover the [Level]'s
    /// image entirely, starting from the origin, then only the region of overlap is updated (note
    /// that the origin of a layered image is the lower left corner on the first layer). For
    /// example, given a [Level] with a width of 256 pixels, a height of 256 pixels and a depth
    /// of 16 layers, and `data` with a width of 128 pixels, a height of 128 pixels and a depth of
    /// 8 layers, then only the lower left quadrants of the first 8 layers of the [Level] are
    /// updated. If the dimensions of the image provided in `data` would, when starting from the
    /// origin, cover more than the [Level]'s image (the width, height and/or depth of `data` is/are
    /// greater than the width, height and/or depth of the [Level]'s image), then any pixels that
    /// would fall outside of the [Level] are ignored. For example, given a [Level] with a width of
    /// 256 pixels, a height of 256 pixels and a depth of 16 layers, and `data` with a width of 256
    /// pixels, height of 512 pixels and a depth of 20 layers, then only the lower halves of the
    /// first 16 layers in `data` are used to update the [Level]; the upper halves of the first 16
    /// layers in `data` are ignored, and the last 4 layers in `data` are ignored entirely.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext + Clone + 'static {
    /// use web_glitz::image::{LayeredImageSource, MipmapLevels};
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    ///
    /// let texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     depth: 16,
    ///     levels: MipmapLevels::Complete
    /// }).unwrap();
    ///
    /// let pixels: Vec<[u8; 3]> = vec![[255, 0, 0]; 256 * 256 * 16];
    /// let data = LayeredImageSource::from_pixels(pixels, 256, 256, 16).unwrap();
    ///
    /// context.submit(texture.base_level().upload_command(data));
    /// # }
    /// ```
    pub fn upload_command<D, T>(
        &self,
        data: LayeredImageSource<D, T>,
    ) -> LevelUploadCommand<D, T, F>
    where
        T: PixelUnpack<F>,
    {
        LevelUploadCommand {
            data,
            texture_data: self.handle.data.clone(),
            level: self.level,
            region: Region3D::Fill,
            _marker: marker::PhantomData,
        }
    }
}

/// Returned from [Level::layers], a reference to the layers of a [Level].
///
/// See [Level::layers] for details.
#[derive(PartialEq, Hash)]
pub struct LevelLayers<'a, F> {
    handle: &'a Texture2DArray<F>,
    level: usize,
    offset: usize,
    len: usize,
}

impl<'a, F> LevelLayers<'a, F>
where
    F: TextureFormat,
{
    /// The number of layers for the level.
    ///
    /// Note that each mipmap level for a [Texture2DArray] as the same number of layers.
    pub fn len(&self) -> usize {
        self.len
    }

    /// Returns a reference to the layer at the `index`, or `None` if the `index` is out of bounds.
    ///
    /// See also [get_unchecked] for an unsafe variant of this method that does not do any bounds
    /// checks.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::MipmapLevels;
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    ///
    /// let texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     depth: 16,
    ///     levels: MipmapLevels::Partial(3)
    /// }).unwrap();
    ///
    /// let base_level = texture.base_level();
    /// let layers = base_level.layers();
    ///
    /// assert_eq!(layers.get(1).map(|l| (l.width(), l.height())), Some((128, 128)));
    /// assert_eq!(layers.get(4).map(|l| (l.width(), l.height())), None);
    /// # }
    /// ```
    pub fn get<'b, I>(&'b self, index: I) -> Option<I::Output>
    where
        I: LevelLayersIndex<'b, F>,
    {
        index.get(self)
    }

    /// Returns a reference to layer at the `index`, without doing any bounds checks.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::MipmapLevels;
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    ///
    /// let texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     depth: 16,
    ///     levels: MipmapLevels::Partial(3)
    /// }).unwrap();
    ///
    /// let base_level = texture.base_level();
    /// let layers = base_level.layers();
    ///
    /// let layer = unsafe { layers.get_unchecked(1) };
    ///
    /// assert_eq!((layer.width(), layer.height()), (128, 128));
    /// # }
    /// ```
    ///
    /// # Unsafe
    ///
    /// The `index` must be in bounds. See also [get] for a safe variant of this method that does
    /// bounds checks.
    pub unsafe fn get_unchecked<'b, I>(&'b self, index: I) -> I::Output
    where
        I: LevelLayersIndex<'b, F>,
    {
        index.get_unchecked(self)
    }

    /// Returns an iterator over the layers.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::MipmapLevels;
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    ///
    /// let texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     depth: 4,
    ///     levels: MipmapLevels::Complete
    /// }).unwrap();
    ///
    /// let base_level = texture.base_level();
    /// let layers = base_level.layers();
    /// let mut iter = layers.iter();
    ///
    /// assert!(iter.next().is_some());
    /// assert!(iter.next().is_some());
    /// assert!(iter.next().is_some());
    /// assert!(iter.next().is_some());
    /// assert!(iter.next().is_none());
    /// # }
    /// ```
    pub fn iter(&self) -> LevelLayersIter<F> {
        LevelLayersIter {
            handle: self.handle,
            level: self.level,
            current_layer: self.offset,
            end_layer: self.offset + self.len,
        }
    }
}

impl<'a, F> IntoIterator for LevelLayers<'a, F>
where
    F: TextureFormat,
{
    type Item = LevelLayer<'a, F>;

    type IntoIter = LevelLayersIter<'a, F>;

    fn into_iter(self) -> Self::IntoIter {
        LevelLayersIter {
            handle: self.handle,
            level: self.level,
            current_layer: self.offset,
            end_layer: self.offset + self.len,
        }
    }
}

/// Iterator over de layers in [LevelLayers].
///
/// See [LevelLayers::iter].
pub struct LevelLayersIter<'a, F> {
    handle: &'a Texture2DArray<F>,
    level: usize,
    current_layer: usize,
    end_layer: usize,
}

impl<'a, F> Iterator for LevelLayersIter<'a, F>
where
    F: TextureFormat,
{
    type Item = LevelLayer<'a, F>;

    fn next(&mut self) -> Option<Self::Item> {
        let layer = self.current_layer;

        if layer < self.end_layer {
            self.current_layer += 1;

            Some(LevelLayer {
                handle: self.handle,
                level: self.level,
                layer,
            })
        } else {
            None
        }
    }
}

/// A helper trait for indexing [LevelLayers].
///
/// See [LevelLayers::get] and [LevelLayers::get_unchecked].
pub trait LevelLayersIndex<'a, F> {
    /// The output type returned by the indexing operations.
    type Output;

    /// Returns the output for this operation if in bounds, or `None` otherwise.
    fn get(self, layers: &'a LevelLayers<F>) -> Option<Self::Output>;

    /// Returns the output for this operation, without performing any bounds checking.
    unsafe fn get_unchecked(self, layers: &'a LevelLayers<F>) -> Self::Output;
}

impl<'a, F> LevelLayersIndex<'a, F> for usize
where
    F: 'a,
{
    type Output = LevelLayer<'a, F>;

    fn get(self, layers: &'a LevelLayers<F>) -> Option<Self::Output> {
        if self < layers.len {
            Some(LevelLayer {
                handle: layers.handle,
                level: layers.level,
                layer: layers.offset + self,
            })
        } else {
            None
        }
    }

    unsafe fn get_unchecked(self, layers: &'a LevelLayers<F>) -> Self::Output {
        LevelLayer {
            handle: layers.handle,
            level: layers.level,
            layer: layers.offset + self,
        }
    }
}

impl<'a, F> LevelLayersIndex<'a, F> for RangeFull
where
    F: 'a,
{
    type Output = LevelLayers<'a, F>;

    fn get(self, layers: &'a LevelLayers<F>) -> Option<Self::Output> {
        Some(LevelLayers {
            handle: layers.handle,
            level: layers.level,
            offset: layers.offset,
            len: layers.len,
        })
    }

    unsafe fn get_unchecked(self, layers: &'a LevelLayers<F>) -> Self::Output {
        LevelLayers {
            handle: layers.handle,
            level: layers.level,
            offset: layers.offset,
            len: layers.len,
        }
    }
}

impl<'a, F> LevelLayersIndex<'a, F> for Range<usize>
where
    F: 'a,
{
    type Output = LevelLayers<'a, F>;

    fn get(self, layers: &'a LevelLayers<F>) -> Option<Self::Output> {
        let Range { start, end } = self;

        if start > end || end > layers.len {
            None
        } else {
            Some(LevelLayers {
                handle: layers.handle,
                level: layers.level,
                offset: layers.offset + start,
                len: end - start,
            })
        }
    }

    unsafe fn get_unchecked(self, layers: &'a LevelLayers<F>) -> Self::Output {
        let Range { start, end } = self;

        LevelLayers {
            handle: layers.handle,
            level: layers.level,
            offset: layers.offset + start,
            len: end - start,
        }
    }
}

impl<'a, F> LevelLayersIndex<'a, F> for RangeInclusive<usize>
where
    F: 'a,
{
    type Output = LevelLayers<'a, F>;

    fn get(self, layers: &'a LevelLayers<F>) -> Option<Self::Output> {
        if *self.end() == usize::max_value() {
            None
        } else {
            <Range<usize> as LevelLayersIndex<'a, F>>::get(*self.start()..self.end() + 1, layers)
        }
    }

    unsafe fn get_unchecked(self, layers: &'a LevelLayers<F>) -> Self::Output {
        <Range<usize> as LevelLayersIndex<'a, F>>::get_unchecked(
            *self.start()..self.end() + 1,
            layers,
        )
    }
}

impl<'a, F> LevelLayersIndex<'a, F> for RangeFrom<usize>
where
    F: 'a,
{
    type Output = LevelLayers<'a, F>;

    fn get(self, layers: &'a LevelLayers<F>) -> Option<Self::Output> {
        <Range<usize> as LevelLayersIndex<'a, F>>::get(self.start..layers.len, layers)
    }

    unsafe fn get_unchecked(self, layers: &'a LevelLayers<F>) -> Self::Output {
        <Range<usize> as LevelLayersIndex<'a, F>>::get_unchecked(self.start..layers.len, layers)
    }
}

impl<'a, F> LevelLayersIndex<'a, F> for RangeTo<usize>
where
    F: 'a,
{
    type Output = LevelLayers<'a, F>;

    fn get(self, layers: &'a LevelLayers<F>) -> Option<Self::Output> {
        <Range<usize> as LevelLayersIndex<'a, F>>::get(0..self.end, layers)
    }

    unsafe fn get_unchecked(self, layers: &'a LevelLayers<F>) -> Self::Output {
        <Range<usize> as LevelLayersIndex<'a, F>>::get_unchecked(0..self.end, layers)
    }
}

impl<'a, F> LevelLayersIndex<'a, F> for RangeToInclusive<usize>
where
    F: 'a,
{
    type Output = LevelLayers<'a, F>;

    fn get(self, layers: &'a LevelLayers<F>) -> Option<Self::Output> {
        <RangeInclusive<usize> as LevelLayersIndex<'a, F>>::get(0..=self.end, layers)
    }

    unsafe fn get_unchecked(self, layers: &'a LevelLayers<F>) -> Self::Output {
        <RangeInclusive<usize> as LevelLayersIndex<'a, F>>::get_unchecked(0..=self.end, layers)
    }
}

/// Reference to a layer of a [Level] of a [Texture2DArray].
#[derive(PartialEq, Hash)]
pub struct LevelLayer<'a, F> {
    handle: &'a Texture2DArray<F>,
    level: usize,
    layer: usize,
}

impl<'a, F> LevelLayer<'a, F>
where
    F: TextureFormat,
{
    pub(crate) fn texture_data(&self) -> &Arc<Texture2DArrayData> {
        &self.handle.data
    }

    /// The index that identifies the [Level] this [LevelLayer] belongs to.
    pub fn level(&self) -> usize {
        self.level
    }

    /// The index that identifies this layer within the [Level] it belongs to.
    ///
    /// Corresponds to the depth at which this [LevelLayer] occurs.
    pub fn layer(&self) -> usize {
        self.layer
    }

    /// The width of the [LevelLayer].
    pub fn width(&self) -> u32 {
        mipmap_size(self.handle.data.width, self.level)
    }

    /// The height of the [LevelLayer].
    pub fn height(&self) -> u32 {
        mipmap_size(self.handle.data.height, self.level)
    }

    /// Returns a reference to the sub-region of this [LevelLayer]'s image described by `region`.
    ///
    /// # Example
    ///
    /// This may for example be used to upload data to only a sub-region of an image, rather than
    /// the complete image:
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext + Clone + 'static {
    /// use web_glitz::image::{Image2DSource, MipmapLevels, Region2D};
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    ///
    /// let texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     depth: 16,
    ///     levels: MipmapLevels::Complete
    /// }).unwrap();
    ///
    /// let base_level = texture.base_level();
    /// let layers = base_level.layers();
    /// let layer = layers.get(0).unwrap();
    /// let sub_image = layer.sub_image(Region2D::Area((0, 0), 128, 128));
    ///
    /// let pixels: Vec<[u8; 3]> = vec![[0, 0, 255]; 128 * 128];
    /// let data = Image2DSource::from_pixels(pixels, 128, 128).unwrap();
    ///
    /// context.submit(sub_image.upload_command(data));
    /// # }
    /// ```
    ///
    /// The lower left quadrant of the first layer of the texture's base level now contains blue
    /// pixels.
    pub fn sub_image(&self, region: Region2D) -> LevelLayerSubImage<F> {
        LevelLayerSubImage {
            handle: self.handle,
            level: self.level,
            layer: self.layer,
            region,
        }
    }

    /// Returns a command which, when executed, replaces the image data in this [LevelLayer]'s image
    /// with the image data provided in `data`.
    ///
    /// The image data must be stored as a [PixelUnpack] type that is suitable for the texture's
    /// [TextureFormat].
    ///
    /// If the dimensions of the image provided in `data` are not sufficient to cover the
    /// [LevelLayer]'s image entirely, starting from the origin, then only the region of overlap is
    /// updated (note that the origin of an image is the lower left corner). For example, given a
    /// [LevelLayer] with a width of 256 pixels and a height of 256 pixels, and `data` with a width
    /// of 128 pixels and a height of 128 pixels, then only the lower left quadrant of the [Level]
    /// is updated. If the dimensions of the image provided in `data` would, when starting from the
    /// origin, cover more than the [LevelLayer]'s image (the width and/or height of `data` is/are
    /// greater than the width and/or height of the [Level]'s image), then any pixels that would
    /// fall outside of the [LevelLayer] are ignored. For example, given a [LevelLayer] with a width
    /// of 256 pixels and a height of 256 pixels, and `data` with a width of 256 pixels and a height
    /// of 512 pixels, then only the lower half of the image in `data` is used to update the
    /// [LevelLayer] and the upper half is ignored.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext + Clone + 'static {
    /// use web_glitz::image::{Image2DSource, MipmapLevels};
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    ///
    /// let texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     depth: 16,
    ///     levels: MipmapLevels::Complete
    /// }).unwrap();
    ///
    /// let base_level = texture.base_level();
    /// let layers = base_level.layers();
    /// let layer = layers.get(0).unwrap();
    ///
    /// let pixels: Vec<[u8; 3]> = vec![[255, 0, 0]; 256 * 256];
    /// let data = Image2DSource::from_pixels(pixels, 256, 256).unwrap();
    ///
    /// context.submit(layer.upload_command(data));
    /// # }
    /// ```
    pub fn upload_command<D, T>(
        &self,
        data: Image2DSource<D, T>,
    ) -> LevelLayerUploadCommand<D, T, F>
    where
        T: PixelUnpack<F>,
    {
        LevelLayerUploadCommand {
            data,
            texture_data: self.handle.data.clone(),
            level: self.level,
            layer: self.layer,
            region: Region2D::Fill,
            _marker: marker::PhantomData,
        }
    }
}

/// Layered sub-image of a [Level].
///
/// See [Level::sub_image] for details.
#[derive(PartialEq, Hash)]
pub struct LevelSubImage<'a, F> {
    handle: &'a Texture2DArray<F>,
    level: usize,
    region: Region3D,
}

impl<'a, F> LevelSubImage<'a, F>
where
    F: TextureFormat,
{
    /// Returns the index of the mipmap level this [LevelSubImage] references.
    pub fn level(&self) -> usize {
        self.level
    }

    /// Returns the [Region3D] of the level this [LevelSubImage] references.
    pub fn region(&self) -> Region3D {
        self.region
    }

    /// Returns the width of this [LevelSubImage].
    pub fn width(&self) -> u32 {
        region_3d_overlap_width(self.handle.data.width, self.level, &self.region)
    }

    /// Returns the height of this [LevelSubImage].
    pub fn height(&self) -> u32 {
        region_3d_overlap_height(self.handle.data.height, self.level, &self.region)
    }

    /// Returns the depth of this [LevelSubImage].
    pub fn depth(&self) -> u32 {
        region_3d_overlap_depth(self.handle.data.depth, &self.region)
    }

    /// Returns a reference to the layers of this [LevelSubImage].
    ///
    /// See also [Level::layers].
    pub fn layers(&self) -> LevelSubImageLayers<F> {
        let (start_layer, end_layer, region) = match self.region {
            Region3D::Area((offset_x, offset_y, offset_z), width, height, depth) => {
                let max_layer = cmp::min(self.handle.data.depth, offset_z + depth);

                (
                    offset_z,
                    max_layer,
                    Region2D::Area((offset_x, offset_y), width, height),
                )
            }
            Region3D::Fill => (0, self.handle.data.depth, Region2D::Fill),
        };

        LevelSubImageLayers {
            handle: self.handle,
            level: self.level,
            offset: start_layer as usize,
            len: (end_layer - start_layer) as usize,
            region,
        }
    }

    /// Returns a reference to the sub-region of this [LevelSubImage]'s image described by `region`.
    ///
    /// See also [Level::sub_image].
    pub fn sub_image(&self, region: Region3D) -> LevelSubImage<F> {
        LevelSubImage {
            handle: self.handle,
            level: self.level,
            region: region_3d_sub_image(self.region, region),
        }
    }

    /// Returns a command which, when executed, replaces the image data in this [LevelSubImage]'s
    /// image region with the image data provided in `data`.
    ///
    /// See also [Level::upload_command].
    pub fn upload_command<D, T>(
        &self,
        data: LayeredImageSource<D, T>,
    ) -> LevelUploadCommand<D, T, F>
    where
        T: PixelUnpack<F>,
    {
        LevelUploadCommand {
            data,
            texture_data: self.handle.data.clone(),
            level: self.level,
            region: self.region,
            _marker: marker::PhantomData,
        }
    }
}

/// Reference to the layers of a [LevelSubImage].
///
/// See [LevelSubImage::layers].
#[derive(PartialEq, Hash)]
pub struct LevelSubImageLayers<'a, F> {
    handle: &'a Texture2DArray<F>,
    level: usize,
    offset: usize,
    len: usize,
    region: Region2D,
}

impl<'a, F> LevelSubImageLayers<'a, F>
where
    F: TextureFormat,
{
    /// The number of layers for the [LevelSubImage].
    pub fn len(&self) -> usize {
        self.len
    }

    /// Returns a reference to the layer at the `index`, or `None` if the `index` is out of bounds.
    ///
    /// See also [get_unchecked] for an unsafe variant of this method that does not do any bounds
    /// checks.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::{MipmapLevels, Region3D};
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    ///
    /// let texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     depth: 16,
    ///     levels: MipmapLevels::Partial(3)
    /// }).unwrap();
    ///
    /// let base_level = texture.base_level();
    /// let sub_image = base_level.sub_image(Region3D::Area((0, 0, 0), 128, 128, 8));
    /// let layers = sub_image.layers();
    ///
    /// assert!(layers.get(1).is_some());
    /// assert!(layers.get(8).is_none());
    /// # }
    /// ```
    pub fn get<'b, I>(&'b self, index: I) -> Option<I::Output>
    where
        I: LevelSubImageLayersIndex<'b, F>,
    {
        index.get(self)
    }

    /// Returns a reference to layer at the `index`, without doing any bounds checks.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::{MipmapLevels, Region3D};
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    ///
    /// let texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     depth: 16,
    ///     levels: MipmapLevels::Partial(3)
    /// }).unwrap();
    ///
    /// let base_level = texture.base_level();
    /// let sub_image = base_level.sub_image(Region3D::Area((0, 0, 0), 128, 128, 8));
    /// let layers = sub_image.layers();
    ///
    /// let layer = unsafe { layers.get_unchecked(1) };
    /// # }
    /// ```
    ///
    /// # Unsafe
    ///
    /// The `index` must be in bounds. See also [get] for a safe variant of this method that does
    /// bounds checks.
    pub unsafe fn get_unchecked<'b, I>(&'b self, index: I) -> I::Output
    where
        I: LevelSubImageLayersIndex<'b, F>,
    {
        index.get_unchecked(self)
    }

    /// Returns an iterator over the layers.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::{MipmapLevels, Region3D};
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    ///
    /// let texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     depth: 16,
    ///     levels: MipmapLevels::Partial(3)
    /// }).unwrap();
    ///
    /// let base_level = texture.base_level();
    /// let sub_image = base_level.sub_image(Region3D::Area((0, 0, 0), 128, 128, 3));
    /// let layers = sub_image.layers();
    /// let mut iter = layers.iter();
    ///
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), Some((128, 128)));
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), Some((128, 128)));
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), Some((128, 128)));
    /// assert!(iter.next().is_none());
    /// # }
    /// ```
    pub fn iter(&self) -> LevelSubImageLayersIter<F> {
        LevelSubImageLayersIter {
            handle: self.handle,
            level: self.level,
            region: self.region,
            current_layer: self.offset as usize,
            end_layer: self.offset + self.len as usize,
        }
    }
}

impl<'a, F> IntoIterator for LevelSubImageLayers<'a, F>
where
    F: TextureFormat,
{
    type Item = LevelLayerSubImage<'a, F>;

    type IntoIter = LevelSubImageLayersIter<'a, F>;

    fn into_iter(self) -> Self::IntoIter {
        LevelSubImageLayersIter {
            handle: self.handle,
            level: self.level,
            region: self.region,
            current_layer: self.offset as usize,
            end_layer: self.offset + self.len as usize,
        }
    }
}

/// An iterator over [LevelSubImageLayers].
///
/// See [LevelSubImageLayers::iter].
pub struct LevelSubImageLayersIter<'a, F> {
    handle: &'a Texture2DArray<F>,
    level: usize,
    region: Region2D,
    current_layer: usize,
    end_layer: usize,
}

impl<'a, F> Iterator for LevelSubImageLayersIter<'a, F>
where
    F: TextureFormat,
{
    type Item = LevelLayerSubImage<'a, F>;

    fn next(&mut self) -> Option<Self::Item> {
        let layer = self.current_layer;

        if layer < self.end_layer {
            self.current_layer += 1;

            Some(LevelLayerSubImage {
                handle: self.handle,
                level: self.level,
                layer,
                region: self.region,
            })
        } else {
            None
        }
    }
}

/// A helper trait for indexing [LevelSubImageLayers].
///
/// See [LevelSubImageLayers::get] and [LevelSubImageLayers::get_unchecked].
pub trait LevelSubImageLayersIndex<'a, F> {
    /// The output type returned by the indexing operations.
    type Output;

    /// Returns the output for this operation if in bounds, or `None` otherwise.
    fn get(self, layers: &'a LevelSubImageLayers<F>) -> Option<Self::Output>;

    /// Returns the output for this operation, without performing any bounds checking.
    unsafe fn get_unchecked(self, layers: &'a LevelSubImageLayers<F>) -> Self::Output;
}

impl<'a, F> LevelSubImageLayersIndex<'a, F> for usize
where
    F: 'a,
{
    type Output = LevelLayerSubImage<'a, F>;

    fn get(self, layers: &'a LevelSubImageLayers<F>) -> Option<Self::Output> {
        if self < layers.len {
            Some(LevelLayerSubImage {
                handle: layers.handle,
                level: layers.level,
                layer: layers.offset + self,
                region: layers.region,
            })
        } else {
            None
        }
    }

    unsafe fn get_unchecked(self, layers: &'a LevelSubImageLayers<F>) -> Self::Output {
        LevelLayerSubImage {
            handle: layers.handle,
            level: layers.level,
            layer: layers.offset + self,
            region: layers.region,
        }
    }
}

impl<'a, F> LevelSubImageLayersIndex<'a, F> for RangeFull
where
    F: 'a,
{
    type Output = LevelSubImageLayers<'a, F>;

    fn get(self, layers: &'a LevelSubImageLayers<F>) -> Option<Self::Output> {
        Some(LevelSubImageLayers {
            handle: layers.handle,
            level: layers.level,
            offset: layers.offset,
            len: layers.len,
            region: layers.region,
        })
    }

    unsafe fn get_unchecked(self, layers: &'a LevelSubImageLayers<F>) -> Self::Output {
        LevelSubImageLayers {
            handle: layers.handle,
            level: layers.level,
            offset: layers.offset,
            len: layers.len,
            region: layers.region,
        }
    }
}

impl<'a, F> LevelSubImageLayersIndex<'a, F> for Range<usize>
where
    F: 'a,
{
    type Output = LevelSubImageLayers<'a, F>;

    fn get(self, layers: &'a LevelSubImageLayers<F>) -> Option<Self::Output> {
        let Range { start, end } = self;

        if start > end || end > layers.len {
            None
        } else {
            Some(LevelSubImageLayers {
                handle: layers.handle,
                level: layers.level,
                offset: layers.offset + start,
                len: end - start,
                region: layers.region,
            })
        }
    }

    unsafe fn get_unchecked(self, layers: &'a LevelSubImageLayers<F>) -> Self::Output {
        let Range { start, end } = self;

        LevelSubImageLayers {
            handle: layers.handle,
            level: layers.level,
            offset: layers.offset + start,
            len: end - start,
            region: layers.region,
        }
    }
}

impl<'a, F> LevelSubImageLayersIndex<'a, F> for RangeInclusive<usize>
where
    F: 'a,
{
    type Output = LevelSubImageLayers<'a, F>;

    fn get(self, layers: &'a LevelSubImageLayers<F>) -> Option<Self::Output> {
        if *self.end() == usize::max_value() {
            None
        } else {
            <Range<usize> as LevelSubImageLayersIndex<'a, F>>::get(
                *self.start()..self.end() + 1,
                layers,
            )
        }
    }

    unsafe fn get_unchecked(self, layers: &'a LevelSubImageLayers<F>) -> Self::Output {
        <Range<usize> as LevelSubImageLayersIndex<'a, F>>::get_unchecked(
            *self.start()..self.end() + 1,
            layers,
        )
    }
}

impl<'a, F> LevelSubImageLayersIndex<'a, F> for RangeFrom<usize>
where
    F: 'a,
{
    type Output = LevelSubImageLayers<'a, F>;

    fn get(self, layers: &'a LevelSubImageLayers<F>) -> Option<Self::Output> {
        <Range<usize> as LevelSubImageLayersIndex<'a, F>>::get(self.start..layers.len, layers)
    }

    unsafe fn get_unchecked(self, layers: &'a LevelSubImageLayers<F>) -> Self::Output {
        <Range<usize> as LevelSubImageLayersIndex<'a, F>>::get_unchecked(
            self.start..layers.len,
            layers,
        )
    }
}

impl<'a, F> LevelSubImageLayersIndex<'a, F> for RangeTo<usize>
where
    F: 'a,
{
    type Output = LevelSubImageLayers<'a, F>;

    fn get(self, layers: &'a LevelSubImageLayers<F>) -> Option<Self::Output> {
        <Range<usize> as LevelSubImageLayersIndex<'a, F>>::get(0..self.end, layers)
    }

    unsafe fn get_unchecked(self, layers: &'a LevelSubImageLayers<F>) -> Self::Output {
        <Range<usize> as LevelSubImageLayersIndex<'a, F>>::get_unchecked(0..self.end, layers)
    }
}

impl<'a, F> LevelSubImageLayersIndex<'a, F> for RangeToInclusive<usize>
where
    F: 'a,
{
    type Output = LevelSubImageLayers<'a, F>;

    fn get(self, layers: &'a LevelSubImageLayers<F>) -> Option<Self::Output> {
        <RangeInclusive<usize> as LevelSubImageLayersIndex<'a, F>>::get(0..=self.end, layers)
    }

    unsafe fn get_unchecked(self, layers: &'a LevelSubImageLayers<F>) -> Self::Output {
        <RangeInclusive<usize> as LevelSubImageLayersIndex<'a, F>>::get_unchecked(
            0..=self.end,
            layers,
        )
    }
}

#[derive(PartialEq, Hash)]
pub struct LevelLayerSubImage<'a, F> {
    handle: &'a Texture2DArray<F>,
    level: usize,
    layer: usize,
    region: Region2D,
}

impl<'a, F> LevelLayerSubImage<'a, F>
where
    F: TextureFormat,
{
    pub(crate) fn level_layer_ref(&self) -> LevelLayer<F> {
        LevelLayer {
            handle: self.handle,
            level: self.level,
            layer: self.layer,
        }
    }

    pub(crate) fn texture_data(&self) -> &Arc<Texture2DArrayData> {
        &self.handle.data
    }

    /// Returns the index of the mipmap level this [LevelLayerSubImage] references.
    pub fn level(&self) -> usize {
        self.level
    }

    /// Returns the index of the layer this [LevelLayerSubImage] references.
    pub fn layer(&self) -> usize {
        self.layer
    }

    /// Returns the [Region2D] of the level-layer this [LevelLayerSubImage] references.
    pub fn region(&self) -> Region2D {
        self.region
    }

    /// Returns the width of this [LevelLayerSubImage].
    pub fn width(&self) -> u32 {
        region_2d_overlap_width(self.handle.data.width, self.level, &self.region)
    }

    /// Returns the height of this [LevelLayerSubImage].
    pub fn height(&self) -> u32 {
        region_2d_overlap_height(self.handle.data.height, self.level, &self.region)
    }

    /// Returns a reference to the sub-region of this [LevelLayerSubImage]'s image described by
    /// `region`.
    ///
    /// See also [LevelLayer::sub_image].
    pub fn sub_image(&self, region: Region2D) -> LevelLayerSubImage<F> {
        LevelLayerSubImage {
            handle: self.handle,
            level: self.level,
            layer: self.layer,
            region: region_2d_sub_image(self.region, region),
        }
    }

    /// Returns a command which, when executed, replaces the image data in this
    /// [LevelLayerSubImage]'s image region with the image data provided in `data`.
    ///
    /// The image data must be stored as a [PixelUnpack] type that is suitable for the texture's
    /// [TextureFormat].
    ///
    /// If the dimensions of the image provided in `data` are not sufficient to cover the
    /// [LevelLayerSubImage] region entirely, starting from the region's origin, then only the
    /// region of overlap is updated (note that the origin of a region is at its lower left corner).
    /// For example, given a [LevelLayerSubImage] with a width of 256 pixels and a height of 256
    /// pixels, and `data` with a width of 128 pixels and a height of 128 pixels, then only the
    /// lower left quadrant of the [LevelLayerSubImage]'s region is updated. If the dimensions of
    /// the image provided in `data` would, when starting from the [LevelLayerSubImage]'s origin,
    /// cover more than the [LevelLayerSubImage]'s region (the width and/or height of `data` is/are
    /// greater than the width and/or height of the [LevelLayerSubImage]'s region), then any pixels
    /// that would fall outside of the [LevelLayerSubImage]'s region are ignored. For example, given
    /// a [LevelLayerSubImage] with a width of 256 pixels and a height of 256 pixels, and `data`
    /// with a width of 256 pixels and a height of 512 pixels, then only the lower half of the image
    /// in `data` is used to update the [LevelLayerSubImage] and the upper half is ignored.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext + Clone + 'static {
    /// use web_glitz::image::{Image2DSource, MipmapLevels, Region2D};
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    ///
    /// let texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     depth: 16,
    ///     levels: MipmapLevels::Complete
    /// }).unwrap();
    ///
    /// let level = texture.base_level();
    /// let layers = level.layers();
    /// let layer = layers.get(0).unwrap();
    /// let sub_image = layer.sub_image(Region2D::Area((0, 0), 128, 128));
    ///
    /// let pixels: Vec<[u8; 3]> = vec![[255, 0, 0]; 128 * 128];
    /// let data = Image2DSource::from_pixels(pixels, 128, 128).unwrap();
    ///
    /// context.submit(sub_image.upload_command(data));
    /// # }
    /// ```
    pub fn upload_command<D, T>(
        &self,
        data: Image2DSource<D, T>,
    ) -> LevelLayerUploadCommand<D, T, F>
    where
        T: PixelUnpack<F>,
    {
        LevelLayerUploadCommand {
            data,
            texture_data: self.handle.data.clone(),
            level: self.level,
            layer: self.layer,
            region: self.region,
            _marker: marker::PhantomData,
        }
    }
}

/// Returned from [Texture2DArray::levels_mut], a mutable reference to the levels of a
/// [Texture2DArray].
///
/// See [Texture2DArray::levels_mut] for details.
///
///
/// [Deref]s to [Levels].
#[derive(PartialEq, Hash)]
pub struct LevelsMut<'a, F> {
    inner: Levels<'a, F>,
}

impl<'a, F> LevelsMut<'a, F> {
    /// Returns a mutable reference to level at the `index`, or `None` if the `index` is out of
    /// bounds.
    ///
    /// See also [get_unchecked_mut] for an unsafe variant or this method that does not do any
    /// bounds checks.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::MipmapLevels;
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    ///
    /// let mut texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     depth: 4,
    ///     levels: MipmapLevels::Partial(3)
    /// }).unwrap();
    ///
    /// let mut levels = texture.levels_mut();
    ///
    /// assert_eq!(levels.get_mut(1).map(|l| (l.width(), l.height())), Some((128, 128)));
    /// assert_eq!(levels.get_mut(4).map(|l| (l.width(), l.height())), None);
    /// # }
    /// ```
    pub fn get_mut<'b, I>(&'b mut self, index: I) -> Option<I::Output>
    where
        I: LevelsMutIndex<'b, F>,
    {
        index.get_mut(self)
    }

    /// Returns a mutable reference to level at the `index`, without doing any bounds checks.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::MipmapLevels;
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    ///
    /// let mut texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     depth: 4,
    ///     levels: MipmapLevels::Partial(3)
    /// }).unwrap();
    ///
    /// let mut levels = texture.levels_mut();
    ///
    /// let level = unsafe { levels.get_unchecked_mut(1) };
    ///
    /// assert_eq!((level.width(), level.height()), (128, 128));
    /// # }
    /// ```
    ///
    /// # Unsafe
    ///
    /// The `index` must be in bounds. See also [get_mut] for a safe variant of this method that
    /// does bounds checks.
    pub unsafe fn get_unchecked_mut<'b, I>(&'b mut self, index: I) -> I::Output
    where
        I: LevelsMutIndex<'b, F>,
    {
        index.get_unchecked_mut(self)
    }

    /// Returns an iterator over mutable references to the levels.
    ///
    /// # Example
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext {
    /// use web_glitz::image::MipmapLevels;
    /// use web_glitz::image::format::RGB8;
    /// use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    ///
    /// let mut texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    ///     format: RGB8,
    ///     width: 256,
    ///     height: 256,
    ///     depth: 3,
    ///     levels: MipmapLevels::Partial(3)
    /// }).unwrap();
    ///
    /// let mut levels = texture.levels_mut();
    /// let mut iter = levels.iter_mut();
    ///
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), Some((128, 128)));
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), Some((64, 64)));
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), Some((32, 32)));
    /// assert_eq!(iter.next().map(|l| (l.width(), l.height())), None);
    /// # }
    /// ```
    pub fn iter_mut(&mut self) -> LevelsMutIter<F> {
        LevelsMutIter {
            handle: self.handle,
            current_level: self.offset,
            end_level: self.offset + self.len,
        }
    }
}

impl<'a, F> Deref for LevelsMut<'a, F> {
    type Target = Levels<'a, F>;

    fn deref(&self) -> &Self::Target {
        &self.inner
    }
}

impl<'a, F> IntoIterator for LevelsMut<'a, F>
where
    F: TextureFormat,
{
    type Item = LevelMut<'a, F>;

    type IntoIter = LevelsMutIter<'a, F>;

    fn into_iter(self) -> Self::IntoIter {
        LevelsMutIter {
            handle: self.handle,
            current_level: self.offset,
            end_level: self.offset + self.len,
        }
    }
}

/// A helper trait for indexing a [LevelsMut].
///
/// See [LevelsMut::get_mut] and [LevelsMut::get_unchecked_mut].
pub trait LevelsMutIndex<'a, F> {
    /// The output type returned by the indexing operations.
    type Output;

    /// Returns the output for this operation if in bounds, or `None` otherwise.
    fn get_mut(self, levels: &'a mut LevelsMut<F>) -> Option<Self::Output>;

    /// Returns the output for this operation, without performing any bounds checking.
    unsafe fn get_unchecked_mut(self, levels: &'a mut LevelsMut<F>) -> Self::Output;
}

impl<'a, F> LevelsMutIndex<'a, F> for usize
where
    F: 'a,
{
    type Output = LevelMut<'a, F>;

    fn get_mut(self, levels: &'a mut LevelsMut<F>) -> Option<Self::Output> {
        if self < levels.len {
            Some(LevelMut {
                inner: Level {
                    handle: levels.handle,
                    level: levels.offset + self,
                },
            })
        } else {
            None
        }
    }

    unsafe fn get_unchecked_mut(self, levels: &'a mut LevelsMut<F>) -> Self::Output {
        LevelMut {
            inner: Level {
                handle: levels.handle,
                level: levels.offset + self,
            },
        }
    }
}

impl<'a, F> LevelsMutIndex<'a, F> for RangeFull
where
    F: 'a,
{
    type Output = LevelsMut<'a, F>;

    fn get_mut(self, levels: &'a mut LevelsMut<F>) -> Option<Self::Output> {
        Some(LevelsMut {
            inner: Levels {
                handle: levels.handle,
                offset: levels.offset,
                len: levels.len,
            },
        })
    }

    unsafe fn get_unchecked_mut(self, levels: &'a mut LevelsMut<F>) -> Self::Output {
        LevelsMut {
            inner: Levels {
                handle: levels.handle,
                offset: levels.offset,
                len: levels.len,
            },
        }
    }
}

impl<'a, F> LevelsMutIndex<'a, F> for Range<usize>
where
    F: 'a,
{
    type Output = LevelsMut<'a, F>;

    fn get_mut(self, levels: &'a mut LevelsMut<F>) -> Option<Self::Output> {
        let Range { start, end } = self;

        if start > end || end > levels.len {
            None
        } else {
            Some(LevelsMut {
                inner: Levels {
                    handle: levels.handle,
                    offset: levels.offset + start,
                    len: end - start,
                },
            })
        }
    }

    unsafe fn get_unchecked_mut(self, levels: &'a mut LevelsMut<F>) -> Self::Output {
        let Range { start, end } = self;

        LevelsMut {
            inner: Levels {
                handle: levels.handle,
                offset: levels.offset + start,
                len: end - start,
            },
        }
    }
}

impl<'a, F> LevelsMutIndex<'a, F> for RangeInclusive<usize>
where
    F: 'a,
{
    type Output = LevelsMut<'a, F>;

    fn get_mut(self, levels: &'a mut LevelsMut<F>) -> Option<Self::Output> {
        if *self.end() == usize::max_value() {
            None
        } else {
            <Range<usize> as LevelsMutIndex<'a, F>>::get_mut(*self.start()..self.end() + 1, levels)
        }
    }

    unsafe fn get_unchecked_mut(self, levels: &'a mut LevelsMut<F>) -> Self::Output {
        <Range<usize> as LevelsMutIndex<'a, F>>::get_unchecked_mut(
            *self.start()..self.end() + 1,
            levels,
        )
    }
}

impl<'a, F> LevelsMutIndex<'a, F> for RangeFrom<usize>
where
    F: 'a,
{
    type Output = LevelsMut<'a, F>;

    fn get_mut(self, levels: &'a mut LevelsMut<F>) -> Option<Self::Output> {
        <Range<usize> as LevelsMutIndex<'a, F>>::get_mut(self.start..levels.len, levels)
    }

    unsafe fn get_unchecked_mut(self, levels: &'a mut LevelsMut<F>) -> Self::Output {
        <Range<usize> as LevelsMutIndex<'a, F>>::get_unchecked_mut(self.start..levels.len, levels)
    }
}

impl<'a, F> LevelsMutIndex<'a, F> for RangeTo<usize>
where
    F: 'a,
{
    type Output = LevelsMut<'a, F>;

    fn get_mut(self, levels: &'a mut LevelsMut<F>) -> Option<Self::Output> {
        <Range<usize> as LevelsMutIndex<'a, F>>::get_mut(0..self.end, levels)
    }

    unsafe fn get_unchecked_mut(self, levels: &'a mut LevelsMut<F>) -> Self::Output {
        <Range<usize> as LevelsMutIndex<'a, F>>::get_unchecked_mut(0..self.end, levels)
    }
}

impl<'a, F> LevelsMutIndex<'a, F> for RangeToInclusive<usize>
where
    F: 'a,
{
    type Output = LevelsMut<'a, F>;

    fn get_mut(self, levels: &'a mut LevelsMut<F>) -> Option<Self::Output> {
        <RangeInclusive<usize> as LevelsMutIndex<'a, F>>::get_mut(0..=self.end, levels)
    }

    unsafe fn get_unchecked_mut(self, levels: &'a mut LevelsMut<F>) -> Self::Output {
        <RangeInclusive<usize> as LevelsMutIndex<'a, F>>::get_unchecked_mut(0..=self.end, levels)
    }
}

/// An iterator over [LevelsMut].
///
/// See [LevelsMut::iter_mut] for details.
pub struct LevelsMutIter<'a, F> {
    handle: &'a Texture2DArray<F>,
    current_level: usize,
    end_level: usize,
}

impl<'a, F> Iterator for LevelsMutIter<'a, F>
where
    F: TextureFormat,
{
    type Item = LevelMut<'a, F>;

    fn next(&mut self) -> Option<Self::Item> {
        let level = self.current_level;

        if level < self.end_level {
            self.current_level += 1;

            Some(LevelMut {
                inner: Level {
                    handle: self.handle,
                    level,
                },
            })
        } else {
            None
        }
    }
}

/// A mutable reference to a [Texture2DArray] mipmap level.
///
/// [Deref]s to [Level].
#[derive(PartialEq, Hash)]
pub struct LevelMut<'a, F> {
    inner: Level<'a, F>,
}

impl<'a, F> LevelMut<'a, F> {
    /// Returns mutable a reference to the layers of this [Level].
    ///
    /// # Examples
    ///
    /// ```rust
    /// # use web_glitz::runtime::RenderingContext;
    /// # use web_glitz::image::MipmapLevels;
    /// # use web_glitz::image::format::RGB8;
    /// # use web_glitz::image::texture_2d_array::Texture2DArrayDescriptor;
    /// # fn wrapper<Rc>(context: &Rc) where Rc: RenderingContext + Clone + 'static {
    /// # let mut texture = context.try_create_texture_2d_array(&Texture2DArrayDescriptor {
    /// #     format: RGB8,
    /// #     width: 256,
    /// #     height: 256,
    /// #     depth: 16,
    /// #     levels: MipmapLevels::Complete
    /// # }).unwrap();
    /// // Returns a mutable reference to layer 2 of the texture's base level if the texture has a 2
    /// // or more layers, or None otherwise:
    /// let layer_2 = texture.base_level_mut().layers_mut().get_mut(2);
    ///
    /// // We can also iterate over mutable references to all layers that exist for the texture's
    /// // base level:
    /// for layer in texture.base_level_mut().layers_mut().iter_mut() {
    ///     // ...
    /// }
    /// # }
    /// ```
    pub fn layers_mut(&mut self) -> LevelLayersMut<'a, F> {
        LevelLayersMut {
            inner: LevelLayers {
                handle: self.handle,
                level: self.level,
                offset: 0,
                len: self.handle.data.depth as usize,
            },
        }
    }
}

impl<'a, F> Deref for LevelMut<'a, F> {
    type Target = Level<'a, F>;

    fn deref(&self) -> &Self::Target {
        &self.inner
    }
}

/// A mutable reference to a the layers of a mipmap level of a [Texture2DArray].
///
/// [Deref]s to [LevelLayers].
pub struct LevelLayersMut<'a, F> {
    inner: LevelLayers<'a, F>,
}

impl<'a, F> LevelLayersMut<'a, F> {
    pub fn get_mut<'b, I>(&'b mut self, index: I) -> Option<I::Output>
    where
        I: LevelLayersMutIndex<'b, F>,
    {
        index.get_mut(self)
    }

    pub unsafe fn get_unchecked_mut<'b, I>(&'b mut self, index: I) -> I::Output
    where
        I: LevelLayersMutIndex<'b, F>,
    {
        index.get_unchecked_mut(self)
    }

    pub fn iter_mut(&mut self) -> LevelLayersMutIter<F> {
        LevelLayersMutIter {
            handle: self.handle,
            level: self.level,
            current_layer: self.offset,
            end_layer: self.offset + self.len,
        }
    }
}

impl<'a, F> IntoIterator for LevelLayersMut<'a, F>
where
    F: TextureFormat,
{
    type Item = LevelLayerMut<'a, F>;

    type IntoIter = LevelLayersMutIter<'a, F>;

    fn into_iter(self) -> Self::IntoIter {
        LevelLayersMutIter {
            handle: self.handle,
            level: self.level,
            current_layer: self.offset,
            end_layer: self.offset + self.len,
        }
    }
}

impl<'a, F> Deref for LevelLayersMut<'a, F> {
    type Target = LevelLayers<'a, F>;

    fn deref(&self) -> &Self::Target {
        &self.inner
    }
}

/// Iterator of a [LevelLayersMut].
///
/// See [LevelLayersMut::iter_mut].
pub struct LevelLayersMutIter<'a, F> {
    handle: &'a Texture2DArray<F>,
    level: usize,
    current_layer: usize,
    end_layer: usize,
}

impl<'a, F> Iterator for LevelLayersMutIter<'a, F>
where
    F: TextureFormat,
{
    type Item = LevelLayerMut<'a, F>;

    fn next(&mut self) -> Option<Self::Item> {
        let layer = self.current_layer;

        if layer < self.end_layer {
            self.current_layer += 1;

            Some(LevelLayerMut {
                inner: LevelLayer {
                    handle: self.handle,
                    level: self.level,
                    layer,
                },
            })
        } else {
            None
        }
    }
}

/// A helper trait for indexing a [LevelLayersMut].
///
/// See [LevelsMut::get_mut] and [LevelsMut::get_unchecked_mut].
pub trait LevelLayersMutIndex<'a, F> {
    /// The output type returned by the indexing operations.
    type Output;

    /// Returns the output for this operation if in bounds, or `None` otherwise.
    fn get_mut(self, layers: &'a mut LevelLayersMut<F>) -> Option<Self::Output>;

    /// Returns the output for this operation, without performing any bounds checking.
    unsafe fn get_unchecked_mut(self, layers: &'a mut LevelLayersMut<F>) -> Self::Output;
}

impl<'a, F> LevelLayersMutIndex<'a, F> for usize
where
    F: 'a,
{
    type Output = LevelLayerMut<'a, F>;

    fn get_mut(self, layers: &'a mut LevelLayersMut<F>) -> Option<Self::Output> {
        if self < layers.len {
            Some(LevelLayerMut {
                inner: LevelLayer {
                    handle: layers.handle,
                    level: layers.level,
                    layer: layers.offset + self,
                },
            })
        } else {
            None
        }
    }

    unsafe fn get_unchecked_mut(self, layers: &'a mut LevelLayersMut<F>) -> Self::Output {
        LevelLayerMut {
            inner: LevelLayer {
                handle: layers.handle,
                level: layers.level,
                layer: layers.offset + self,
            },
        }
    }
}

impl<'a, F> LevelLayersMutIndex<'a, F> for RangeFull
where
    F: 'a,
{
    type Output = LevelLayersMut<'a, F>;

    fn get_mut(self, layers: &'a mut LevelLayersMut<F>) -> Option<Self::Output> {
        Some(LevelLayersMut {
            inner: LevelLayers {
                handle: layers.handle,
                level: layers.level,
                offset: layers.offset,
                len: layers.len,
            },
        })
    }

    unsafe fn get_unchecked_mut(self, layers: &'a mut LevelLayersMut<F>) -> Self::Output {
        LevelLayersMut {
            inner: LevelLayers {
                handle: layers.handle,
                level: layers.level,
                offset: layers.offset,
                len: layers.len,
            },
        }
    }
}

impl<'a, F> LevelLayersMutIndex<'a, F> for Range<usize>
where
    F: 'a,
{
    type Output = LevelLayersMut<'a, F>;

    fn get_mut(self, layers: &'a mut LevelLayersMut<F>) -> Option<Self::Output> {
        let Range { start, end } = self;

        if start > end || end > layers.len {
            None
        } else {
            Some(LevelLayersMut {
                inner: LevelLayers {
                    handle: layers.handle,
                    level: layers.level,
                    offset: layers.offset + start,
                    len: end - start,
                },
            })
        }
    }

    unsafe fn get_unchecked_mut(self, layers: &'a mut LevelLayersMut<F>) -> Self::Output {
        let Range { start, end } = self;

        LevelLayersMut {
            inner: LevelLayers {
                handle: layers.handle,
                level: layers.level,
                offset: layers.offset + start,
                len: end - start,
            },
        }
    }
}

impl<'a, F> LevelLayersMutIndex<'a, F> for RangeInclusive<usize>
where
    F: 'a,
{
    type Output = LevelLayersMut<'a, F>;

    fn get_mut(self, layers: &'a mut LevelLayersMut<F>) -> Option<Self::Output> {
        if *self.end() == usize::max_value() {
            None
        } else {
            <Range<usize> as LevelLayersMutIndex<'a, F>>::get_mut(
                *self.start()..self.end() + 1,
                layers,
            )
        }
    }

    unsafe fn get_unchecked_mut(self, layers: &'a mut LevelLayersMut<F>) -> Self::Output {
        <Range<usize> as LevelLayersMutIndex<'a, F>>::get_unchecked_mut(
            *self.start()..self.end() + 1,
            layers,
        )
    }
}

impl<'a, F> LevelLayersMutIndex<'a, F> for RangeFrom<usize>
where
    F: 'a,
{
    type Output = LevelLayersMut<'a, F>;

    fn get_mut(self, layers: &'a mut LevelLayersMut<F>) -> Option<Self::Output> {
        <Range<usize> as LevelLayersMutIndex<'a, F>>::get_mut(self.start..layers.len, layers)
    }

    unsafe fn get_unchecked_mut(self, layers: &'a mut LevelLayersMut<F>) -> Self::Output {
        <Range<usize> as LevelLayersMutIndex<'a, F>>::get_unchecked_mut(
            self.start..layers.len,
            layers,
        )
    }
}

impl<'a, F> LevelLayersMutIndex<'a, F> for RangeTo<usize>
where
    F: 'a,
{
    type Output = LevelLayersMut<'a, F>;

    fn get_mut(self, layers: &'a mut LevelLayersMut<F>) -> Option<Self::Output> {
        <Range<usize> as LevelLayersMutIndex<'a, F>>::get_mut(0..self.end, layers)
    }

    unsafe fn get_unchecked_mut(self, layers: &'a mut LevelLayersMut<F>) -> Self::Output {
        <Range<usize> as LevelLayersMutIndex<'a, F>>::get_unchecked_mut(0..self.end, layers)
    }
}

impl<'a, F> LevelLayersMutIndex<'a, F> for RangeToInclusive<usize>
where
    F: 'a,
{
    type Output = LevelLayersMut<'a, F>;

    fn get_mut(self, layers: &'a mut LevelLayersMut<F>) -> Option<Self::Output> {
        <RangeInclusive<usize> as LevelLayersMutIndex<'a, F>>::get_mut(0..=self.end, layers)
    }

    unsafe fn get_unchecked_mut(self, layers: &'a mut LevelLayersMut<F>) -> Self::Output {
        <RangeInclusive<usize> as LevelLayersMutIndex<'a, F>>::get_unchecked_mut(
            0..=self.end,
            layers,
        )
    }
}

/// A mutable reference to a layer of a [Texture2DArray] mipmap level.
///
/// [Deref]s to [LevelLayer].
#[derive(PartialEq, Hash)]
pub struct LevelLayerMut<'a, F> {
    inner: LevelLayer<'a, F>,
}

impl<'a, F> Deref for LevelLayerMut<'a, F> {
    type Target = LevelLayer<'a, F>;

    fn deref(&self) -> &Self::Target {
        &self.inner
    }
}

struct AllocateCommand<F> {
    data: Arc<Texture2DArrayData>,
    _marker: marker::PhantomData<[F]>,
}

unsafe impl<F> GpuTask<Connection> for AllocateCommand<F>
where
    F: TextureFormat,
{
    type Output = ();

    fn context_id(&self) -> ContextId {
        ContextId::Id(self.data.context_id)
    }

    fn progress(&mut self, connection: &mut Connection) -> Progress<Self::Output> {
        let (gl, state) = unsafe { connection.unpack_mut() };
        let data = &self.data;

        let texture_object = gl.create_texture().unwrap();

        state.set_active_texture_lru().apply(gl).unwrap();
        state
            .bind_texture_2d_array(Some(&texture_object))
            .apply(gl)
            .unwrap();

        let levels = data.levels as i32;

        gl.tex_storage_3d(
            Gl::TEXTURE_2D_ARRAY,
            levels,
            F::ID,
            data.width as i32,
            data.height as i32,
            data.depth as i32,
        );

        gl.tex_parameteri(Gl::TEXTURE_2D_ARRAY, Gl::TEXTURE_MAX_LEVEL, levels);

        unsafe {
            *data.id.get() = Some(JsId::from_value(texture_object.into()));
        }

        Progress::Finished(())
    }
}

/// Uploads data to a [Level] or [LevelSubImage].
///
/// See [Level::upload_command] and [LevelSubImage::upload_command] for details.
pub struct LevelUploadCommand<D, T, F> {
    data: LayeredImageSource<D, T>,
    texture_data: Arc<Texture2DArrayData>,
    level: usize,
    region: Region3D,
    _marker: marker::PhantomData<[F]>,
}

unsafe impl<D, T, F> GpuTask<Connection> for LevelUploadCommand<D, T, F>
where
    D: Borrow<[T]>,
    T: PixelUnpack<F>,
    F: TextureFormat,
{
    type Output = ();

    fn context_id(&self) -> ContextId {
        ContextId::Id(self.texture_data.context_id)
    }

    fn progress(&mut self, connection: &mut Connection) -> Progress<Self::Output> {
        let mut width = region_3d_overlap_width(self.texture_data.width, self.level, &self.region);
        let mut height =
            region_3d_overlap_height(self.texture_data.height, self.level, &self.region);
        let depth = region_3d_overlap_depth(self.texture_data.height, &self.region);

        if width == 0 || height == 0 || depth == 0 {
            return Progress::Finished(());
        }

        let (gl, state) = unsafe { connection.unpack_mut() };

        match &self.data.internal {
            LayeredImageSourceInternal::PixelData {
                data,
                row_length,
                image_height,
                alignment,
                ..
            } => {
                state.set_active_texture_lru().apply(gl).unwrap();

                unsafe {
                    self.texture_data
                        .id()
                        .unwrap()
                        .with_value_unchecked(|texture_object| {
                            state
                                .bind_texture_2d_array(Some(texture_object))
                                .apply(gl)
                                .unwrap();
                        });
                }

                state
                    .set_pixel_unpack_alignment((*alignment).into())
                    .apply(gl)
                    .unwrap();

                if width < *row_length {
                    state
                        .set_pixel_unpack_row_length(*row_length as i32)
                        .apply(gl)
                        .unwrap();
                } else {
                    width = *row_length;

                    state.set_pixel_unpack_row_length(0).apply(gl).unwrap();
                }

                if height < *image_height {
                    state
                        .set_pixel_unpack_image_height(*image_height as i32)
                        .apply(gl)
                        .unwrap();
                } else {
                    height = *image_height;

                    state.set_pixel_unpack_image_height(0).apply(gl).unwrap();
                }

                let (offset_x, offset_y, offset_z) = match self.region {
                    Region3D::Fill => (0, 0, 0),
                    Region3D::Area(offset, ..) => offset,
                };

                let elements = *row_length as usize * *image_height as usize * depth as usize;
                let data_buffer = texture_data_as_js_buffer(data.borrow(), elements);

                gl.tex_sub_image_3d_with_opt_array_buffer_view(
                    Gl::TEXTURE_2D_ARRAY,
                    self.level as i32,
                    offset_x as i32,
                    offset_y as i32,
                    offset_z as i32,
                    width as i32,
                    height as i32,
                    depth as i32,
                    T::FORMAT_ID,
                    T::TYPE_ID,
                    Some(&data_buffer),
                )
                .unwrap();
            }
        }

        Progress::Finished(())
    }
}

/// Uploads data to a [LevelLayer] or [LevelLayerSubImage].
///
/// See [LevelLayer::upload_command] and [LevelLayerSubImage::upload_command] for details.
pub struct LevelLayerUploadCommand<D, T, F> {
    data: Image2DSource<D, T>,
    texture_data: Arc<Texture2DArrayData>,
    level: usize,
    layer: usize,
    region: Region2D,
    _marker: marker::PhantomData<[F]>,
}

unsafe impl<D, T, F> GpuTask<Connection> for LevelLayerUploadCommand<D, T, F>
where
    D: Borrow<[T]>,
    T: PixelUnpack<F>,
    F: TextureFormat,
{
    type Output = ();

    fn context_id(&self) -> ContextId {
        ContextId::Id(self.texture_data.context_id)
    }

    fn progress(&mut self, connection: &mut Connection) -> Progress<Self::Output> {
        let mut width = region_2d_overlap_width(self.texture_data.width, self.level, &self.region);
        let height = region_2d_overlap_height(self.texture_data.height, self.level, &self.region);

        if width == 0 || height == 0 {
            return Progress::Finished(());
        }

        let (gl, state) = unsafe { connection.unpack_mut() };

        match &self.data.internal {
            Image2DSourceInternal::PixelData {
                data,
                row_length,
                alignment,
                ..
            } => {
                state.set_active_texture_lru().apply(gl).unwrap();

                unsafe {
                    self.texture_data
                        .id()
                        .unwrap()
                        .with_value_unchecked(|texture_object| {
                            state
                                .bind_texture_2d_array(Some(texture_object))
                                .apply(gl)
                                .unwrap();
                        });
                }

                state
                    .set_pixel_unpack_alignment((*alignment).into())
                    .apply(gl)
                    .unwrap();

                if width < *row_length {
                    state
                        .set_pixel_unpack_row_length(*row_length as i32)
                        .apply(gl)
                        .unwrap();
                } else {
                    width = *row_length;

                    state.set_pixel_unpack_row_length(0).apply(gl).unwrap();
                }

                let (offset_x, offset_y) = match self.region {
                    Region2D::Fill => (0, 0),
                    Region2D::Area(offset, ..) => offset,
                };

                let elements = *row_length as usize * height as usize;
                let data_buffer = texture_data_as_js_buffer(data.borrow(), elements);

                gl.tex_sub_image_3d_with_opt_array_buffer_view(
                    Gl::TEXTURE_2D_ARRAY,
                    self.level as i32,
                    offset_x as i32,
                    offset_y as i32,
                    self.layer as i32,
                    width as i32,
                    height as i32,
                    1,
                    T::FORMAT_ID,
                    T::TYPE_ID,
                    Some(&data_buffer),
                )
                .unwrap();
            }
        }

        Progress::Finished(())
    }
}

/// Returned from [Texture2DArray::generate_mipmap_command], generates the image data for a
/// [Texture2DArray]'s mipmap chain.
///
/// See [Texture2DArray::generate_mipmap_command] for details.
pub struct GenerateMipmapCommand {
    texture_data: Arc<Texture2DArrayData>,
}

unsafe impl GpuTask<Connection> for GenerateMipmapCommand {
    type Output = ();

    fn context_id(&self) -> ContextId {
        ContextId::Id(self.texture_data.context_id)
    }

    fn progress(&mut self, connection: &mut Connection) -> Progress<Self::Output> {
        let (gl, state) = unsafe { connection.unpack_mut() };

        unsafe {
            self.texture_data
                .id()
                .unwrap()
                .with_value_unchecked(|texture_object| {
                    state
                        .bind_texture_2d_array(Some(texture_object))
                        .apply(gl)
                        .unwrap();
                });
        }

        gl.generate_mipmap(Gl::TEXTURE_2D_ARRAY);

        Progress::Finished(())
    }
}
