use chrono::{DateTime, Datelike, Local, NaiveDateTime, Utc, Date, TimeZone};
use fastly::geo::Geo;
use maplit::hashmap;

pub enum Season {
    Summer,
    Autumn,
    Winter,
    Spring,
}

// Transform the OpenWeather icons to the template ones.
pub fn get_feather_weather_icon(icon: &str) -> String {
    let map = hashmap! {
        "01d" => "sun",
        "02d" => "cloud",
        "03d" => "cloud",
        "04d" => "cloud",
        "09d" => "cloud-drizzle",
        "10d" => "cloud-rain",
        "11d" => "cloud-lightning",
        "13d" => "cloud-snow",
        "50d" => "align-center",
        "01n" => "sun",
        "02n" => "cloud",
        "03n" => "cloud",
        "04n" => "cloud",
        "09n" => "cloud-drizzle",
        "10n" => "cloud-rain",
        "11n" => "cloud-lightning",
        "13n" => "cloud-snow",
        "50n" => "align-center",
    };

    match map.get(icon.replace("\"", "").as_str()) {
        Some(i) => String::from(i.trim()),
        None => String::from("sun"),
    }
}

// Don't know why Chrono doesn't have this option.
pub fn weekday_full(weekday: String) -> String {
    match weekday.as_str() {
        "Mon" => String::from("Monday"),
        "Tue" => String::from("Tuesday"),
        "Wed" => String::from("Wednesday"),
        "Thu" => String::from("Thursday"),
        "Fri" => String::from("Friday"),
        "Sat" => String::from("Saturday"),
        "Sun" => String::from("Sunday"),
        _ => String::from("Error day"),
    }
}

pub fn datetime_to_day(datetime: String) -> String {
    let naive = NaiveDateTime::from_timestamp(datetime.parse::<i64>().unwrap(), 0);
    let datetime: DateTime<Utc> = DateTime::from_utc(naive, Utc);
    let local = datetime.with_timezone(&Local);
    local.weekday().to_string()
}

// Get your current season depending in your location.
// Season start dates are fixed despite varying in real life by 1 day depending on the year.
pub fn get_season(location: Geo, local: Date<Local>) -> Season {
    let spring_start = Local.ymd(local.year(), 3, 20);
    let summer_start = Local.ymd(local.year(), 6, 21);
    let autumn_start = Local.ymd(local.year(), 9, 22);
    let winter_start = Local.ymd(local.year(), 12, 21);

    // Let's make winter the default season, why not.
    let mut output = Season::Winter;

    if (local >= spring_start) && (local < summer_start) {
        if location.latitude().is_sign_positive() {
            output = Season::Spring
        } else {
            output = Season::Autumn
        }
    } else if (local >= summer_start) && (local < autumn_start) {
        if location.latitude().is_sign_positive() {
            output = Season::Summer
        } else {
            output = Season::Winter
        }
    } else if (local >= autumn_start) && (local < winter_start) {
        if location.latitude().is_sign_positive() {
            output = Season::Autumn
        } else {
            output = Season::Spring
        }
    } else if (local >= winter_start) && (local < spring_start) {
        if location.latitude().is_sign_positive() {
            output = Season::Winter
        } else {
            output = Season::Summer
        }
    }

    output
}
