#![allow(unused_macros, unused_attributes, dead_code)]

use crate::util::conf::config;
use serde::{Deserialize, Serialize};
use std::path::Path;

#[derive(Debug, Deserialize, Serialize)]
pub struct Log {
    pub log_level: String,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct Stream {
    pub max_cache: usize,
}

// #[derive(Debug, Deserialize, Serialize)]
// pub struct DB {
//     pub pg_url: String,
//     pub max_conn: u32,
// }

#[derive(Debug, Deserialize, Serialize)]
pub struct Server {
    pub app_name: String,
    pub addr: String,
}

#[derive(Debug, Deserialize, Serialize)]
pub struct Config {
    pub log: Log,
    pub server: Server,
    pub stream: Stream,
}

/// load config file to struct
/// supported formats
/// [x] yaml
/// [x] json
/// [x] toml
/// [x] http
///
/// example
///
/// ```rust
///#[derive(Debug, Deserialize)]
/// pub struct Config {
///     pub log: i32,
///     pub server: String,
///     pub stream: String,
/// }
/// let conf:Config = load_config("./src/util/conf/config.toml").unwrap();
/// println!("{:?}",conf)
/// ```
#[allow(unused_unsafe)]
pub fn load_config<T:for<'a> Deserialize<'a>>(path:impl AsRef<Path>)->Result<T, String>{
    let path:&'static str = unsafe {
        let s = path.as_ref().to_string_lossy().to_string();
        Box::leak(Box::new(s))
    };
    let acres: Result<T, config::Error> =
        config::load_config_from_path(config::File::new(path));
    match acres {
        Err(e) => Err(e.error()),
        Ok(o) => Ok(o),
    }
}

impl Config {
    pub fn new(path: &'static str) -> Result<Self, String> {
        let acres: Result<Config, config::Error> =
            config::load_config_from_path(config::File::new(path));
        match acres {
            Err(e) => Err(e.error()),
            Ok(o) => Ok(o),
        }
    }
}
