use crate::util::cmd::args::Flags;
use std::sync::atomic::{AtomicBool, Ordering};
use std::sync::Arc;
use wd_event::{Context, EventHandle, EventManage};
use std::str::FromStr;

pub const APPLICATION_EXIT: &'static str = "WD_RUN_APPLICATION_EXIT";
pub const APPLICATION_INIT: &'static str = "WD_RUN_APPLICATION_INIT";

pub const GLOBAL_FLAG: &'static str = "WD_RUN_GLOBAL_FLAG";

pub const SERVER_DESCRIPTION: &'static str = "WD_RUN_SERVER_DESCRIPTION";

#[derive(Clone, Debug)]
pub struct CmdInfo {
    pub cmd: String,
    pub help: String,
    pub flags: Vec<(String, String, String)>,
}
impl CmdInfo {
    pub fn new<S: ToString>(cmd: S, help: S) -> Self {
        CmdInfo {
            cmd: cmd.to_string(),
            help: help.to_string(),
            flags: vec![],
        }
    }
    pub fn add<K: ToString,Def: ToString,Des: ToString>(mut self, key: K, default: Def, desc: Des) -> Self {
        self.flags
            .push((key.to_string(), default.to_string(), desc.to_string()));
        self
    }
}
pub struct ArgsManager {
    description: String,
    event_manager: EventManage,
    cmds: Vec<CmdInfo>,
    flags: Vec<(String, String, String)>,
}
impl ArgsManager {
    pub fn new() -> ArgsManager {
        ArgsManager {
            description: "wd_run".to_string(),
            event_manager: EventManage::new(),
            cmds: vec![],
            flags: vec![],
        }
    }
    pub fn add_global_flag<K: ToString,Def: ToString,Des: ToString>(mut self, key: K, default: Def, desc: Des) -> Self {
        self.flags.push((key.to_string(),default.to_string(),desc.to_string()));self
    }
    pub fn register_cmd<F>(mut self, cmd: CmdInfo, f: F) -> Self
    where
        F: EventHandle + Send + Sync + 'static,
    {
        self.event_manager.register_event(&cmd.cmd, f);
        self.cmds.push(cmd);
        self
    }
    pub fn register_init<F>(mut self, f: F) -> Self
    where
        F: EventHandle + Send + Sync + 'static,
    {
        self.event_manager.register_event(APPLICATION_INIT, f);
        self
    }
    pub fn register_exit<F>(mut self, f: F) -> Self
    where
        F: EventHandle + Send + Sync + 'static,
    {
        self.event_manager.register_event(APPLICATION_EXIT, f);
        self
    }

    pub async fn run(self) -> Context {
        //注册一个help 事件
        let event = Arc::new(self.event_manager);
        let cmds = self.cmds;
        let global_flags = self.flags;
        let args = Flags::parse_args();
        //打印帮助
        if args.cmds.is_empty() {
            Self::show_help(self.description, cmds, global_flags);
            return Context::new();
        }
        //组装context
        let ctx = Self::generate_context(global_flags, args.clone(), self.description).await;
        Self::run_raw(event, args, cmds, ctx).await
    }
    pub async fn async_run(self) -> Context {
        //注册一个help 事件
        let event = Arc::new(self.event_manager);
        let cmds = self.cmds;
        let global_flags = self.flags;
        let args = Flags::parse_args();
        //打印帮助
        if args.cmds.is_empty() {
            Self::show_help(self.description, cmds, global_flags);
            return Context::new();
        }
        //组装context
        let ctx = Self::generate_context(global_flags, args.clone(), self.description).await;
        let async_ctx = ctx.clone();
        tokio::spawn(async move {
            Self::run_raw(event, args, cmds, async_ctx).await;
        });
        return ctx;
    }
    pub(crate) async fn generate_context(
        global_flags: Vec<(String, String, String)>,
        args: Flags,
        description: String,
    ) -> Context {
        let ctx = Context::new();
        ctx.set(SERVER_DESCRIPTION, description).await;
        for (cmd, def, _des) in global_flags.iter() {
            ctx.set(cmd.clone(), def.clone()).await;
        }
        for (key, value) in args.args.iter() {
            ctx.set(key.clone(), value.clone()).await;
        }
        ctx.set(GLOBAL_FLAG, args.clone()).await;
        return ctx;
    }
    pub(crate) async fn run_raw(
        event: Arc<EventManage>,
        args: Flags,
        cmds: Vec<CmdInfo>,
        mut ctx: Context,
    ) -> Context {
        let term = Arc::new(AtomicBool::new(false));
        ctx = event.dispatch(APPLICATION_INIT, ctx).await;
        let async_ctx = ctx.clone();
        let event_ayc = event.clone();
        let run_over = term.clone();
        tokio::spawn(async move {
            for cmd in args.cmds.into_iter() {
                let ctx = async_ctx.clone();
                for ci in cmds.iter() {
                    for (c, d, _s) in ci.flags.iter() {
                        ctx.set(c, d.clone()).await;
                    }
                }
                for (k, v) in cmd.args.into_iter() {
                    ctx.set(k, v).await;
                }
                //填充默认值
                event_ayc.dispatch(&cmd.cmd.clone(), ctx).await;
            }
            run_over.swap(true,Ordering::Relaxed);
        });
        //监听退出信号
        default_exit_lister(term).await;
        event.dispatch(APPLICATION_EXIT, ctx).await
    }
    pub(crate) fn show_help(
        description: String,
        cmds: Vec<CmdInfo>,
        global_flags: Vec<(String, String, String)>,
    ) {
        println!("server description:[{}]", description);
        println!("set global parameters as follows:");
        for (cmd, def, des) in global_flags {
            println!("    -[{}] default:{} help:{}", cmd, def, des);
        }
        for cmd in cmds.into_iter() {
            println!("{}    help:{}", cmd.cmd, cmd.help);
            for (cmd, def, des) in cmd.flags.into_iter() {
                println!("      -[{}] default:{} help:{}", cmd, def, des);
            }
        }
    }
}

pub async fn parse_args<K:ToString,A:FromStr>(ctx:&Context,k:K)->Option<A>{
    let value = ctx.copy::<_,String>(k).await?;
    if let Ok(o) = A::from_str(&value){
        return Some(o);
    };None
}

async fn default_exit_lister(term:Arc<AtomicBool>) {
    // let term = Arc::new(AtomicBool::new(false));
    signal_hook::flag::register(signal_hook::SIGTERM, Arc::clone(&term))
        .expect("系统信号SIGTERM(15) 监听失败");
    signal_hook::flag::register(signal_hook::SIGINT, Arc::clone(&term))
        .expect("系统信号SIGINT(2) 监听失败");
    while !term.load(Ordering::Relaxed) {
        tokio::time::sleep(std::time::Duration::from_secs(1)).await;
    }
}
