#![allow(unused_must_use, dead_code)]

use crate::util::cmd::cmd_event::{get_event, init_event};
use crate::util::cmd::flag::parse;
use std::collections::HashMap;
use std::sync::atomic::{AtomicBool, Ordering};
use std::sync::Arc;
use tokio::sync::oneshot::Receiver;
use wd_event::*;

pub const KE_DEFAULT_EXEC: &str = "default_exec_function"; //默认的执行方法

pub const KE_ARGS: &str = "args"; //全局参数加当前命令参数
pub const KE_FLAG_TREE: &str = "flag_tree"; //所有输入参数的语法树

//只供显示，不具备任何作用，默认值不会被赋值
#[derive(Clone, Debug)]
pub struct CmdInfo {
    pub cmd: String,
    pub help: String,
    pub default: String,
    pub flags: Vec<Self>,
}
impl CmdInfo {
    pub fn new(c: &str, d: &str, h: &str) -> Self {
        CmdInfo {
            cmd: c.to_string(),
            help: h.to_string(),
            default: d.to_string(),
            flags: vec![],
        }
    }
    pub fn set_flag(&mut self, cmd: &str, default: &str, help: &str) {
        self.flags.push(Self::new(cmd, default, help))
    }
}

pub struct ExecManager {
    flags: parse::FlagTree,
    cmds: HashMap<String, CmdInfo>,
    run_end: Option<Receiver<bool>>,
}

fn default_run(_: HashMap<String, String>, _: &parse::FlagTree) -> Result<(), String> {
    Ok(())
}

async fn default_exit_lister(event: Arc<wd_event::EventManage<String>>) {
    let term = Arc::new(AtomicBool::new(false));
    signal_hook::flag::register(signal_hook::SIGTERM, Arc::clone(&term))
        .expect("系统信号SIGTERM(15) 监听失败");
    signal_hook::flag::register(signal_hook::SIGINT, Arc::clone(&term))
        .expect("系统信号SIGINT(2) 监听失败");
    while !term.load(Ordering::Relaxed) {
        tokio::time::sleep(std::time::Duration::from_secs(1)).await;
    }
    // core.exec_sequence_async(&"exit".to_string(),Some(Context::new_null())).await;
    event
        .exec_immediately(&"exit".to_string(), Some(Context::new_null()))
        .await;
    std::process::exit(1)
}

impl ExecManager {
    pub async fn new() -> Self {
        init_event().await;
        let flags = parse::load().expect("输入参数解析错误");
        ExecManager {
            flags,
            cmds: HashMap::new(),
            run_end: None,
        }
    }
    //设置默认方法
    pub fn set_default<T: wd_event::AsyncEvent + 'static + Send>(
        mut self,
        d: T,
        h: Option<CmdInfo>,
    ) -> Self {
        get_event()
            .add_async(KE_DEFAULT_EXEC.to_string(), d)
            .unwrap();
        if let Some(h) = h {
            self.cmds.insert(KE_DEFAULT_EXEC.to_string(), h);
        }
        self
    }
    //注册执行方法
    pub fn registered<T: wd_event::AsyncEvent + 'static + Send>(&mut self, r: T, h: CmdInfo) {
        get_event().add_async(h.cmd.clone(), r).unwrap();
        self.cmds.insert(h.cmd.clone(), h);
    }
    pub async fn exit_sign_lister() {
        let term = Arc::new(AtomicBool::new(false));
        signal_hook::flag::register(signal_hook::SIGTERM, Arc::clone(&term))
            .expect("系统信号SIGTERM(15) 监听失败");
        signal_hook::flag::register(signal_hook::SIGINT, Arc::clone(&term))
            .expect("系统信号SIGINT(2) 监听失败");
        while !term.load(Ordering::Relaxed) {
            tokio::time::sleep(std::time::Duration::from_secs(1)).await;
        }
        get_event()
            .exec_sequence_async(
                &"exit".to_string(),
                Some(Context::new_add_msg(String::from("exit"))),
            )
            .await;
        std::process::exit(1);
    }
    pub async fn async_exit_sign_lister() {
        let term = Arc::new(AtomicBool::new(false));
        signal_hook::flag::register(signal_hook::SIGTERM, Arc::clone(&term))
            .expect("系统信号SIGTERM(15) 监听失败");
        signal_hook::flag::register(signal_hook::SIGINT, Arc::clone(&term))
            .expect("系统信号SIGINT(2) 监听失败");
        while !term.load(Ordering::Relaxed) {
            tokio::time::sleep(std::time::Duration::from_secs(1)).await;
        }
        std::process::exit(1);
    }
    pub async fn run(mut self) -> Result<(), String> {
        //注册help函数
        self.registr_help_to_event();
        //初始化全局参数
        let mut args = HashMap::new();
        if let Some(s) = self.cmds.get(&KE_DEFAULT_EXEC.to_string()) {
            for info in s.flags.iter() {
                args.insert(info.cmd.clone(), info.default.clone());
            }
        }
        for i in self.flags.global.iter() {
            args.insert(i.1.key.clone(), i.1.value.clone());
        }
        //执行默认
        let ctx = Context::new_null();
        //TODO 全局执行没有命令参数
        ctx.set_value(KE_ARGS.to_string(), args.clone());
        ctx.set_value(KE_FLAG_TREE.to_string(), self.flags.clone());
        get_event()
            .exec_sequence_async(&KE_DEFAULT_EXEC.to_string(), Option::from(ctx))
            .await
            .expect("default function executy error");
        //顺序阻塞执行所有任务
        for (cmd, flag) in self.flags.flags.iter() {
            let mut child_args = args.clone();
            if let Some(s) = self.cmds.get(cmd) {
                for info in s.flags.iter() {
                    child_args.insert(info.cmd.clone(), info.default.clone());
                }
            }
            if let Some(child_flag) = &flag.chile_flags {
                for item in child_flag {
                    child_args.insert(item.key.clone(), item.value.clone());
                }
            }
            //将命令参数添加到事件默认值
            let ctx = Context::new_add_msg(cmd.clone());
            ctx.set_value(KE_ARGS, child_args);
            ctx.set_value(KE_FLAG_TREE, self.flags.clone());
            get_event()
                .exec_immediately(&cmd.to_string(), Option::from(ctx))
                .await;
        }
        //异步执行退出函数
        ExecManager::exit_sign_lister().await;
        Ok(())
    }
    pub async fn async_run(mut self) -> Result<(), String> {
        //注册help函数
        self.registr_help_to_event();
        //初始化全局参数
        let mut args = HashMap::new();
        if let Some(s) = self.cmds.get(&KE_DEFAULT_EXEC.to_string()) {
            for info in s.flags.iter() {
                args.insert(info.cmd.clone(), info.default.clone());
            }
        }
        for i in self.flags.global.iter() {
            args.insert(i.1.key.clone(), i.1.value.clone());
        }
        //执行默认
        let ctx = Context::new_null();
        //TODO 全局执行没有命令参数
        ctx.set_value(KE_ARGS.to_string(), args.clone());
        ctx.set_value(KE_FLAG_TREE.to_string(), self.flags.clone());
        get_event()
            .exec_sequence_async(&KE_DEFAULT_EXEC.to_string(), Option::from(ctx))
            .await
            .expect("default function executy error");
        //顺序阻塞执行所有任务
        for (cmd, flag) in self.flags.flags.iter() {
            let mut child_args = args.clone();
            if let Some(s) = self.cmds.get(cmd) {
                for info in s.flags.iter() {
                    child_args.insert(info.cmd.clone(), info.default.clone());
                }
            }
            if let Some(child_flag) = &flag.chile_flags {
                for item in child_flag {
                    child_args.insert(item.key.clone(), item.value.clone());
                }
            }
            //将命令参数添加到事件默认值
            let ctx = Context::new_add_msg(cmd.clone());
            ctx.set_value(KE_ARGS, child_args);
            ctx.set_value(KE_FLAG_TREE, self.flags.clone());
            get_event()
                .exec_immediately(&cmd.to_string(), Option::from(ctx))
                .await;
        }
        //异步执行退出函数
        tokio::spawn(async{
            ExecManager::async_exit_sign_lister().await;
        });
        Ok(())
    }
    fn registr_help_to_event(&mut self) {
        let mut ch = CmdHelp {
            cmdr: self.cmds.clone(),
        };
        let ci = CmdInfo::new("help", "", "or -h :service related help information");
        ch.cmdr.insert("help".to_string(), ci);
        get_event().add_async("help".to_string(), ch.clone());
        if let None = self.cmds.get(&KE_DEFAULT_EXEC.to_string()) {
            get_event().add_async(KE_DEFAULT_EXEC.to_string().clone(), ch);
        }
    }
}
//------------------------help 对应的信息打印-------------------------------
#[derive(Clone)]
struct CmdHelp {
    cmdr: HashMap<String, CmdInfo>,
}
#[wd_event::event_trait]
impl AsyncEvent for CmdHelp {
    async fn handle(&self, ctx: Arc<Context>) {
        if let None = ctx.get_msg::<String>() {
            if let Some(flags) =
                ctx.get_value::<String, HashMap<String, String>>(&KE_ARGS.to_string())
            {
                if let None = flags.get(&"-h".to_string()) {
                    return;
                }
            } else {
                return;
            }
        }
        println!(" parameter specification:");
        for (_, info) in self.cmdr.iter() {
            if info.cmd != String::new() {
                println!("    {}: ({})", info.cmd, info.help);
            }
            if info.flags.len() != 0 {
                for a in info.flags.iter() {
                    println!("        {}  ({})  :{}", a.cmd, a.default, a.help);
                }
            }
        }
        std::process::exit(1);
    }
}
