use std::env;
use std::fs::read_to_string;
use std::path::Path;

use devices::ChipFamily;

// Ref: https://github.com/probe-rs/probe-rs/blob/master/probe-rs/build.rs
fn main() {
    // Only rerun build.rs if something inside devices/ has changed. (By default
    // cargo reruns build.rs if any file under the crate root has changed)
    // This improves build times and IDE responsivity when not editing targets.
    println!("cargo:rerun-if-changed=devices");

    let mut families: Vec<ChipFamily> = Vec::new();
    let files = vec![
        "devices/0x10-CH56x.yaml",
        "devices/0x11-CH55x.yaml",
        "devices/0x12-CH54x.yaml",
        "devices/0x13-CH57x.yaml",
        "devices/0x14-CH32F103.yaml",
        "devices/0x15-CH32V103.yaml",
        "devices/0x16-CH58x.yaml",
        "devices/0x17-CH32V30x.yaml",
        "devices/0x18-CH32F20x.yaml",
        "devices/0x19-CH32V20x.yaml",
        "devices/0x20-CH32F20x-Compact.yaml",
    ];
    for file in files {
        let string = read_to_string(&file).expect(
            "Chip family definition file could not be read. This is a bug. Please report it.",
        );

        let yaml: Result<ChipFamily, _> = serde_yaml::from_str(&string);

        match yaml {
            Ok(familiy) => families.push(familiy),
            Err(e) => panic!("Failed to parse target file: {:?} because:\n{}", file, e),
        }
    }

    let families_bin =
        bincode::serialize(&families).expect("Failed to serialize families as bincode");

    let out_dir = env::var("OUT_DIR").unwrap();
    let dest_path = Path::new(&out_dir).join("targets.bincode");
    std::fs::write(dest_path, &families_bin).unwrap();

    let _: Vec<ChipFamily> = bincode::deserialize(&families_bin).unwrap();
}
