mod clients;
pub mod constants;
pub mod pressers;
pub mod records;
mod test_server;

#[cfg(test)]
mod tests {
    use crate::pressers::HtmlPresser;
    use crate::test_server::TestServer;
    use serial_test::serial;
    use std::thread;

    macro_rules! aw {
        ($e:expr) => {
            tokio_test::block_on($e)
        };
    }

    #[test]
    #[serial]
    fn press_urls() {
        let press_urls = aw!(HtmlPresser::press_urls("https://example.com", 1, 1));
        let url = press_urls.unwrap();

        assert_eq!(url.len(), 1);
    }

    #[test]
    #[serial]
    fn press_curated_urls_nm() {
        let pressed_urls = aw!(HtmlPresser::press_curated_urls(
            "https://example.com",
            "no_match123",
            1,
            1
        ));
        let url = pressed_urls.unwrap();

        assert_eq!(url.len(), 0);
    }

    #[test]
    #[serial]
    fn press_curated_urls_m() {
        thread::spawn(|| {
            TestServer::test_anchors_server();
        });
        let pressed_urls = aw!(HtmlPresser::press_curated_urls(
            "http://127.0.0.1:7878",
            "not-example.com",
            1,
            1
        ));
        let url = pressed_urls.unwrap();
        println!("{:?}", url);
        assert_eq!(url.len(), 6);
    }

    #[test]
    #[serial]
    fn press_recs_blind() {
        let pressed_recs = aw!(HtmlPresser::press_records_blind("https://funnyjunk.com", 2));
        let rec = pressed_recs.unwrap();

        assert_eq!(rec.len(), 2);
    }

    #[test]
    #[serial]
    fn press_recs_vector() {
        let pressed_recs = aw!(HtmlPresser::press_records(vec!["https://example.com"]));
        let rec = pressed_recs.unwrap();

        assert_eq!(rec.len(), 1);
    }

    #[test]
    #[serial]
    fn press_recs_single() {
        let pressed_recs = aw!(HtmlPresser::press_record("https://example.com"));
        let rec = pressed_recs.unwrap();

        assert_eq!(rec.origin, "https://example.com/");
    }

    /*

    record tests

     */
    #[test]
    #[serial]
    fn test_server_works() {
        thread::spawn(|| {
            TestServer::test_anchors_server();
        });

        let pressed_rec = aw!(HtmlPresser::press_record("http://127.0.0.1:7878"));
        assert_eq!(pressed_rec.is_ok(), true);
    }

    #[test]
    #[serial]
    fn record_domain_anchors() {
        thread::spawn(move || {
            TestServer::test_anchors_server();
        });

        let pressed_rec = aw!(HtmlPresser::press_record("http://127.0.0.1:7878"));

        let mut rec = pressed_rec.unwrap();
        rec.origin = "https://example.com".to_string();
        let domain_anchors = rec.domain_anchors();

        assert_eq!(domain_anchors.unwrap().len(), 5);
    }

    #[test]
    #[serial]
    fn record_non_domain_anchors() {
        thread::spawn(|| {
            TestServer::test_anchors_server();
        });

        let pressed_rec_result = aw!(HtmlPresser::press_record("http://127.0.0.1:7878"));
        let mut rec = pressed_rec_result.unwrap();
        rec.origin = "https://example.com".to_string();
        let domain_anchors = rec.non_domain_anchors().unwrap();

        assert_eq!(domain_anchors.len(), 5);
    }

    #[test]
    #[serial]
    fn record_tag_text() {
        thread::spawn(|| {
            TestServer::test_anchors_server();
        });

        let pressed_rec_result = aw!(HtmlPresser::press_record("http://127.0.0.1:7878"));
        let rec = pressed_rec_result.unwrap();
        let anchors = rec.tag_text("a").unwrap();

        assert_eq!(anchors.len(), 10);
    }

    #[test]
    #[serial]
    fn record_tag_html() {
        thread::spawn(|| {
            TestServer::test_anchors_server();
        });

        let pressed_rec_result = aw!(HtmlPresser::press_record("http://127.0.0.1:7878"));
        let rec = pressed_rec_result.unwrap();
        let anchors = rec.tag_html("a");

        assert_eq!(anchors.unwrap().len(), 10);
    }

    #[test]
    #[serial]
    fn record_get_emails() {
        thread::spawn(|| {
            TestServer::test_anchors_server();
        });
        let pressed_rec_result = aw!(HtmlPresser::press_record("http://127.0.0.1:7878"));
        let rec = pressed_rec_result.unwrap();
        let emails = rec.get_emails().unwrap();

        assert_eq!(emails.len(), 2);
    }
    #[test]
    #[serial]
    fn doc_test_one() {
        thread::spawn(|| {
            TestServer::test_anchors_server();
        });
        let pressed_rec_result = aw!(HtmlPresser::press_record("http://127.0.0.1:7878"));
        let mut rec = pressed_rec_result.unwrap();
        let emails = rec.get_emails().unwrap();
        assert_eq!(emails.len(), 2);
        let anchors = rec.anchors().unwrap();
        assert_eq!(anchors.len(), 10);
        rec.origin = "https://example.com".to_string();
        let anchors = rec.domain_anchors().unwrap();
        assert_eq!(anchors.len(), 5);
        let anchors = rec.non_domain_anchors().unwrap();
        assert_eq!(anchors.len(), 5);
        let anchors = rec.anchors_curate("example").unwrap();
        assert_eq!(anchors.len(), 10);
        let d_anchors = rec.anchors_curate("not-example.com").unwrap();
        assert_eq!(d_anchors.len(), 5);
        let meta = rec.html_meta();
        assert!(meta.is_some());
    }
}
