use std::io::Read as _;
use std::io::Write as _;

const ANSII_SAVE: &[u8] = b"\x1B[s";
const ANSII_DIM: &[u8] = b"\x1B[2m";
const ANSII_LINE_CLEAR: &[u8] = b"\x1B[2K";
const ANSII_LINE_CLEAR_END: &[u8] = b"\x1B[K";
const ANSII_RESTORE: &[u8] = b"\x1B[u";
const ANSII_MOVE_UP: &[u8] = b"\x1B[A";
const ANSII_MOVE_COL_1: &[u8] = b"\x1B[G";

fn maybe_duration<'d, D: serde::Deserializer<'d>>(d: D)
	-> Result<Option<Option<std::time::Duration>>, D::Error>
{
	struct V;

	impl<'v> serde::de::Visitor<'v> for V {
		type Value = Option<std::time::Duration>;

		fn expecting(&self, formatter: &mut std::fmt::Formatter) -> std::fmt::Result {
			formatter.write_str("expecting duration or \"never\"")
		}

		fn visit_str<E: serde::de::Error>(self, s: &str) -> Result<Self::Value, E> {
			if s == "never" {
				Ok(None)
			} else {
				humanize_rs::duration::parse(s)
					.map(Some)
					.map_err(|_| serde::de::Error::invalid_value(serde::de::Unexpected::Str(s), &self))
			}
		}
	}

	d.deserialize_str(V).map(Some)
}

#[derive(Debug,serde_derive::Deserialize)]
#[serde(deny_unknown_fields)]
#[serde(rename_all="kebab-case")]
struct Config {
	/// Enable watchlog when there is no output for this long.
	#[serde(default)]
	#[serde(with="serde_humanize_rs")]
	delay: Option<std::time::Duration>,

	/// Insert a timestamp into the output after a delay of at least this long.
	///
	/// The literal string "never" can be used to disable this feature.
	///
	/// Note that the timestamp is never inserted after breaks which occur mid-line.
	#[serde(default)]
	#[serde(deserialize_with="maybe_duration")]
	permanent_delay: Option<Option<std::time::Duration>>,
}

fn main() -> Result<(), Box<dyn std::error::Error>> {
	assert_eq!(std::env::args().len(), 1, "No argument are accepted. Pipe into stdin.");

	let paths = standard_paths::StandardPaths::new()
		.locate_all(
			standard_paths::LocationType::AppConfigLocation,
			"config.ini",
			standard_paths::LocateOption::LocateFile)?;

	let mut cfg: Config = serde_ini::from_str("").unwrap();
	for path in paths.into_iter().flatten() {
		let Config{delay, permanent_delay} = serde_ini::from_str(&std::fs::read_to_string(path)?)?;
		cfg.delay = cfg.delay.or(delay);
		cfg.permanent_delay = cfg.permanent_delay.or(permanent_delay);
	}

	let Config{delay, permanent_delay} = cfg;
	let delay = delay.unwrap_or(std::time::Duration::from_secs(3));
	let permanent_delay = permanent_delay.unwrap_or(Some(std::time::Duration::from_secs(10)));

	let mut buf = Vec::new();
	let stdin = std::io::stdin();
	let mut stdin = stdin.lock();
	let stdout = std::io::stdout();
	let mut stdout = stdout.lock();

	let mut pollfd = [
		nix::poll::PollFd::new(
			std::os::unix::io::AsRawFd::as_raw_fd(&stdin),
			nix::poll::PollFlags::POLLIN),
	];

	loop {
		let existing_size = buf.len();
		buf.resize(4*1024, 0);
		let size = stdin.read(&mut buf[existing_size..])?;
		buf.truncate(existing_size + size);

		stdout.write_all(&buf)?;
		if size == 0 {
			return Ok(()) // EOF
		}
		stdout.flush()?;
		let new_line = buf.last().unwrap() == &b'\n';
		buf.clear();
		let last_read = std::time::Instant::now();

		let mut target_time = delay;

		nix::poll::poll(&mut pollfd, target_time.as_millis() as i32)?;
		if pollfd[0].revents() == Some(nix::poll::PollFlags::POLLIN) ||
			pollfd[0].revents() == Some(nix::poll::PollFlags::POLLHUP) {
			continue
		} else if pollfd[0].revents() != Some(nix::poll::PollFlags::empty()) {
			return Err(string_error::into_err(
				format!("Unexpected events in poll {:?}", pollfd[0].revents())))
		}

		buf.extend(ANSII_SAVE);
		buf.extend(ANSII_DIM);
		if !new_line {
			buf.extend("…\n".as_bytes())
		}
		write!(&mut buf, "Last output {:.0?} ago.", last_read.elapsed())?;
		stdout.write_all(&buf)?;
		stdout.flush()?;
		buf.clear();

		loop {
			target_time += std::time::Duration::from_secs(1);
			let elapsed = last_read.elapsed();
			let wait = target_time - elapsed.min(target_time);

			nix::poll::poll(&mut pollfd, wait.as_millis() as i32)?;
			if pollfd[0].revents() == Some(nix::poll::PollFlags::POLLIN) ||
				pollfd[0].revents() == Some(nix::poll::PollFlags::POLLHUP) {
				if permanent_delay.is_some()
					&& new_line
					&& last_read.elapsed() > permanent_delay.unwrap()
				{
					buf.extend(ANSII_MOVE_COL_1);
					buf.extend(ANSII_LINE_CLEAR_END);
					write!(&mut buf,
						"\t{}\n",
						chrono::offset::Local::now().format("%F %R:%S"))?;
				} else {
					buf.extend(ANSII_LINE_CLEAR);
				}
				buf.extend(ANSII_RESTORE);
				if !new_line {
					buf.extend(ANSII_MOVE_UP);
					buf.extend(ANSII_LINE_CLEAR_END);
				}
				break
			} else if pollfd[0].revents() != Some(nix::poll::PollFlags::empty()) {
				return Err(string_error::into_err(
					format!("Unexpected events in poll {:?}", pollfd[0].revents())))
			}

			buf.extend(ANSII_MOVE_COL_1);
			buf.extend(ANSII_LINE_CLEAR_END);
			write!(&mut buf,
				"Last output {:.0?} ago.",
				last_read.elapsed())?;
			stdout.write_all(&buf)?;
			stdout.flush()?;
			buf.clear();
		}
	}
}
