use ed25519_dalek::{Keypair, PublicKey, Signature, Verifier};
use wasmium_errors::{WasmiumError, WasmiumResult};

pub type Ed25519PublicKeyArray = [u8; 32];
pub type Ed25519SignatureArray = [u8; 64];
pub type Keeper = [u8; 32];
pub type Arbitrator = [u8; 32];
pub type Base58PublicKey<'wm> = &'wm str;

#[derive(Debug, Clone, PartialEq, Eq)]
pub struct WasmiumSignature<'ws> {
    message: &'ws [u8],
    signature: Signature,
    public_key: PublicKey,
}

impl<'ws> WasmiumSignature<'ws> {
    pub fn add_message(&mut self, message: &'ws [u8]) -> &mut WasmiumSignature<'ws> {
        self.message = message;

        self
    }

    pub fn add_signature(
        &mut self,
        signature_array: Ed25519SignatureArray,
    ) -> Result<&mut WasmiumSignature<'ws>, WasmiumOutcome> {
        match Utilities::array_to_ed25519_signature(signature_array) {
            Ok(signature) => {
                self.signature = signature;
                Ok(self)
            }
            Err(error_outcome) => return Err(error_outcome),
        }
    }

    pub fn add_public_key(
        &mut self,
        public_key_array: Ed25519PublicKeyArray,
    ) -> Result<&mut WasmiumSignature<'ws>, WasmiumOutcome> {
        match Utilities::array_to_ed25519_public_key(public_key_array) {
            Ok(public_key) => {
                self.public_key = public_key;
                Ok(self)
            }
            Err(error_outcome) => return Err(error_outcome),
        }
    }

    pub fn verify_signature(&self) -> WasmiumOutcome {
        match self.public_key.verify(&self.message, &self.signature) {
            Ok(_) => WasmiumOutcome::ValidSignature,
            Err(_) => WasmiumOutcome::InvalidSignature,
        }
    }
}

impl<'ws> Default for WasmiumSignature<'ws> {
    fn default() -> Self {
        WasmiumSignature {
            message: b"WASMIUM_KEEPER",
            signature: Signature::from_bytes(&[0_u8; 64]).unwrap(), //Never fails
            public_key: PublicKey::from_bytes(&[0_u8; 32]).unwrap(), //Never fails
        }
    }
}

#[derive(Debug, Clone, Copy, PartialEq, Eq, PartialOrd, Ord)]
pub enum WasmiumOutcome {
    ValidSignature,
    InvalidSignature,
    InvalidSignatureBytes,
    InvalidPublicKeyBytes,
    InvalidKeypairBytes,
}

#[derive(Debug)]
pub struct Utilities;

impl Utilities {
    pub fn base58_to_public_key_array(value: Base58PublicKey) -> WasmiumResult<[u8; 32]> {
        let decoded_value = match bs58::decode(&value).into_vec() {
            Ok(decoded) => decoded,
            Err(_) => return Err(WasmiumError::InvalidBase58ForPublickKey),
        };

        let converted_value: [u8; 32] = match decoded_value.try_into() {
            Ok(public) => public,
            Err(_) => return Err(WasmiumError::ErrorConvertingToU832),
        };

        Ok(converted_value)
    }

    pub fn array_to_ed25519_keypair(keypair_array: [u8; 64]) -> Result<Keypair, WasmiumOutcome> {
        match Keypair::from_bytes(&keypair_array) {
            Ok(keypair) => Ok(keypair),
            Err(_) => return Err(WasmiumOutcome::InvalidKeypairBytes),
        }
    }

    pub fn array_to_ed25519_public_key(
        public_key_array: [u8; 32],
    ) -> Result<PublicKey, WasmiumOutcome> {
        match PublicKey::from_bytes(&public_key_array) {
            Ok(public_key) => Ok(public_key),
            Err(_) => return Err(WasmiumOutcome::InvalidPublicKeyBytes),
        }
    }

    pub fn array_to_ed25519_signature(
        signature_array: [u8; 64],
    ) -> Result<Signature, WasmiumOutcome> {
        match Signature::from_bytes(&signature_array) {
            Ok(signature) => Ok(signature),
            Err(_) => Err(WasmiumOutcome::InvalidSignatureBytes),
        }
    }
}

#[cfg(test)]
mod public_key_tests {

    const KEYPAIR: [u8; 64] = [
        14, 206, 93, 54, 79, 221, 151, 197, 84, 158, 215, 171, 115, 97, 58, 53, 87, 127, 189, 113,
        11, 146, 146, 121, 253, 96, 36, 197, 212, 224, 116, 30, 246, 19, 75, 34, 193, 84, 183, 255,
        190, 28, 118, 154, 2, 111, 203, 103, 119, 178, 122, 11, 3, 96, 156, 47, 142, 207, 83, 173,
        129, 103, 108, 92,
    ];

    #[test]
    fn is_signer() {
        use crate::{Utilities, WasmiumOutcome, WasmiumSignature};

        let message = b"Foo Bar";
        let keypair = Utilities::array_to_ed25519_keypair(KEYPAIR).unwrap();

        use ed25519_dalek::Signer;
        let signature = keypair.sign(message);
        let public_key = keypair.public;

        let mut sig = WasmiumSignature::default();
        sig.add_message(message);
        sig.add_public_key(*public_key.as_bytes()).unwrap();
        sig.add_signature(signature.to_bytes()).unwrap();

        assert_eq!(WasmiumOutcome::ValidSignature, sig.verify_signature());
    }
}
