use crate::sys::module::Module;
use crate::sys::store::Store;
use crate::sys::{HostEnvInitError, LinkError, RuntimeError};
use crate::{ExportError, NativeFunc, WasmTypeList};
use std::fmt;
use std::sync::{Arc, Mutex};
use thiserror::Error;
use wasmer_types::InstanceConfig;
use wasmer_vm::{InstanceHandle, Resolver, VMContext};

use super::exports::ExportableWithGenerics;

/// A WebAssembly Instance is a stateful, executable
/// instance of a WebAssembly [`Module`].
///
/// Instance objects contain all the exported WebAssembly
/// functions, memories, tables and globals that allow
/// interacting with WebAssembly.
///
/// Spec: <https://webassembly.github.io/spec/core/exec/runtime.html#module-instances>
#[derive(Clone)]
pub struct Instance {
    handle: Arc<Mutex<InstanceHandle>>,
    module: Module,
}

#[cfg(test)]
mod send_test {
    use super::*;

    fn is_send<T: Send>() -> bool {
        true
    }

    #[test]
    fn instance_is_send() {
        assert!(is_send::<Instance>());
    }
}

/// An error while instantiating a module.
///
/// This is not a common WebAssembly error, however
/// we need to differentiate from a `LinkError` (an error
/// that happens while linking, on instantiation), a
/// Trap that occurs when calling the WebAssembly module
/// start function, and an error when initializing the user's
/// host environments.
#[derive(Error, Debug)]
pub enum InstantiationError {
    /// A linking ocurred during instantiation.
    #[error(transparent)]
    Link(LinkError),

    /// Could not instantiate the artifact.
    #[error("could not instantiate the artifact: {0}")]
    Instantiation(Box<dyn std::error::Error + Send + Sync>),

    /// A runtime error occured while invoking the start function
    #[error("could not invoke the start function: {0}")]
    Start(RuntimeError),

    /// The module was compiled with a CPU feature that is not available on
    /// the current host.
    #[error("missing requires CPU features: {0:?}")]
    CpuFeature(String),

    /// Error occurred when initializing the host environment.
    #[error(transparent)]
    HostEnvInitialization(HostEnvInitError),
}

impl From<wasmer_engine::InstantiationError> for InstantiationError {
    fn from(other: wasmer_engine::InstantiationError) -> Self {
        match other {
            wasmer_engine::InstantiationError::Link(e) => Self::Link(e),
            wasmer_engine::InstantiationError::Start(e) => Self::Start(e),
            wasmer_engine::InstantiationError::CpuFeature(e) => Self::CpuFeature(e),
            wasmer_engine::InstantiationError::CreateInstance(e) => {
                Self::Instantiation(Box::new(e).into())
            }
        }
    }
}

impl From<HostEnvInitError> for InstantiationError {
    fn from(other: HostEnvInitError) -> Self {
        Self::HostEnvInitialization(other)
    }
}

impl Instance {
    /// Creates a new `Instance` from a WebAssembly [`Module`] and a
    /// set of imports resolved by the [`Resolver`].
    ///
    /// The resolver can be anything that implements the [`Resolver`] trait,
    /// so you can plug custom resolution for the imports, if you wish not
    /// to use [`ImportObject`].
    ///
    /// The [`ImportObject`] is the easiest way to provide imports to the instance.
    ///
    /// [`ImportObject`]: crate::ImportObject
    ///
    /// ```
    /// # use wasmer::{imports, Store, Module, Global, Value, Instance};
    /// # fn main() -> anyhow::Result<()> {
    /// let store = Store::default();
    /// let module = Module::new(&store, "(module)")?;
    /// let imports = imports!{
    ///   "host" => {
    ///     "var" => Global::new(&store, Value::I32(2))
    ///   }
    /// };
    /// let instance = Instance::new(&module, &imports)?;
    /// # Ok(())
    /// # }
    /// ```
    ///
    /// ## Errors
    ///
    /// The function can return [`InstantiationError`]s.
    ///
    /// Those are, as defined by the spec:
    ///  * Link errors that happen when plugging the imports into the instance
    ///  * Runtime errors that happen when running the module `start` function.
    pub fn new(module: &Module, resolver: &dyn Resolver) -> Result<Self, InstantiationError> {
        Instance::new_with_config(module, InstanceConfig::default(), resolver)
    }

    /// New instance with config.
    pub fn new_with_config(
        module: &Module,
        config: InstanceConfig,
        resolver: &dyn Resolver,
    ) -> Result<Self, InstantiationError> {
        unsafe {
            if (*config.gas_counter).opcode_cost > i32::MAX as u64 {
                // Fast gas counter logic assumes that individual opcode cost is not too big.
                return Err(InstantiationError::HostEnvInitialization(
                    HostEnvInitError::IncorrectGasMeteringConfig,
                ));
            }
        }
        let handle = module.instantiate(resolver, config)?;
        let instance = Self {
            handle: Arc::new(Mutex::new(handle)),
            module: module.clone(),
        };

        // # Safety
        // `initialize_host_envs` should be called after instantiation but before
        // returning an `Instance` to the user. We set up the host environments
        // via `WasmerEnv::init_with_instance`.
        //
        // This usage is correct because we pass a valid pointer to `instance` and the
        // correct error type returned by `WasmerEnv::init_with_instance` as a generic
        // parameter.
        unsafe {
            wasmer_vm::initialize_host_envs::<HostEnvInitError>(
                &*instance.handle,
                &instance as *const _ as *const _,
            )?;
        }

        Ok(instance)
    }

    /// Gets the [`Module`] associated with this instance.
    pub fn module(&self) -> &Module {
        &self.module
    }

    /// Returns the [`Store`] where the `Instance` belongs.
    pub fn store(&self) -> &Store {
        self.module.store()
    }

    /// Lookup an exported entity by its name.
    pub fn lookup(&self, field: &str) -> Option<crate::Export> {
        let vmextern = self.handle.lock().unwrap().lookup(field)?;
        Some(vmextern.into())
    }

    /// Lookup an exported function by its name.
    pub fn lookup_function(&self, field: &str) -> Option<crate::Function> {
        if let crate::Export::Function(f) = self.lookup(field)? {
            Some(crate::Function::from_vm_export(self.store(), f))
        } else {
            None
        }
    }

    /// Get an export as a `NativeFunc`.
    pub fn get_native_function<Args, Rets>(
        &self,
        name: &str,
    ) -> Result<NativeFunc<Args, Rets>, ExportError>
    where
        Args: WasmTypeList,
        Rets: WasmTypeList,
    {
        match self.lookup(name) {
            Some(crate::Export::Function(f)) => crate::Function::from_vm_export(self.store(), f)
                .native()
                .map_err(|_| ExportError::IncompatibleType),
            Some(_) => Err(ExportError::IncompatibleType),
            None => Err(ExportError::Missing("not found".into())),
        }
    }

    /// Hack to get this working with nativefunc too
    pub fn get_with_generics<'a, T, Args, Rets>(&'a self, name: &str) -> Result<T, ExportError>
    where
        Args: WasmTypeList,
        Rets: WasmTypeList,
        T: ExportableWithGenerics<'a, Args, Rets>,
    {
        let export = self
            .lookup(name)
            .ok_or_else(|| ExportError::Missing(name.to_string()))?;
        let ext = crate::Extern::from_vm_export(self.store(), export);
        T::get_self_from_extern_with_generics(ext)
    }

    /// Like `get_with_generics` but with a WeakReference to the `InstanceRef` internally.
    /// This is useful for passing data into `WasmerEnv`, for example.
    pub fn get_with_generics_weak<'a, T, Args, Rets>(&'a self, name: &str) -> Result<T, ExportError>
    where
        Args: WasmTypeList,
        Rets: WasmTypeList,
        T: ExportableWithGenerics<'a, Args, Rets>,
    {
        let mut out: T = self.get_with_generics(name)?;
        out.into_weak_instance_ref();
        Ok(out)
    }

    #[doc(hidden)]
    pub fn vmctx_ptr(&self) -> *mut VMContext {
        self.handle.lock().unwrap().vmctx_ptr()
    }
}

impl fmt::Debug for Instance {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        f.debug_struct("Instance").finish()
    }
}
