// Copyright Claudio Mattera 2022.
//
// Distributed under the MIT License or the Apache 2.0 License at your option.
// See the accompanying files License-MIT.txt and License-Apache-2.0.txt, or
// online at
// https://opensource.org/licenses/MIT
// https://opensource.org/licenses/Apache-2.0

use wasm4fun_core::{
    TONE_MODE1, TONE_MODE2, TONE_MODE3, TONE_MODE4, TONE_NOISE, TONE_PULSE1, TONE_PULSE2,
    TONE_TRIANGLE,
};

#[cfg(not(feature = "nosound"))]
use wasm4fun_core::tone;

/// A wave form
#[derive(Clone, Copy)]
pub enum WaveForm {
    /// First pulse wave (square wave). The classic chiptune sound.
    Pulse1,

    /// Second pulse wave (square wave). The classic chiptune sound.
    Pulse2,

    /// Triangle wave channel. A softer sound, great for bass.
    Triangle,

    /// Noise channel. A harsh sound, for percussion and effects.
    Noise,
}

impl WaveForm {
    fn to_flag(self) -> u32 {
        match self {
            Self::Pulse1 => TONE_PULSE1,
            Self::Pulse2 => TONE_PULSE2,
            Self::Triangle => TONE_TRIANGLE,
            Self::Noise => TONE_NOISE,
        }
    }
}

/// A duty cycle
#[derive(Clone, Copy)]
pub enum DutyCycle {
    /// 12.5% duty cycle
    OneEight,

    /// 25% duty cycle
    OneQuarter,

    /// 50% duty cycle
    Half,

    /// 75% duty cycle
    ThreeQuarters,
}

impl DutyCycle {
    fn to_flag(self) -> u32 {
        match self {
            Self::OneEight => TONE_MODE1,
            Self::OneQuarter => TONE_MODE2,
            Self::Half => TONE_MODE3,
            Self::ThreeQuarters => TONE_MODE4,
        }
    }
}

/// A tone
///
/// Tones are constructed with a builder pattern, where their frequencies,
/// duration, volume, duty cycle and wave form can be specified.
///
/// ```no_run
/// use wasm4fun_sound::{DutyCycle, Tone, WaveForm};
///
/// let tone = Tone::new()
///     .with_second_frequency(523)
///     .with_first_frequency(262)
///     .with_wave_form(WaveForm::Triangle)
///     .with_duty_cycle(DutyCycle::Half)
///     .with_sustain(60)
///     .with_release(30)
///     .with_decay(15)
///     .with_attack(45);
///
/// tone.play();
/// ```
#[derive(Clone)]
pub struct Tone {
    frequency: u32,
    duration: u32,
    volume: u32,
    flags: u32,
}

impl Tone {
    /// Create a tone
    pub fn new() -> Self {
        Self {
            frequency: 0,
            duration: 0,
            volume: 0,
            flags: DutyCycle::OneQuarter.to_flag() & WaveForm::Pulse1.to_flag(),
        }
    }

    /// Set tone's first frequency
    pub fn with_first_frequency(mut self, first_frequency: u16) -> Self {
        let first_frequency: u32 = first_frequency as u32;
        let second_frequency: u32 = self.frequency & 0xffff0000;
        self.frequency = first_frequency | second_frequency;
        self
    }

    /// Set tone's second frequency
    pub fn with_second_frequency(mut self, second_frequency: u16) -> Self {
        let first_frequency: u32 = self.frequency & 0x0000ffff;
        let second_frequency: u32 = (second_frequency as u32) << 16;
        self.frequency = first_frequency | second_frequency;
        self
    }

    /// Set tone's sustain duration
    ///
    /// "Sustain" is the time to hold the tone steady at volume.
    pub fn with_sustain(mut self, sustain: u8) -> Self {
        let sustain: u32 = sustain as u32;
        let attack: u32 = self.duration & 0xff000000;
        let release: u32 = self.duration & 0x0000ff00;
        let decay: u32 = self.duration & 0x00ff0000;
        self.duration = sustain | attack | release | decay;
        self
    }

    /// Set tone's attack duration
    ///
    /// "Attack" is the time it takes to initially ramp up from 0 volume to
    /// 100% volume.
    pub fn with_attack(mut self, attack: u8) -> Self {
        let sustain: u32 = self.duration & 0x00000ff;
        let attack: u32 = (attack as u32) << 24;
        let release: u32 = self.duration & 0x0000ff00;
        let decay: u32 = self.duration & 0x00ff0000;
        self.duration = sustain | attack | release | decay;
        self
    }

    /// Set tone's release duration
    ///
    /// "Release" is the time to ramp back down to 0 volume.
    pub fn with_release(mut self, release: u8) -> Self {
        let sustain: u32 = self.duration & 0x00000ff;
        let attack: u32 = self.duration & 0xff000000;
        let release: u32 = (release as u32) << 8;
        let decay: u32 = self.duration & 0x00ff0000;
        self.duration = sustain | attack | release | decay;
        self
    }

    /// Set tone's decay duration
    ///
    /// "Decay" is the time taken to ramp down from 100% volume to the tone
    /// volume parameter.
    pub fn with_decay(mut self, decay: u8) -> Self {
        let sustain: u32 = self.duration & 0x00000ff;
        let attack: u32 = self.duration & 0xff000000;
        let release: u32 = self.duration & 0x0000ff00;
        let decay: u32 = (decay as u32) << 16;
        self.duration = sustain | attack | release | decay;
        self
    }

    /// Set tone's volume
    pub fn with_volume(mut self, volume: u32) -> Self {
        self.volume = volume;
        self
    }

    /// Set tone's wave form
    pub fn with_wave_form(mut self, wave_form: WaveForm) -> Self {
        let wave_form = wave_form.to_flag();
        let duty_cycle = self.flags & 0b1100;
        self.flags = wave_form | duty_cycle;
        self
    }

    /// Set tone's duty cycle
    pub fn with_duty_cycle(mut self, duty_cycle: DutyCycle) -> Self {
        let duty_cycle = duty_cycle.to_flag();
        let wave_form = self.flags & 0b0011;
        self.flags = wave_form | duty_cycle;
        self
    }

    /// Play the tone
    ///
    /// If feature `nosound` is enabled, this function does nothing.
    #[inline]
    pub fn play(&self) {
        #[cfg(not(feature = "nosound"))]
        tone(self.frequency, self.duration, self.volume, self.flags)
    }
}

impl Default for Tone {
    fn default() -> Self {
        Self::new()
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_tone_new() {
        let tone = Tone::new();
        assert_eq!(tone.frequency, 0);
        assert_eq!(tone.duration, 0);
        assert_eq!(tone.volume, 0);
        assert_eq!(tone.flags, TONE_PULSE1 | TONE_MODE1);
    }

    #[test]
    fn test_tone_with_frequency() {
        let tone = Tone::new()
            .with_first_frequency(262)
            .with_second_frequency(523);
        assert_eq!(tone.frequency, 262 | (523 << 16));
        assert_eq!(tone.duration, 0);
        assert_eq!(tone.volume, 0);
        assert_eq!(tone.flags, TONE_PULSE1 | TONE_MODE1);

        // Check again in opposite order
        let tone = Tone::new()
            .with_second_frequency(523)
            .with_first_frequency(262);
        assert_eq!(tone.frequency, 262 | (523 << 16));
        assert_eq!(tone.duration, 0);
        assert_eq!(tone.volume, 0);
        assert_eq!(tone.flags, TONE_PULSE1 | TONE_MODE1);
    }

    #[test]
    fn test_tone_with_duration() {
        let tone = Tone::new()
            .with_sustain(60)
            .with_release(30)
            .with_decay(15)
            .with_attack(45);
        assert_eq!(tone.frequency, 0);
        assert_eq!(tone.duration, 60 | (30 << 8) | (15 << 16) | (45 << 24));
        assert_eq!(tone.volume, 0);
        assert_eq!(tone.flags, TONE_PULSE1 | TONE_MODE1);

        // Check again in opposite order
        let tone = Tone::new()
            .with_attack(45)
            .with_decay(15)
            .with_release(30)
            .with_sustain(60);
        assert_eq!(tone.frequency, 0);
        assert_eq!(tone.duration, 60 | (30 << 8) | (15 << 16) | (45 << 24));
        assert_eq!(tone.volume, 0);
        assert_eq!(tone.flags, TONE_PULSE1 | TONE_MODE1);
    }

    #[test]
    fn test_tone_with_flags() {
        let tone = Tone::new()
            .with_wave_form(WaveForm::Triangle)
            .with_duty_cycle(DutyCycle::Half);
        assert_eq!(tone.frequency, 0);
        assert_eq!(tone.duration, 0);
        assert_eq!(tone.volume, 0);
        assert_eq!(tone.flags, TONE_TRIANGLE | TONE_MODE3);

        // Check again in opposite order
        let tone = Tone::new()
            .with_duty_cycle(DutyCycle::ThreeQuarters)
            .with_wave_form(WaveForm::Noise);
        assert_eq!(tone.frequency, 0);
        assert_eq!(tone.duration, 0);
        assert_eq!(tone.volume, 0);
        assert_eq!(tone.flags, TONE_NOISE | TONE_MODE4);
    }
}
