// Copyright Bruno Garcia 2022.
//
// Distributed under the ISC License.
// See the accompanying files License-ISC.txt, or
// online at
// https://opensource.org/licenses/ISC
//
// WASM-4: https://wasm4.org/docs

//! Primitives and constants from WASM-4 fantasy console
//!
//! [WASM-4](https://wasm4.org/) ships a `w4` application that can generate a
//! preconfigured Rust crate to build a game.
//! That crate contains a `wasm4` module that exports all primitives and
//! constants exposed by the console.
//!
//! This crate replicates that module, documenting each public symbol and
//! adding few Rust attributes where needed, so that `w4` is no longer
//! necessary to build a game.
//!
//! Note that `w4` application is still required to play the game or to
//! generate a self-contained cartridge from the final `.wasm` artefact.

#![no_std]

// ┌───────────────────────────────────────────────────────────────────────────┐
// │                                                                           │
// │ Platform Constants                                                        │
// │                                                                           │
// └───────────────────────────────────────────────────────────────────────────┘

/// Width and height of the screen in pixels
pub const SCREEN_SIZE: u32 = 160;

// ┌───────────────────────────────────────────────────────────────────────────┐
// │                                                                           │
// │ Memory Addresses                                                          │
// │                                                                           │
// └───────────────────────────────────────────────────────────────────────────┘

/// System palette memory region
pub static mut PALETTE: *mut [u32; 4] = 0x04 as *mut [u32; 4];

/// Address of current drawing colors
pub const DRAW_COLORS: *mut u16 = 0x14 as *mut u16;

/// Address of first gamepad
pub const GAMEPAD1: *const u8 = 0x16 as *const u8;

/// Address of second gamepad
pub const GAMEPAD2: *const u8 = 0x17 as *const u8;

/// Address of third gamepad
pub const GAMEPAD3: *const u8 = 0x18 as *const u8;

/// Address of fourth gamepad
pub const GAMEPAD4: *const u8 = 0x19 as *const u8;

/// Address of mouse x coordinate
pub const MOUSE_X: *const i16 = 0x1a as *const i16;

/// Address of mouse y coordinate
pub const MOUSE_Y: *const i16 = 0x1c as *const i16;

/// Address of mouse buttons
pub const MOUSE_BUTTONS: *const u8 = 0x1e as *const u8;

/// System flags
pub const SYSTEM_FLAGS: *mut u8 = 0x1f as *mut u8;

/// Framebuffer memory region
pub static mut FRAMEBUFFER: *mut [u8; 6400] = 0xa0 as *mut [u8; 6400];

/// Gamepad X button state bitmask
///
/// Non-zero if button is pressed.
pub const BUTTON_1: u8 = 1;

/// Gamepad Z button state bitmask
///
/// Non-zero if button is pressed.
pub const BUTTON_2: u8 = 2;

/// Gamepad left button state bitmask
///
/// Non-zero if button is pressed.
pub const BUTTON_LEFT: u8 = 16;

/// Gamepad right button state bitmask
///
/// Non-zero if button is pressed.
pub const BUTTON_RIGHT: u8 = 32;

/// Gamepad up button state bitmask
///
/// Non-zero if button is pressed.
pub const BUTTON_UP: u8 = 64;

/// Gamepad down button state bitmask
///
/// Non-zero if button is pressed.
pub const BUTTON_DOWN: u8 = 128;

/// Mouse left button state bitmask
///
/// Non-zero if button is pressed.
pub const MOUSE_LEFT: u8 = 1;

/// Mouse right button state bitmask
///
/// Non-zero if button is pressed.
pub const MOUSE_RIGHT: u8 = 2;

/// Mouse middle button state bitmask
///
/// Non-zero if button is pressed.
pub const MOUSE_MIDDLE: u8 = 4;

/// Flag for preserving framebuffer content between frames
///
/// If set, the framebuffer is not cleared between frames.
pub const SYSTEM_PRESERVE_FRAMEBUFFER: u8 = 1;

/// Flag for hiding gamepad overlay
///
/// If set, the gamepad overlay is not shown on mobile.
pub const SYSTEM_HIDE_GAMEPAD_OVERLAY: u8 = 2;

// ┌───────────────────────────────────────────────────────────────────────────┐
// │                                                                           │
// │ Drawing Functions                                                         │
// │                                                                           │
// └───────────────────────────────────────────────────────────────────────────┘

/// Copy pixels to the framebuffer
pub fn blit(sprite: &[u8], x: i32, y: i32, width: u32, height: u32, flags: u32) {
    unsafe { extern_blit(sprite.as_ptr(), x, y, width, height, flags) }
}
extern "C" {
    #[link_name = "blit"]
    fn extern_blit(sprite: *const u8, x: i32, y: i32, width: u32, height: u32, flags: u32);
}

/// Copy a subregion within a larger sprite atlas to the framebuffer
#[allow(clippy::too_many_arguments)]
pub fn blit_sub(
    sprite: &[u8],
    x: i32,
    y: i32,
    width: u32,
    height: u32,
    src_x: u32,
    src_y: u32,
    stride: u32,
    flags: u32,
) {
    unsafe {
        extern_blit_sub(
            sprite.as_ptr(),
            x,
            y,
            width,
            height,
            src_x,
            src_y,
            stride,
            flags,
        )
    }
}
extern "C" {
    #[link_name = "blitSub"]
    fn extern_blit_sub(
        sprite: *const u8,
        x: i32,
        y: i32,
        width: u32,
        height: u32,
        src_x: u32,
        src_y: u32,
        stride: u32,
        flags: u32,
    );
}

/// Flag for 2-bits-per-pixel sprites
///
/// This value must be passed to functions [`blit`] and [`blit_sub`] when
/// blitting 2-bits-per-pixel sprites.
pub const BLIT_2BPP: u32 = 1;

/// Flag for 1-bit-per-pixel sprites
///
/// This value must be passed to functions [`blit`] and [`blit_sub`] when
/// blitting 1-bit-per-pixel sprites.
pub const BLIT_1BPP: u32 = 0;

/// Flag for horizontally flipping sprites
///
/// This value can be passed to functions [`blit`] and [`blit_sub`].
pub const BLIT_FLIP_X: u32 = 2;

/// Flag for vertically flipping sprites
///
/// This value can be passed to functions [`blit`] and [`blit_sub`].
pub const BLIT_FLIP_Y: u32 = 4;

/// Flag for counter-clockwise rotating sprites
///
/// This value can be passed to functions [`blit`] and [`blit_sub`].
pub const BLIT_ROTATE: u32 = 8;

/// Draw a line between two points
pub fn line(x1: i32, y1: i32, x2: i32, y2: i32) {
    unsafe { extern_line(x1, y1, x2, y2) }
}
extern "C" {
    #[link_name = "line"]
    fn extern_line(x1: i32, y1: i32, x2: i32, y2: i32);
}

/// Draw an oval (or circle)
pub fn oval(x: i32, y: i32, width: u32, height: u32) {
    unsafe { extern_oval(x, y, width, height) }
}
extern "C" {
    #[link_name = "oval"]
    fn extern_oval(x: i32, y: i32, width: u32, height: u32);
}

/// Draw a rectangle
pub fn rect(x: i32, y: i32, width: u32, height: u32) {
    unsafe { extern_rect(x, y, width, height) }
}
extern "C" {
    #[link_name = "rect"]
    fn extern_rect(x: i32, y: i32, width: u32, height: u32);
}

/// Draw text using the built-in system font
pub fn text<T: AsRef<str>>(text: T, x: i32, y: i32) {
    let text_ref = text.as_ref();
    unsafe { extern_text(text_ref.as_ptr(), text_ref.len(), x, y) }
}
extern "C" {
    #[link_name = "textUtf8"]
    fn extern_text(text: *const u8, length: usize, x: i32, y: i32);
}

/// Draw a vertical line
pub fn vline(x: i32, y: i32, len: u32) {
    unsafe {
        extern_vline(x, y, len);
    }
}

extern "C" {
    #[link_name = "vline"]
    fn extern_vline(x: i32, y: i32, len: u32);
}

/// Draw a horizontal line
pub fn hline(x: i32, y: i32, len: u32) {
    unsafe {
        extern_hline(x, y, len);
    }
}

extern "C" {
    #[link_name = "hline"]
    fn extern_hline(x: i32, y: i32, len: u32);
}

// ┌───────────────────────────────────────────────────────────────────────────┐
// │                                                                           │
// │ Sound Functions                                                           │
// │                                                                           │
// └───────────────────────────────────────────────────────────────────────────┘

/// Play a sound tone
pub fn tone(frequency: u32, duration: u32, volume: u32, flags: u32) {
    unsafe { extern_tone(frequency, duration, volume, flags) }
}
extern "C" {
    #[link_name = "tone"]
    fn extern_tone(frequency: u32, duration: u32, volume: u32, flags: u32);
}

/// Select first pulse wave form
///
/// This value can be passed to function [`tone`].
pub const TONE_PULSE1: u32 = 0;

/// Select second pulse wave form
///
/// This value can be passed to function [`tone`].
pub const TONE_PULSE2: u32 = 1;

/// Select triangle wave form
///
/// This value can be passed to function [`tone`].
pub const TONE_TRIANGLE: u32 = 2;

/// Select noise wave form
///
/// This value can be passed to function [`tone`].
pub const TONE_NOISE: u32 = 3;

/// Select 12.5% duty cycle
///
/// This value can be passed to function [`tone`].
pub const TONE_MODE1: u32 = 0;

/// Select 25% duty cycle
///
/// This value can be passed to function [`tone`].
pub const TONE_MODE2: u32 = 4;

/// Select 50% duty cycle
///
/// This value can be passed to function [`tone`].
pub const TONE_MODE3: u32 = 8;

/// Select 75% duty cycle
///
/// This value can be passed to function [`tone`].
pub const TONE_MODE4: u32 = 12;

// ┌───────────────────────────────────────────────────────────────────────────┐
// │                                                                           │
// │ Storage Functions                                                         │
// │                                                                           │
// └───────────────────────────────────────────────────────────────────────────┘

extern "C" {
    /// Read up to `size` bytes from persistent storage into the pointer `dest`
    pub fn diskr(dest: *mut u8, size: u32) -> u32;

    /// Write up to `size` bytes from the pointer `src` into persistent storage
    pub fn diskw(src: *const u8, size: u32) -> u32;
}

// ┌───────────────────────────────────────────────────────────────────────────┐
// │                                                                           │
// │ Other Functions                                                           │
// │                                                                           │
// └───────────────────────────────────────────────────────────────────────────┘

/// Print a message to the debug console
pub fn trace<T: AsRef<str>>(text: T) {
    let text_ref = text.as_ref();
    unsafe { extern_trace(text_ref.as_ptr(), text_ref.len()) }
}
extern "C" {
    #[link_name = "traceUtf8"]
    fn extern_trace(trace: *const u8, length: usize);
}
