//! Provides wasm3 wrappers for abstract [`api`] types
//! to support runtime testing in the wasm3 environment

use core::slice;
use std::ptr;
use std::ffi::{CString, c_void};
use log::{debug};
use wasm_embedded_lib::*;

use crate::mock::MockCtx;
use crate::api::{I2c};

mod i2c;


/// Run a mock application using the wasm3 runtime
pub fn run(ctx: &mut MockCtx, bin: &str) -> anyhow::Result<()> {

    // Load WASI binary
    debug!("Loading binary: {}", bin);
    let b = std::fs::read(bin)?;

    // Setup task to execute
    let name = CString::new(bin)?;
    let task = wasme_task_t{
        name: name.as_ptr(),
        data: b.as_ptr(),
        data_len: b.len() as u32,
    };
    
    // Initialise WASME context
    let mut wasme_ctx = unsafe { WASME_init(&task, 10 * 1024) };
    if wasme_ctx.is_null() {
        return Err(anyhow::anyhow!("Failed to create WASME ctx"));
    }

    // Bind mock drivers
    let i2c_ctx: *mut c_void = ctx as *mut _ as *mut c_void;
    debug!("I2C_CTX: {:p}", ctx);
    let res = unsafe { WASME_bind_i2c(wasme_ctx, &DRV_I2C, i2c_ctx) };
    if res < 0 {
        unsafe { WASME_deinit(&mut wasme_ctx) };
        return Err(anyhow::anyhow!("Error binding I2C driver: {}", res));
    }

    // Run WASME instance
    let entry = CString::new("_start")?;
    let res = unsafe { WASME_run(wasme_ctx, entry.as_ptr(), 0, ptr::null_mut()) };
    if res < 0 {
        unsafe { WASME_deinit(&mut wasme_ctx) };
        return Err(anyhow::anyhow!("Error executing WASME instance: {}", res));
    }

    debug!("WASME execution complete!");

    unsafe { WASME_deinit(&mut wasme_ctx) };

    Ok(())
}
