
use structopt::StructOpt;
use log::LevelFilter;

use wasm_embedded_rt::{Mode, Runtime, linux::LinuxCtx, mock::MockCtx};

#[derive(Clone, PartialEq, Debug, StructOpt)]
struct Options {
    /// Operating mode
    #[structopt(long, default_value="dynamic")]
    mode: Mode,

    /// Runtime
    #[structopt(long, default_value="wasmtime")]
    runtime: Runtime,

    /// Configuration file (toml)
    #[structopt(long)]
    config: Option<String>,

    /// WASM binary to execute
    #[structopt()]
    bin: String,

    #[structopt(long = "log-level", default_value = "info")]
    /// Configure app logging levels (warn, info, debug, trace)
    pub log_level: LevelFilter,
}


fn main() -> Result<(), anyhow::Error> {
    // Load options
    let opts = Options::from_args();

    // Setup logging
    let log_config = simplelog::ConfigBuilder::new()
        .add_filter_ignore_str("cranelift_codegen")
        .add_filter_ignore_str("regalloc")
        .add_filter_ignore_str("cranelift_wasm")
        .build();
    let _ = simplelog::SimpleLogger::init(opts.log_level, log_config);

    match (&opts.runtime, &opts.mode) {
        #[cfg(feature="rt-wasmtime")]
        (Runtime::Wasmtime, Mode::Mock) => {
            // Load mock configuration
            let cfg = match &opts.config {
                Some(c) => c,
                None => return Err(anyhow::anyhow!("mock mode requires --config file")),
            };
            let ctx = MockCtx::load(&cfg)?;

            wasm_embedded_rt::run(ctx, MockCtx::bind, &opts.bin)?;
        },
        #[cfg(feature="rt-wasmtime")]
        (Runtime::Wasmtime, Mode::Linux) => {
            // Load linux configuration
            // TODO: config files?
            let ctx = LinuxCtx::new();

            wasm_embedded_rt::run(ctx, LinuxCtx::bind, &opts.bin)?;
        },
        #[cfg(not(feature="rt-wasmtime"))]
        (Runtime::Wasmtime, _) => {
            return Err(anyhow::anyhow!("Runtime was not built with wasmtime enabled"));
        },
        #[cfg(feature="rt-wasm3")]
        (Runtime::Wasm3, Mode::Mock) => {
            // Load mock configuration
            let cfg = match &opts.config {
                Some(c) => c,
                None => return Err(anyhow::anyhow!("mock mode requires --config file")),
            };
            let ctx = MockCtx::load(&cfg)?;

            wasm_embedded_rt::wasm3::run(&mut ctx, &opts.bin)?
        },
        #[cfg(feature="rt-wasm3")]
        (Runtime::Wasm3, Mode::Linux) => {
            todo!()
        },
        #[cfg(not(feature="rt-wasm3"))]
        (Runtime::Wasm3, _) => {
            return Err(anyhow::anyhow!("Runtime was not built with wasm3 enabled"))
        },
    }

    Ok(())
}
