#[cfg(feature = "rt-wasmtime")]
use wasmtime_wasi::{WasiCtx, WasiCtxBuilder};

use crate::api::{self, types, Error};

pub mod i2c;
pub mod spi;


/// Linux wasm-embedded context
pub struct LinuxCtx {
    spi: spi::LinuxSpi,
    i2c: i2c::LinuxI2c,
    gpio: crate::gpio::GpioCtx,
    dev: crate::dev::DeviceCtx,
    #[cfg(feature = "rt-wasmtime")]
    wasi: WasiCtx,
}


impl LinuxCtx {
    pub fn new() -> Self {
        #[cfg(feature = "rt-wasmtime")]
        let wasi = WasiCtxBuilder::new()
            .inherit_stdio()
            .inherit_args().unwrap()
            .build();

        Self{
            spi: spi::LinuxSpi{},
            i2c: i2c::LinuxI2c{},
            gpio: crate::gpio::GpioCtx{},
            dev: crate::dev::DeviceCtx{},
            #[cfg(feature = "rt-wasmtime")]
            wasi,
        }
    }

    #[cfg(feature = "rt-wasmtime")]
    pub fn bind(linker: &mut wasmtime::Linker<Self>) -> anyhow::Result<()> {
        wasmtime_wasi::add_to_linker(linker, |ctx: &mut Self| &mut ctx.wasi )?;

        // Bind embedded APIs
        api::spi::add_to_linker(linker, |m: &mut Self| &mut m.spi )?;
        api::i2c::add_to_linker(linker, |m: &mut Self| &mut m.i2c )?;
        api::gpio::add_to_linker(linker, |m| &mut m.gpio )?;
        
        Ok(())
    }
}
