
use embedded_hal::blocking::spi::Operation;

use super::*;

/// SPI context abstraction.
/// This hides wiggle-generated implementation details to simplify implementing I2C contexts.
/// Hopefully one day generation is improved so we don't _need_ this any more
pub trait Spi {
    fn init(&mut self, dev: u32, baud: u32, mosi: i32, miso: i32, sck: i32, cs: i32) -> Result<i32, Error>;

    fn deinit(&mut self, handle: i32) -> Result<(), Error>;

    fn write<'a>(&mut self, handle: i32, data: &[u8]) -> Result<(), Error>;

    fn transfer<'a>(&mut self, handle: i32, data: &mut [u8]) -> Result<(), Error>;

    fn exec<'a>(&mut self, handle: i32, ops: &[Operation<u8>]) -> Result<(), Error>;
}

impl <T: Spi> crate::api::spi::Spi for T {
    fn init(&mut self, dev: u32, baud: u32, mosi: i32, miso: i32, sck: i32, cs: i32) -> Result<i32, Error> {
        println!("Opening SPI device: {} (baud: {} mosi: {} miso: {} sck: {} cs: {})", 
                dev, baud, mosi, miso, sck, cs);
        Spi::init(self, dev, baud, mosi, miso, sck, cs)
    }

    fn deinit(&mut self, handle: i32) -> Result<(), Error> {
        println!("Closing SPI handle: {}", handle);
        Spi::deinit(self, handle)
    }

    fn write<'a>(&mut self, handle: i32, data: &types::Wbytes<'a>) -> Result<(), Error> {
        let d = data.ptr.as_array(data.len);
        let d1 = d.as_slice().unwrap();

        println!("Write SPI {} data: {:02x?}", handle, d1.deref());
        Spi::write(self, handle, d1.deref())
    }

    fn transfer<'a>(&mut self, handle: i32, data: &types::Rbytes<'a>) -> Result<(), Error> {
        let d = data.ptr.as_array(data.len);
        let mut d1 = d.as_slice_mut().unwrap();

        println!("Transfer SPI {} data: {:02x?}", handle, d1.deref());
        Spi::transfer(self, handle, d1.deref_mut())
    }

    fn exec<'a>(&mut self, _handle: i32, ops: &GuestPtr<'a, [types::Op]>) -> Result<(), Error> {
        let _num_ops = ops.len();
        // TODO: idk yet how guest types etc. are going to work here
        todo!()
    }
}
