
use super::*;

/// I2C context abstraction.
/// This hides wiggle-generated implementation details to simplify implementing I2C contexts.
/// Hopefully one day generation is improved so we don't _need_ this any more
pub trait I2c {
    fn init(&mut self, dev: u32, _baud: u32, sda: i32, sck: i32) -> Result<i32, Error>;

    fn deinit(&mut self, handle: i32) -> Result<(), Error>;

    fn write(&mut self, handle: i32, addr: u16, data: &[u8]) -> Result<(), Error>;

    fn read(&mut self, handle: i32, addr: u16, buff: &mut [u8]) -> Result<(), Error>;

    fn write_read(&mut self, handle: i32, addr: u16, data: &[u8], buff: &mut [u8]) -> Result<(), Error>;   
}

impl <T: I2c> crate::api::i2c::I2c for T {

    fn init(&mut self, port: u32, baud: u32, sda: i32, scl: i32) -> Result<i32, Error> {
        println!("Opening I2C port: {} (baud: {} sda: {} scl: {})", port, baud, sda, scl);
        I2c::init(self, port, baud, sda, scl)
    }

    fn deinit(&mut self, handle: i32) -> Result<(), Error> {
        println!("Closing I2C handle: {}", handle);
        I2c::deinit(self, handle)
    }

    /// Write to an I2c device
    fn write(&mut self, handle: i32, addr: u16, data: &types::Rbytes) -> Result<(), Error> {
        let d = data.ptr.as_array(data.len);
        let d1 = d.as_slice_mut().unwrap();

        println!("I2C write handle: {} addr: {} data: {:02x?}", handle, addr, d1.deref());

        I2c::write(self, handle, addr, d1.deref())
    }

    /// Read from an I2c device
    fn read(&mut self, handle: i32, addr: u16, buff: &types::Wbytes) -> Result<(), Error> {
        let b = buff.ptr.as_array(buff.len);
        let mut b1 = b.as_slice_mut().unwrap();

        println!("I2C read handle: {} addr: {}", handle, addr);

        I2c::read(self, handle, addr, b1.deref_mut())
    }

    /// Write to and read from an I2c device on the specified peripheral
    fn write_read(&mut self, handle: i32, addr: u16, data: &types::Rbytes, buff: &types::Wbytes) -> Result<(), Error> {
        let d = data.ptr.as_array(data.len);
        let d1 = d.as_slice().unwrap();

        let b = buff.ptr.as_array(buff.len);
        let mut b1 = b.as_slice_mut().unwrap();

        println!("I2C write_read dev: {} addr: {} write: {:02x?}", handle, addr, d1.deref());

        I2c::write_read(self, handle, addr, d1.deref(), b1.deref_mut())
    }
}
