#ifndef WASME_I2C_H
#define WASME_I2C_H

#include <stdlib.h>
#include <stdint.h>

#ifdef __cplusplus
extern "C"
{
#endif

/// WASME context forward-declaration
typedef struct wasme_ctx_s wasme_ctx_t;

/// Init, takes device index and config and returns handle (or error)
typedef int32_t i2c_init_f(void *ctx, uint32_t dev, uint32_t baud, int32_t sda, int32_t scl);

/// Deinit, takes device handle and deinitialises
typedef int32_t i2c_deinit_f(void *ctx, int32_t handle);

/// Write, takes device handle, writes data_out to address
typedef int32_t i2c_write_f(void *ctx, int32_t handle, uint16_t address, uint8_t *data_out, uint32_t length_out);

/// Read, takes device handle, reads data_in from address
typedef int32_t i2c_read_f(void *ctx, int32_t handle, uint16_t address, uint8_t *data_in, uint32_t length_in);

/// WriteRead, takes device handle, writes data_out then reads data_in from address
typedef int32_t i2c_write_read_f(void *ctx, int32_t handle, uint16_t address,
                                uint8_t *data_out, uint32_t length_out,
                                uint8_t *data_in, uint32_t length_in);

/// WASME I2C driver object
typedef struct
{
    i2c_init_f *init;
    i2c_deinit_f *deinit;
    i2c_write_f *write;
    i2c_read_f *read;
    i2c_write_read_f *write_read;
} wasme_i2c_drv_t;

/// Bind the provided I2C driver to the WASM3 module for use
int32_t WASME_bind_i2c(wasme_ctx_t* ctx, const wasme_i2c_drv_t *drv, void* drv_ctx);

#ifdef __cplusplus
}
#endif

#ifdef __cplusplus

/// WASME C++ I2C driver object
class I2cDriver
{
public:
    virtual int32_t init(uint32_t dev, uint32_t baud, int32_t sda, int32_t scl){ return -1; };
    virtual int32_t deinit(int32_t handle){ return -1; };

    virtual int32_t write(int32_t handle, uint16_t address,
                          uint8_t *data_out, uint32_t length_out){ return -1; };

    virtual int32_t read(int32_t handle, uint16_t address,
                         uint8_t *data_in, uint32_t length_in){ return -1; };

    virtual int32_t write_read(int32_t handle, uint16_t address,
                               uint8_t *data_out, uint32_t length_out,
                               uint8_t *data_in, uint32_t length_in){ return -1; };
};

/// C/C++ wrapper driver
extern const wasme_i2c_drv_t wasme_i2c_drv_cxx;

#endif

#endif
