use core::marker::PhantomData;

use embedded_hal::digital::*;

use crate::Error;
use api::{GpioMode};

mod api {
    #[repr(C)]
    #[derive(Copy, Clone, PartialEq, Debug)]
    pub enum GpioMode {
        Input = 0,
        Output = 1,
    }

    #[repr(C)]
    #[derive(Copy, Clone, PartialEq, Debug)]
    pub enum GpioValue {
        Low = 0,
        High = 1,
    }

    #[link(wasm_import_module = "gpio")]
    extern "C" {
        /// Initialise the provided GPIO pin in input or output mode
        pub fn init(port: u32, pin: u32, mode: u32, handle: &mut i32) -> i32;

        /// Deinitialise the specified GPIO pin
        pub fn deinit(handle: i32) -> i32;

        /// Write to a GPIO pin
        pub fn write(handle: i32, value: u32) -> i32;

        // Read from a GPIO pin
        pub fn read(handle: i32, value: *const u32) -> i32;
    }
}


// GPIO pin instance
#[repr(C)]
#[derive(Clone, PartialEq, Debug)]
pub struct Gpio<MODE> {
    pub(crate) handle: i32,
    pub(crate) mode: PhantomData<MODE>,
}


impl <MODE> Default for Gpio<MODE> {
    fn default() -> Self {
        Self { handle: -1, mode: PhantomData }
    }
}


/// Input marker type
#[derive(Copy, Clone, PartialEq, Debug)]
pub struct Input;

/// Output marker type
#[derive(Copy, Clone, PartialEq, Debug)]
pub struct Output;

impl Gpio<Input> {
    /// Initialise a GPIO input pin
    pub fn input(port: u32, pin: u32) -> Result<Self, Error> {
        let mut handle = 0;
        
        let res = unsafe { api::init(port, pin, GpioMode::Input as u32, &mut handle) };
        if res < 0 {
            return Err(Error::Runtime(res))
        }

        Ok(Self{handle, mode: PhantomData})
    }
}

impl Gpio<Output> {
    /// Initialise a GPIO output pin
    pub fn output(port: u32, pin: u32) -> Result<Self, Error> {
        let mut handle = 0;
        
        let res = unsafe { api::init(port, pin, GpioMode::Output as u32, &mut handle) };
        if res < 0 {
            return Err(Error::Runtime(res))
        }

        Ok(Self{handle, mode: PhantomData})
    }
}

impl <MODE> Gpio<MODE> {
    pub fn deinit(&mut self) {
        unsafe {
            api::deinit(self.handle);
        }
    }
}

impl OutputPin for Gpio<Output> {
    type Error = Error;

    fn try_set_high(&mut self) -> Result<(), Self::Error> {

        let res = unsafe { api::write(self.handle, 1) };

        if res < 0 {
            return Err(Error::Runtime(res))
        }
        
        Ok(())
    }

    fn try_set_low(&mut self) -> Result<(), Self::Error> {

        let res = unsafe { api::write(self.handle, 0) };

        if res < 0 {
            return Err(Error::Runtime(res))
        }
        
        Ok(())
    }
}

impl InputPin for Gpio<Input> {
    type Error = Error;

    fn try_is_high(&self) -> Result<bool, Self::Error> {
        let mut v: u32 = 0;
        let p = &mut v;

        let res = unsafe { api::read(self.handle, p) };

        if res < 0 {
            return Err(Error::Runtime(res))
        }
        
        Ok(v != 0)
    }

    fn try_is_low(&self) -> Result<bool, Self::Error> {
        let v = self.try_is_high()?;

        Ok(v == false)
    }
}

